//===-- Unittests for cospif ----------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "hdr/errno_macros.h"
#include "src/math/cospif.h"
#include "test/UnitTest/FPMatcher.h"
#include "test/src/math/sdcomp26094.h"
#include "utils/MPFRWrapper/MPFRUtils.h"

using LlvmLibcCospifTest = LIBC_NAMESPACE::testing::FPTest<float>;

using LIBC_NAMESPACE::testing::SDCOMP26094_VALUES;

namespace mpfr = LIBC_NAMESPACE::testing::mpfr;

TEST_F(LlvmLibcCospifTest, SpecialNumbers) {
  EXPECT_FP_EQ(aNaN, LIBC_NAMESPACE::cospif(aNaN));
  EXPECT_MATH_ERRNO(0);

  EXPECT_FP_EQ(1.0f, LIBC_NAMESPACE::cospif(0.0f));
  EXPECT_MATH_ERRNO(0);

  EXPECT_FP_EQ(1.0f, LIBC_NAMESPACE::cospif(-0.0f));
  EXPECT_MATH_ERRNO(0);

  EXPECT_FP_EQ(aNaN, LIBC_NAMESPACE::cospif(inf));
  EXPECT_MATH_ERRNO(EDOM);

  EXPECT_FP_EQ(aNaN, LIBC_NAMESPACE::cospif(neg_inf));
  EXPECT_MATH_ERRNO(EDOM);
}

TEST_F(LlvmLibcCospifTest, SpecificBitPatterns) {
  constexpr int N = 38;
  constexpr uint32_t INPUTS[N] = {
      0x3f00'0000U, // x = 0.5
      0x461d'd600U, // x = 10101.5
      0x3f06'0a92U, // x = pi/6
      0x3f3a'dc51U, // x = 0x1.75b8a2p-1f
      0x3f49'0fdbU, // x = pi/4
      0x3f86'0a92U, // x = pi/3
      0x3fa7'832aU, // x = 0x1.4f0654p+0f
      0x3fc9'0fdbU, // x = pi/2
      0x4017'1973U, // x = 0x1.2e32e6p+1f
      0x4049'0fdbU, // x = pi
      0x4096'cbe4U, // x = 0x1.2d97c8p+2f
      0x40c9'0fdbU, // x = 2*pi
      0x433b'7490U, // x = 0x1.76e92p+7f
      0x437c'e5f1U, // x = 0x1.f9cbe2p+7f
      0x4619'9998U, // x = 0x1.33333p+13f
      0x474d'246fU, // x = 0x1.9a48dep+15f
      0x4afd'ece4U, // x = 0x1.fbd9c8p+22f
      0x4c23'32e9U, // x = 0x1.4665d2p+25f
      0x50a3'e87fU, // x = 0x1.47d0fep+34f
      0x5239'47f6U, // x = 0x1.728fecp+37f
      0x53b1'46a6U, // x = 0x1.628d4cp+40f
      0x55ca'fb2aU, // x = 0x1.95f654p+44f
      0x588e'f060U, // x = 0x1.1de0cp+50f
      0x5c07'bcd0U, // x = 0x1.0f79ap+57f
      0x5ebc'fddeU, // x = 0x1.79fbbcp+62f
      0x5fa6'eba7U, // x = 0x1.4dd74ep+64f
      0x61a4'0b40U, // x = 0x1.48168p+68f
      0x6386'134eU, // x = 0x1.0c269cp+72f
      0x6589'8498U, // x = 0x1.13093p+76f
      0x6600'0001U, // x = 0x1.000002p+77f
      0x664e'46e4U, // x = 0x1.9c8dc8p+77f
      0x66b0'14aaU, // x = 0x1.602954p+78f
      0x67a9'242bU, // x = 0x1.524856p+80f
      0x6a19'76f1U, // x = 0x1.32ede2p+85f
      0x6c55'da58U, // x = 0x1.abb4bp+89f
      0x6f79'be45U, // x = 0x1.f37c8ap+95f
      0x7276'69d4U, // x = 0x1.ecd3a8p+101f
      0x7758'4625U, // x = 0x1.b08c4ap+111f
  };

  for (int i = 0; i < N; ++i) {
    float x = FPBits(INPUTS[i]).get_val();
    EXPECT_MPFR_MATCH_ALL_ROUNDING(mpfr::Operation::Cospi, x,
                                   LIBC_NAMESPACE::cospif(x), 0.5);
    EXPECT_MPFR_MATCH_ALL_ROUNDING(mpfr::Operation::Cospi, -x,
                                   LIBC_NAMESPACE::cospif(-x), 0.5);
  }
}

// For small values, sinpi(x) is pi * x.
TEST_F(LlvmLibcCospifTest, SmallValues) {
  float x = FPBits(0x1780'0000U).get_val();
  EXPECT_MPFR_MATCH_ALL_ROUNDING(mpfr::Operation::Cospi, x,
                                 LIBC_NAMESPACE::cospif(x), 0.5);

  x = FPBits(0x0040'0000U).get_val();
  EXPECT_MPFR_MATCH_ALL_ROUNDING(mpfr::Operation::Cospi, x,
                                 LIBC_NAMESPACE::cospif(x), 0.5);
}

// SDCOMP-26094: check cospif in the cases for which the range reducer
// returns values furthest beyond its nominal upper bound of pi/4.
TEST_F(LlvmLibcCospifTest, SDCOMP_26094) {
  for (uint32_t v : SDCOMP26094_VALUES) {
    float x = FPBits((v)).get_val();
    EXPECT_MPFR_MATCH_ALL_ROUNDING(mpfr::Operation::Cospi, x,
                                   LIBC_NAMESPACE::cospif(x), 0.5);
  }
}

// sinpi(+n + 0.5) = sinpi(-n + 0.5) = +0.0
TEST_F(LlvmLibcCospifTest, SignedZeros) {
  EXPECT_FP_EQ(0.0, LIBC_NAMESPACE::cospif(100.5f));
  EXPECT_FP_EQ(0.0, LIBC_NAMESPACE::cospif(-100.5f));
  EXPECT_FP_EQ(0.0, LIBC_NAMESPACE::cospif(45678.5f));
  EXPECT_FP_EQ(0.0, LIBC_NAMESPACE::cospif(-45678.5f));
  EXPECT_FP_EQ(0.0, LIBC_NAMESPACE::cospif(8000000.5f));
  EXPECT_FP_EQ(0.0, LIBC_NAMESPACE::cospif(-8000000.5f));
}
