# This file is part of lyx2lyx
# Copyright (C) 2003-2004 José Matos <jamatos@lyx.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

"""Convert files to the file format generated by lyx 0.12"""

import re

from parser_tools import check_token, find_re, find_token


def space_before_layout(document):
    "Remove empty line before \\layout."
    lines = document.body
    i = 2  # skip first layout
    while True:
        i = find_token(lines, "\\layout", i)
        if i == -1:
            break

        prot_space = lines[i - 2].find("\\protected_separator")
        if lines[i - 1] == "" and prot_space == -1:
            del lines[i - 1]
        i = i + 1


def formula_inset_space_eat(document):
    "Remove space after inset formula."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset Formula", i)
        if i == -1:
            break

        if len(lines[i]) > 22 and lines[i][21] == " ":
            lines[i] = lines[i][:20] + lines[i][21:]
        i = i + 1


def update_tabular(document):
    "Update from tabular format 1 or 2 to 4."
    lines = document.body
    lyxtable_re = re.compile(r".*\\LyXTable$")
    i = 0
    while True:
        i = find_re(lines, lyxtable_re, i)
        if i == -1:
            break
        i = i + 1
        format = lines[i][8:]

        lines[i] = "multicol4"
        i = i + 1
        rows = int(lines[i].split()[0])
        columns = int(lines[i].split()[1])

        lines[i] = lines[i] + " 0 0 -1 -1 -1 -1"
        i = i + 1

        for j in range(rows):
            lines[i] = lines[i] + " 0 0"
            i = i + 1

        for j in range(columns):
            lines[i] = lines[i] + " "
            i = i + 1

        while lines[i].strip():
            if not format:
                lines[i] = lines[i] + " 1 1"
            lines[i] = lines[i] + " 0 0 0"
            i = i + 1

        lines[i] = lines[i].strip()


def final_dot(document):
    "Merge lines if the dot is the final character."
    lines = document.body
    i = 0
    while i < len(lines):
        if (
            lines[i][-1:] == "."
            and lines[i + 1][:1] != "\\"
            and lines[i + 1][:1] != " "
            and len(lines[i]) + len(lines[i + 1]) <= 72
            and lines[i + 1] != ""
        ):
            lines[i] = lines[i] + lines[i + 1]
            del lines[i + 1]
        else:
            i = i + 1


def update_inset_label(document):
    "Update inset Label."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset Label", i)
        if i == -1:
            return
        lines[i] = "\\begin_inset LatexCommand \\label{" + lines[i][19:] + "}"
        i = i + 1


def update_latexdel(document):
    "Update inset LatexDel."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset LatexDel", i)
        if i == -1:
            return
        lines[i] = lines[i].replace("\\begin_inset LatexDel", "\\begin_inset LatexCommand")
        i = i + 1


def update_vfill(document):
    "Update fill_top and fill_bottom."
    lines = document.body
    for i in range(len(lines)):
        lines[i] = lines[i].replace("\\fill_top", "\\added_space_top vfill")
        lines[i] = lines[i].replace("\\fill_bottom", "\\added_space_bottom vfill")


def update_space_units(document):
    "Update space units."
    lines = document.body
    added_space_bottom = re.compile(r"\\added_space_bottom ([^ ]*)")
    added_space_top = re.compile(r"\\added_space_top ([^ ]*)")
    for i in range(len(lines)):
        result = added_space_bottom.search(lines[i])
        if result:
            old = "\\added_space_bottom " + result.group(1)
            new = "\\added_space_bottom " + str(float(result.group(1))) + "cm"
            lines[i] = lines[i].replace(old, new)

        result = added_space_top.search(lines[i])
        if result:
            old = "\\added_space_top " + result.group(1)
            new = "\\added_space_top " + str(float(result.group(1))) + "cm"
            lines[i] = lines[i].replace(old, new)


def remove_cursor(document):
    "Remove cursor, it is not saved on the file anymore."
    lines = document.body
    i = 0
    cursor_re = re.compile(r".*(\\cursor \d*)")
    while True:
        i = find_re(lines, cursor_re, i)
        if i == -1:
            break
        cursor = cursor_re.search(lines[i]).group(1)
        lines[i] = lines[i].replace(cursor, "")
        i = i + 1


def remove_empty_insets(document):
    "Remove empty insets."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\begin_inset ", i)
        if i == -1:
            break
        if lines[i] == "\\begin_inset " and lines[i + 1] == "\\end_inset ":
            del lines[i]
            del lines[i]
        i = i + 1


def remove_formula_latex(document):
    "Remove formula latex."
    lines = document.body
    i = 0
    while True:
        i = find_token(lines, "\\latex formula_latex ", i)
        if i == -1:
            break
        del lines[i]

        i = find_token(lines, "\\latex default", i)
        if i == -1:
            break
        del lines[i]


def add_end_document(document):
    "Add \\the_end to the end of the document."
    lines = document.body
    i = find_token(lines, "\\the_end", 0)
    if i == -1:
        lines.append("\\the_end")


def header_update(document):
    "Update document header."
    lines = document.header
    i = 0
    l = len(lines)
    while i < l:
        if lines[i][-1:] == " ":
            lines[i] = lines[i][:-1]

        if check_token(lines[i], "\\epsfig"):
            lines[i] = lines[i].replace("\\epsfig", "\\graphics")
            i = i + 1
            continue

        if check_token(lines[i], "\\papersize"):
            size = lines[i].split()[1]
            new_size = size
            paperpackage = ""

            if size == "usletter":
                new_size = "letterpaper"
            if size == "a4wide":
                new_size = "Default"
                paperpackage = "widemarginsa4"

            lines[i] = "\\papersize " + new_size
            i = i + 1
            if paperpackage:
                lines.insert(i, "\\paperpackage " + paperpackage)
                i = i + 1

            lines.insert(i, "\\use_geometry 0")
            lines.insert(i + 1, "\\use_amsmath 0")
            i = i + 2
            continue

        if check_token(lines[i], "\\baselinestretch"):
            size = lines[i].split()[1]
            if size == "1.00":
                name = "single"
            elif size == "1.50":
                name = "onehalf"
            elif size == "2.00":
                name = "double"
            else:
                name = "other " + size
            lines[i] = "\\spacing %s " % name
            i = i + 1
            continue

        i = i + 1


def update_latexaccents(document):
    "Update latex accent insets."
    body = document.body
    i = 1
    while True:
        i = find_token(body, "\\i ", i)
        if i == -1:
            return

        contents = body[i][2:].strip()

        if contents.find("{") != -1 and contents.find("}") != -1:
            i = i + 1
            continue

        if len(contents) == 2:
            contents = contents + "{}"
        elif len(contents) == 3:
            contents = contents[:2] + "{" + contents[2] + "}"
        elif len(contents) == 4:
            if contents[2] == " ":
                contents = contents[:2] + "{" + contents[3] + "}"
            elif contents[2:4] == "\\i" or contents[2:4] == "\\j":
                contents = contents[:2] + "{" + contents[2:] + "}"

        body[i] = "\\i " + contents
        i = i + 1


def obsolete_latex_title(document):
    "Replace layout Latex_Title with Title."
    body = document.body
    i = 0
    while True:
        i = find_token(body, "\\layout", i)
        if i == -1:
            return

        if body[i].lower().find("latex_title") != -1:
            body[i] = "\\layout Title"

        i = i + 1


def remove_inset_latex(document):
    "Replace inset latex with layout LaTeX"
    body = document.body

    i = 0
    while True:
        i = find_token(body, "\\begin_inset Latex", i)
        if i == -1:
            return

        body[i] = body[i].replace("\\begin_inset Latex", "\\layout LaTeX")
        i = find_token(body, "\\end_inset", i)
        if i == -1:
            # this should not happen
            return
        del body[i]


supported_versions = ["0.12.0", "0.12.1", "0.12"]
convert = [
    [
        215,
        [
            header_update,
            add_end_document,
            remove_cursor,
            final_dot,
            update_inset_label,
            update_latexdel,
            update_space_units,
            space_before_layout,
            formula_inset_space_eat,
            update_tabular,
            update_vfill,
            remove_empty_insets,
            remove_formula_latex,
            update_latexaccents,
            obsolete_latex_title,
            remove_inset_latex,
        ],
    ]
]
revert = []


if __name__ == "__main__":
    pass
