/*
 * Copyright (C) Internet Systems Consortium, Inc. ("ISC")
 *
 * SPDX-License-Identifier: MPL-2.0
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, you can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * See the COPYRIGHT file distributed with this work for additional
 * information regarding copyright ownership.
 */

#pragma once

/*! \file isc/stdio.h */

/*%
 * These functions are wrappers around the corresponding stdio functions.
 *
 * They return a detailed error code in the form of an an isc_result_t.  ANSI C
 * does not guarantee that stdio functions set errno, hence these functions
 * must use platform dependent methods (e.g., the POSIX errno) to construct the
 * error code.
 */

#include <stdio.h>

#include <isc/result.h>

/*% Open */
isc_result_t
isc_stdio_open(const char *filename, const char *mode, FILE **fp);

/*% Close */
isc_result_t
isc_stdio_close(FILE *f);

/*% Seek */
isc_result_t
isc_stdio_seek(FILE *f, off_t offset, int whence);

/*% Tell */
isc_result_t
isc_stdio_tell(FILE *f, off_t *offsetp);

/*% Read */
isc_result_t
isc_stdio_read(void *ptr, size_t size, size_t nmemb, FILE *f, size_t *nret);

/*% Write */
isc_result_t
isc_stdio_write(const void *ptr, size_t size, size_t nmemb, FILE *f,
		size_t *nret);

/*% Flush */
isc_result_t
isc_stdio_flush(FILE *f);

isc_result_t
isc_stdio_sync(FILE *f);
/*%<
 * Invoke fsync() on the file descriptor underlying an stdio stream, or an
 * equivalent system-dependent operation.  Note that this function has no
 * direct counterpart in the stdio library.
 */
