/*
 * Copyright (C) 2011 Google Inc. All rights reserved.
 * Copyright (C) 2014 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "StyleCalculationValue.h"

#include "StyleCalculationTree+Copy.h"
#include "StyleCalculationTree+Evaluation.h"
#include "StyleZoomPrimitives.h"
#include <cmath>
#include <wtf/text/TextStream.h>

namespace WebCore {
namespace Style {
namespace Calculation {

Ref<Value> Value::create(CSS::Category category, CSS::Range range, Tree&& tree)
{
    return adoptRef(*new Value(category, range, WTFMove(tree)));
}

Value::Value(CSS::Category category, CSS::Range range, Tree&& tree)
    : m_category(category)
    , m_range(range)
    , m_tree(WTFMove(tree))
{
}

Value::~Value() = default;

double Value::evaluate(double percentResolutionLength, const ZoomFactor& usedZoom) const
{
    auto result = Calculation::evaluate(m_tree, percentResolutionLength, usedZoom);
    if (std::isnan(result))
        return 0;
    return std::clamp(result, m_range.min, m_range.max);
}

double Value::evaluate(double percentResolutionLength, const ZoomNeeded& zoomNeeded) const
{
    auto result = Calculation::evaluate(m_tree, percentResolutionLength, zoomNeeded);
    if (std::isnan(result))
        return 0;
    return std::clamp(result, m_range.min, m_range.max);
}

Tree Value::copyTree() const
{
    return Calculation::copy(m_tree);
}

Child Value::copyRoot() const
{
    auto tree = copyTree();
    return { WTFMove(tree.root) };
}

bool Value::operator==(const Value& other) const
{
    return m_tree == other.m_tree;
}

TextStream& operator<<(TextStream& ts, const Value& value)
{
    return ts << "calc("_s << value.tree() << ')';
}

} // namespace Calculation
} // namespace Style
} // namespace WebCore
