import passlib.utils.handlers as uh
from passlib.crypto.digest import compile_hmac
from passlib.utils.binary import h64

__all__: list[str] = []
_BNULL = b"\x00"


class sha1_crypt(  # type: ignore[misc]
    uh.HasManyBackends,
    uh.HasRounds,
    uh.HasSalt,
    uh.GenericHandler,
):
    """This class implements the SHA1-Crypt password hash, and follows the :ref:`password-hash-api`.

    It supports a variable-length salt, and a variable number of rounds.

    The :meth:`~passlib.ifc.PasswordHash.using` method accepts the following optional keywords:

    :type salt: str
    :param salt:
        Optional salt string.
        If not specified, an 8 character one will be autogenerated (this is recommended).
        If specified, it must be 0-64 characters, drawn from the regexp range ``[./0-9A-Za-z]``.

    :type salt_size: int
    :param salt_size:
        Optional number of bytes to use when autogenerating new salts.
        Defaults to 8 bytes, but can be any value between 0 and 64.

    :type rounds: int
    :param rounds:
        Optional number of rounds to use.
        Defaults to 480000, must be between 1 and 4294967295, inclusive.

    :type relaxed: bool
    :param relaxed:
        By default, providing an invalid value for one of the other
        keywords will result in a :exc:`ValueError`. If ``relaxed=True``,
        and the error can be corrected, a :exc:`~passlib.exc.PasslibHashWarning`
        will be issued instead. Correctable errors include ``rounds``
        that are too small or too large, and ``salt`` strings that are too long.

        .. versionadded:: 1.6
    """

    # --GenericHandler--
    name = "sha1_crypt"
    setting_kwds = ("salt", "salt_size", "rounds")
    ident = "$sha1$"
    checksum_size = 28
    checksum_chars = uh.HASH64_CHARS

    # --HasSalt--
    default_salt_size = 8
    max_salt_size = 64
    salt_chars = uh.HASH64_CHARS

    # --HasRounds--
    default_rounds = 480000  # current passlib default
    min_rounds = 1  # really, this should be higher.
    max_rounds = 4294967295  # 32-bit integer limit
    rounds_cost = "linear"

    @classmethod
    def from_string(cls, hash):
        rounds, salt, chk = uh.parse_mc3(hash, cls.ident, handler=cls)
        return cls(rounds=rounds, salt=salt, checksum=chk)

    def to_string(self, config=False):
        chk = None if config else self.checksum
        return uh.render_mc3(self.ident, self.rounds, self.salt, chk)

    backends = ("builtin",)

    # ---------------------------------------------------------------
    # builtin backend
    # ---------------------------------------------------------------
    @classmethod
    def _load_backend_builtin(cls):
        cls._set_calc_checksum_backend(cls._calc_checksum_builtin)
        return True

    def _calc_checksum_builtin(self, secret):
        if isinstance(secret, str):
            secret = secret.encode("utf-8")
        if _BNULL in secret:
            raise uh.exc.NullPasswordError(self)
        rounds = self.rounds
        # NOTE: this seed value is NOT the same as the config string
        result = (f"{self.salt}$sha1${rounds}").encode("ascii")
        # NOTE: this algorithm is essentially PBKDF1, modified to use HMAC.
        keyed_hmac = compile_hmac("sha1", secret)
        for _ in range(rounds):
            result = keyed_hmac(result)
        return h64.encode_transposed_bytes(result, self._chk_offsets).decode("ascii")

    _chk_offsets = [
        2,
        1,
        0,
        5,
        4,
        3,
        8,
        7,
        6,
        11,
        10,
        9,
        14,
        13,
        12,
        17,
        16,
        15,
        0,
        19,
        18,
    ]
