#
# @BEGIN LICENSE
#
# Psi4: an open-source quantum chemistry software package
#
# Copyright (c) 2007-2019 The Psi4 Developers.
#
# The copyrights for code used from other parties are included in
# the corresponding files.
#
# This file is part of Psi4.
#
# Psi4 is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, version 3.
#
# Psi4 is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License along
# with Psi4; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# @END LICENSE
#

"""
| Database (Hobza) of interaction energies for bimolecular complexes.
| Geometries and reference interaction energies from Rezac et al. JCTC 7 2427 (2011).
| Note that the S66by8-N-1.0 members are not quite the same geometries as S66-N.

- **cp**  ``'off'`` || ``'on'``

- **rlxd** ``'off'``

- **subset**

  - ``'small'``
  - ``'large'``
  - ``'equilibrium'``
  - ``'HB'`` eight-point (0.9, 0.95, 1.0, 1.05, 1.1, 1.25, 1.5, 2.0) :math:`\\times R_{eq}` dissociation curve hydrogen-bonded systems
  - ``'MX'`` eight-point (0.9, 0.95, 1.0, 1.05, 1.1, 1.25, 1.5, 2.0) :math:`\\times R_{eq}` dissociation curve mixed-influence systems
  - ``'DD'`` eight-point (0.9, 0.95, 1.0, 1.05, 1.1, 1.25, 1.5, 2.0) :math:`\\times R_{eq}` dissociation curve dispersion-dominated systems


"""
import re
import qcdb

# <<< S66by8 Database Module >>>
dbse = 'S66by8'

# <<< Database Members >>>
dist = ['-0.9', '-0.95', '-1.0', '-1.05', '-1.1', '-1.25', '-1.5', '-2.0']

HRXN = [str(rxn) + d for rxn in range(1,  67) for d in dist]
HRXN_EQ = [str(rxn) + '-1.0' for rxn in range(1,  67)]
HRXN_SM = ['1-0.9', '12-1.0', '59-2.0']
HRXN_LG = ['26-0.9', '34-1.0', '66-2.0']
HB = [str(rxn) + d for rxn in range(1,  24) for d in dist]
DD = [str(rxn) + d for rxn in range(24, 47) for d in dist]
MX = [str(rxn) + d for rxn in range(47, 67) for d in dist]

# <<< Chemical Systems Involved >>>
RXNM = {}     # reaction matrix of reagent contributions per reaction
ACTV = {}     # order of active reagents per reaction
ACTV_CP = {}  # order of active reagents per counterpoise-corrected reaction
ACTV_SA = {}  # order of active reagents for non-supermolecular calculations
for rxn in HRXN:

    RXNM[   '%s-%s' % (dbse, rxn)] = {'%s-%s-dimer'      % (dbse, rxn) : +1,
                                      '%s-%s-monoA-CP'   % (dbse, rxn) : -1,
                                      '%s-%s-monoB-CP'   % (dbse, rxn) : -1,
                                      '%s-%s-monoA-unCP' % (dbse, rxn) : -1,
                                      '%s-%s-monoB-unCP' % (dbse, rxn) : -1 }

    ACTV_SA['%s-%s' % (dbse, rxn)] = ['%s-%s-dimer'      % (dbse, rxn) ]

    ACTV_CP['%s-%s' % (dbse, rxn)] = ['%s-%s-dimer'      % (dbse, rxn),
                                      '%s-%s-monoA-CP'   % (dbse, rxn),
                                      '%s-%s-monoB-CP'   % (dbse, rxn) ]

    ACTV[   '%s-%s' % (dbse, rxn)] = ['%s-%s-dimer'      % (dbse, rxn),
                                      '%s-%s-monoA-unCP' % (dbse, rxn),
                                      '%s-%s-monoB-unCP' % (dbse, rxn) ]

# <<< Reference Values [kcal/mol] >>>
BIND = {}
BIND['%s-%s'            % (dbse, '1-0.9'                 )] =   -4.570
BIND['%s-%s'            % (dbse, '1-0.95'                )] =   -4.880
BIND['%s-%s'            % (dbse, '1-1.0'                 )] =   -4.890
BIND['%s-%s'            % (dbse, '1-1.05'                )] =   -4.720
BIND['%s-%s'            % (dbse, '1-1.1'                 )] =   -4.450
BIND['%s-%s'            % (dbse, '1-1.25'                )] =   -3.460
BIND['%s-%s'            % (dbse, '1-1.5'                 )] =   -2.110
BIND['%s-%s'            % (dbse, '1-2.0'                 )] =   -0.870
BIND['%s-%s'            % (dbse, '2-0.9'                 )] =   -5.190
BIND['%s-%s'            % (dbse, '2-0.95'                )] =   -5.550
BIND['%s-%s'            % (dbse, '2-1.0'                 )] =   -5.570
BIND['%s-%s'            % (dbse, '2-1.05'                )] =   -5.380
BIND['%s-%s'            % (dbse, '2-1.1'                 )] =   -5.080
BIND['%s-%s'            % (dbse, '2-1.25'                )] =   -3.940
BIND['%s-%s'            % (dbse, '2-1.5'                 )] =   -2.390
BIND['%s-%s'            % (dbse, '2-2.0'                 )] =   -0.950
BIND['%s-%s'            % (dbse, '3-0.9'                 )] =   -6.490
BIND['%s-%s'            % (dbse, '3-0.95'                )] =   -6.870
BIND['%s-%s'            % (dbse, '3-1.0'                 )] =   -6.880
BIND['%s-%s'            % (dbse, '3-1.05'                )] =   -6.650
BIND['%s-%s'            % (dbse, '3-1.1'                 )] =   -6.280
BIND['%s-%s'            % (dbse, '3-1.25'                )] =   -4.920
BIND['%s-%s'            % (dbse, '3-1.5'                 )] =   -2.980
BIND['%s-%s'            % (dbse, '3-2.0'                 )] =   -1.140
BIND['%s-%s'            % (dbse, '4-0.9'                 )] =   -7.630
BIND['%s-%s'            % (dbse, '4-0.95'                )] =   -8.060
BIND['%s-%s'            % (dbse, '4-1.0'                 )] =   -8.080
BIND['%s-%s'            % (dbse, '4-1.05'                )] =   -7.840
BIND['%s-%s'            % (dbse, '4-1.1'                 )] =   -7.460
BIND['%s-%s'            % (dbse, '4-1.25'                )] =   -5.990
BIND['%s-%s'            % (dbse, '4-1.5'                 )] =   -3.830
BIND['%s-%s'            % (dbse, '4-2.0'                 )] =   -1.440
BIND['%s-%s'            % (dbse, '5-0.9'                 )] =   -5.280
BIND['%s-%s'            % (dbse, '5-0.95'                )] =   -5.690
BIND['%s-%s'            % (dbse, '5-1.0'                 )] =   -5.750
BIND['%s-%s'            % (dbse, '5-1.05'                )] =   -5.580
BIND['%s-%s'            % (dbse, '5-1.1'                 )] =   -5.280
BIND['%s-%s'            % (dbse, '5-1.25'                )] =   -4.140
BIND['%s-%s'            % (dbse, '5-1.5'                 )] =   -2.520
BIND['%s-%s'            % (dbse, '5-2.0'                 )] =   -1.010
BIND['%s-%s'            % (dbse, '6-0.9'                 )] =   -6.970
BIND['%s-%s'            % (dbse, '6-0.95'                )] =   -7.470
BIND['%s-%s'            % (dbse, '6-1.0'                 )] =   -7.540
BIND['%s-%s'            % (dbse, '6-1.05'                )] =   -7.340
BIND['%s-%s'            % (dbse, '6-1.1'                 )] =   -6.970
BIND['%s-%s'            % (dbse, '6-1.25'                )] =   -5.500
BIND['%s-%s'            % (dbse, '6-1.5'                 )] =   -3.350
BIND['%s-%s'            % (dbse, '6-2.0'                 )] =   -1.270
BIND['%s-%s'            % (dbse, '7-0.9'                 )] =   -7.630
BIND['%s-%s'            % (dbse, '7-0.95'                )] =   -8.140
BIND['%s-%s'            % (dbse, '7-1.0'                 )] =   -8.220
BIND['%s-%s'            % (dbse, '7-1.05'                )] =   -8.020
BIND['%s-%s'            % (dbse, '7-1.1'                 )] =   -7.660
BIND['%s-%s'            % (dbse, '7-1.25'                )] =   -6.180
BIND['%s-%s'            % (dbse, '7-1.5'                 )] =   -3.650
BIND['%s-%s'            % (dbse, '7-2.0'                 )] =   -1.100
BIND['%s-%s'            % (dbse, '8-0.9'                 )] =   -4.590
BIND['%s-%s'            % (dbse, '8-0.95'                )] =   -4.950
BIND['%s-%s'            % (dbse, '8-1.0'                 )] =   -5.000
BIND['%s-%s'            % (dbse, '8-1.05'                )] =   -4.850
BIND['%s-%s'            % (dbse, '8-1.1'                 )] =   -4.590
BIND['%s-%s'            % (dbse, '8-1.25'                )] =   -3.590
BIND['%s-%s'            % (dbse, '8-1.5'                 )] =   -2.200
BIND['%s-%s'            % (dbse, '8-2.0'                 )] =   -0.910
BIND['%s-%s'            % (dbse, '9-0.9'                 )] =   -2.810
BIND['%s-%s'            % (dbse, '9-0.95'                )] =   -3.040
BIND['%s-%s'            % (dbse, '9-1.0'                 )] =   -3.040
BIND['%s-%s'            % (dbse, '9-1.05'                )] =   -2.900
BIND['%s-%s'            % (dbse, '9-1.1'                 )] =   -2.690
BIND['%s-%s'            % (dbse, '9-1.25'                )] =   -1.980
BIND['%s-%s'            % (dbse, '9-1.5'                 )] =   -1.100
BIND['%s-%s'            % (dbse, '9-2.0'                 )] =   -0.390
BIND['%s-%s'            % (dbse, '10-0.9'                )] =   -3.700
BIND['%s-%s'            % (dbse, '10-0.95'               )] =   -4.090
BIND['%s-%s'            % (dbse, '10-1.0'                )] =   -4.150
BIND['%s-%s'            % (dbse, '10-1.05'               )] =   -4.010
BIND['%s-%s'            % (dbse, '10-1.1'                )] =   -3.750
BIND['%s-%s'            % (dbse, '10-1.25'               )] =   -2.790
BIND['%s-%s'            % (dbse, '10-1.5'                )] =   -1.310
BIND['%s-%s'            % (dbse, '10-2.0'                )] =   -0.390
BIND['%s-%s'            % (dbse, '11-0.9'                )] =   -4.950
BIND['%s-%s'            % (dbse, '11-0.95'               )] =   -5.360
BIND['%s-%s'            % (dbse, '11-1.0'                )] =   -5.410
BIND['%s-%s'            % (dbse, '11-1.05'               )] =   -5.240
BIND['%s-%s'            % (dbse, '11-1.1'                )] =   -4.940
BIND['%s-%s'            % (dbse, '11-1.25'               )] =   -3.210
BIND['%s-%s'            % (dbse, '11-1.5'                )] =   -1.400
BIND['%s-%s'            % (dbse, '11-2.0'                )] =   -0.460
BIND['%s-%s'            % (dbse, '12-0.9'                )] =   -6.750
BIND['%s-%s'            % (dbse, '12-0.95'               )] =   -7.200
BIND['%s-%s'            % (dbse, '12-1.0'                )] =   -7.250
BIND['%s-%s'            % (dbse, '12-1.05'               )] =   -7.030
BIND['%s-%s'            % (dbse, '12-1.1'                )] =   -6.660
BIND['%s-%s'            % (dbse, '12-1.25'               )] =   -5.220
BIND['%s-%s'            % (dbse, '12-1.5'                )] =   -3.150
BIND['%s-%s'            % (dbse, '12-2.0'                )] =   -1.200
BIND['%s-%s'            % (dbse, '13-0.9'                )] =   -5.670
BIND['%s-%s'            % (dbse, '13-0.95'               )] =   -6.110
BIND['%s-%s'            % (dbse, '13-1.0'                )] =   -6.180
BIND['%s-%s'            % (dbse, '13-1.05'               )] =   -6.030
BIND['%s-%s'            % (dbse, '13-1.1'                )] =   -5.740
BIND['%s-%s'            % (dbse, '13-1.25'               )] =   -4.600
BIND['%s-%s'            % (dbse, '13-1.5'                )] =   -2.950
BIND['%s-%s'            % (dbse, '13-2.0'                )] =   -1.300
BIND['%s-%s'            % (dbse, '14-0.9'                )] =   -6.810
BIND['%s-%s'            % (dbse, '14-0.95'               )] =   -7.340
BIND['%s-%s'            % (dbse, '14-1.0'                )] =   -7.450
BIND['%s-%s'            % (dbse, '14-1.05'               )] =   -7.290
BIND['%s-%s'            % (dbse, '14-1.1'                )] =   -6.970
BIND['%s-%s'            % (dbse, '14-1.25'               )] =   -5.610
BIND['%s-%s'            % (dbse, '14-1.5'                )] =   -3.550
BIND['%s-%s'            % (dbse, '14-2.0'                )] =   -1.490
BIND['%s-%s'            % (dbse, '15-0.9'                )] =   -8.020
BIND['%s-%s'            % (dbse, '15-0.95'               )] =   -8.530
BIND['%s-%s'            % (dbse, '15-1.0'                )] =   -8.620
BIND['%s-%s'            % (dbse, '15-1.05'               )] =   -8.450
BIND['%s-%s'            % (dbse, '15-1.1'                )] =   -8.100
BIND['%s-%s'            % (dbse, '15-1.25'               )] =   -6.660
BIND['%s-%s'            % (dbse, '15-1.5'                )] =   -4.420
BIND['%s-%s'            % (dbse, '15-2.0'                )] =   -1.770
BIND['%s-%s'            % (dbse, '16-0.9'                )] =   -4.700
BIND['%s-%s'            % (dbse, '16-0.95'               )] =   -5.060
BIND['%s-%s'            % (dbse, '16-1.0'                )] =   -5.120
BIND['%s-%s'            % (dbse, '16-1.05'               )] =   -4.980
BIND['%s-%s'            % (dbse, '16-1.1'                )] =   -4.740
BIND['%s-%s'            % (dbse, '16-1.25'               )] =   -3.790
BIND['%s-%s'            % (dbse, '16-1.5'                )] =   -2.450
BIND['%s-%s'            % (dbse, '16-2.0'                )] =   -1.130
BIND['%s-%s'            % (dbse, '17-0.9'                )] =  -15.700
BIND['%s-%s'            % (dbse, '17-0.95'               )] =  -16.900
BIND['%s-%s'            % (dbse, '17-1.0'                )] =  -17.180
BIND['%s-%s'            % (dbse, '17-1.05'               )] =  -16.860
BIND['%s-%s'            % (dbse, '17-1.1'                )] =  -16.170
BIND['%s-%s'            % (dbse, '17-1.25'               )] =  -13.150
BIND['%s-%s'            % (dbse, '17-1.5'                )] =   -8.350
BIND['%s-%s'            % (dbse, '17-2.0'                )] =   -3.330
BIND['%s-%s'            % (dbse, '18-0.9'                )] =   -6.390
BIND['%s-%s'            % (dbse, '18-0.95'               )] =   -6.800
BIND['%s-%s'            % (dbse, '18-1.0'                )] =   -6.830
BIND['%s-%s'            % (dbse, '18-1.05'               )] =   -6.620
BIND['%s-%s'            % (dbse, '18-1.1'                )] =   -6.270
BIND['%s-%s'            % (dbse, '18-1.25'               )] =   -4.930
BIND['%s-%s'            % (dbse, '18-1.5'                )] =   -3.010
BIND['%s-%s'            % (dbse, '18-2.0'                )] =   -1.190
BIND['%s-%s'            % (dbse, '19-0.9'                )] =   -6.780
BIND['%s-%s'            % (dbse, '19-0.95'               )] =   -7.310
BIND['%s-%s'            % (dbse, '19-1.0'                )] =   -7.400
BIND['%s-%s'            % (dbse, '19-1.05'               )] =   -7.230
BIND['%s-%s'            % (dbse, '19-1.1'                )] =   -6.880
BIND['%s-%s'            % (dbse, '19-1.25'               )] =   -5.480
BIND['%s-%s'            % (dbse, '19-1.5'                )] =   -3.380
BIND['%s-%s'            % (dbse, '19-2.0'                )] =   -1.330
BIND['%s-%s'            % (dbse, '20-0.9'                )] =  -17.450
BIND['%s-%s'            % (dbse, '20-0.95'               )] =  -18.790
BIND['%s-%s'            % (dbse, '20-1.0'                )] =  -19.090
BIND['%s-%s'            % (dbse, '20-1.05'               )] =  -18.720
BIND['%s-%s'            % (dbse, '20-1.1'                )] =  -17.940
BIND['%s-%s'            % (dbse, '20-1.25'               )] =  -14.560
BIND['%s-%s'            % (dbse, '20-1.5'                )] =   -9.210
BIND['%s-%s'            % (dbse, '20-2.0'                )] =   -3.590
BIND['%s-%s'            % (dbse, '21-0.9'                )] =  -14.930
BIND['%s-%s'            % (dbse, '21-0.95'               )] =  -16.030
BIND['%s-%s'            % (dbse, '21-1.0'                )] =  -16.260
BIND['%s-%s'            % (dbse, '21-1.05'               )] =  -15.940
BIND['%s-%s'            % (dbse, '21-1.1'                )] =  -15.260
BIND['%s-%s'            % (dbse, '21-1.25'               )] =  -12.420
BIND['%s-%s'            % (dbse, '21-1.5'                )] =   -8.000
BIND['%s-%s'            % (dbse, '21-2.0'                )] =   -3.000
BIND['%s-%s'            % (dbse, '22-0.9'                )] =  -17.900
BIND['%s-%s'            % (dbse, '22-0.95'               )] =  -19.190
BIND['%s-%s'            % (dbse, '22-1.0'                )] =  -19.490
BIND['%s-%s'            % (dbse, '22-1.05'               )] =  -19.150
BIND['%s-%s'            % (dbse, '22-1.1'                )] =  -18.410
BIND['%s-%s'            % (dbse, '22-1.25'               )] =  -15.140
BIND['%s-%s'            % (dbse, '22-1.5'                )] =   -9.860
BIND['%s-%s'            % (dbse, '22-2.0'                )] =   -4.150
BIND['%s-%s'            % (dbse, '23-0.9'                )] =  -17.700
BIND['%s-%s'            % (dbse, '23-0.95'               )] =  -18.890
BIND['%s-%s'            % (dbse, '23-1.0'                )] =  -19.190
BIND['%s-%s'            % (dbse, '23-1.05'               )] =  -18.900
BIND['%s-%s'            % (dbse, '23-1.1'                )] =  -18.220
BIND['%s-%s'            % (dbse, '23-1.25'               )] =  -15.210
BIND['%s-%s'            % (dbse, '23-1.5'                )] =  -10.230
BIND['%s-%s'            % (dbse, '23-2.0'                )] =   -4.650
BIND['%s-%s'            % (dbse, '24-0.9'                )] =   -0.230
BIND['%s-%s'            % (dbse, '24-0.95'               )] =   -2.070
BIND['%s-%s'            % (dbse, '24-1.0'                )] =   -2.740
BIND['%s-%s'            % (dbse, '24-1.05'               )] =   -2.800
BIND['%s-%s'            % (dbse, '24-1.1'                )] =   -2.580
BIND['%s-%s'            % (dbse, '24-1.25'               )] =   -1.540
BIND['%s-%s'            % (dbse, '24-1.5'                )] =   -0.490
BIND['%s-%s'            % (dbse, '24-2.0'                )] =   -0.060
BIND['%s-%s'            % (dbse, '25-0.9'                )] =   -1.320
BIND['%s-%s'            % (dbse, '25-0.95'               )] =   -3.170
BIND['%s-%s'            % (dbse, '25-1.0'                )] =   -3.830
BIND['%s-%s'            % (dbse, '25-1.05'               )] =   -3.860
BIND['%s-%s'            % (dbse, '25-1.1'                )] =   -3.580
BIND['%s-%s'            % (dbse, '25-1.25'               )] =   -2.360
BIND['%s-%s'            % (dbse, '25-1.5'                )] =   -0.960
BIND['%s-%s'            % (dbse, '25-2.0'                )] =   -0.240
BIND['%s-%s'            % (dbse, '26-0.9'                )] =   -7.860
BIND['%s-%s'            % (dbse, '26-0.95'               )] =   -9.500
BIND['%s-%s'            % (dbse, '26-1.0'                )] =   -9.820
BIND['%s-%s'            % (dbse, '26-1.05'               )] =   -9.430
BIND['%s-%s'            % (dbse, '26-1.1'                )] =   -8.690
BIND['%s-%s'            % (dbse, '26-1.25'               )] =   -6.070
BIND['%s-%s'            % (dbse, '26-1.5'                )] =   -3.100
BIND['%s-%s'            % (dbse, '26-2.0'                )] =   -0.990
BIND['%s-%s'            % (dbse, '27-0.9'                )] =   -0.730
BIND['%s-%s'            % (dbse, '27-0.95'               )] =   -2.690
BIND['%s-%s'            % (dbse, '27-1.0'                )] =   -3.370
BIND['%s-%s'            % (dbse, '27-1.05'               )] =   -3.400
BIND['%s-%s'            % (dbse, '27-1.1'                )] =   -3.130
BIND['%s-%s'            % (dbse, '27-1.25'               )] =   -1.970
BIND['%s-%s'            % (dbse, '27-1.5'                )] =   -0.720
BIND['%s-%s'            % (dbse, '27-2.0'                )] =   -0.150
BIND['%s-%s'            % (dbse, '28-0.9'                )] =   -3.530
BIND['%s-%s'            % (dbse, '28-0.95'               )] =   -5.220
BIND['%s-%s'            % (dbse, '28-1.0'                )] =   -5.710
BIND['%s-%s'            % (dbse, '28-1.05'               )] =   -5.550
BIND['%s-%s'            % (dbse, '28-1.1'                )] =   -5.080
BIND['%s-%s'            % (dbse, '28-1.25'               )] =   -3.290
BIND['%s-%s'            % (dbse, '28-1.5'                )] =   -1.360
BIND['%s-%s'            % (dbse, '28-2.0'                )] =   -0.250
BIND['%s-%s'            % (dbse, '29-0.9'                )] =   -3.740
BIND['%s-%s'            % (dbse, '29-0.95'               )] =   -6.170
BIND['%s-%s'            % (dbse, '29-1.0'                )] =   -6.810
BIND['%s-%s'            % (dbse, '29-1.05'               )] =   -6.580
BIND['%s-%s'            % (dbse, '29-1.1'                )] =   -5.970
BIND['%s-%s'            % (dbse, '29-1.25'               )] =   -3.870
BIND['%s-%s'            % (dbse, '29-1.5'                )] =   -1.780
BIND['%s-%s'            % (dbse, '29-2.0'                )] =   -0.530
BIND['%s-%s'            % (dbse, '30-0.9'                )] =   -0.260
BIND['%s-%s'            % (dbse, '30-0.95'               )] =   -1.130
BIND['%s-%s'            % (dbse, '30-1.0'                )] =   -1.410
BIND['%s-%s'            % (dbse, '30-1.05'               )] =   -1.400
BIND['%s-%s'            % (dbse, '30-1.1'                )] =   -1.260
BIND['%s-%s'            % (dbse, '30-1.25'               )] =   -0.690
BIND['%s-%s'            % (dbse, '30-1.5'                )] =   -0.170
BIND['%s-%s'            % (dbse, '30-2.0'                )] =    0.010
BIND['%s-%s'            % (dbse, '31-0.9'                )] =   -2.570
BIND['%s-%s'            % (dbse, '31-0.95'               )] =   -3.240
BIND['%s-%s'            % (dbse, '31-1.0'                )] =   -3.380
BIND['%s-%s'            % (dbse, '31-1.05'               )] =   -3.240
BIND['%s-%s'            % (dbse, '31-1.1'                )] =   -2.960
BIND['%s-%s'            % (dbse, '31-1.25'               )] =   -1.990
BIND['%s-%s'            % (dbse, '31-1.5'                )] =   -0.930
BIND['%s-%s'            % (dbse, '31-2.0'                )] =   -0.250
BIND['%s-%s'            % (dbse, '32-0.9'                )] =   -2.790
BIND['%s-%s'            % (dbse, '32-0.95'               )] =   -3.560
BIND['%s-%s'            % (dbse, '32-1.0'                )] =   -3.740
BIND['%s-%s'            % (dbse, '32-1.05'               )] =   -3.590
BIND['%s-%s'            % (dbse, '32-1.1'                )] =   -3.290
BIND['%s-%s'            % (dbse, '32-1.25'               )] =   -2.230
BIND['%s-%s'            % (dbse, '32-1.5'                )] =   -1.040
BIND['%s-%s'            % (dbse, '32-2.0'                )] =   -0.270
BIND['%s-%s'            % (dbse, '33-0.9'                )] =   -0.890
BIND['%s-%s'            % (dbse, '33-0.95'               )] =   -1.620
BIND['%s-%s'            % (dbse, '33-1.0'                )] =   -1.860
BIND['%s-%s'            % (dbse, '33-1.05'               )] =   -1.830
BIND['%s-%s'            % (dbse, '33-1.1'                )] =   -1.670
BIND['%s-%s'            % (dbse, '33-1.25'               )] =   -1.030
BIND['%s-%s'            % (dbse, '33-1.5'                )] =   -0.370
BIND['%s-%s'            % (dbse, '33-2.0'                )] =   -0.050
BIND['%s-%s'            % (dbse, '34-0.9'                )] =   -2.860
BIND['%s-%s'            % (dbse, '34-0.95'               )] =   -3.620
BIND['%s-%s'            % (dbse, '34-1.0'                )] =   -3.770
BIND['%s-%s'            % (dbse, '34-1.05'               )] =   -3.610
BIND['%s-%s'            % (dbse, '34-1.1'                )] =   -3.300
BIND['%s-%s'            % (dbse, '34-1.25'               )] =   -2.240
BIND['%s-%s'            % (dbse, '34-1.5'                )] =   -1.050
BIND['%s-%s'            % (dbse, '34-2.0'                )] =   -0.270
BIND['%s-%s'            % (dbse, '35-0.9'                )] =   -1.850
BIND['%s-%s'            % (dbse, '35-0.95'               )] =   -2.490
BIND['%s-%s'            % (dbse, '35-1.0'                )] =   -2.610
BIND['%s-%s'            % (dbse, '35-1.05'               )] =   -2.480
BIND['%s-%s'            % (dbse, '35-1.1'                )] =   -2.250
BIND['%s-%s'            % (dbse, '35-1.25'               )] =   -1.500
BIND['%s-%s'            % (dbse, '35-1.5'                )] =   -0.700
BIND['%s-%s'            % (dbse, '35-2.0'                )] =   -0.180
BIND['%s-%s'            % (dbse, '36-0.9'                )] =   -1.450
BIND['%s-%s'            % (dbse, '36-0.95'               )] =   -1.740
BIND['%s-%s'            % (dbse, '36-1.0'                )] =   -1.770
BIND['%s-%s'            % (dbse, '36-1.05'               )] =   -1.680
BIND['%s-%s'            % (dbse, '36-1.1'                )] =   -1.540
BIND['%s-%s'            % (dbse, '36-1.25'               )] =   -1.050
BIND['%s-%s'            % (dbse, '36-1.5'                )] =   -0.500
BIND['%s-%s'            % (dbse, '36-2.0'                )] =   -0.140
BIND['%s-%s'            % (dbse, '37-0.9'                )] =   -1.610
BIND['%s-%s'            % (dbse, '37-0.95'               )] =   -2.250
BIND['%s-%s'            % (dbse, '37-1.0'                )] =   -2.410
BIND['%s-%s'            % (dbse, '37-1.05'               )] =   -2.330
BIND['%s-%s'            % (dbse, '37-1.1'                )] =   -2.140
BIND['%s-%s'            % (dbse, '37-1.25'               )] =   -1.470
BIND['%s-%s'            % (dbse, '37-1.5'                )] =   -0.710
BIND['%s-%s'            % (dbse, '37-2.0'                )] =   -0.190
BIND['%s-%s'            % (dbse, '38-0.9'                )] =   -2.240
BIND['%s-%s'            % (dbse, '38-0.95'               )] =   -2.840
BIND['%s-%s'            % (dbse, '38-1.0'                )] =   -3.000
BIND['%s-%s'            % (dbse, '38-1.05'               )] =   -2.860
BIND['%s-%s'            % (dbse, '38-1.1'                )] =   -2.600
BIND['%s-%s'            % (dbse, '38-1.25'               )] =   -1.710
BIND['%s-%s'            % (dbse, '38-1.5'                )] =   -0.790
BIND['%s-%s'            % (dbse, '38-2.0'                )] =   -0.200
BIND['%s-%s'            % (dbse, '39-0.9'                )] =   -2.160
BIND['%s-%s'            % (dbse, '39-0.95'               )] =   -3.240
BIND['%s-%s'            % (dbse, '39-1.0'                )] =   -3.570
BIND['%s-%s'            % (dbse, '39-1.05'               )] =   -3.470
BIND['%s-%s'            % (dbse, '39-1.1'                )] =   -3.170
BIND['%s-%s'            % (dbse, '39-1.25'               )] =   -2.070
BIND['%s-%s'            % (dbse, '39-1.5'                )] =   -0.900
BIND['%s-%s'            % (dbse, '39-2.0'                )] =   -0.190
BIND['%s-%s'            % (dbse, '40-0.9'                )] =   -1.860
BIND['%s-%s'            % (dbse, '40-0.95'               )] =   -2.680
BIND['%s-%s'            % (dbse, '40-1.0'                )] =   -2.890
BIND['%s-%s'            % (dbse, '40-1.05'               )] =   -2.800
BIND['%s-%s'            % (dbse, '40-1.1'                )] =   -2.570
BIND['%s-%s'            % (dbse, '40-1.25'               )] =   -1.700
BIND['%s-%s'            % (dbse, '40-1.5'                )] =   -0.770
BIND['%s-%s'            % (dbse, '40-2.0'                )] =   -0.190
BIND['%s-%s'            % (dbse, '41-0.9'                )] =   -3.830
BIND['%s-%s'            % (dbse, '41-0.95'               )] =   -4.690
BIND['%s-%s'            % (dbse, '41-1.0'                )] =   -4.840
BIND['%s-%s'            % (dbse, '41-1.05'               )] =   -4.600
BIND['%s-%s'            % (dbse, '41-1.1'                )] =   -4.080
BIND['%s-%s'            % (dbse, '41-1.25'               )] =   -2.440
BIND['%s-%s'            % (dbse, '41-1.5'                )] =   -0.980
BIND['%s-%s'            % (dbse, '41-2.0'                )] =   -0.210
BIND['%s-%s'            % (dbse, '42-0.9'                )] =   -3.050
BIND['%s-%s'            % (dbse, '42-0.95'               )] =   -3.970
BIND['%s-%s'            % (dbse, '42-1.0'                )] =   -4.130
BIND['%s-%s'            % (dbse, '42-1.05'               )] =   -3.920
BIND['%s-%s'            % (dbse, '42-1.1'                )] =   -3.540
BIND['%s-%s'            % (dbse, '42-1.25'               )] =   -2.290
BIND['%s-%s'            % (dbse, '42-1.5'                )] =   -1.020
BIND['%s-%s'            % (dbse, '42-2.0'                )] =   -0.250
BIND['%s-%s'            % (dbse, '43-0.9'                )] =   -2.870
BIND['%s-%s'            % (dbse, '43-0.95'               )] =   -3.600
BIND['%s-%s'            % (dbse, '43-1.0'                )] =   -3.700
BIND['%s-%s'            % (dbse, '43-1.05'               )] =   -3.490
BIND['%s-%s'            % (dbse, '43-1.1'                )] =   -3.150
BIND['%s-%s'            % (dbse, '43-1.25'               )] =   -2.050
BIND['%s-%s'            % (dbse, '43-1.5'                )] =   -0.920
BIND['%s-%s'            % (dbse, '43-2.0'                )] =   -0.230
BIND['%s-%s'            % (dbse, '44-0.9'                )] =   -1.660
BIND['%s-%s'            % (dbse, '44-0.95'               )] =   -1.970
BIND['%s-%s'            % (dbse, '44-1.0'                )] =   -1.990
BIND['%s-%s'            % (dbse, '44-1.05'               )] =   -1.860
BIND['%s-%s'            % (dbse, '44-1.1'                )] =   -1.670
BIND['%s-%s'            % (dbse, '44-1.25'               )] =   -1.090
BIND['%s-%s'            % (dbse, '44-1.5'                )] =   -0.490
BIND['%s-%s'            % (dbse, '44-2.0'                )] =   -0.120
BIND['%s-%s'            % (dbse, '45-0.9'                )] =   -1.130
BIND['%s-%s'            % (dbse, '45-0.95'               )] =   -1.630
BIND['%s-%s'            % (dbse, '45-1.0'                )] =   -1.750
BIND['%s-%s'            % (dbse, '45-1.05'               )] =   -1.680
BIND['%s-%s'            % (dbse, '45-1.1'                )] =   -1.520
BIND['%s-%s'            % (dbse, '45-1.25'               )] =   -0.980
BIND['%s-%s'            % (dbse, '45-1.5'                )] =   -0.420
BIND['%s-%s'            % (dbse, '45-2.0'                )] =   -0.100
BIND['%s-%s'            % (dbse, '46-0.9'                )] =   -3.750
BIND['%s-%s'            % (dbse, '46-0.95'               )] =   -4.210
BIND['%s-%s'            % (dbse, '46-1.0'                )] =   -4.240
BIND['%s-%s'            % (dbse, '46-1.05'               )] =   -4.050
BIND['%s-%s'            % (dbse, '46-1.1'                )] =   -3.730
BIND['%s-%s'            % (dbse, '46-1.25'               )] =   -2.630
BIND['%s-%s'            % (dbse, '46-1.5'                )] =   -1.190
BIND['%s-%s'            % (dbse, '46-2.0'                )] =   -0.290
BIND['%s-%s'            % (dbse, '47-0.9'                )] =   -1.640
BIND['%s-%s'            % (dbse, '47-0.95'               )] =   -2.580
BIND['%s-%s'            % (dbse, '47-1.0'                )] =   -2.870
BIND['%s-%s'            % (dbse, '47-1.05'               )] =   -2.820
BIND['%s-%s'            % (dbse, '47-1.1'                )] =   -2.600
BIND['%s-%s'            % (dbse, '47-1.25'               )] =   -1.770
BIND['%s-%s'            % (dbse, '47-1.5'                )] =   -0.830
BIND['%s-%s'            % (dbse, '47-2.0'                )] =   -0.220
BIND['%s-%s'            % (dbse, '48-0.9'                )] =   -2.530
BIND['%s-%s'            % (dbse, '48-0.95'               )] =   -3.310
BIND['%s-%s'            % (dbse, '48-1.0'                )] =   -3.530
BIND['%s-%s'            % (dbse, '48-1.05'               )] =   -3.450
BIND['%s-%s'            % (dbse, '48-1.1'                )] =   -3.220
BIND['%s-%s'            % (dbse, '48-1.25'               )] =   -2.290
BIND['%s-%s'            % (dbse, '48-1.5'                )] =   -1.180
BIND['%s-%s'            % (dbse, '48-2.0'                )] =   -0.370
BIND['%s-%s'            % (dbse, '49-0.9'                )] =   -2.090
BIND['%s-%s'            % (dbse, '49-0.95'               )] =   -3.040
BIND['%s-%s'            % (dbse, '49-1.0'                )] =   -3.320
BIND['%s-%s'            % (dbse, '49-1.05'               )] =   -3.260
BIND['%s-%s'            % (dbse, '49-1.1'                )] =   -3.020
BIND['%s-%s'            % (dbse, '49-1.25'               )] =   -2.100
BIND['%s-%s'            % (dbse, '49-1.5'                )] =   -1.050
BIND['%s-%s'            % (dbse, '49-2.0'                )] =   -0.330
BIND['%s-%s'            % (dbse, '50-0.9'                )] =   -1.870
BIND['%s-%s'            % (dbse, '50-0.95'               )] =   -2.650
BIND['%s-%s'            % (dbse, '50-1.0'                )] =   -2.860
BIND['%s-%s'            % (dbse, '50-1.05'               )] =   -2.790
BIND['%s-%s'            % (dbse, '50-1.1'                )] =   -2.580
BIND['%s-%s'            % (dbse, '50-1.25'               )] =   -1.790
BIND['%s-%s'            % (dbse, '50-1.5'                )] =   -0.890
BIND['%s-%s'            % (dbse, '50-2.0'                )] =   -0.270
BIND['%s-%s'            % (dbse, '51-0.9'                )] =   -1.210
BIND['%s-%s'            % (dbse, '51-0.95'               )] =   -1.470
BIND['%s-%s'            % (dbse, '51-1.0'                )] =   -1.520
BIND['%s-%s'            % (dbse, '51-1.05'               )] =   -1.460
BIND['%s-%s'            % (dbse, '51-1.1'                )] =   -1.350
BIND['%s-%s'            % (dbse, '51-1.25'               )] =   -0.930
BIND['%s-%s'            % (dbse, '51-1.5'                )] =   -0.460
BIND['%s-%s'            % (dbse, '51-2.0'                )] =   -0.140
BIND['%s-%s'            % (dbse, '52-0.9'                )] =   -3.970
BIND['%s-%s'            % (dbse, '52-0.95'               )] =   -4.580
BIND['%s-%s'            % (dbse, '52-1.0'                )] =   -4.700
BIND['%s-%s'            % (dbse, '52-1.05'               )] =   -4.550
BIND['%s-%s'            % (dbse, '52-1.1'                )] =   -4.250
BIND['%s-%s'            % (dbse, '52-1.25'               )] =   -3.130
BIND['%s-%s'            % (dbse, '52-1.5'                )] =   -1.700
BIND['%s-%s'            % (dbse, '52-2.0'                )] =   -0.550
BIND['%s-%s'            % (dbse, '53-0.9'                )] =   -3.790
BIND['%s-%s'            % (dbse, '53-0.95'               )] =   -4.270
BIND['%s-%s'            % (dbse, '53-1.0'                )] =   -4.360
BIND['%s-%s'            % (dbse, '53-1.05'               )] =   -4.230
BIND['%s-%s'            % (dbse, '53-1.1'                )] =   -3.970
BIND['%s-%s'            % (dbse, '53-1.25'               )] =   -2.970
BIND['%s-%s'            % (dbse, '53-1.5'                )] =   -1.650
BIND['%s-%s'            % (dbse, '53-2.0'                )] =   -0.480
BIND['%s-%s'            % (dbse, '54-0.9'                )] =   -2.800
BIND['%s-%s'            % (dbse, '54-0.95'               )] =   -3.210
BIND['%s-%s'            % (dbse, '54-1.0'                )] =   -3.270
BIND['%s-%s'            % (dbse, '54-1.05'               )] =   -3.130
BIND['%s-%s'            % (dbse, '54-1.1'                )] =   -2.900
BIND['%s-%s'            % (dbse, '54-1.25'               )] =   -2.100
BIND['%s-%s'            % (dbse, '54-1.5'                )] =   -1.160
BIND['%s-%s'            % (dbse, '54-2.0'                )] =   -0.420
BIND['%s-%s'            % (dbse, '55-0.9'                )] =   -3.450
BIND['%s-%s'            % (dbse, '55-0.95'               )] =   -4.050
BIND['%s-%s'            % (dbse, '55-1.0'                )] =   -4.190
BIND['%s-%s'            % (dbse, '55-1.05'               )] =   -4.070
BIND['%s-%s'            % (dbse, '55-1.1'                )] =   -3.800
BIND['%s-%s'            % (dbse, '55-1.25'               )] =   -2.800
BIND['%s-%s'            % (dbse, '55-1.5'                )] =   -1.530
BIND['%s-%s'            % (dbse, '55-2.0'                )] =   -0.520
BIND['%s-%s'            % (dbse, '56-0.9'                )] =   -2.510
BIND['%s-%s'            % (dbse, '56-0.95'               )] =   -3.090
BIND['%s-%s'            % (dbse, '56-1.0'                )] =   -3.230
BIND['%s-%s'            % (dbse, '56-1.05'               )] =   -3.120
BIND['%s-%s'            % (dbse, '56-1.1'                )] =   -2.860
BIND['%s-%s'            % (dbse, '56-1.25'               )] =   -1.950
BIND['%s-%s'            % (dbse, '56-1.5'                )] =   -0.940
BIND['%s-%s'            % (dbse, '56-2.0'                )] =   -0.260
BIND['%s-%s'            % (dbse, '57-0.9'                )] =   -3.750
BIND['%s-%s'            % (dbse, '57-0.95'               )] =   -4.960
BIND['%s-%s'            % (dbse, '57-1.0'                )] =   -5.280
BIND['%s-%s'            % (dbse, '57-1.05'               )] =   -5.140
BIND['%s-%s'            % (dbse, '57-1.1'                )] =   -4.780
BIND['%s-%s'            % (dbse, '57-1.25'               )] =   -3.410
BIND['%s-%s'            % (dbse, '57-1.5'                )] =   -1.800
BIND['%s-%s'            % (dbse, '57-2.0'                )] =   -0.620
BIND['%s-%s'            % (dbse, '58-0.9'                )] =   -2.800
BIND['%s-%s'            % (dbse, '58-0.95'               )] =   -3.820
BIND['%s-%s'            % (dbse, '58-1.0'                )] =   -4.150
BIND['%s-%s'            % (dbse, '58-1.05'               )] =   -3.890
BIND['%s-%s'            % (dbse, '58-1.1'                )] =   -3.450
BIND['%s-%s'            % (dbse, '58-1.25'               )] =   -2.180
BIND['%s-%s'            % (dbse, '58-1.5'                )] =   -1.010
BIND['%s-%s'            % (dbse, '58-2.0'                )] =   -0.280
BIND['%s-%s'            % (dbse, '59-0.9'                )] =   -2.520
BIND['%s-%s'            % (dbse, '59-0.95'               )] =   -2.800
BIND['%s-%s'            % (dbse, '59-1.0'                )] =   -2.850
BIND['%s-%s'            % (dbse, '59-1.05'               )] =   -2.760
BIND['%s-%s'            % (dbse, '59-1.1'                )] =   -2.600
BIND['%s-%s'            % (dbse, '59-1.25'               )] =   -1.980
BIND['%s-%s'            % (dbse, '59-1.5'                )] =   -1.170
BIND['%s-%s'            % (dbse, '59-2.0'                )] =   -0.460
BIND['%s-%s'            % (dbse, '60-0.9'                )] =   -4.280
BIND['%s-%s'            % (dbse, '60-0.95'               )] =   -4.780
BIND['%s-%s'            % (dbse, '60-1.0'                )] =   -4.860
BIND['%s-%s'            % (dbse, '60-1.05'               )] =   -4.690
BIND['%s-%s'            % (dbse, '60-1.1'                )] =   -4.390
BIND['%s-%s'            % (dbse, '60-1.25'               )] =   -3.260
BIND['%s-%s'            % (dbse, '60-1.5'                )] =   -1.770
BIND['%s-%s'            % (dbse, '60-2.0'                )] =   -0.560
BIND['%s-%s'            % (dbse, '61-0.9'                )] =   -2.700
BIND['%s-%s'            % (dbse, '61-0.95'               )] =   -2.900
BIND['%s-%s'            % (dbse, '61-1.0'                )] =   -2.880
BIND['%s-%s'            % (dbse, '61-1.05'               )] =   -2.730
BIND['%s-%s'            % (dbse, '61-1.1'                )] =   -2.520
BIND['%s-%s'            % (dbse, '61-1.25'               )] =   -1.790
BIND['%s-%s'            % (dbse, '61-1.5'                )] =   -0.780
BIND['%s-%s'            % (dbse, '61-2.0'                )] =   -0.170
BIND['%s-%s'            % (dbse, '62-0.9'                )] =   -3.150
BIND['%s-%s'            % (dbse, '62-0.95'               )] =   -3.500
BIND['%s-%s'            % (dbse, '62-1.0'                )] =   -3.510
BIND['%s-%s'            % (dbse, '62-1.05'               )] =   -3.320
BIND['%s-%s'            % (dbse, '62-1.1'                )] =   -3.040
BIND['%s-%s'            % (dbse, '62-1.25'               )] =   -2.120
BIND['%s-%s'            % (dbse, '62-1.5'                )] =   -1.040
BIND['%s-%s'            % (dbse, '62-2.0'                )] =   -0.270
BIND['%s-%s'            % (dbse, '63-0.9'                )] =   -2.730
BIND['%s-%s'            % (dbse, '63-0.95'               )] =   -3.600
BIND['%s-%s'            % (dbse, '63-1.0'                )] =   -3.800
BIND['%s-%s'            % (dbse, '63-1.05'               )] =   -3.650
BIND['%s-%s'            % (dbse, '63-1.1'                )] =   -3.340
BIND['%s-%s'            % (dbse, '63-1.25'               )] =   -2.240
BIND['%s-%s'            % (dbse, '63-1.5'                )] =   -1.020
BIND['%s-%s'            % (dbse, '63-2.0'                )] =   -0.260
BIND['%s-%s'            % (dbse, '64-0.9'                )] =   -2.600
BIND['%s-%s'            % (dbse, '64-0.95'               )] =   -2.950
BIND['%s-%s'            % (dbse, '64-1.0'                )] =   -2.990
BIND['%s-%s'            % (dbse, '64-1.05'               )] =   -2.850
BIND['%s-%s'            % (dbse, '64-1.1'                )] =   -2.630
BIND['%s-%s'            % (dbse, '64-1.25'               )] =   -1.860
BIND['%s-%s'            % (dbse, '64-1.5'                )] =   -0.880
BIND['%s-%s'            % (dbse, '64-2.0'                )] =   -0.190
BIND['%s-%s'            % (dbse, '65-0.9'                )] =   -3.540
BIND['%s-%s'            % (dbse, '65-0.95'               )] =   -3.910
BIND['%s-%s'            % (dbse, '65-1.0'                )] =   -3.990
BIND['%s-%s'            % (dbse, '65-1.05'               )] =   -3.890
BIND['%s-%s'            % (dbse, '65-1.1'                )] =   -3.680
BIND['%s-%s'            % (dbse, '65-1.25'               )] =   -2.840
BIND['%s-%s'            % (dbse, '65-1.5'                )] =   -1.670
BIND['%s-%s'            % (dbse, '65-2.0'                )] =   -0.620
BIND['%s-%s'            % (dbse, '66-0.9'                )] =   -3.450
BIND['%s-%s'            % (dbse, '66-0.95'               )] =   -3.880
BIND['%s-%s'            % (dbse, '66-1.0'                )] =   -3.970
BIND['%s-%s'            % (dbse, '66-1.05'               )] =   -3.850
BIND['%s-%s'            % (dbse, '66-1.1'                )] =   -3.620
BIND['%s-%s'            % (dbse, '66-1.25'               )] =   -2.710
BIND['%s-%s'            % (dbse, '66-1.5'                )] =   -1.500
BIND['%s-%s'            % (dbse, '66-2.0'                )] =   -0.490

# <<< Comment Lines >>>
TAGL = {}
TAGL['%s-%s'            % (dbse, '1-0.9'                 )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-0.9'                 )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-0.9'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-0.9'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-0.9'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-0.9'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-0.95'                )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-0.95'                )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-0.95'                )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-0.95'                )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-0.95'                )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-0.95'                )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-1.0'                 )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-1.0'                 )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-1.0'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-1.0'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-1.0'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-1.0'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-1.05'                )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-1.05'                )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-1.05'                )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-1.05'                )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-1.05'                )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-1.05'                )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-1.1'                 )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-1.1'                 )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-1.1'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-1.1'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-1.1'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-1.1'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-1.25'                )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-1.25'                )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-1.25'                )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-1.25'                )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-1.25'                )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-1.25'                )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-1.5'                 )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-1.5'                 )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-1.5'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-1.5'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-1.5'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-1.5'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '1-2.0'                 )] = """Water ... Water """
TAGL['%s-%s-dimer'      % (dbse, '1-2.0'                 )] = """Dimer from Water ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '1-2.0'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '1-2.0'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '1-2.0'                 )] = """Monomer A from Water ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '1-2.0'                 )] = """Monomer B from Water ... Water """
TAGL['%s-%s'            % (dbse, '2-0.9'                 )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-0.9'                 )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-0.9'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-0.9'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-0.9'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-0.9'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-0.95'                )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-0.95'                )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-0.95'                )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-0.95'                )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-0.95'                )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-0.95'                )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-1.0'                 )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-1.0'                 )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-1.0'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-1.0'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-1.0'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-1.0'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-1.05'                )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-1.05'                )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-1.05'                )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-1.05'                )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-1.05'                )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-1.05'                )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-1.1'                 )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-1.1'                 )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-1.1'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-1.1'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-1.1'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-1.1'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-1.25'                )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-1.25'                )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-1.25'                )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-1.25'                )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-1.25'                )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-1.25'                )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-1.5'                 )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-1.5'                 )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-1.5'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-1.5'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-1.5'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-1.5'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '2-2.0'                 )] = """Water ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '2-2.0'                 )] = """Dimer from Water ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '2-2.0'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '2-2.0'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '2-2.0'                 )] = """Monomer A from Water ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '2-2.0'                 )] = """Monomer B from Water ... MeOH """
TAGL['%s-%s'            % (dbse, '3-0.9'                 )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-0.9'                 )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-0.9'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-0.9'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-0.9'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-0.9'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-0.95'                )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-0.95'                )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-0.95'                )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-0.95'                )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-0.95'                )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-0.95'                )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-1.0'                 )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-1.0'                 )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-1.0'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-1.0'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-1.0'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-1.0'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-1.05'                )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-1.05'                )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-1.05'                )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-1.05'                )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-1.05'                )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-1.05'                )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-1.1'                 )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-1.1'                 )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-1.1'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-1.1'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-1.1'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-1.1'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-1.25'                )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-1.25'                )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-1.25'                )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-1.25'                )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-1.25'                )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-1.25'                )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-1.5'                 )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-1.5'                 )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-1.5'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-1.5'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-1.5'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-1.5'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '3-2.0'                 )] = """Water ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '3-2.0'                 )] = """Dimer from Water ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '3-2.0'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '3-2.0'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '3-2.0'                 )] = """Monomer A from Water ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '3-2.0'                 )] = """Monomer B from Water ... MeNH2 """
TAGL['%s-%s'            % (dbse, '4-0.9'                 )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-0.9'                 )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-0.9'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-0.9'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-0.9'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-0.9'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-0.95'                )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-0.95'                )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-0.95'                )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-0.95'                )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-0.95'                )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-0.95'                )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-1.0'                 )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-1.0'                 )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-1.0'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-1.0'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-1.0'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-1.0'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-1.05'                )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-1.05'                )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-1.05'                )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-1.05'                )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-1.05'                )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-1.05'                )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-1.1'                 )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-1.1'                 )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-1.1'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-1.1'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-1.1'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-1.1'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-1.25'                )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-1.25'                )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-1.25'                )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-1.25'                )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-1.25'                )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-1.25'                )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-1.5'                 )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-1.5'                 )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-1.5'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-1.5'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-1.5'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-1.5'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '4-2.0'                 )] = """Water ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '4-2.0'                 )] = """Dimer from Water ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '4-2.0'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '4-2.0'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '4-2.0'                 )] = """Monomer A from Water ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '4-2.0'                 )] = """Monomer B from Water ... Peptide """
TAGL['%s-%s'            % (dbse, '5-0.9'                 )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-0.9'                 )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-0.9'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-0.9'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-0.9'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-0.9'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-0.95'                )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-0.95'                )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-0.95'                )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-0.95'                )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-0.95'                )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-0.95'                )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-1.0'                 )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-1.0'                 )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-1.0'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-1.0'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-1.0'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-1.0'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-1.05'                )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-1.05'                )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-1.05'                )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-1.05'                )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-1.05'                )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-1.05'                )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-1.1'                 )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-1.1'                 )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-1.1'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-1.1'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-1.1'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-1.1'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-1.25'                )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-1.25'                )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-1.25'                )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-1.25'                )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-1.25'                )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-1.25'                )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-1.5'                 )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-1.5'                 )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-1.5'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-1.5'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-1.5'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-1.5'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '5-2.0'                 )] = """MeOH ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '5-2.0'                 )] = """Dimer from MeOH ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '5-2.0'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '5-2.0'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '5-2.0'                 )] = """Monomer A from MeOH ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '5-2.0'                 )] = """Monomer B from MeOH ... MeOH """
TAGL['%s-%s'            % (dbse, '6-0.9'                 )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-0.9'                 )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-0.9'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-0.9'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-0.9'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-0.9'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-0.95'                )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-0.95'                )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-0.95'                )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-0.95'                )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-0.95'                )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-0.95'                )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-1.0'                 )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-1.0'                 )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-1.0'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-1.0'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-1.0'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-1.0'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-1.05'                )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-1.05'                )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-1.05'                )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-1.05'                )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-1.05'                )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-1.05'                )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-1.1'                 )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-1.1'                 )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-1.1'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-1.1'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-1.1'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-1.1'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-1.25'                )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-1.25'                )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-1.25'                )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-1.25'                )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-1.25'                )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-1.25'                )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-1.5'                 )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-1.5'                 )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-1.5'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-1.5'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-1.5'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-1.5'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '6-2.0'                 )] = """MeOH ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '6-2.0'                 )] = """Dimer from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '6-2.0'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '6-2.0'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '6-2.0'                 )] = """Monomer A from MeOH ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '6-2.0'                 )] = """Monomer B from MeOH ... MeNH2 """
TAGL['%s-%s'            % (dbse, '7-0.9'                 )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-0.9'                 )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-0.9'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-0.9'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-0.9'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-0.9'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-0.95'                )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-0.95'                )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-0.95'                )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-0.95'                )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-0.95'                )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-0.95'                )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-1.0'                 )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-1.0'                 )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-1.0'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-1.0'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-1.0'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-1.0'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-1.05'                )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-1.05'                )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-1.05'                )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-1.05'                )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-1.05'                )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-1.05'                )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-1.1'                 )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-1.1'                 )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-1.1'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-1.1'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-1.1'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-1.1'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-1.25'                )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-1.25'                )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-1.25'                )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-1.25'                )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-1.25'                )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-1.25'                )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-1.5'                 )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-1.5'                 )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-1.5'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-1.5'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-1.5'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-1.5'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '7-2.0'                 )] = """MeOH ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '7-2.0'                 )] = """Dimer from MeOH ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '7-2.0'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '7-2.0'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '7-2.0'                 )] = """Monomer A from MeOH ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '7-2.0'                 )] = """Monomer B from MeOH ... Peptide """
TAGL['%s-%s'            % (dbse, '8-0.9'                 )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-0.9'                 )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-0.9'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-0.9'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-0.9'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-0.9'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-0.95'                )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-0.95'                )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-0.95'                )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-0.95'                )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-0.95'                )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-0.95'                )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-1.0'                 )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-1.0'                 )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-1.0'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-1.0'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-1.0'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-1.0'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-1.05'                )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-1.05'                )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-1.05'                )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-1.05'                )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-1.05'                )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-1.05'                )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-1.1'                 )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-1.1'                 )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-1.1'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-1.1'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-1.1'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-1.1'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-1.25'                )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-1.25'                )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-1.25'                )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-1.25'                )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-1.25'                )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-1.25'                )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-1.5'                 )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-1.5'                 )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-1.5'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-1.5'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-1.5'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-1.5'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '8-2.0'                 )] = """MeOH ... Water """
TAGL['%s-%s-dimer'      % (dbse, '8-2.0'                 )] = """Dimer from MeOH ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '8-2.0'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '8-2.0'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '8-2.0'                 )] = """Monomer A from MeOH ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '8-2.0'                 )] = """Monomer B from MeOH ... Water """
TAGL['%s-%s'            % (dbse, '9-0.9'                 )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-0.9'                 )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-0.9'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-0.9'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-0.9'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-0.9'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-0.95'                )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-0.95'                )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-0.95'                )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-0.95'                )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-0.95'                )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-0.95'                )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-1.0'                 )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-1.0'                 )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-1.0'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-1.0'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-1.0'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-1.0'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-1.05'                )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-1.05'                )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-1.05'                )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-1.05'                )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-1.05'                )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-1.05'                )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-1.1'                 )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-1.1'                 )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-1.1'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-1.1'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-1.1'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-1.1'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-1.25'                )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-1.25'                )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-1.25'                )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-1.25'                )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-1.25'                )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-1.25'                )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-1.5'                 )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-1.5'                 )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-1.5'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-1.5'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-1.5'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-1.5'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '9-2.0'                 )] = """MeNH2 ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '9-2.0'                 )] = """Dimer from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '9-2.0'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '9-2.0'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '9-2.0'                 )] = """Monomer A from MeNH2 ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '9-2.0'                 )] = """Monomer B from MeNH2 ... MeOH """
TAGL['%s-%s'            % (dbse, '10-0.9'                )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-0.9'                )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-0.9'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-0.9'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-0.9'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-0.9'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-0.95'               )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-0.95'               )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-0.95'               )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-0.95'               )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-0.95'               )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-0.95'               )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-1.0'                )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-1.0'                )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-1.0'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-1.0'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-1.0'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-1.0'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-1.05'               )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-1.05'               )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-1.05'               )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-1.05'               )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-1.05'               )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-1.05'               )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-1.1'                )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-1.1'                )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-1.1'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-1.1'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-1.1'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-1.1'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-1.25'               )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-1.25'               )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-1.25'               )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-1.25'               )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-1.25'               )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-1.25'               )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-1.5'                )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-1.5'                )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-1.5'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-1.5'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-1.5'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-1.5'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '10-2.0'                )] = """MeNH2 ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '10-2.0'                )] = """Dimer from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '10-2.0'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '10-2.0'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '10-2.0'                )] = """Monomer A from MeNH2 ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '10-2.0'                )] = """Monomer B from MeNH2 ... MeNH2 """
TAGL['%s-%s'            % (dbse, '11-0.9'                )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-0.9'                )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-0.9'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-0.9'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-0.9'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-0.9'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-0.95'               )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-0.95'               )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-0.95'               )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-0.95'               )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-0.95'               )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-0.95'               )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-1.0'                )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-1.0'                )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-1.0'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-1.0'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-1.0'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-1.0'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-1.05'               )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-1.05'               )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-1.05'               )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-1.05'               )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-1.05'               )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-1.05'               )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-1.1'                )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-1.1'                )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-1.1'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-1.1'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-1.1'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-1.1'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-1.25'               )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-1.25'               )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-1.25'               )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-1.25'               )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-1.25'               )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-1.25'               )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-1.5'                )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-1.5'                )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-1.5'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-1.5'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-1.5'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-1.5'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '11-2.0'                )] = """MeNH2 ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '11-2.0'                )] = """Dimer from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '11-2.0'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '11-2.0'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '11-2.0'                )] = """Monomer A from MeNH2 ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '11-2.0'                )] = """Monomer B from MeNH2 ... Peptide """
TAGL['%s-%s'            % (dbse, '12-0.9'                )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-0.9'                )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-0.9'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-0.9'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-0.9'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-0.9'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-0.95'               )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-0.95'               )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-0.95'               )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-0.95'               )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-0.95'               )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-0.95'               )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-1.0'                )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-1.0'                )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-1.0'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-1.0'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-1.0'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-1.0'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-1.05'               )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-1.05'               )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-1.05'               )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-1.05'               )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-1.05'               )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-1.05'               )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-1.1'                )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-1.1'                )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-1.1'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-1.1'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-1.1'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-1.1'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-1.25'               )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-1.25'               )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-1.25'               )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-1.25'               )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-1.25'               )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-1.25'               )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-1.5'                )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-1.5'                )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-1.5'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-1.5'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-1.5'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-1.5'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '12-2.0'                )] = """MeNH2 ... Water """
TAGL['%s-%s-dimer'      % (dbse, '12-2.0'                )] = """Dimer from MeNH2 ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '12-2.0'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '12-2.0'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '12-2.0'                )] = """Monomer A from MeNH2 ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '12-2.0'                )] = """Monomer B from MeNH2 ... Water """
TAGL['%s-%s'            % (dbse, '13-0.9'                )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-0.9'                )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-0.9'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-0.9'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-0.9'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-0.9'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-0.95'               )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-0.95'               )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-0.95'               )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-0.95'               )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-0.95'               )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-0.95'               )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-1.0'                )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-1.0'                )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-1.0'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-1.0'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-1.0'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-1.0'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-1.05'               )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-1.05'               )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-1.05'               )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-1.05'               )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-1.05'               )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-1.05'               )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-1.1'                )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-1.1'                )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-1.1'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-1.1'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-1.1'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-1.1'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-1.25'               )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-1.25'               )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-1.25'               )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-1.25'               )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-1.25'               )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-1.25'               )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-1.5'                )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-1.5'                )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-1.5'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-1.5'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-1.5'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-1.5'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '13-2.0'                )] = """Peptide ... MeOH """
TAGL['%s-%s-dimer'      % (dbse, '13-2.0'                )] = """Dimer from Peptide ... MeOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '13-2.0'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '13-2.0'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '13-2.0'                )] = """Monomer A from Peptide ... MeOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '13-2.0'                )] = """Monomer B from Peptide ... MeOH """
TAGL['%s-%s'            % (dbse, '14-0.9'                )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-0.9'                )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-0.9'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-0.9'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-0.9'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-0.9'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-0.95'               )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-0.95'               )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-0.95'               )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-0.95'               )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-0.95'               )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-0.95'               )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-1.0'                )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-1.0'                )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-1.0'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-1.0'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-1.0'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-1.0'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-1.05'               )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-1.05'               )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-1.05'               )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-1.05'               )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-1.05'               )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-1.05'               )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-1.1'                )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-1.1'                )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-1.1'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-1.1'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-1.1'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-1.1'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-1.25'               )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-1.25'               )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-1.25'               )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-1.25'               )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-1.25'               )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-1.25'               )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-1.5'                )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-1.5'                )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-1.5'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-1.5'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-1.5'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-1.5'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '14-2.0'                )] = """Peptide ... MeNH2 """
TAGL['%s-%s-dimer'      % (dbse, '14-2.0'                )] = """Dimer from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '14-2.0'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '14-2.0'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '14-2.0'                )] = """Monomer A from Peptide ... MeNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '14-2.0'                )] = """Monomer B from Peptide ... MeNH2 """
TAGL['%s-%s'            % (dbse, '15-0.9'                )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-0.9'                )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-0.9'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-0.9'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-0.9'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-0.9'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-0.95'               )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-0.95'               )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-0.95'               )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-0.95'               )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-0.95'               )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-0.95'               )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-1.0'                )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-1.0'                )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-1.0'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-1.0'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-1.0'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-1.0'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-1.05'               )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-1.05'               )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-1.05'               )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-1.05'               )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-1.05'               )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-1.05'               )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-1.1'                )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-1.1'                )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-1.1'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-1.1'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-1.1'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-1.1'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-1.25'               )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-1.25'               )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-1.25'               )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-1.25'               )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-1.25'               )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-1.25'               )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-1.5'                )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-1.5'                )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-1.5'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-1.5'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-1.5'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-1.5'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '15-2.0'                )] = """Peptide ... Peptide """
TAGL['%s-%s-dimer'      % (dbse, '15-2.0'                )] = """Dimer from Peptide ... Peptide """
TAGL['%s-%s-monoA-CP'   % (dbse, '15-2.0'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-CP'   % (dbse, '15-2.0'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s-monoA-unCP' % (dbse, '15-2.0'                )] = """Monomer A from Peptide ... Peptide """
TAGL['%s-%s-monoB-unCP' % (dbse, '15-2.0'                )] = """Monomer B from Peptide ... Peptide """
TAGL['%s-%s'            % (dbse, '16-0.9'                )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-0.9'                )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-0.9'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-0.9'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-0.9'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-0.9'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-0.95'               )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-0.95'               )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-0.95'               )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-0.95'               )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-0.95'               )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-0.95'               )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-1.0'                )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-1.0'                )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-1.0'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-1.0'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-1.0'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-1.0'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-1.05'               )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-1.05'               )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-1.05'               )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-1.05'               )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-1.05'               )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-1.05'               )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-1.1'                )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-1.1'                )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-1.1'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-1.1'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-1.1'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-1.1'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-1.25'               )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-1.25'               )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-1.25'               )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-1.25'               )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-1.25'               )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-1.25'               )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-1.5'                )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-1.5'                )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-1.5'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-1.5'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-1.5'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-1.5'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '16-2.0'                )] = """Peptide ... Water """
TAGL['%s-%s-dimer'      % (dbse, '16-2.0'                )] = """Dimer from Peptide ... Water """
TAGL['%s-%s-monoA-CP'   % (dbse, '16-2.0'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-CP'   % (dbse, '16-2.0'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s-monoA-unCP' % (dbse, '16-2.0'                )] = """Monomer A from Peptide ... Water """
TAGL['%s-%s-monoB-unCP' % (dbse, '16-2.0'                )] = """Monomer B from Peptide ... Water """
TAGL['%s-%s'            % (dbse, '17-0.9'                )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-0.9'                )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-0.9'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-0.9'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-0.9'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-0.9'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-0.95'               )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-0.95'               )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-0.95'               )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-0.95'               )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-0.95'               )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-0.95'               )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-1.0'                )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-1.0'                )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-1.0'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-1.0'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-1.0'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-1.0'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-1.05'               )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-1.05'               )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-1.05'               )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-1.05'               )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-1.05'               )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-1.05'               )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-1.1'                )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-1.1'                )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-1.1'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-1.1'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-1.1'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-1.1'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-1.25'               )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-1.25'               )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-1.25'               )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-1.25'               )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-1.25'               )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-1.25'               )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-1.5'                )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-1.5'                )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-1.5'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-1.5'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-1.5'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-1.5'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '17-2.0'                )] = """Uracil ... Uracil (BP) """
TAGL['%s-%s-dimer'      % (dbse, '17-2.0'                )] = """Dimer from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-CP'   % (dbse, '17-2.0'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-CP'   % (dbse, '17-2.0'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoA-unCP' % (dbse, '17-2.0'                )] = """Monomer A from Uracil ... Uracil (BP) """
TAGL['%s-%s-monoB-unCP' % (dbse, '17-2.0'                )] = """Monomer B from Uracil ... Uracil (BP) """
TAGL['%s-%s'            % (dbse, '18-0.9'                )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-0.9'                )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-0.9'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-0.9'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-0.9'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-0.9'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-0.95'               )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-0.95'               )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-0.95'               )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-0.95'               )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-0.95'               )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-0.95'               )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-1.0'                )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-1.0'                )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-1.0'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-1.0'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-1.0'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-1.0'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-1.05'               )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-1.05'               )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-1.05'               )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-1.05'               )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-1.05'               )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-1.05'               )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-1.1'                )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-1.1'                )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-1.1'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-1.1'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-1.1'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-1.1'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-1.25'               )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-1.25'               )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-1.25'               )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-1.25'               )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-1.25'               )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-1.25'               )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-1.5'                )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-1.5'                )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-1.5'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-1.5'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-1.5'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-1.5'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '18-2.0'                )] = """Water ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '18-2.0'                )] = """Dimer from Water ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '18-2.0'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '18-2.0'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '18-2.0'                )] = """Monomer A from Water ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '18-2.0'                )] = """Monomer B from Water ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-0.9'                )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-0.9'                )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-0.9'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-0.9'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-0.9'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-0.9'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-0.95'               )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-0.95'               )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-0.95'               )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-0.95'               )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-0.95'               )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-0.95'               )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-1.0'                )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-1.0'                )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-1.0'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-1.0'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-1.0'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-1.0'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-1.05'               )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-1.05'               )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-1.05'               )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-1.05'               )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-1.05'               )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-1.05'               )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-1.1'                )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-1.1'                )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-1.1'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-1.1'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-1.1'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-1.1'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-1.25'               )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-1.25'               )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-1.25'               )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-1.25'               )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-1.25'               )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-1.25'               )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-1.5'                )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-1.5'                )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-1.5'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-1.5'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-1.5'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-1.5'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '19-2.0'                )] = """MeOH ... Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '19-2.0'                )] = """Dimer from MeOH ... Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '19-2.0'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '19-2.0'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '19-2.0'                )] = """Monomer A from MeOH ... Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '19-2.0'                )] = """Monomer B from MeOH ... Pyridine """
TAGL['%s-%s'            % (dbse, '20-0.9'                )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-0.9'                )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-0.9'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-0.9'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-0.9'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-0.9'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-0.95'               )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-0.95'               )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-0.95'               )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-0.95'               )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-0.95'               )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-0.95'               )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-1.0'                )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-1.0'                )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-1.0'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-1.0'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-1.0'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-1.0'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-1.05'               )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-1.05'               )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-1.05'               )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-1.05'               )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-1.05'               )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-1.05'               )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-1.1'                )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-1.1'                )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-1.1'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-1.1'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-1.1'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-1.1'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-1.25'               )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-1.25'               )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-1.25'               )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-1.25'               )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-1.25'               )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-1.25'               )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-1.5'                )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-1.5'                )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-1.5'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-1.5'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-1.5'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-1.5'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '20-2.0'                )] = """AcOH ... AcOH """
TAGL['%s-%s-dimer'      % (dbse, '20-2.0'                )] = """Dimer from AcOH ... AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '20-2.0'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '20-2.0'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '20-2.0'                )] = """Monomer A from AcOH ... AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '20-2.0'                )] = """Monomer B from AcOH ... AcOH """
TAGL['%s-%s'            % (dbse, '21-0.9'                )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-0.9'                )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-0.9'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-0.9'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-0.9'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-0.9'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-0.95'               )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-0.95'               )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-0.95'               )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-0.95'               )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-0.95'               )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-0.95'               )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-1.0'                )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-1.0'                )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-1.0'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-1.0'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-1.0'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-1.0'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-1.05'               )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-1.05'               )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-1.05'               )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-1.05'               )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-1.05'               )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-1.05'               )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-1.1'                )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-1.1'                )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-1.1'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-1.1'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-1.1'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-1.1'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-1.25'               )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-1.25'               )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-1.25'               )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-1.25'               )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-1.25'               )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-1.25'               )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-1.5'                )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-1.5'                )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-1.5'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-1.5'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-1.5'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-1.5'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '21-2.0'                )] = """AcNH2 ... AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '21-2.0'                )] = """Dimer from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '21-2.0'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '21-2.0'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '21-2.0'                )] = """Monomer A from AcNH2 ... AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '21-2.0'                )] = """Monomer B from AcNH2 ... AcNH2 """
TAGL['%s-%s'            % (dbse, '22-0.9'                )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-0.9'                )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-0.9'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-0.9'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-0.9'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-0.9'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-0.95'               )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-0.95'               )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-0.95'               )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-0.95'               )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-0.95'               )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-0.95'               )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-1.0'                )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-1.0'                )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-1.0'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-1.0'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-1.0'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-1.0'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-1.05'               )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-1.05'               )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-1.05'               )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-1.05'               )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-1.05'               )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-1.05'               )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-1.1'                )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-1.1'                )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-1.1'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-1.1'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-1.1'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-1.1'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-1.25'               )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-1.25'               )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-1.25'               )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-1.25'               )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-1.25'               )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-1.25'               )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-1.5'                )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-1.5'                )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-1.5'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-1.5'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-1.5'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-1.5'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '22-2.0'                )] = """AcOH ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '22-2.0'                )] = """Dimer from AcOH ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '22-2.0'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '22-2.0'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '22-2.0'                )] = """Monomer A from AcOH ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '22-2.0'                )] = """Monomer B from AcOH ... Uracil """
TAGL['%s-%s'            % (dbse, '23-0.9'                )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-0.9'                )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-0.9'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-0.9'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-0.9'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-0.9'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-0.95'               )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-0.95'               )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-0.95'               )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-0.95'               )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-0.95'               )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-0.95'               )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-1.0'                )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-1.0'                )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-1.0'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-1.0'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-1.0'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-1.0'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-1.05'               )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-1.05'               )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-1.05'               )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-1.05'               )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-1.05'               )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-1.05'               )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-1.1'                )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-1.1'                )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-1.1'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-1.1'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-1.1'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-1.1'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-1.25'               )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-1.25'               )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-1.25'               )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-1.25'               )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-1.25'               )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-1.25'               )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-1.5'                )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-1.5'                )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-1.5'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-1.5'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-1.5'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-1.5'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '23-2.0'                )] = """AcNH2 ... Uracil """
TAGL['%s-%s-dimer'      % (dbse, '23-2.0'                )] = """Dimer from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-CP'   % (dbse, '23-2.0'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-CP'   % (dbse, '23-2.0'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s-monoA-unCP' % (dbse, '23-2.0'                )] = """Monomer A from AcNH2 ... Uracil """
TAGL['%s-%s-monoB-unCP' % (dbse, '23-2.0'                )] = """Monomer B from AcNH2 ... Uracil """
TAGL['%s-%s'            % (dbse, '24-0.9'                )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-0.9'                )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-0.9'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-0.9'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-0.9'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-0.9'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-0.95'               )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-0.95'               )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-0.95'               )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-0.95'               )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-0.95'               )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-0.95'               )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-1.0'                )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-1.0'                )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-1.0'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-1.0'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-1.0'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-1.0'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-1.05'               )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-1.05'               )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-1.05'               )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-1.05'               )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-1.05'               )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-1.05'               )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-1.1'                )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-1.1'                )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-1.1'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-1.1'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-1.1'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-1.1'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-1.25'               )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-1.25'               )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-1.25'               )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-1.25'               )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-1.25'               )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-1.25'               )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-1.5'                )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-1.5'                )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-1.5'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-1.5'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-1.5'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-1.5'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '24-2.0'                )] = """Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '24-2.0'                )] = """Dimer from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '24-2.0'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '24-2.0'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '24-2.0'                )] = """Monomer A from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '24-2.0'                )] = """Monomer B from Benzene ...  Benzene (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-0.9'                )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-0.9'                )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-0.9'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-0.9'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-0.9'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-0.9'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-0.95'               )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-0.95'               )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-0.95'               )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-0.95'               )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-0.95'               )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-0.95'               )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-1.0'                )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-1.0'                )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-1.0'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-1.0'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-1.0'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-1.0'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-1.05'               )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-1.05'               )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-1.05'               )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-1.05'               )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-1.05'               )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-1.05'               )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-1.1'                )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-1.1'                )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-1.1'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-1.1'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-1.1'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-1.1'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-1.25'               )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-1.25'               )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-1.25'               )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-1.25'               )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-1.25'               )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-1.25'               )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-1.5'                )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-1.5'                )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-1.5'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-1.5'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-1.5'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-1.5'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '25-2.0'                )] = """Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '25-2.0'                )] = """Dimer from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '25-2.0'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '25-2.0'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '25-2.0'                )] = """Monomer A from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '25-2.0'                )] = """Monomer B from Pyridine ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-0.9'                )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-0.9'                )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-0.9'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-0.9'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-0.9'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-0.9'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-0.95'               )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-0.95'               )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-0.95'               )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-0.95'               )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-0.95'               )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-0.95'               )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-1.0'                )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-1.0'                )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-1.0'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-1.0'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-1.0'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-1.0'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-1.05'               )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-1.05'               )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-1.05'               )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-1.05'               )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-1.05'               )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-1.05'               )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-1.1'                )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-1.1'                )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-1.1'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-1.1'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-1.1'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-1.1'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-1.25'               )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-1.25'               )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-1.25'               )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-1.25'               )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-1.25'               )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-1.25'               )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-1.5'                )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-1.5'                )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-1.5'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-1.5'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-1.5'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-1.5'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '26-2.0'                )] = """Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '26-2.0'                )] = """Dimer from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '26-2.0'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '26-2.0'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '26-2.0'                )] = """Monomer A from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '26-2.0'                )] = """Monomer B from Uracil ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-0.9'                )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-0.9'                )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-0.9'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-0.9'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-0.9'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-0.9'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-0.95'               )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-0.95'               )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-0.95'               )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-0.95'               )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-0.95'               )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-0.95'               )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-1.0'                )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-1.0'                )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-1.0'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-1.0'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-1.0'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-1.0'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-1.05'               )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-1.05'               )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-1.05'               )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-1.05'               )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-1.05'               )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-1.05'               )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-1.1'                )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-1.1'                )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-1.1'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-1.1'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-1.1'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-1.1'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-1.25'               )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-1.25'               )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-1.25'               )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-1.25'               )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-1.25'               )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-1.25'               )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-1.5'                )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-1.5'                )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-1.5'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-1.5'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-1.5'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-1.5'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '27-2.0'                )] = """Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '27-2.0'                )] = """Dimer from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '27-2.0'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '27-2.0'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '27-2.0'                )] = """Monomer A from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '27-2.0'                )] = """Monomer B from Benzene ...  Pyridine (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-0.9'                )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-0.9'                )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-0.9'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-0.9'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-0.9'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-0.9'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-0.95'               )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-0.95'               )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-0.95'               )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-0.95'               )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-0.95'               )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-0.95'               )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-1.0'                )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-1.0'                )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-1.0'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-1.0'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-1.0'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-1.0'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-1.05'               )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-1.05'               )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-1.05'               )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-1.05'               )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-1.05'               )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-1.05'               )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-1.1'                )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-1.1'                )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-1.1'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-1.1'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-1.1'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-1.1'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-1.25'               )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-1.25'               )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-1.25'               )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-1.25'               )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-1.25'               )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-1.25'               )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-1.5'                )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-1.5'                )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-1.5'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-1.5'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-1.5'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-1.5'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '28-2.0'                )] = """Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '28-2.0'                )] = """Dimer from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '28-2.0'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '28-2.0'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '28-2.0'                )] = """Monomer A from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '28-2.0'                )] = """Monomer B from Benzene ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-0.9'                )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-0.9'                )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-0.9'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-0.9'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-0.9'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-0.9'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-0.95'               )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-0.95'               )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-0.95'               )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-0.95'               )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-0.95'               )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-0.95'               )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-1.0'                )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-1.0'                )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-1.0'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-1.0'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-1.0'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-1.0'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-1.05'               )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-1.05'               )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-1.05'               )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-1.05'               )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-1.05'               )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-1.05'               )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-1.1'                )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-1.1'                )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-1.1'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-1.1'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-1.1'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-1.1'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-1.25'               )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-1.25'               )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-1.25'               )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-1.25'               )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-1.25'               )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-1.25'               )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-1.5'                )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-1.5'                )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-1.5'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-1.5'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-1.5'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-1.5'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '29-2.0'                )] = """Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-dimer'      % (dbse, '29-2.0'                )] = """Dimer from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '29-2.0'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '29-2.0'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '29-2.0'                )] = """Monomer A from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '29-2.0'                )] = """Monomer B from Pyridine ...  Uracil (pi-pi) """
TAGL['%s-%s'            % (dbse, '30-0.9'                )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-0.9'                )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-0.9'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-0.9'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-0.9'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-0.9'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-0.95'               )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-0.95'               )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-0.95'               )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-0.95'               )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-0.95'               )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-0.95'               )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-1.0'                )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-1.0'                )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-1.0'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-1.0'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-1.0'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-1.0'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-1.05'               )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-1.05'               )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-1.05'               )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-1.05'               )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-1.05'               )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-1.05'               )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-1.1'                )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-1.1'                )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-1.1'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-1.1'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-1.1'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-1.1'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-1.25'               )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-1.25'               )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-1.25'               )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-1.25'               )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-1.25'               )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-1.25'               )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-1.5'                )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-1.5'                )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-1.5'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-1.5'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-1.5'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-1.5'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '30-2.0'                )] = """Benzene ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '30-2.0'                )] = """Dimer from Benzene ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '30-2.0'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '30-2.0'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '30-2.0'                )] = """Monomer A from Benzene ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '30-2.0'                )] = """Monomer B from Benzene ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-0.9'                )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-0.9'                )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-0.9'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-0.9'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-0.9'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-0.9'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-0.95'               )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-0.95'               )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-0.95'               )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-0.95'               )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-0.95'               )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-0.95'               )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-1.0'                )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-1.0'                )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-1.0'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-1.0'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-1.0'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-1.0'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-1.05'               )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-1.05'               )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-1.05'               )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-1.05'               )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-1.05'               )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-1.05'               )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-1.1'                )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-1.1'                )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-1.1'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-1.1'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-1.1'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-1.1'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-1.25'               )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-1.25'               )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-1.25'               )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-1.25'               )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-1.25'               )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-1.25'               )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-1.5'                )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-1.5'                )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-1.5'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-1.5'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-1.5'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-1.5'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '31-2.0'                )] = """Uracil ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '31-2.0'                )] = """Dimer from Uracil ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '31-2.0'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '31-2.0'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '31-2.0'                )] = """Monomer A from Uracil ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '31-2.0'                )] = """Monomer B from Uracil ...  Ethene """
TAGL['%s-%s'            % (dbse, '32-0.9'                )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-0.9'                )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-0.9'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-0.9'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-0.9'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-0.9'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-0.95'               )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-0.95'               )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-0.95'               )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-0.95'               )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-0.95'               )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-0.95'               )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-1.0'                )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-1.0'                )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-1.0'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-1.0'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-1.0'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-1.0'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-1.05'               )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-1.05'               )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-1.05'               )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-1.05'               )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-1.05'               )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-1.05'               )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-1.1'                )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-1.1'                )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-1.1'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-1.1'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-1.1'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-1.1'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-1.25'               )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-1.25'               )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-1.25'               )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-1.25'               )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-1.25'               )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-1.25'               )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-1.5'                )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-1.5'                )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-1.5'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-1.5'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-1.5'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-1.5'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '32-2.0'                )] = """Uracil ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '32-2.0'                )] = """Dimer from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '32-2.0'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '32-2.0'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '32-2.0'                )] = """Monomer A from Uracil ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '32-2.0'                )] = """Monomer B from Uracil ...  Ethyne """
TAGL['%s-%s'            % (dbse, '33-0.9'                )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-0.9'                )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-0.9'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-0.9'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-0.9'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-0.9'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-0.95'               )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-0.95'               )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-0.95'               )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-0.95'               )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-0.95'               )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-0.95'               )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-1.0'                )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-1.0'                )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-1.0'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-1.0'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-1.0'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-1.0'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-1.05'               )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-1.05'               )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-1.05'               )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-1.05'               )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-1.05'               )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-1.05'               )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-1.1'                )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-1.1'                )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-1.1'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-1.1'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-1.1'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-1.1'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-1.25'               )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-1.25'               )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-1.25'               )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-1.25'               )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-1.25'               )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-1.25'               )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-1.5'                )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-1.5'                )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-1.5'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-1.5'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-1.5'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-1.5'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '33-2.0'                )] = """Pyridine ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '33-2.0'                )] = """Dimer from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '33-2.0'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '33-2.0'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '33-2.0'                )] = """Monomer A from Pyridine ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '33-2.0'                )] = """Monomer B from Pyridine ...  Ethene """
TAGL['%s-%s'            % (dbse, '34-0.9'                )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-0.9'                )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-0.9'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-0.9'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-0.9'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-0.9'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-0.95'               )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-0.95'               )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-0.95'               )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-0.95'               )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-0.95'               )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-0.95'               )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-1.0'                )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-1.0'                )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-1.0'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-1.0'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-1.0'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-1.0'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-1.05'               )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-1.05'               )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-1.05'               )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-1.05'               )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-1.05'               )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-1.05'               )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-1.1'                )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-1.1'                )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-1.1'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-1.1'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-1.1'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-1.1'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-1.25'               )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-1.25'               )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-1.25'               )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-1.25'               )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-1.25'               )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-1.25'               )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-1.5'                )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-1.5'                )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-1.5'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-1.5'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-1.5'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-1.5'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '34-2.0'                )] = """Pentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '34-2.0'                )] = """Dimer from Pentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '34-2.0'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '34-2.0'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '34-2.0'                )] = """Monomer A from Pentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '34-2.0'                )] = """Monomer B from Pentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-0.9'                )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-0.9'                )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-0.9'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-0.9'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-0.9'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-0.9'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-0.95'               )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-0.95'               )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-0.95'               )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-0.95'               )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-0.95'               )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-0.95'               )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-1.0'                )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-1.0'                )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-1.0'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-1.0'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-1.0'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-1.0'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-1.05'               )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-1.05'               )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-1.05'               )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-1.05'               )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-1.05'               )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-1.05'               )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-1.1'                )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-1.1'                )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-1.1'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-1.1'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-1.1'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-1.1'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-1.25'               )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-1.25'               )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-1.25'               )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-1.25'               )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-1.25'               )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-1.25'               )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-1.5'                )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-1.5'                )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-1.5'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-1.5'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-1.5'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-1.5'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '35-2.0'                )] = """Neopentane ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '35-2.0'                )] = """Dimer from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '35-2.0'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '35-2.0'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '35-2.0'                )] = """Monomer A from Neopentane ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '35-2.0'                )] = """Monomer B from Neopentane ...  Pentane """
TAGL['%s-%s'            % (dbse, '36-0.9'                )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-0.9'                )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-0.9'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-0.9'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-0.9'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-0.9'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-0.95'               )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-0.95'               )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-0.95'               )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-0.95'               )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-0.95'               )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-0.95'               )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-1.0'                )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-1.0'                )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-1.0'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-1.0'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-1.0'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-1.0'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-1.05'               )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-1.05'               )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-1.05'               )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-1.05'               )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-1.05'               )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-1.05'               )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-1.1'                )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-1.1'                )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-1.1'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-1.1'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-1.1'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-1.1'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-1.25'               )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-1.25'               )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-1.25'               )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-1.25'               )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-1.25'               )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-1.25'               )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-1.5'                )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-1.5'                )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-1.5'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-1.5'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-1.5'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-1.5'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '36-2.0'                )] = """Neopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '36-2.0'                )] = """Dimer from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '36-2.0'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '36-2.0'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '36-2.0'                )] = """Monomer A from Neopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '36-2.0'                )] = """Monomer B from Neopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-0.9'                )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-0.9'                )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-0.9'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-0.9'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-0.9'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-0.9'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-0.95'               )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-0.95'               )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-0.95'               )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-0.95'               )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-0.95'               )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-0.95'               )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-1.0'                )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-1.0'                )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-1.0'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-1.0'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-1.0'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-1.0'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-1.05'               )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-1.05'               )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-1.05'               )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-1.05'               )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-1.05'               )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-1.05'               )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-1.1'                )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-1.1'                )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-1.1'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-1.1'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-1.1'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-1.1'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-1.25'               )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-1.25'               )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-1.25'               )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-1.25'               )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-1.25'               )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-1.25'               )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-1.5'                )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-1.5'                )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-1.5'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-1.5'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-1.5'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-1.5'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '37-2.0'                )] = """Cyclopentane ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '37-2.0'                )] = """Dimer from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '37-2.0'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '37-2.0'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '37-2.0'                )] = """Monomer A from Cyclopentane ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '37-2.0'                )] = """Monomer B from Cyclopentane ...  Neopentane """
TAGL['%s-%s'            % (dbse, '38-0.9'                )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-0.9'                )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-0.9'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-0.9'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-0.9'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-0.9'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-0.95'               )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-0.95'               )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-0.95'               )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-0.95'               )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-0.95'               )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-0.95'               )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-1.0'                )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-1.0'                )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-1.0'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-1.0'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-1.0'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-1.0'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-1.05'               )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-1.05'               )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-1.05'               )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-1.05'               )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-1.05'               )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-1.05'               )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-1.1'                )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-1.1'                )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-1.1'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-1.1'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-1.1'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-1.1'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-1.25'               )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-1.25'               )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-1.25'               )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-1.25'               )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-1.25'               )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-1.25'               )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-1.5'                )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-1.5'                )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-1.5'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-1.5'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-1.5'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-1.5'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '38-2.0'                )] = """Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '38-2.0'                )] = """Dimer from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '38-2.0'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '38-2.0'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '38-2.0'                )] = """Monomer A from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '38-2.0'                )] = """Monomer B from Cyclopentane ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-0.9'                )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-0.9'                )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-0.9'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-0.9'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-0.9'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-0.9'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-0.95'               )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-0.95'               )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-0.95'               )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-0.95'               )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-0.95'               )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-0.95'               )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-1.0'                )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-1.0'                )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-1.0'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-1.0'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-1.0'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-1.0'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-1.05'               )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-1.05'               )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-1.05'               )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-1.05'               )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-1.05'               )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-1.05'               )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-1.1'                )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-1.1'                )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-1.1'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-1.1'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-1.1'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-1.1'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-1.25'               )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-1.25'               )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-1.25'               )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-1.25'               )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-1.25'               )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-1.25'               )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-1.5'                )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-1.5'                )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-1.5'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-1.5'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-1.5'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-1.5'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '39-2.0'                )] = """Benzene ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '39-2.0'                )] = """Dimer from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '39-2.0'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '39-2.0'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '39-2.0'                )] = """Monomer A from Benzene ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '39-2.0'                )] = """Monomer B from Benzene ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '40-0.9'                )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-0.9'                )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-0.9'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-0.9'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-0.9'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-0.9'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-0.95'               )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-0.95'               )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-0.95'               )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-0.95'               )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-0.95'               )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-0.95'               )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-1.0'                )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-1.0'                )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-1.0'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-1.0'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-1.0'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-1.0'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-1.05'               )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-1.05'               )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-1.05'               )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-1.05'               )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-1.05'               )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-1.05'               )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-1.1'                )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-1.1'                )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-1.1'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-1.1'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-1.1'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-1.1'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-1.25'               )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-1.25'               )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-1.25'               )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-1.25'               )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-1.25'               )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-1.25'               )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-1.5'                )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-1.5'                )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-1.5'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-1.5'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-1.5'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-1.5'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '40-2.0'                )] = """Benzene ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '40-2.0'                )] = """Dimer from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '40-2.0'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '40-2.0'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '40-2.0'                )] = """Monomer A from Benzene ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '40-2.0'                )] = """Monomer B from Benzene ...  Neopentane """
TAGL['%s-%s'            % (dbse, '41-0.9'                )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-0.9'                )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-0.9'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-0.9'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-0.9'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-0.9'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-0.95'               )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-0.95'               )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-0.95'               )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-0.95'               )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-0.95'               )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-0.95'               )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-1.0'                )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-1.0'                )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-1.0'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-1.0'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-1.0'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-1.0'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-1.05'               )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-1.05'               )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-1.05'               )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-1.05'               )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-1.05'               )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-1.05'               )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-1.1'                )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-1.1'                )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-1.1'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-1.1'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-1.1'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-1.1'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-1.25'               )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-1.25'               )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-1.25'               )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-1.25'               )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-1.25'               )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-1.25'               )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-1.5'                )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-1.5'                )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-1.5'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-1.5'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-1.5'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-1.5'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '41-2.0'                )] = """Uracil ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '41-2.0'                )] = """Dimer from Uracil ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '41-2.0'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '41-2.0'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '41-2.0'                )] = """Monomer A from Uracil ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '41-2.0'                )] = """Monomer B from Uracil ...  Pentane """
TAGL['%s-%s'            % (dbse, '42-0.9'                )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-0.9'                )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-0.9'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-0.9'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-0.9'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-0.9'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-0.95'               )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-0.95'               )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-0.95'               )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-0.95'               )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-0.95'               )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-0.95'               )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-1.0'                )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-1.0'                )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-1.0'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-1.0'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-1.0'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-1.0'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-1.05'               )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-1.05'               )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-1.05'               )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-1.05'               )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-1.05'               )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-1.05'               )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-1.1'                )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-1.1'                )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-1.1'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-1.1'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-1.1'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-1.1'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-1.25'               )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-1.25'               )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-1.25'               )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-1.25'               )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-1.25'               )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-1.25'               )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-1.5'                )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-1.5'                )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-1.5'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-1.5'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-1.5'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-1.5'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '42-2.0'                )] = """Uracil ...  Cyclopentane """
TAGL['%s-%s-dimer'      % (dbse, '42-2.0'                )] = """Dimer from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '42-2.0'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '42-2.0'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '42-2.0'                )] = """Monomer A from Uracil ...  Cyclopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '42-2.0'                )] = """Monomer B from Uracil ...  Cyclopentane """
TAGL['%s-%s'            % (dbse, '43-0.9'                )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-0.9'                )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-0.9'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-0.9'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-0.9'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-0.9'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-0.95'               )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-0.95'               )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-0.95'               )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-0.95'               )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-0.95'               )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-0.95'               )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-1.0'                )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-1.0'                )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-1.0'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-1.0'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-1.0'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-1.0'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-1.05'               )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-1.05'               )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-1.05'               )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-1.05'               )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-1.05'               )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-1.05'               )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-1.1'                )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-1.1'                )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-1.1'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-1.1'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-1.1'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-1.1'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-1.25'               )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-1.25'               )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-1.25'               )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-1.25'               )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-1.25'               )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-1.25'               )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-1.5'                )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-1.5'                )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-1.5'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-1.5'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-1.5'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-1.5'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '43-2.0'                )] = """Uracil ...  Neopentane """
TAGL['%s-%s-dimer'      % (dbse, '43-2.0'                )] = """Dimer from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '43-2.0'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '43-2.0'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '43-2.0'                )] = """Monomer A from Uracil ...  Neopentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '43-2.0'                )] = """Monomer B from Uracil ...  Neopentane """
TAGL['%s-%s'            % (dbse, '44-0.9'                )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-0.9'                )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-0.9'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-0.9'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-0.9'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-0.9'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-0.95'               )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-0.95'               )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-0.95'               )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-0.95'               )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-0.95'               )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-0.95'               )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-1.0'                )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-1.0'                )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-1.0'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-1.0'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-1.0'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-1.0'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-1.05'               )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-1.05'               )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-1.05'               )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-1.05'               )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-1.05'               )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-1.05'               )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-1.1'                )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-1.1'                )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-1.1'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-1.1'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-1.1'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-1.1'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-1.25'               )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-1.25'               )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-1.25'               )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-1.25'               )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-1.25'               )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-1.25'               )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-1.5'                )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-1.5'                )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-1.5'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-1.5'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-1.5'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-1.5'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '44-2.0'                )] = """Ethene ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '44-2.0'                )] = """Dimer from Ethene ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '44-2.0'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '44-2.0'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '44-2.0'                )] = """Monomer A from Ethene ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '44-2.0'                )] = """Monomer B from Ethene ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-0.9'                )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-0.9'                )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-0.9'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-0.9'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-0.9'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-0.9'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-0.95'               )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-0.95'               )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-0.95'               )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-0.95'               )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-0.95'               )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-0.95'               )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-1.0'                )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-1.0'                )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-1.0'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-1.0'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-1.0'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-1.0'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-1.05'               )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-1.05'               )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-1.05'               )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-1.05'               )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-1.05'               )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-1.05'               )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-1.1'                )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-1.1'                )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-1.1'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-1.1'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-1.1'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-1.1'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-1.25'               )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-1.25'               )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-1.25'               )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-1.25'               )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-1.25'               )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-1.25'               )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-1.5'                )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-1.5'                )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-1.5'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-1.5'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-1.5'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-1.5'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '45-2.0'                )] = """Ethyne ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '45-2.0'                )] = """Dimer from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '45-2.0'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '45-2.0'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '45-2.0'                )] = """Monomer A from Ethyne ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '45-2.0'                )] = """Monomer B from Ethyne ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-0.9'                )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-0.9'                )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-0.9'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-0.9'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-0.9'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-0.9'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-0.95'               )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-0.95'               )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-0.95'               )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-0.95'               )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-0.95'               )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-0.95'               )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-1.0'                )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-1.0'                )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-1.0'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-1.0'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-1.0'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-1.0'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-1.05'               )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-1.05'               )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-1.05'               )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-1.05'               )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-1.05'               )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-1.05'               )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-1.1'                )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-1.1'                )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-1.1'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-1.1'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-1.1'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-1.1'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-1.25'               )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-1.25'               )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-1.25'               )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-1.25'               )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-1.25'               )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-1.25'               )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-1.5'                )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-1.5'                )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-1.5'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-1.5'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-1.5'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-1.5'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '46-2.0'                )] = """Peptide ...  Pentane """
TAGL['%s-%s-dimer'      % (dbse, '46-2.0'                )] = """Dimer from Peptide ...  Pentane """
TAGL['%s-%s-monoA-CP'   % (dbse, '46-2.0'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-CP'   % (dbse, '46-2.0'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s-monoA-unCP' % (dbse, '46-2.0'                )] = """Monomer A from Peptide ...  Pentane """
TAGL['%s-%s-monoB-unCP' % (dbse, '46-2.0'                )] = """Monomer B from Peptide ...  Pentane """
TAGL['%s-%s'            % (dbse, '47-0.9'                )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-0.9'                )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-0.9'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-0.9'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-0.9'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-0.9'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-0.95'               )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-0.95'               )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-0.95'               )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-0.95'               )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-0.95'               )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-0.95'               )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-1.0'                )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-1.0'                )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-1.0'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-1.0'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-1.0'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-1.0'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-1.05'               )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-1.05'               )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-1.05'               )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-1.05'               )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-1.05'               )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-1.05'               )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-1.1'                )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-1.1'                )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-1.1'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-1.1'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-1.1'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-1.1'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-1.25'               )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-1.25'               )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-1.25'               )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-1.25'               )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-1.25'               )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-1.25'               )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-1.5'                )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-1.5'                )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-1.5'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-1.5'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-1.5'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-1.5'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '47-2.0'                )] = """Benzene ...  Benzene (TS) """
TAGL['%s-%s-dimer'      % (dbse, '47-2.0'                )] = """Dimer from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '47-2.0'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '47-2.0'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '47-2.0'                )] = """Monomer A from Benzene ...  Benzene (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '47-2.0'                )] = """Monomer B from Benzene ...  Benzene (TS) """
TAGL['%s-%s'            % (dbse, '48-0.9'                )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-0.9'                )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-0.9'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-0.9'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-0.9'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-0.9'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-0.95'               )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-0.95'               )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-0.95'               )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-0.95'               )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-0.95'               )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-0.95'               )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-1.0'                )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-1.0'                )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-1.0'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-1.0'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-1.0'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-1.0'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-1.05'               )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-1.05'               )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-1.05'               )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-1.05'               )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-1.05'               )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-1.05'               )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-1.1'                )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-1.1'                )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-1.1'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-1.1'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-1.1'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-1.1'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-1.25'               )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-1.25'               )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-1.25'               )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-1.25'               )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-1.25'               )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-1.25'               )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-1.5'                )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-1.5'                )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-1.5'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-1.5'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-1.5'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-1.5'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '48-2.0'                )] = """Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '48-2.0'                )] = """Dimer from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '48-2.0'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '48-2.0'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '48-2.0'                )] = """Monomer A from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '48-2.0'                )] = """Monomer B from Pyridine ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-0.9'                )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-0.9'                )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-0.9'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-0.9'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-0.9'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-0.9'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-0.95'               )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-0.95'               )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-0.95'               )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-0.95'               )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-0.95'               )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-0.95'               )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-1.0'                )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-1.0'                )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-1.0'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-1.0'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-1.0'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-1.0'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-1.05'               )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-1.05'               )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-1.05'               )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-1.05'               )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-1.05'               )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-1.05'               )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-1.1'                )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-1.1'                )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-1.1'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-1.1'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-1.1'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-1.1'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-1.25'               )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-1.25'               )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-1.25'               )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-1.25'               )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-1.25'               )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-1.25'               )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-1.5'                )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-1.5'                )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-1.5'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-1.5'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-1.5'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-1.5'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '49-2.0'                )] = """Benzene ...  Pyridine (TS) """
TAGL['%s-%s-dimer'      % (dbse, '49-2.0'                )] = """Dimer from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '49-2.0'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '49-2.0'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '49-2.0'                )] = """Monomer A from Benzene ...  Pyridine (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '49-2.0'                )] = """Monomer B from Benzene ...  Pyridine (TS) """
TAGL['%s-%s'            % (dbse, '50-0.9'                )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-0.9'                )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-0.9'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-0.9'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-0.9'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-0.9'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-0.95'               )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-0.95'               )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-0.95'               )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-0.95'               )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-0.95'               )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-0.95'               )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-1.0'                )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-1.0'                )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-1.0'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-1.0'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-1.0'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-1.0'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-1.05'               )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-1.05'               )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-1.05'               )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-1.05'               )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-1.05'               )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-1.05'               )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-1.1'                )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-1.1'                )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-1.1'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-1.1'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-1.1'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-1.1'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-1.25'               )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-1.25'               )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-1.25'               )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-1.25'               )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-1.25'               )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-1.25'               )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-1.5'                )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-1.5'                )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-1.5'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-1.5'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-1.5'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-1.5'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '50-2.0'                )] = """Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '50-2.0'                )] = """Dimer from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '50-2.0'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '50-2.0'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '50-2.0'                )] = """Monomer A from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '50-2.0'                )] = """Monomer B from Benzene ...  Ethyne (CH-pi) """
TAGL['%s-%s'            % (dbse, '51-0.9'                )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-0.9'                )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-0.9'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-0.9'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-0.9'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-0.9'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-0.95'               )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-0.95'               )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-0.95'               )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-0.95'               )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-0.95'               )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-0.95'               )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-1.0'                )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-1.0'                )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-1.0'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-1.0'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-1.0'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-1.0'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-1.05'               )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-1.05'               )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-1.05'               )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-1.05'               )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-1.05'               )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-1.05'               )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-1.1'                )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-1.1'                )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-1.1'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-1.1'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-1.1'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-1.1'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-1.25'               )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-1.25'               )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-1.25'               )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-1.25'               )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-1.25'               )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-1.25'               )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-1.5'                )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-1.5'                )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-1.5'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-1.5'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-1.5'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-1.5'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '51-2.0'                )] = """Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-dimer'      % (dbse, '51-2.0'                )] = """Dimer from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-CP'   % (dbse, '51-2.0'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-CP'   % (dbse, '51-2.0'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoA-unCP' % (dbse, '51-2.0'                )] = """Monomer A from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s-monoB-unCP' % (dbse, '51-2.0'                )] = """Monomer B from Ethyne ...  Ethyne (TS) """
TAGL['%s-%s'            % (dbse, '52-0.9'                )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-0.9'                )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-0.9'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-0.9'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-0.9'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-0.9'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-0.95'               )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-0.95'               )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-0.95'               )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-0.95'               )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-0.95'               )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-0.95'               )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-1.0'                )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-1.0'                )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-1.0'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-1.0'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-1.0'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-1.0'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-1.05'               )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-1.05'               )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-1.05'               )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-1.05'               )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-1.05'               )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-1.05'               )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-1.1'                )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-1.1'                )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-1.1'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-1.1'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-1.1'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-1.1'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-1.25'               )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-1.25'               )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-1.25'               )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-1.25'               )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-1.25'               )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-1.25'               )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-1.5'                )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-1.5'                )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-1.5'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-1.5'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-1.5'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-1.5'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '52-2.0'                )] = """Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '52-2.0'                )] = """Dimer from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '52-2.0'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '52-2.0'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '52-2.0'                )] = """Monomer A from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '52-2.0'                )] = """Monomer B from Benzene ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '53-0.9'                )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-0.9'                )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-0.9'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-0.9'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-0.9'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-0.9'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-0.95'               )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-0.95'               )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-0.95'               )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-0.95'               )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-0.95'               )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-0.95'               )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-1.0'                )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-1.0'                )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-1.0'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-1.0'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-1.0'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-1.0'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-1.05'               )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-1.05'               )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-1.05'               )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-1.05'               )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-1.05'               )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-1.05'               )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-1.1'                )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-1.1'                )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-1.1'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-1.1'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-1.1'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-1.1'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-1.25'               )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-1.25'               )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-1.25'               )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-1.25'               )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-1.25'               )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-1.25'               )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-1.5'                )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-1.5'                )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-1.5'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-1.5'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-1.5'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-1.5'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '53-2.0'                )] = """Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '53-2.0'                )] = """Dimer from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '53-2.0'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '53-2.0'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '53-2.0'                )] = """Monomer A from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '53-2.0'                )] = """Monomer B from Benzene ... AcNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '54-0.9'                )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-0.9'                )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-0.9'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-0.9'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-0.9'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-0.9'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-0.95'               )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-0.95'               )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-0.95'               )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-0.95'               )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-0.95'               )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-0.95'               )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-1.0'                )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-1.0'                )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-1.0'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-1.0'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-1.0'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-1.0'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-1.05'               )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-1.05'               )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-1.05'               )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-1.05'               )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-1.05'               )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-1.05'               )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-1.1'                )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-1.1'                )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-1.1'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-1.1'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-1.1'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-1.1'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-1.25'               )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-1.25'               )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-1.25'               )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-1.25'               )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-1.25'               )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-1.25'               )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-1.5'                )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-1.5'                )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-1.5'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-1.5'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-1.5'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-1.5'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '54-2.0'                )] = """Benzene ...  Water (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '54-2.0'                )] = """Dimer from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '54-2.0'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '54-2.0'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '54-2.0'                )] = """Monomer A from Benzene ...  Water (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '54-2.0'                )] = """Monomer B from Benzene ...  Water (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-0.9'                )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-0.9'                )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-0.9'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-0.9'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-0.9'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-0.9'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-0.95'               )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-0.95'               )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-0.95'               )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-0.95'               )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-0.95'               )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-0.95'               )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-1.0'                )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-1.0'                )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-1.0'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-1.0'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-1.0'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-1.0'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-1.05'               )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-1.05'               )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-1.05'               )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-1.05'               )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-1.05'               )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-1.05'               )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-1.1'                )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-1.1'                )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-1.1'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-1.1'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-1.1'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-1.1'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-1.25'               )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-1.25'               )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-1.25'               )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-1.25'               )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-1.25'               )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-1.25'               )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-1.5'                )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-1.5'                )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-1.5'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-1.5'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-1.5'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-1.5'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '55-2.0'                )] = """Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '55-2.0'                )] = """Dimer from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '55-2.0'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '55-2.0'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '55-2.0'                )] = """Monomer A from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '55-2.0'                )] = """Monomer B from Benzene ...  MeOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '56-0.9'                )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-0.9'                )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-0.9'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-0.9'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-0.9'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-0.9'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-0.95'               )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-0.95'               )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-0.95'               )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-0.95'               )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-0.95'               )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-0.95'               )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-1.0'                )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-1.0'                )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-1.0'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-1.0'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-1.0'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-1.0'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-1.05'               )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-1.05'               )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-1.05'               )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-1.05'               )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-1.05'               )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-1.05'               )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-1.1'                )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-1.1'                )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-1.1'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-1.1'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-1.1'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-1.1'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-1.25'               )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-1.25'               )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-1.25'               )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-1.25'               )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-1.25'               )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-1.25'               )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-1.5'                )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-1.5'                )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-1.5'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-1.5'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-1.5'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-1.5'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '56-2.0'                )] = """Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '56-2.0'                )] = """Dimer from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '56-2.0'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '56-2.0'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '56-2.0'                )] = """Monomer A from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '56-2.0'                )] = """Monomer B from Benzene ...  MeNH2 (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-0.9'                )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-0.9'                )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-0.9'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-0.9'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-0.9'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-0.9'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-0.95'               )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-0.95'               )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-0.95'               )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-0.95'               )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-0.95'               )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-0.95'               )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-1.0'                )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-1.0'                )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-1.0'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-1.0'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-1.0'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-1.0'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-1.05'               )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-1.05'               )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-1.05'               )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-1.05'               )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-1.05'               )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-1.05'               )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-1.1'                )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-1.1'                )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-1.1'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-1.1'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-1.1'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-1.1'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-1.25'               )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-1.25'               )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-1.25'               )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-1.25'               )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-1.25'               )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-1.25'               )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-1.5'                )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-1.5'                )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-1.5'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-1.5'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-1.5'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-1.5'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '57-2.0'                )] = """Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '57-2.0'                )] = """Dimer from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '57-2.0'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '57-2.0'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '57-2.0'                )] = """Monomer A from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '57-2.0'                )] = """Monomer B from Benzene ...  Peptide (NH-pi) """
TAGL['%s-%s'            % (dbse, '58-0.9'                )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-0.9'                )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-0.9'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-0.9'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-0.9'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-0.9'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-0.95'               )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-0.95'               )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-0.95'               )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-0.95'               )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-0.95'               )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-0.95'               )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-1.0'                )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-1.0'                )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-1.0'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-1.0'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-1.0'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-1.0'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-1.05'               )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-1.05'               )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-1.05'               )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-1.05'               )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-1.05'               )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-1.05'               )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-1.1'                )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-1.1'                )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-1.1'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-1.1'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-1.1'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-1.1'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-1.25'               )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-1.25'               )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-1.25'               )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-1.25'               )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-1.25'               )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-1.25'               )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-1.5'                )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-1.5'                )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-1.5'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-1.5'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-1.5'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-1.5'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '58-2.0'                )] = """Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-dimer'      % (dbse, '58-2.0'                )] = """Dimer from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-CP'   % (dbse, '58-2.0'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-CP'   % (dbse, '58-2.0'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoA-unCP' % (dbse, '58-2.0'                )] = """Monomer A from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s-monoB-unCP' % (dbse, '58-2.0'                )] = """Monomer B from Pyridine ...  Pyridine (CH-N) """
TAGL['%s-%s'            % (dbse, '59-0.9'                )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-0.9'                )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-0.9'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-0.9'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-0.9'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-0.9'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-0.95'               )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-0.95'               )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-0.95'               )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-0.95'               )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-0.95'               )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-0.95'               )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-1.0'                )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-1.0'                )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-1.0'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-1.0'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-1.0'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-1.0'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-1.05'               )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-1.05'               )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-1.05'               )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-1.05'               )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-1.05'               )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-1.05'               )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-1.1'                )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-1.1'                )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-1.1'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-1.1'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-1.1'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-1.1'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-1.25'               )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-1.25'               )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-1.25'               )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-1.25'               )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-1.25'               )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-1.25'               )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-1.5'                )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-1.5'                )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-1.5'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-1.5'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-1.5'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-1.5'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '59-2.0'                )] = """Ethyne ...  Water (CH-O) """
TAGL['%s-%s-dimer'      % (dbse, '59-2.0'                )] = """Dimer from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-CP'   % (dbse, '59-2.0'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-CP'   % (dbse, '59-2.0'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoA-unCP' % (dbse, '59-2.0'                )] = """Monomer A from Ethyne ...  Water (CH-O) """
TAGL['%s-%s-monoB-unCP' % (dbse, '59-2.0'                )] = """Monomer B from Ethyne ...  Water (CH-O) """
TAGL['%s-%s'            % (dbse, '60-0.9'                )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-0.9'                )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-0.9'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-0.9'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-0.9'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-0.9'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-0.95'               )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-0.95'               )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-0.95'               )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-0.95'               )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-0.95'               )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-0.95'               )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-1.0'                )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-1.0'                )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-1.0'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-1.0'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-1.0'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-1.0'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-1.05'               )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-1.05'               )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-1.05'               )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-1.05'               )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-1.05'               )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-1.05'               )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-1.1'                )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-1.1'                )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-1.1'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-1.1'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-1.1'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-1.1'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-1.25'               )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-1.25'               )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-1.25'               )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-1.25'               )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-1.25'               )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-1.25'               )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-1.5'                )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-1.5'                )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-1.5'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-1.5'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-1.5'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-1.5'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '60-2.0'                )] = """Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-dimer'      % (dbse, '60-2.0'                )] = """Dimer from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-CP'   % (dbse, '60-2.0'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-CP'   % (dbse, '60-2.0'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoA-unCP' % (dbse, '60-2.0'                )] = """Monomer A from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s-monoB-unCP' % (dbse, '60-2.0'                )] = """Monomer B from Ethyne ...  AcOH (OH-pi) """
TAGL['%s-%s'            % (dbse, '61-0.9'                )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-0.9'                )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-0.9'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-0.9'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-0.9'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-0.9'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-0.95'               )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-0.95'               )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-0.95'               )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-0.95'               )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-0.95'               )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-0.95'               )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-1.0'                )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-1.0'                )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-1.0'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-1.0'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-1.0'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-1.0'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-1.05'               )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-1.05'               )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-1.05'               )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-1.05'               )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-1.05'               )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-1.05'               )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-1.1'                )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-1.1'                )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-1.1'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-1.1'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-1.1'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-1.1'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-1.25'               )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-1.25'               )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-1.25'               )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-1.25'               )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-1.25'               )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-1.25'               )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-1.5'                )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-1.5'                )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-1.5'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-1.5'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-1.5'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-1.5'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '61-2.0'                )] = """Pentane ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '61-2.0'                )] = """Dimer from Pentane ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '61-2.0'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '61-2.0'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '61-2.0'                )] = """Monomer A from Pentane ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '61-2.0'                )] = """Monomer B from Pentane ...  AcOH """
TAGL['%s-%s'            % (dbse, '62-0.9'                )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-0.9'                )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-0.9'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-0.9'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-0.9'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-0.9'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-0.95'               )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-0.95'               )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-0.95'               )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-0.95'               )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-0.95'               )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-0.95'               )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-1.0'                )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-1.0'                )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-1.0'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-1.0'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-1.0'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-1.0'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-1.05'               )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-1.05'               )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-1.05'               )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-1.05'               )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-1.05'               )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-1.05'               )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-1.1'                )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-1.1'                )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-1.1'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-1.1'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-1.1'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-1.1'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-1.25'               )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-1.25'               )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-1.25'               )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-1.25'               )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-1.25'               )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-1.25'               )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-1.5'                )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-1.5'                )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-1.5'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-1.5'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-1.5'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-1.5'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '62-2.0'                )] = """Pentane ...  AcNH2 """
TAGL['%s-%s-dimer'      % (dbse, '62-2.0'                )] = """Dimer from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-CP'   % (dbse, '62-2.0'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-CP'   % (dbse, '62-2.0'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s-monoA-unCP' % (dbse, '62-2.0'                )] = """Monomer A from Pentane ...  AcNH2 """
TAGL['%s-%s-monoB-unCP' % (dbse, '62-2.0'                )] = """Monomer B from Pentane ...  AcNH2 """
TAGL['%s-%s'            % (dbse, '63-0.9'                )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-0.9'                )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-0.9'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-0.9'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-0.9'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-0.9'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-0.95'               )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-0.95'               )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-0.95'               )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-0.95'               )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-0.95'               )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-0.95'               )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-1.0'                )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-1.0'                )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-1.0'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-1.0'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-1.0'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-1.0'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-1.05'               )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-1.05'               )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-1.05'               )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-1.05'               )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-1.05'               )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-1.05'               )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-1.1'                )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-1.1'                )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-1.1'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-1.1'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-1.1'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-1.1'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-1.25'               )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-1.25'               )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-1.25'               )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-1.25'               )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-1.25'               )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-1.25'               )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-1.5'                )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-1.5'                )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-1.5'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-1.5'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-1.5'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-1.5'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '63-2.0'                )] = """Benzene ...  AcOH """
TAGL['%s-%s-dimer'      % (dbse, '63-2.0'                )] = """Dimer from Benzene ...  AcOH """
TAGL['%s-%s-monoA-CP'   % (dbse, '63-2.0'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-CP'   % (dbse, '63-2.0'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s-monoA-unCP' % (dbse, '63-2.0'                )] = """Monomer A from Benzene ...  AcOH """
TAGL['%s-%s-monoB-unCP' % (dbse, '63-2.0'                )] = """Monomer B from Benzene ...  AcOH """
TAGL['%s-%s'            % (dbse, '64-0.9'                )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-0.9'                )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-0.9'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-0.9'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-0.9'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-0.9'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-0.95'               )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-0.95'               )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-0.95'               )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-0.95'               )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-0.95'               )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-0.95'               )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-1.0'                )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-1.0'                )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-1.0'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-1.0'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-1.0'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-1.0'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-1.05'               )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-1.05'               )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-1.05'               )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-1.05'               )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-1.05'               )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-1.05'               )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-1.1'                )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-1.1'                )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-1.1'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-1.1'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-1.1'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-1.1'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-1.25'               )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-1.25'               )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-1.25'               )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-1.25'               )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-1.25'               )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-1.25'               )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-1.5'                )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-1.5'                )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-1.5'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-1.5'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-1.5'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-1.5'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '64-2.0'                )] = """Peptide ...  Ethene """
TAGL['%s-%s-dimer'      % (dbse, '64-2.0'                )] = """Dimer from Peptide ...  Ethene """
TAGL['%s-%s-monoA-CP'   % (dbse, '64-2.0'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-CP'   % (dbse, '64-2.0'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s-monoA-unCP' % (dbse, '64-2.0'                )] = """Monomer A from Peptide ...  Ethene """
TAGL['%s-%s-monoB-unCP' % (dbse, '64-2.0'                )] = """Monomer B from Peptide ...  Ethene """
TAGL['%s-%s'            % (dbse, '65-0.9'                )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-0.9'                )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-0.9'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-0.9'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-0.9'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-0.9'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-0.95'               )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-0.95'               )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-0.95'               )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-0.95'               )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-0.95'               )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-0.95'               )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-1.0'                )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-1.0'                )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-1.0'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-1.0'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-1.0'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-1.0'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-1.05'               )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-1.05'               )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-1.05'               )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-1.05'               )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-1.05'               )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-1.05'               )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-1.1'                )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-1.1'                )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-1.1'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-1.1'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-1.1'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-1.1'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-1.25'               )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-1.25'               )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-1.25'               )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-1.25'               )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-1.25'               )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-1.25'               )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-1.5'                )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-1.5'                )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-1.5'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-1.5'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-1.5'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-1.5'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '65-2.0'                )] = """Pyridine ...  Ethyne """
TAGL['%s-%s-dimer'      % (dbse, '65-2.0'                )] = """Dimer from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-CP'   % (dbse, '65-2.0'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-CP'   % (dbse, '65-2.0'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s-monoA-unCP' % (dbse, '65-2.0'                )] = """Monomer A from Pyridine ...  Ethyne """
TAGL['%s-%s-monoB-unCP' % (dbse, '65-2.0'                )] = """Monomer B from Pyridine ...  Ethyne """
TAGL['%s-%s'            % (dbse, '66-0.9'                )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-0.9'                )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-0.9'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-0.9'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-0.9'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-0.9'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-0.95'               )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-0.95'               )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-0.95'               )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-0.95'               )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-0.95'               )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-0.95'               )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-1.0'                )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-1.0'                )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-1.0'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-1.0'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-1.0'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-1.0'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-1.05'               )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-1.05'               )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-1.05'               )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-1.05'               )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-1.05'               )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-1.05'               )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-1.1'                )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-1.1'                )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-1.1'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-1.1'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-1.1'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-1.1'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-1.25'               )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-1.25'               )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-1.25'               )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-1.25'               )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-1.25'               )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-1.25'               )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-1.5'                )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-1.5'                )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-1.5'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-1.5'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-1.5'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-1.5'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s'            % (dbse, '66-2.0'                )] = """MeNH2 ...  Pyridine """
TAGL['%s-%s-dimer'      % (dbse, '66-2.0'                )] = """Dimer from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-CP'   % (dbse, '66-2.0'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-CP'   % (dbse, '66-2.0'                )] = """Monomer B from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoA-unCP' % (dbse, '66-2.0'                )] = """Monomer A from MeNH2 ...  Pyridine """
TAGL['%s-%s-monoB-unCP' % (dbse, '66-2.0'                )] = """Monomer B from MeNH2 ...  Pyridine """

# <<< Geometry Specification Strings >>>
GEOS = {}

GEOS['%s-%s-%s' % (dbse, '1-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.423835444108    -0.044407032752     0.004633480398
    H               -1.743832444108     0.858428967248    -0.016797519602
    H               -0.464118444108     0.053773967248    -0.000089519602
    --
    0 1
    O                1.346132555892     0.039570967248    -0.004329519602
    H                1.722754555892    -0.398809032752     0.761794480398
    H                1.718397555892    -0.436642032752    -0.749731519602

""")

GEOS['%s-%s-%s' % (dbse, '1-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.474155888150    -0.044012032752     0.004751452419
    H               -1.794152888150     0.858823967248    -0.016679547581
    H               -0.514438888150     0.054168967248     0.000028452419
    --
    0 1
    O                1.396453111850     0.039175967248    -0.004447547581
    H                1.773074111850    -0.399204032752     0.761677452419
    H                1.768717111850    -0.437037032752    -0.749849547581

""")

GEOS['%s-%s-%s' % (dbse, '1-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.524389888150    -0.043618032752     0.004868980398
    H               -1.844386888150     0.859217967248    -0.016562019602
    H               -0.564672888150     0.054562967248     0.000145980398
    --
    0 1
    O                1.446687111850     0.038781967248    -0.004565019602
    H                1.823308111850    -0.399598032752     0.761558980398
    H                1.818951111850    -0.437431032752    -0.749967019602

""")

GEOS['%s-%s-%s' % (dbse, '1-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.574723888150    -0.043223032752     0.004986952419
    H               -1.894720888150     0.859612967248    -0.016444047581
    H               -0.615006888150     0.054957967248     0.000263952419
    --
    0 1
    O                1.497021111850     0.038386967248    -0.004683047581
    H                1.873642111850    -0.399993032752     0.761441952419
    H                1.869285111850    -0.437826032752    -0.750085047581

""")

GEOS['%s-%s-%s' % (dbse, '1-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.625007888150    -0.042828532752     0.005104508377
    H               -1.945004888150     0.860007467248    -0.016326491623
    H               -0.665290888150     0.055352467248     0.000381508377
    --
    0 1
    O                1.547305111850     0.037992467248    -0.004800491623
    H                1.923926111850    -0.400387532752     0.761323508377
    H                1.919569111850    -0.438220532752    -0.750203491623

""")

GEOS['%s-%s-%s' % (dbse, '1-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.775859888150    -0.041645032752     0.005457980398
    H               -2.095856888150     0.861190967248    -0.015973019602
    H               -0.816142888150     0.056535967248     0.000734980398
    --
    0 1
    O                1.698157111850     0.036808967248    -0.005154019602
    H                2.074778111850    -0.401571032752     0.760969980398
    H                2.070421111850    -0.439404032752    -0.750556019602

""")

GEOS['%s-%s-%s' % (dbse, '1-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.027279888150    -0.039672532752     0.006046952419
    H               -2.347276888150     0.863163467248    -0.015384047581
    H               -1.067562888150     0.058508467248     0.001323952419
    --
    0 1
    O                1.949577111850     0.034836467248    -0.005743047581
    H                2.326198111850    -0.403543532752     0.760381952419
    H                2.321841111850    -0.441376532752    -0.751145047581

""")

GEOS['%s-%s-%s' % (dbse, '1-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.530119888150    -0.035727032752     0.007224508377
    H               -2.850116888150     0.867108967248    -0.014206491623
    H               -1.570402888150     0.062453967248     0.002501508377
    --
    0 1
    O                2.452417111850     0.030890967248    -0.006920491623
    H                2.829038111850    -0.407489032752     0.759203508377
    H                2.824681111850    -0.445322032752    -0.752323491623

""")

GEOS['%s-%s-%s' % (dbse, '2-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.885743876758     0.078739054744     0.134663990326
    H               -2.302420876758     0.877612054744     0.460433990326
    H               -0.956716876758     0.189496054744     0.375612990326
    --
    0 1
    O                0.804364123242     0.176345054744     0.509494990326
    H                1.172347123242    -0.395732945256     1.187022990326
    C                1.269369123242    -0.295284945256    -0.752664009674
    H                0.838552123242     0.355809054744    -1.505778009674
    H                2.355333123242    -0.244491945256    -0.826976009674
    H                0.941026123242    -1.315154945256    -0.951745009674

""")

GEOS['%s-%s-%s' % (dbse, '2-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.948361608847     0.079206893554     0.129903872320
    H               -2.365038608847     0.878079893554     0.455673872320
    H               -1.019334608847     0.189963893554     0.370852872320
    --
    0 1
    O                0.839578391153     0.176081893554     0.512171872320
    H                1.207561391153    -0.395995106446     1.189700872320
    C                1.304583391153    -0.295548106446    -0.749987127680
    H                0.873766391153     0.355545893554    -1.503101127680
    H                2.390547391153    -0.244755106446    -0.824299127680
    H                0.976241391153    -1.315417106446    -0.949067127680

""")

GEOS['%s-%s-%s' % (dbse, '2-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.010915935358     0.079673892770     0.125148274972
    H               -2.427592935358     0.878546892770     0.450918274972
    H               -1.081888935358     0.190430892770     0.366097274972
    --
    0 1
    O                0.874757064642     0.175818892770     0.514846274972
    H                1.242740064642    -0.396258107230     1.192375274972
    C                1.339762064642    -0.295810107230    -0.747312725028
    H                0.908945064642     0.355282892770    -1.500426725028
    H                2.425726064642    -0.245018107230    -0.821624725028
    H                1.011419064642    -1.315680107230    -0.946392725028

""")

GEOS['%s-%s-%s' % (dbse, '2-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.073597652653     0.080142011687     0.120383076822
    H               -2.490274652653     0.879015011687     0.446153076822
    H               -1.144570652653     0.190899011687     0.361332076822
    --
    0 1
    O                0.910007347347     0.175556011687     0.517526076822
    H                1.277990347347    -0.396520988313     1.195055076822
    C                1.375012347347    -0.296073988313    -0.744632923178
    H                0.944195347347     0.355020011687    -1.497746923178
    H                2.460976347347    -0.245280988313    -0.818944923178
    H                1.046669347347    -1.315942988313    -0.943712923178

""")

GEOS['%s-%s-%s' % (dbse, '2-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.136215384742     0.080609331293     0.115622399329
    H               -2.552892384742     0.879482331293     0.441392399329
    H               -1.207188384742     0.191366331293     0.356571399329
    --
    0 1
    O                0.945221615258     0.175293331293     0.520203399329
    H                1.313204615258    -0.396784668707     1.197731399329
    C                1.410226615258    -0.296336668707    -0.741955600671
    H                0.979409615258     0.354756331293    -1.495069600671
    H                2.496190615258    -0.245544668707    -0.816267600671
    H                1.081884615258    -1.316206668707    -0.941036600671

""")

GEOS['%s-%s-%s' % (dbse, '2-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.324069160637     0.082012288236     0.101341485825
    H               -2.740746160637     0.880885288236     0.427111485825
    H               -1.395042160637     0.192769288236     0.342290485825
    --
    0 1
    O                1.050864839363     0.174504288236     0.528234485825
    H                1.418847839363    -0.397573711764     1.205763485825
    C                1.515869839363    -0.297125711764    -0.733924514175
    H                1.085052839363     0.353968288236    -1.487038514175
    H                2.601833839363    -0.246332711764    -0.808236514175
    H                1.187527839363    -1.316995711764    -0.933004514175

""")

GEOS['%s-%s-%s' % (dbse, '2-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.637158980339     0.084350363311     0.077539217335
    H               -3.053835980339     0.883223363311     0.403309217335
    H               -1.708131980339     0.195107363311     0.318488217335
    --
    0 1
    O                1.226937019661     0.173189363311     0.541620217335
    H                1.594920019661    -0.398887636689     1.219148217335
    C                1.691942019661    -0.298440636689    -0.720538782665
    H                1.261125019661     0.352653363311    -1.473652782665
    H                2.777906019661    -0.247647636689    -0.794850782665
    H                1.363599019661    -1.318309636689    -0.919619782665

""")

GEOS['%s-%s-%s' % (dbse, '2-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.263338019971     0.089026594027     0.029935499807
    H               -3.680015019971     0.887899594027     0.355705499807
    H               -2.334311019971     0.199783594027     0.270884499807
    --
    0 1
    O                1.579080980029     0.170559594027     0.568391499807
    H                1.947063980029    -0.401517405973     1.245919499807
    C                2.044085980029    -0.301070405973    -0.693768500193
    H                1.613268980029     0.350023594027    -1.446882500193
    H                3.130049980029    -0.250277405973    -0.768079500193
    H                1.715742980029    -1.320939405973    -0.892848500193

""")

GEOS['%s-%s-%s' % (dbse, '3-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.989635769081    -0.554343019751    -0.004130727853
    H               -2.348292769081     0.333338980249     0.028201272147
    H               -1.028127769081    -0.416748019751     0.011997272147
    --
    0 1
    N                0.781310230919    -0.345354019751     0.020630272147
    H                1.076675230919    -0.897992019751    -0.774160727853
    H                1.091391230919    -0.856075019751     0.837489272147
    C                1.440646230919     0.963009980249    -0.019611727853
    H                1.130103230919     1.544734980249     0.844903272147
    H                2.531955230919     0.923714980249    -0.028567727853
    H                1.113933230919     1.498926980249    -0.907435727853

""")

GEOS['%s-%s-%s' % (dbse, '3-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.053247247274    -0.556852834738    -0.004434181594
    H               -2.411904247274     0.330829165262     0.027897818406
    H               -1.091739247274    -0.419257834738     0.011693818406
    --
    0 1
    N                0.818217752726    -0.343897834738     0.020805818406
    H                1.113582752726    -0.896535834738    -0.773985181594
    H                1.128298752726    -0.854618834738     0.837665818406
    C                1.477552752726     0.964466165262    -0.019435181594
    H                1.167010752726     1.546191165262     0.845079818406
    H                2.568862752726     0.925171165262    -0.028391181594
    H                1.150840752726     1.500383165262    -0.907259181594

""")

GEOS['%s-%s-%s' % (dbse, '3-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.116795686819    -0.559360118395    -0.004737308518
    H               -2.475452686819     0.328321881605     0.027594691482
    H               -1.155287686819    -0.421765118395     0.011390691482
    --
    0 1
    N                0.855088313181    -0.342443118395     0.020981691482
    H                1.150453313181    -0.895081118395    -0.773809308518
    H                1.165169313181    -0.853164118395     0.837841691482
    C                1.514423313181     0.965920881605    -0.019259308518
    H                1.203881313181     1.547645881605     0.845255691482
    H                2.605733313181     0.926625881605    -0.028215308518
    H                1.187711313181     1.501837881605    -0.907083308518

""")

GEOS['%s-%s-%s' % (dbse, '3-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.180470631292    -0.561872464714    -0.005041088820
    H               -2.539127631292     0.325809535286     0.027290911180
    H               -1.218962631292    -0.424277464714     0.011086911180
    --
    0 1
    N                0.892032368708    -0.340985464714     0.021157911180
    H                1.187396368708    -0.893623464714    -0.773632088820
    H                1.202112368708    -0.851706464714     0.838017911180
    C                1.551367368708     0.967378535286    -0.019083088820
    H                1.240825368708     1.549103535286     0.845431911180
    H                2.642676368708     0.928083535286    -0.028039088820
    H                1.224655368708     1.503295535286    -0.906907088820

""")

GEOS['%s-%s-%s' % (dbse, '3-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.244082354120    -0.564382279701    -0.005344562851
    H               -2.602739354120     0.323299720299     0.026987437149
    H               -1.282574354120    -0.426787279701     0.010783437149
    --
    0 1
    N                0.928939645880    -0.339529279701     0.021334437149
    H                1.224303645880    -0.892167279701    -0.773456562851
    H                1.239019645880    -0.850250279701     0.838193437149
    C                1.588274645880     0.968834720299    -0.018907562851
    H                1.277732645880     1.550559720299     0.845608437149
    H                2.679583645880     0.929539720299    -0.027862562851
    H                1.261562645880     1.504751720299    -0.906730562851

""")

GEOS['%s-%s-%s' % (dbse, '3-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.434916951407    -0.571911745209    -0.006255168196
    H               -2.793573951407     0.315770254791     0.026076831804
    H               -1.473408951407    -0.434316745209     0.009872831804
    --
    0 1
    N                1.039661048593    -0.335160745209     0.021862831804
    H                1.335026048593    -0.887798745209    -0.772928168196
    H                1.349742048593    -0.845881745209     0.838721831804
    C                1.698996048593     0.973203254791    -0.018379168196
    H                1.388454048593     1.554928254791     0.846136831804
    H                2.790306048593     0.933908254791    -0.027335168196
    H                1.372284048593     1.509121254791    -0.906203168196

""")

GEOS['%s-%s-%s' % (dbse, '3-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.752975177347    -0.584461432434    -0.007772680766
    H               -3.111632177347     0.303220567566     0.024559319234
    H               -1.791467177347    -0.446866432434     0.008355319234
    --
    0 1
    N                1.224196822653    -0.327879432434     0.022743319234
    H                1.519561822653    -0.880517432434    -0.772047680766
    H                1.534277822653    -0.838600432434     0.839602319234
    C                1.883532822653     0.980484567566    -0.017498680766
    H                1.572989822653     1.562209567566     0.847016319234
    H                2.974841822653     0.941189567566    -0.026454680766
    H                1.556819822653     1.516401567566    -0.905322680766

""")

GEOS['%s-%s-%s' % (dbse, '3-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.389091466518    -0.609560235688    -0.010807461530
    H               -3.747748466518     0.278121764312     0.021524538470
    H               -2.427583466518    -0.471965235688     0.005320538470
    --
    0 1
    N                1.593269533482    -0.313317235688     0.024503538470
    H                1.888633533482    -0.865955235688    -0.770286461530
    H                1.903349533482    -0.824038235688     0.841363538470
    C                2.252604533482     0.995046764312    -0.015737461530
    H                1.942062533482     1.576771764312     0.848777538470
    H                3.343913533482     0.955751764312    -0.024693461530
    H                1.925892533482     1.530964764312    -0.903561461530

""")

GEOS['%s-%s-%s' % (dbse, '4-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.937197541271     0.329131236719     0.135202465955
    H               -3.456640541271     1.127662236719     0.236780465955
    H               -2.020275541271     0.645365236719     0.149642465955
    --
    0 1
    C               -0.493683541271    -1.547067763281    -0.168682534045
    H                0.156279458729    -2.408179763281    -0.301597534045
    H               -1.174659541271    -1.469595763281    -1.012547534045
    H               -1.106700541271    -1.685863763281     0.718326465955
    C                0.261048458729    -0.249054763281    -0.036088534045
    O               -0.315666541271     0.826240236719     0.120500465955
    N                1.611681458729    -0.346556763281    -0.100706534045
    H                2.012633458729    -1.257090763281    -0.230221534045
    C                2.479923458729     0.805413236719     0.008736465955
    H                1.843429458729     1.673713236719     0.143521465955
    H                3.145513458729     0.713958236719     0.864829465955
    H                3.074389458729     0.934834236719    -0.893522534045

""")

GEOS['%s-%s-%s' % (dbse, '4-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.013163702229     0.321070525665     0.136501259076
    H               -3.532606702229     1.119601525665     0.238079259076
    H               -2.096241702229     0.637304525665     0.150941259076
    --
    0 1
    C               -0.474954702229    -1.545080474335    -0.169002740924
    H                0.175008297771    -2.406191474335    -0.301917740924
    H               -1.155930702229    -1.467608474335    -1.012867740924
    H               -1.087971702229    -1.683876474335     0.718006259076
    C                0.279777297771    -0.247067474335    -0.036408740924
    O               -0.296937702229     0.828227525665     0.120180259076
    N                1.630410297771    -0.344569474335    -0.101026740924
    H                2.031362297771    -1.255103474335    -0.230541740924
    C                2.498652297771     0.807400525665     0.008416259076
    H                1.862158297771     1.675700525665     0.143201259076
    H                3.164242297771     0.715945525665     0.864509259076
    H                3.093118297771     0.936821525665    -0.893842740924

""")

GEOS['%s-%s-%s' % (dbse, '4-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.089050267825     0.313018045913     0.137798634471
    H               -3.608493267825     1.111549045913     0.239376634471
    H               -2.172128267825     0.629252045913     0.152238634471
    --
    0 1
    C               -0.456245267825    -1.543094954087    -0.169322365529
    H                0.193717732175    -2.404206954087    -0.302237365529
    H               -1.137221267825    -1.465622954087    -1.013187365529
    H               -1.069262267825    -1.681890954087     0.717686634471
    C                0.298486732175    -0.245081954087    -0.036728365529
    O               -0.278229267825     0.830213045913     0.119859634471
    N                1.649119732175    -0.342584954087    -0.101346365529
    H                2.050071732175    -1.253118954087    -0.230861365529
    C                2.517361732175     0.809386045913     0.008096634471
    H                1.880867732175     1.677686045913     0.142880634471
    H                3.182951732175     0.717930045913     0.864189634471
    H                3.111827732175     0.938806045913    -0.894162365529

""")

GEOS['%s-%s-%s' % (dbse, '4-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.165096024145     0.304949114623     0.139098713541
    H               -3.684539024145     1.103480114623     0.240676713541
    H               -2.248174024145     0.621183114623     0.153538713541
    --
    0 1
    C               -0.437497024145    -1.541105885377    -0.169642286459
    H                0.212465975855    -2.402216885377    -0.302558286459
    H               -1.118473024145    -1.463633885377    -1.013508286459
    H               -1.050514024145    -1.679901885377     0.717365713541
    C                0.317234975855    -0.243092885377    -0.037049286459
    O               -0.259480024145     0.832202114623     0.119539713541
    N                1.667867975855    -0.340594885377    -0.101667286459
    H                2.068819975855    -1.251128885377    -0.231182286459
    C                2.536109975855     0.811375114623     0.007775713541
    H                1.899615975855     1.679676114623     0.142560713541
    H                3.201699975855     0.719920114623     0.863868713541
    H                3.130575975855     0.940796114623    -0.894483286459

""")

GEOS['%s-%s-%s' % (dbse, '4-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.241025107367     0.296892271821     0.140396836219
    H               -3.760468107367     1.095423271821     0.241974836219
    H               -2.324103107367     0.613126271821     0.154836836219
    --
    0 1
    C               -0.418777107367    -1.539118728179    -0.169963163781
    H                0.231185892633    -2.400230728179    -0.302878163781
    H               -1.099753107367    -1.461646728179    -1.013828163781
    H               -1.031794107367    -1.677914728179     0.717045836219
    C                0.335954892633    -0.241106728179    -0.037369163781
    O               -0.240761107367     0.834188271821     0.119219836219
    N                1.686587892633    -0.338608728179    -0.101987163781
    H                2.087539892633    -1.249142728179    -0.231502163781
    C                2.554829892633     0.813361271821     0.007455836219
    H                1.918335892633     1.681662271821     0.142240836219
    H                3.220419892633     0.721906271821     0.863548836219
    H                3.149295892633     0.942782271821    -0.894803163781

""")

GEOS['%s-%s-%s' % (dbse, '4-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.468960667976     0.272706303606     0.144293874958
    H               -3.988403667976     1.071237303606     0.245871874958
    H               -2.552038667976     0.588940303606     0.158733874958
    --
    0 1
    C               -0.362581667976    -1.533156696394    -0.170923125042
    H                0.287381332024    -2.394267696394    -0.303839125042
    H               -1.043557667976    -1.455683696394    -1.014788125042
    H               -0.975598667976    -1.671952696394     0.716084874958
    C                0.392150332024    -0.235143696394    -0.038330125042
    O               -0.184564667976     0.840151303606     0.118258874958
    N                1.742783332024    -0.332645696394    -0.102948125042
    H                2.143735332024    -1.243179696394    -0.232463125042
    C                2.611025332024     0.819324303606     0.006494874958
    H                1.974531332024     1.687625303606     0.141279874958
    H                3.276615332024     0.727869303606     0.862587874958
    H                3.205491332024     0.948745303606    -0.895764125042

""")

GEOS['%s-%s-%s' % (dbse, '4-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.848791472766     0.232402660825     0.150787401102
    H               -4.368234472766     1.030933660825     0.252365401102
    H               -2.931869472766     0.548636660825     0.165227401102
    --
    0 1
    C               -0.268937472766    -1.523219339175    -0.172524598898
    H                0.381025527234    -2.384331339175    -0.305439598898
    H               -0.949913472766    -1.445747339175    -1.016389598898
    H               -0.881954472766    -1.662015339175     0.714483401102
    C                0.485794527234    -0.225207339175    -0.039930598898
    O               -0.090920472766     0.850087660825     0.116657401102
    N                1.836427527234    -0.322709339175    -0.104548598898
    H                2.237379527234    -1.233243339175    -0.234064598898
    C                2.704669527234     0.829260660825     0.004894401102
    H                2.068175527234     1.697561660825     0.139678401102
    H                3.370259527234     0.737805660825     0.860987401102
    H                3.299135527234     0.958681660825    -0.897365598898

""")

GEOS['%s-%s-%s' % (dbse, '4-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -4.737942977435     0.138055588881     0.165988798331
    H               -5.257385977435     0.936586588881     0.267566798331
    H               -3.821020977435     0.454289588881     0.180428798331
    --
    0 1
    C               -0.049724977435    -1.499959411119    -0.176272201669
    H                0.600238022565    -2.361071411119    -0.309187201669
    H               -0.730699977435    -1.422487411119    -1.020137201669
    H               -0.662740977435    -1.638755411119     0.710735798331
    C                0.705008022565    -0.201946411119    -0.043679201669
    O                0.128292022565     0.873348588881     0.112909798331
    N                2.055641022565    -0.299449411119    -0.108296201669
    H                2.456592022565    -1.209982411119    -0.237812201669
    C                2.923883022565     0.852521588881     0.001146798331
    H                2.287389022565     1.720821588881     0.135930798331
    H                3.589472022565     0.761066588881     0.857238798331
    H                3.518349022565     0.981941588881    -0.901113201669

""")

GEOS['%s-%s-%s' % (dbse, '5-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.360570413711     0.317448184975    -0.106672844671
    H               -0.416338413711     0.293651184975    -0.310807844671
    C               -1.876500413711    -0.970557815025    -0.372011844671
    H               -2.934380413711    -0.955534815025    -0.123315844671
    H               -1.781045413711    -1.251940815025    -1.423462844671
    H               -1.399271413711    -1.745542815025     0.233239155329
    --
    0 1
    O                1.323593586289     0.225014184975    -0.391813844671
    H                1.792382586289     0.839399184975    -0.961479844671
    C                1.832427586289     0.382033184975     0.929903155329
    H                1.278858586289    -0.306162815025     1.560034155329
    H                2.891423586289     0.131594184975     0.986626155329
    H                1.679987586289     1.392727184975     1.307796155329

""")

GEOS['%s-%s-%s' % (dbse, '5-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.408898960914     0.319354684975    -0.104422828937
    H               -0.464666960914     0.295557684975    -0.308557828937
    C               -1.924828960914    -0.968651315025    -0.369761828937
    H               -2.982708960914    -0.953628315025    -0.121065828937
    H               -1.829373960914    -1.250034315025    -1.421212828937
    H               -1.447599960914    -1.743636315025     0.235489171063
    --
    0 1
    O                1.371922039086     0.223107684975    -0.394063828937
    H                1.840712039086     0.837492684975    -0.963729828937
    C                1.880756039086     0.380126684975     0.927653171063
    H                1.327187039086    -0.308069315025     1.557784171063
    H                2.939753039086     0.129687684975     0.984376171063
    H                1.728317039086     1.390820684975     1.305545171063

""")

GEOS['%s-%s-%s' % (dbse, '5-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.457177913711     0.321259184975    -0.102175047752
    H               -0.512945913711     0.297462184975    -0.306310047752
    C               -1.973107913711    -0.966746815025    -0.367514047752
    H               -3.030987913711    -0.951723815025    -0.118818047752
    H               -1.877652913711    -1.248129815025    -1.418965047752
    H               -1.495878913711    -1.741731815025     0.237736952248
    --
    0 1
    O                1.420201086289     0.221203184975    -0.396312047752
    H                1.888990086289     0.835588184975    -0.965978047752
    C                1.929035086289     0.378222184975     0.925405952248
    H                1.375466086289    -0.309973815025     1.555535952248
    H                2.988031086289     0.127783184975     0.982128952248
    H                1.776595086289     1.388916184975     1.303297952248

""")

GEOS['%s-%s-%s' % (dbse, '5-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.505556413711     0.323167684975    -0.099922579221
    H               -0.561324413711     0.299370684975    -0.304057579221
    C               -2.021486413711    -0.964838315025    -0.365261579221
    H               -3.079366413711    -0.949815315025    -0.116565579221
    H               -1.926031413711    -1.246221315025    -1.416712579221
    H               -1.544257413711    -1.739823315025     0.239989420779
    --
    0 1
    O                1.468579586289     0.219294684975    -0.398564579221
    H                1.937368586289     0.833679684975    -0.968229579221
    C                1.977413586289     0.376313684975     0.923153420779
    H                1.423844586289    -0.311882315025     1.553284420779
    H                3.036409586289     0.125874684975     0.979876420779
    H                1.824973586289     1.387007684975     1.301045420779

""")

GEOS['%s-%s-%s' % (dbse, '5-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.553884960914     0.325074184975    -0.097672563487
    H               -0.609652960914     0.301277184975    -0.301807563487
    C               -2.069814960914    -0.962931815025    -0.363011563487
    H               -3.127694960914    -0.947908815025    -0.114315563487
    H               -1.974359960914    -1.244314815025    -1.414462563487
    H               -1.592585960914    -1.737916815025     0.242239436513
    --
    0 1
    O                1.516908039086     0.217388184975    -0.400814563487
    H                1.985698039086     0.831773184975    -0.970480563487
    C                2.025742039086     0.374407184975     0.920903436513
    H                1.472173039086    -0.313788815025     1.551034436513
    H                3.084739039086     0.123968184975     0.977626436513
    H                1.873303039086     1.385101184975     1.298795436513

""")

GEOS['%s-%s-%s' % (dbse, '5-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.698891960914     0.330794216444    -0.090921547752
    H               -0.754659960914     0.306997216444    -0.295056547752
    C               -2.214821960914    -0.957211783556    -0.356260547752
    H               -3.272701960914    -0.942188783556    -0.107564547752
    H               -2.119366960914    -1.238594783556    -1.407711547752
    H               -1.737592960914    -1.732196783556     0.248990452248
    --
    0 1
    O                1.661915039086     0.211668216444    -0.407565547752
    H                2.130705039086     0.826052216444    -0.977231547752
    C                2.170749039086     0.368687216444     0.914152452248
    H                1.617180039086    -0.319509783556     1.544282452248
    H                3.229746039086     0.118248216444     0.970875452248
    H                2.018310039086     1.379381216444     1.292044452248

""")

GEOS['%s-%s-%s' % (dbse, '5-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.940514413711     0.340325716444    -0.079672344671
    H               -0.996282413711     0.316528716444    -0.283807344671
    C               -2.456444413711    -0.947680283556    -0.345011344671
    H               -3.514324413711    -0.932657283556    -0.096315344671
    H               -2.360989413711    -1.229063283556    -1.396462344671
    H               -1.979215413711    -1.722665283556     0.260239655329
    --
    0 1
    O                1.903537586289     0.202136716444    -0.418814344671
    H                2.372326586289     0.816520716444    -0.988480344671
    C                2.412371586289     0.359155716444     0.902902655329
    H                1.858802586289    -0.329041283556     1.533033655329
    H                3.471367586289     0.108716716444     0.959625655329
    H                2.259931586289     1.369849716444     1.280795655329

""")

GEOS['%s-%s-%s' % (dbse, '5-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.423800945179     0.359390684975    -0.057171860406
    H               -1.479568945179     0.335593684975    -0.261306860406
    C               -2.939730945179    -0.928615315025    -0.322510860406
    H               -3.997610945179    -0.913592315025    -0.073814860406
    H               -2.844275945179    -1.209998315025    -1.373961860406
    H               -2.462501945179    -1.703600315025     0.282740139594
    --
    0 1
    O                2.386824054821     0.183071684975    -0.441314860406
    H                2.855614054821     0.797456684975    -1.010980860406
    C                2.895658054821     0.340090684975     0.880402139594
    H                2.342089054821    -0.348105315025     1.510533139594
    H                3.954654054821     0.089651684975     0.937126139594
    H                2.743219054821     1.350784684975     1.258295139594

""")

GEOS['%s-%s-%s' % (dbse, '6-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.401759927035     0.709941925083    -0.003128475550
    H               -0.429215927035     0.735821925083    -0.008175475550
    C               -1.771510927035    -0.649804074917    -0.007900475550
    H               -2.857763927035    -0.699084074917    -0.003649475550
    H               -1.418245927035    -1.180540074917    -0.897257475550
    H               -1.410919927035    -1.188709074917     0.873565524450
    --
    0 1
    N                1.340544072965     0.636957925083    -0.012452475550
    H                1.711164072965     1.144734925083     0.780680524450
    H                1.711287072965     1.101296925083    -0.831710475550
    C                1.818296072965    -0.749075074917     0.025009524450
    H                1.426412072965    -1.283101074917    -0.837182475550
    H                2.905008072965    -0.855810074917     0.028043524450
    H                1.426170072965    -1.235805074917     0.914641524450

""")

GEOS['%s-%s-%s' % (dbse, '6-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.450150390505     0.712645128964    -0.003011570239
    H               -0.477606390505     0.738525128964    -0.008058570239
    C               -1.819901390505    -0.647100871036    -0.007783570239
    H               -2.906154390505    -0.696380871036    -0.003532570239
    H               -1.466636390505    -1.177836871036    -0.897140570239
    H               -1.459310390505    -1.186005871036     0.873682429761
    --
    0 1
    N                1.390468609495     0.634169128964    -0.012572570239
    H                1.761087609495     1.141945128964     0.780559429761
    H                1.761211609495     1.098508128964    -0.831831570239
    C                1.868220609495    -0.751863871036     0.024888429761
    H                1.476335609495    -1.285889871036    -0.837303570239
    H                2.954931609495    -0.858599871036     0.027923429761
    H                1.476093609495    -1.238594871036     0.914520429761

""")

GEOS['%s-%s-%s' % (dbse, '6-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.498491682033     0.715345759993    -0.002894903138
    H               -0.525947682033     0.741225759993    -0.007941903138
    C               -1.868242682033    -0.644400240007    -0.007666903138
    H               -2.954495682033    -0.693680240007    -0.003415903138
    H               -1.514977682033    -1.175136240007    -0.897023903138
    H               -1.507651682033    -1.183305240007     0.873799096862
    --
    0 1
    N                1.440342317967     0.631382759993    -0.012692903138
    H                1.810961317967     1.139159759993     0.780439096862
    H                1.811084317967     1.095721759993    -0.831951903138
    C                1.918094317967    -0.754650240007     0.024768096862
    H                1.526209317967    -1.288675240007    -0.837423903138
    H                3.004805317967    -0.861385240007     0.027802096862
    H                1.525967317967    -1.241380240007     0.914400096862

""")

GEOS['%s-%s-%s' % (dbse, '6-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.546930953087     0.718051647098    -0.002777759818
    H               -0.574386953087     0.743931647098    -0.007824759818
    C               -1.916681953087    -0.641694352902    -0.007549759818
    H               -3.002934953087    -0.690974352902    -0.003298759818
    H               -1.563416953087    -1.172430352902    -0.896906759818
    H               -1.556090953087    -1.180599352902     0.873916240182
    --
    0 1
    N                1.490317046913     0.628591647098    -0.012813759818
    H                1.860936046913     1.136367647098     0.780318240182
    H                1.861060046913     1.092929647098    -0.832072759818
    C                1.968069046913    -0.757442352902     0.024647240182
    H                1.576184046913    -1.291467352902    -0.837544759818
    H                3.054780046913    -0.864177352902     0.027681240182
    H                1.575942046913    -1.244172352902     0.914279240182

""")

GEOS['%s-%s-%s' % (dbse, '6-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.595321052198     0.720754803040    -0.002660680417
    H               -0.622777052198     0.746634803040    -0.007707680417
    C               -1.965072052198    -0.638991196960    -0.007432680417
    H               -3.051325052198    -0.688271196960    -0.003181680417
    H               -1.611807052198    -1.169727196960    -0.896789680417
    H               -1.604481052198    -1.177896196960     0.874033319583
    --
    0 1
    N                1.540240947802     0.625802803040    -0.012934680417
    H                1.910860947802     1.133578803040     0.780198319583
    H                1.910983947802     1.090140803040    -0.832193680417
    C                2.017992947802    -0.760231196960     0.024526319583
    H                1.626108947802    -1.294256196960    -0.837664680417
    H                3.104704947802    -0.866966196960     0.027561319583
    H                1.625866947802    -1.246961196960     0.914159319583

""")

GEOS['%s-%s-%s' % (dbse, '6-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.740492078251     0.728864508875    -0.002309948905
    H               -0.767948078251     0.754744508875    -0.007356948905
    C               -2.110243078251    -0.630881491125    -0.007081948905
    H               -3.196496078251    -0.680161491125    -0.002830948905
    H               -1.756978078251    -1.161617491125    -0.896438948905
    H               -1.749652078251    -1.169786491125     0.874384051095
    --
    0 1
    N                1.690013921749     0.617435508875    -0.013296948905
    H                2.060632921749     1.125212508875     0.779836051095
    H                2.060756921749     1.081774508875    -0.832554948905
    C                2.167765921749    -0.768597491125     0.024165051095
    H                1.775880921749    -1.302623491125    -0.838026948905
    H                3.254476921749    -0.875332491125     0.027199051095
    H                1.775638921749    -1.255327491125     0.913797051095

""")

GEOS['%s-%s-%s' % (dbse, '6-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.982443270566     0.742380764803    -0.001725216702
    H               -1.009899270566     0.768260764803    -0.006772216702
    C               -2.352194270566    -0.617365235197    -0.006497216702
    H               -3.438447270566    -0.666645235197    -0.002246216702
    H               -1.998929270566    -1.148101235197    -0.895854216702
    H               -1.991603270566    -1.156270235197     0.874968783298
    --
    0 1
    N                1.939634729434     0.603490764803    -0.013900216702
    H                2.310254729434     1.111267764803     0.779232783298
    H                2.310377729434     1.067829764803    -0.833158216702
    C                2.417386729434    -0.782542235197     0.023561783298
    H                2.025501729434    -1.316567235197    -0.838630216702
    H                3.504097729434    -0.889277235197     0.026595783298
    H                2.025260729434    -1.269272235197     0.913193783298

""")

GEOS['%s-%s-%s' % (dbse, '6-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.466346083475     0.769412832401    -0.000555974326
    H               -1.493802083475     0.795292832401    -0.005602974326
    C               -2.836097083475    -0.590333167599    -0.005327974326
    H               -3.922350083475    -0.639613167599    -0.001076974326
    H               -2.482832083475    -1.121069167599    -0.894684974326
    H               -2.475506083475    -1.129238167599     0.876138025674
    --
    0 1
    N                2.438876916525     0.575601832401    -0.015105974326
    H                2.809496916525     1.083378832401     0.778026025674
    H                2.809619916525     1.039940832401    -0.834364974326
    C                2.916628916525    -0.810431167599     0.022355025674
    H                2.524743916525    -1.344457167599    -0.839836974326
    H                4.003339916525    -0.917166167599     0.025389025674
    H                2.524502916525    -1.297161167599     0.911987025674

""")

GEOS['%s-%s-%s' % (dbse, '7-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.259645029681    -0.519040829348    -0.909344298797
    H               -1.339742029681    -0.689061829348    -0.650900298797
    C               -3.073041029681    -1.043340829348     0.117607701203
    H               -4.108359029681    -0.840876829348    -0.145535298797
    H               -2.958611029681    -2.123995829348     0.231598701203
    H               -2.875499029681    -0.577664829348     1.087706701203
    --
    0 1
    C               -0.230756029681     1.599719170652     0.502429701203
    H                0.275636970319     2.561168170652     0.467558701203
    H               -0.641301029681     1.453583170652     1.500305701203
    H               -1.060404029681     1.590206170652    -0.199076298797
    C                0.684873970319     0.444780170652     0.186512701203
    O                0.260678970319    -0.675130829348    -0.098568298797
    N                2.011010970319     0.706504170652     0.245712701203
    H                2.309022970319     1.616933170652     0.543399701203
    C                2.993416970319    -0.333606829348     0.022968701203
    H                2.636859970319    -0.984543829348    -0.769208298797
    H                3.933785970319     0.119917170652    -0.275545298797
    H                3.152926970319    -0.938133829348     0.915043701203

""")

GEOS['%s-%s-%s' % (dbse, '7-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.321454757660    -0.519578918243    -0.930675815475
    H               -1.401551757660    -0.689599918243    -0.672231815475
    C               -3.134850757660    -1.043878918243     0.096276184525
    H               -4.170168757660    -0.841414918243    -0.166866815475
    H               -3.020420757660    -2.124533918243     0.210267184525
    H               -2.937308757660    -0.578202918243     1.066375184525
    --
    0 1
    C               -0.203658757660     1.599955081757     0.511781184525
    H                0.302734242340     2.561404081757     0.476911184525
    H               -0.614202757660     1.453819081757     1.509658184525
    H               -1.033306757660     1.590442081757    -0.189724815475
    C                0.711971242340     0.445016081757     0.195865184525
    O                0.287776242340    -0.674894918243    -0.089216815475
    N                2.038108242340     0.706740081757     0.255064184525
    H                2.336120242340     1.617168081757     0.552751184525
    C                3.020514242340    -0.333370918243     0.032320184525
    H                2.663957242340    -0.984307918243    -0.759856815475
    H                3.960883242340     0.120153081757    -0.266192815475
    H                3.180024242340    -0.937897918243     0.924396184525

""")

GEOS['%s-%s-%s' % (dbse, '7-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.383198849350    -0.520116321510    -0.951984780059
    H               -1.463295849350    -0.690137321510    -0.693540780059
    C               -3.196594849350    -1.044416321510     0.074967219941
    H               -4.231912849350    -0.841952321510    -0.188175780059
    H               -3.082164849350    -2.125071321510     0.188958219941
    H               -2.999052849350    -0.578740321510     1.045066219941
    --
    0 1
    C               -0.176590849350     1.600190678490     0.521123219941
    H                0.329803150650     2.561639678490     0.486252219941
    H               -0.587134849350     1.454054678490     1.518999219941
    H               -1.006238849350     1.590677678490    -0.180382780059
    C                0.739040150650     0.445251678490     0.205206219941
    O                0.314845150650    -0.674659321510    -0.079874780059
    N                2.065176150650     0.706975678490     0.264406219941
    H                2.363189150650     1.617403678490     0.562093219941
    C                3.047583150650    -0.333135321510     0.041662219941
    H                2.691025150650    -0.984072321510    -0.750514780059
    H                3.987952150650     0.120388678490    -0.256851780059
    H                3.207093150650    -0.937662321510     0.933737219941

""")

GEOS['%s-%s-%s' % (dbse, '7-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.445074194435    -0.520655096031    -0.973338972156
    H               -1.525171194435    -0.690676096031    -0.714894972156
    C               -3.258470194435    -1.044955096031     0.053613027844
    H               -4.293788194435    -0.842491096031    -0.209529972156
    H               -3.144040194435    -2.125610096031     0.167604027844
    H               -3.060928194435    -0.579279096031     1.023712027844
    --
    0 1
    C               -0.149464194435     1.600426903969     0.530484027844
    H                0.356928805565     2.561875903969     0.495614027844
    H               -0.560008194435     1.454290903969     1.528361027844
    H               -0.979112194435     1.590912903969    -0.171020972156
    C                0.766165805565     0.445487903969     0.214568027844
    O                0.341970805565    -0.674423096031    -0.070512972156
    N                2.092302805565     0.707211903969     0.273768027844
    H                2.390314805565     1.617639903969     0.571455027844
    C                3.074708805565    -0.332899096031     0.051024027844
    H                2.718151805565    -0.983837096031    -0.741152972156
    H                4.015077805565     0.120624903969    -0.247489972156
    H                3.234218805565    -0.937426096031     0.943099027844

""")

GEOS['%s-%s-%s' % (dbse, '7-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.506883922414    -0.521193194516    -0.994670574261
    H               -1.586980922414    -0.691214194516    -0.736226574261
    C               -3.320279922414    -1.045493194516     0.032281425739
    H               -4.355597922414    -0.843029194516    -0.230861574261
    H               -3.205849922414    -2.126148194516     0.146272425739
    H               -3.122737922414    -0.579817194516     1.002380425739
    --
    0 1
    C               -0.122366922414     1.600662805484     0.539836425739
    H                0.384026077586     2.562111805484     0.504965425739
    H               -0.532910922414     1.454526805484     1.537712425739
    H               -0.952014922414     1.591148805484    -0.161669574261
    C                0.793263077586     0.445723805484     0.223920425739
    O                0.369068077586    -0.674187194516    -0.061161574261
    N                2.119400077586     0.707447805484     0.283119425739
    H                2.417413077586     1.617875805484     0.580806425739
    C                3.101806077586    -0.332663194516     0.060375425739
    H                2.745249077586    -0.983601194516    -0.731801574261
    H                4.042175077586     0.120860805484    -0.238138574261
    H                3.261316077586    -0.937190194516     0.952451425739

""")

GEOS['%s-%s-%s' % (dbse, '7-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.692313087167    -0.522807109008    -1.058665361821
    H               -1.772410087167    -0.692828109008    -0.800221361821
    C               -3.505709087167    -1.047107109008    -0.031713361821
    H               -4.541027087167    -0.844643109008    -0.294856361821
    H               -3.391279087167    -2.127762109008     0.082277638179
    H               -3.308167087167    -0.581431109008     0.938385638179
    --
    0 1
    C               -0.041075087167     1.601369890992     0.567891638179
    H                0.465318912833     2.562818890992     0.533020638179
    H               -0.451619087167     1.455233890992     1.565767638179
    H               -0.870723087167     1.591856890992    -0.133614361821
    C                0.874554912833     0.446430890992     0.251975638179
    O                0.450359912833    -0.673479109008    -0.033106361821
    N                2.200691912833     0.708155890992     0.311174638179
    H                2.498704912833     1.618583890992     0.608861638179
    C                3.183097912833    -0.331956109008     0.088430638179
    H                2.826540912833    -0.982893109008    -0.703746361821
    H                4.123466912833     0.121567890992    -0.210083361821
    H                3.342607912833    -0.936483109008     0.980506638179

""")

GEOS['%s-%s-%s' % (dbse, '7-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.050751505017    -0.525927245962    -1.182368263082
    H               -2.130848505017    -0.695948245962    -0.923924263082
    C               -3.864147505017    -1.050227245962    -0.155416263082
    H               -4.899465505017    -0.847763245962    -0.418559263082
    H               -3.749717505017    -2.130882245962    -0.041425263082
    H               -3.666605505017    -0.584551245962     0.814682736918
    --
    0 1
    C                0.116063494983     1.602737754038     0.622122736918
    H                0.622457494983     2.564186754038     0.587252736918
    H               -0.294480505017     1.456601754038     1.619999736918
    H               -0.713584505017     1.593224754038    -0.079382263082
    C                1.031693494983     0.447798754038     0.306206736918
    O                0.607499494983    -0.672111245962     0.021124736918
    N                2.357830494983     0.709523754038     0.365405736918
    H                2.655843494983     1.619951754038     0.663092736918
    C                3.340236494983    -0.330588245962     0.142661736918
    H                2.983679494983    -0.981525245962    -0.649515263082
    H                4.280606494983     0.122935754038    -0.155851263082
    H                3.499747494983    -0.935115245962     1.034737736918

""")

GEOS['%s-%s-%s' % (dbse, '7-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.933338493811    -0.533609791895    -1.486963947407
    H               -3.013435493811    -0.703630791895    -1.228519947407
    C               -4.746734493811    -1.057909791895    -0.460011947407
    H               -5.782052493811    -0.855445791895    -0.723154947407
    H               -4.632304493811    -2.138564791895    -0.346020947407
    H               -4.549192493811    -0.592233791895     0.510087052593
    --
    0 1
    C                0.502988506189     1.606106208105     0.755657052593
    H                1.009381506189     2.567555208105     0.720786052593
    H                0.092443506189     1.459970208105     1.753533052593
    H               -0.326659493811     1.596592208105     0.054151052593
    C                1.418618506189     0.451167208105     0.439741052593
    O                0.994423506189    -0.668743791895     0.154659052593
    N                2.744755506189     0.712891208105     0.498940052593
    H                3.042767506189     1.623319208105     0.796627052593
    C                3.727161506189    -0.327219791895     0.276196052593
    H                3.370604506189    -0.978156791895    -0.515980947407
    H                4.667530506189     0.126304208105    -0.022317947407
    H                3.886671506189    -0.931746791895     1.168272052593

""")

GEOS['%s-%s-%s' % (dbse, '8-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -0.879503680518     0.440155753873    -0.003304839796
    H                0.084765319482     0.407683753873    -0.002950839796
    C               -1.340932680518    -0.895296246127     0.004968160204
    H               -2.427218680518    -0.863909246127     0.004105160204
    H               -1.018907680518    -1.454780246127    -0.876560839796
    H               -1.019969680518    -1.443475246127     0.893961160204
    --
    0 1
    O                1.870593319482     0.349077753873    -0.001033839796
    H                2.309797319482     0.737229753873    -0.760906839796
    H                2.308477319482     0.751681753873     0.752055160204

""")

GEOS['%s-%s-%s' % (dbse, '8-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -0.915212895037     0.441327739745    -0.003343014269
    H                0.049056104963     0.408855739745    -0.002989014269
    C               -1.376641895037    -0.894124260255     0.004929985731
    H               -2.462927895037    -0.862737260255     0.004066985731
    H               -1.054616895037    -1.453608260255    -0.876599014269
    H               -1.055678895037    -1.442303260255     0.893922985731
    --
    0 1
    O                1.934091104963     0.346993739745    -0.000966014269
    H                2.373295104963     0.735145739745    -0.760839014269
    H                2.371975104963     0.749597739745     0.752123985731

""")

GEOS['%s-%s-%s' % (dbse, '8-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -0.950886114904     0.442498285832    -0.003381188742
    H                0.013382885096     0.410026285832    -0.003027188742
    C               -1.412315114904    -0.892953714168     0.004891811258
    H               -2.498601114904    -0.861566714168     0.004028811258
    H               -1.090290114904    -1.452437714168    -0.876637188742
    H               -1.091352114904    -1.441132714168     0.893884811258
    --
    0 1
    O                1.997524885096     0.344912285832    -0.000898188742
    H                2.436728885096     0.733064285832    -0.760770188742
    H                2.435408885096     0.747516285832     0.752191811258

""")

GEOS['%s-%s-%s' % (dbse, '8-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -0.986631324076     0.443671351544    -0.003419682878
    H               -0.022362324076     0.411199351544    -0.003065682878
    C               -1.448060324076    -0.891780648456     0.004853317122
    H               -2.534346324076    -0.860393648456     0.003990317122
    H               -1.126035324076    -1.451264648456    -0.876675682878
    H               -1.127097324076    -1.439959648456     0.893846317122
    --
    0 1
    O                2.061086675924     0.342826351544    -0.000829682878
    H                2.500290675924     0.730978351544    -0.760702682878
    H                2.498970675924     0.745430351544     0.752260317122

""")

GEOS['%s-%s-%s' % (dbse, '8-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.022340538595     0.444843337417    -0.003458177014
    H               -0.058071538595     0.412371337417    -0.003104177014
    C               -1.483769538595    -0.890608662583     0.004814822986
    H               -2.570055538595    -0.859221662583     0.003951822986
    H               -1.161744538595    -1.450092662583    -0.876714177014
    H               -1.162806538595    -1.438787662583     0.893807822986
    --
    0 1
    O                2.124584461405     0.340742337417    -0.000761177014
    H                2.563788461405     0.728894337417    -0.760634177014
    H                2.562468461405     0.743346337417     0.752327822986

""")

GEOS['%s-%s-%s' % (dbse, '8-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.129468222436     0.448358935089    -0.003573020096
    H               -0.165199222436     0.415886935089    -0.003219020096
    C               -1.590897222436    -0.887093064911     0.004699979904
    H               -2.677183222436    -0.855706064911     0.003836979904
    H               -1.268872222436    -1.446577064911    -0.876829020096
    H               -1.269934222436    -1.435272064911     0.893692979904
    --
    0 1
    O                2.315077777564     0.334490935089    -0.000557020096
    H                2.754282777564     0.722642935089    -0.760430020096
    H                2.752962777564     0.737094935089     0.752532979904

""")

GEOS['%s-%s-%s' % (dbse, '8-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.308014614696     0.454218504506    -0.003764531787
    H               -0.343745614696     0.421746504506    -0.003410531787
    C               -1.769443614696    -0.881233495494     0.004508468213
    H               -2.855729614696    -0.849846495494     0.003645468213
    H               -1.447418614696    -1.440717495494    -0.877020531787
    H               -1.448480614696    -1.429412495494     0.893501468213
    --
    0 1
    O                2.632567385304     0.324071504506    -0.000216531787
    H                3.071771385304     0.712223504506    -0.760088531787
    H                3.070451385304     0.726675504506     0.752873468213

""")

GEOS['%s-%s-%s' % (dbse, '8-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -1.665107119835     0.465937283393    -0.004147874832
    H               -0.700838119835     0.433465283393    -0.003793874832
    C               -2.126536119835    -0.869514716607     0.004125125168
    H               -3.212822119835    -0.838127716607     0.003262125168
    H               -1.804511119835    -1.428998716607    -0.877403874832
    H               -1.805573119835    -1.417693716607     0.893118125168
    --
    0 1
    O                3.267545880165     0.303233283393     0.000465125168
    H                3.706749880165     0.691385283393    -0.759406874832
    H                3.705429880165     0.705837283393     0.753555125168

""")

GEOS['%s-%s-%s' % (dbse, '9-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.569069043166    -0.409106692533    -0.573994583455
    H               -0.578670043166    -0.621321692533    -0.602067583455
    H               -2.044057043166    -1.298653692533    -0.497159583455
    C               -1.851400043166     0.392645307467     0.614239416545
    H               -2.917245043166     0.606953307467     0.664027416545
    H               -1.556408043166    -0.061015692533     1.566211416545
    H               -1.338624043166     1.349066307467     0.529811416545
    --
    0 1
    O                1.446502956834    -0.428218692533    -0.510511583455
    H                1.728528956834    -0.058269692533    -1.350375583455
    C                1.832416956834     0.483365307467     0.508924416545
    H                1.509232956834     0.053733307467     1.451991416545
    H                2.913471956834     0.622449307467     0.547964416545
    H                1.353941956834     1.456963307467     0.393748416545

""")

GEOS['%s-%s-%s' % (dbse, '9-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.626198722641    -0.414554335609    -0.576577260236
    H               -0.635799722641    -0.626769335609    -0.604650260236
    H               -2.101186722641    -1.304101335609    -0.499742260236
    C               -1.908529722641     0.387197664391     0.611656739764
    H               -2.974374722641     0.601505664391     0.661444739764
    H               -1.613537722641    -0.066463335609     1.563628739764
    H               -1.395753722641     1.343618664391     0.527228739764
    --
    0 1
    O                1.501877277359    -0.422938335609    -0.508008260236
    H                1.783903277359    -0.052990335609    -1.347872260236
    C                1.887791277359     0.488645664391     0.511427739764
    H                1.564608277359     0.059012664391     1.454494739764
    H                2.968846277359     0.627728664391     0.550467739764
    H                1.409316277359     1.462243664391     0.396251739764

""")

GEOS['%s-%s-%s' % (dbse, '9-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.683277621998    -0.419996948612    -0.579157905812
    H               -0.692878621998    -0.632211948612    -0.607230905812
    H               -2.158265621998    -1.309543948612    -0.502322905812
    C               -1.965608621998     0.381755051388     0.609076094188
    H               -3.031453621998     0.596063051388     0.658864094188
    H               -1.670616621998    -0.071905948612     1.561048094188
    H               -1.452832621998     1.338176051388     0.524648094188
    --
    0 1
    O                1.557202378002    -0.417662948612    -0.505506905812
    H                1.839228378002    -0.047714948612    -1.345370905812
    C                1.943116378002     0.493921051388     0.513929094188
    H                1.619933378002     0.064288051388     1.456996094188
    H                3.024171378002     0.633004051388     0.552969094188
    H                1.464642378002     1.467519051388     0.398753094188

""")

GEOS['%s-%s-%s' % (dbse, '9-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.740458049630    -0.425449209838    -0.581743089639
    H               -0.750059049630    -0.637664209838    -0.609816089639
    H               -2.215446049630    -1.314996209838    -0.504908089639
    C               -2.022789049630     0.376302790162     0.606490910361
    H               -3.088634049630     0.590610790162     0.656278910361
    H               -1.727797049630    -0.077358209838     1.558462910361
    H               -1.510013049630     1.332723790162     0.522062910361
    --
    0 1
    O                1.612625950370    -0.412378209838    -0.503001089639
    H                1.894651950370    -0.042430209838    -1.342865089639
    C                1.998539950370     0.499205790162     0.516434910361
    H                1.675356950370     0.069572790162     1.459500910361
    H                3.079594950370     0.638288790162     0.555474910361
    H                1.520064950370     1.472803790162     0.401257910361

""")

GEOS['%s-%s-%s' % (dbse, '9-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.797587713125    -0.430896440991    -0.584325782399
    H               -0.807188713125    -0.643111440991    -0.612398782399
    H               -2.272575713125    -1.320443440991    -0.507490782399
    C               -2.079918713125     0.370855559009     0.603908217601
    H               -3.145763713125     0.585163559009     0.653696217601
    H               -1.784926713125    -0.082805440991     1.555880217601
    H               -1.567142713125     1.327276559009     0.519480217601
    --
    0 1
    O                1.668000286875    -0.407098440991    -0.500497782399
    H                1.950026286875    -0.037149440991    -1.340361782399
    C                2.053914286875     0.504485559009     0.518938217601
    H                1.730731286875     0.074853559009     1.462005217601
    H                3.134969286875     0.643569559009     0.557978217601
    H                1.575439286875     1.478083559009     0.403761217601

""")

GEOS['%s-%s-%s' % (dbse, '9-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.968976703609    -0.447238704684    -0.592074034971
    H               -0.978577703609    -0.659453704684    -0.620147034971
    H               -2.443964703609    -1.336785704684    -0.515239034971
    C               -2.251307703609     0.354513295316     0.596159965029
    H               -3.317152703609     0.568821295316     0.645947965029
    H               -1.956315703609    -0.099147704684     1.548131965029
    H               -1.738531703609     1.310934295316     0.511731965029
    --
    0 1
    O                1.834123296391    -0.391258704684    -0.492988034971
    H                2.116149296391    -0.021309704684    -1.332852034971
    C                2.220037296391     0.520326295316     0.526448965029
    H                1.896854296391     0.090693295316     1.469514965029
    H                3.301092296391     0.659409295316     0.565488965029
    H                1.741562296391     1.493924295316     0.411271965029

""")

GEOS['%s-%s-%s' % (dbse, '9-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.254625037062    -0.474475889968    -0.604987910695
    H               -1.264226037062    -0.686690889968    -0.633060910695
    H               -2.729613037062    -1.364022889968    -0.528152910695
    C               -2.536956037062     0.327276110032     0.583246089305
    H               -3.602801037062     0.541584110032     0.633034089305
    H               -2.241964037062    -0.126384889968     1.535218089305
    H               -2.024180037062     1.283697110032     0.498818089305
    --
    0 1
    O                2.110994962938    -0.364857889968    -0.480470910695
    H                2.393020962938     0.005090110032    -1.320334910695
    C                2.496908962938     0.546726110032     0.538966089305
    H                2.173725962938     0.117093110032     1.482032089305
    H                3.577963962938     0.685809110032     0.578005089305
    H                2.018434962938     1.520324110032     0.423789089305

""")

GEOS['%s-%s-%s' % (dbse, '9-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.825922163830    -0.528949801253    -0.630815503835
    H               -1.835523163830    -0.741164801253    -0.658888503835
    H               -3.300910163830    -1.418496801253    -0.553980503835
    C               -3.108253163830     0.272802198747     0.557418496165
    H               -4.174098163830     0.487110198747     0.607206496165
    H               -2.813261163830    -0.180858801253     1.509390496165
    H               -2.595477163830     1.229223198747     0.472990496165
    --
    0 1
    O                2.664738836170    -0.312057801253    -0.455436503835
    H                2.946764836170     0.057891198747    -1.295300503835
    C                3.050652836170     0.599526198747     0.563999496165
    H                2.727469836170     0.169894198747     1.507066496165
    H                4.131707836170     0.738610198747     0.603039496165
    H                2.572177836170     1.573124198747     0.448823496165

""")

GEOS['%s-%s-%s' % (dbse, '10-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.337124360659     0.201608256532    -0.622612167452
    H               -0.508529360659    -0.383244743468    -0.672954167452
    H               -1.998952360659    -0.188050743468    -1.280848167452
    C               -1.897142360659     0.122091256532     0.724766832548
    H               -2.791221360659     0.739937256532     0.785986832548
    H               -2.161229360659    -0.885151743468     1.062271832548
    H               -1.178759360659     0.529367256532     1.433906832548
    --
    0 1
    N                1.508159639341    -0.695099743468    -0.487787167452
    H                1.804428639341    -0.420561743468    -1.415773167452
    H                2.131709639341    -1.442789743468    -0.212592167452
    C                1.668389639341     0.440111256532     0.425887832548
    H                1.398199639341     0.125415256532     1.431540832548
    H                2.675799639341     0.861887256532     0.462859832548
    H                0.975849639341     1.225886256532     0.133230832548

""")

GEOS['%s-%s-%s' % (dbse, '10-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.393140795726     0.210270724066    -0.627755199918
    H               -0.564545795726    -0.374582275934    -0.678097199918
    H               -2.054968795726    -0.179388275934    -1.285991199918
    C               -1.953158795726     0.130753724066     0.719623800082
    H               -2.847237795726     0.748599724066     0.780843800082
    H               -2.217245795726    -0.876489275934     1.057128800082
    H               -1.234775795726     0.538029724066     1.428763800082
    --
    0 1
    N                1.564176204274    -0.703762275934    -0.482644199918
    H                1.860444204274    -0.429224275934    -1.410629199918
    H                2.187725204274    -1.451452275934    -0.207449199918
    C                1.724406204274     0.431448724066     0.431030800082
    H                1.454215204274     0.116752724066     1.436684800082
    H                2.731816204274     0.853225724066     0.468002800082
    H                1.031865204274     1.217224724066     0.138373800082

""")

GEOS['%s-%s-%s' % (dbse, '10-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.449107860659     0.218925256532    -0.632894167452
    H               -0.620512860659    -0.365927743468    -0.683236167452
    H               -2.110935860659    -0.170733743468    -1.291130167452
    C               -2.009125860659     0.139408256532     0.714484832548
    H               -2.903204860659     0.757254256532     0.775704832548
    H               -2.273212860659    -0.867834743468     1.051989832548
    H               -1.290742860659     0.546684256532     1.423624832548
    --
    0 1
    N                1.620143139341    -0.712416743468    -0.477505167452
    H                1.916412139341    -0.437878743468    -1.405491167452
    H                2.243693139341    -1.460106743468    -0.202310167452
    C                1.780373139341     0.422794256532     0.436169832548
    H                1.510183139341     0.108098256532     1.441822832548
    H                2.787783139341     0.844570256532     0.473141832548
    H                1.087833139341     1.208569256532     0.143512832548

""")

GEOS['%s-%s-%s' % (dbse, '10-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.505173376892     0.227594998517    -0.638041699918
    H               -0.676578376892    -0.357258001483    -0.688383699918
    H               -2.167001376892    -0.162064001483    -1.296277699918
    C               -2.065191376892     0.148077998517     0.709337300082
    H               -2.959270376892     0.765923998517     0.770557300082
    H               -2.329278376892    -0.859165001483     1.046842300082
    H               -1.346808376892     0.555353998517     1.418477300082
    --
    0 1
    N                1.676208623108    -0.721086001483    -0.472357699918
    H                1.972477623108    -0.446549001483    -1.400342699918
    H                2.299758623108    -1.468777001483    -0.197162699918
    C                1.836438623108     0.414123998517     0.441317300082
    H                1.566248623108     0.099427998517     1.446971300082
    H                2.843849623108     0.835900998517     0.478289300082
    H                1.143898623108     1.199899998517     0.148660300082

""")

GEOS['%s-%s-%s' % (dbse, '10-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.561189860659     0.236257256532    -0.643185167452
    H               -0.732594860659    -0.348595743468    -0.693527167452
    H               -2.223017860659    -0.153401743468    -1.301421167452
    C               -2.121207860659     0.156740256532     0.704193832548
    H               -3.015286860659     0.774586256532     0.765413832548
    H               -2.385294860659    -0.850502743468     1.041698832548
    H               -1.402824860659     0.564016256532     1.413333832548
    --
    0 1
    N                1.732225139341    -0.729748743468    -0.467214167452
    H                2.028494139341    -0.455210743468    -1.395200167452
    H                2.355775139341    -1.477438743468    -0.192019167452
    C                1.892455139341     0.405462256532     0.446460832548
    H                1.622265139341     0.090766256532     1.452113832548
    H                2.899865139341     0.827238256532     0.483432832548
    H                1.199915139341     1.191237256532     0.153803832548

""")

GEOS['%s-%s-%s' % (dbse, '10-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.729238876892     0.262243756532    -0.658614893203
    H               -0.900643876892    -0.322609243468    -0.708956893203
    H               -2.391066876892    -0.127415243468    -1.316850893203
    C               -2.289256876892     0.182726756532     0.688764106797
    H               -3.183335876892     0.800572756532     0.749984106797
    H               -2.553343876892    -0.824516243468     1.026269106797
    H               -1.570873876892     0.590002756532     1.397904106797
    --
    0 1
    N                1.900274123108    -0.755735243468    -0.451784893203
    H                2.196543123108    -0.481197243468    -1.379769893203
    H                2.523824123108    -1.503425243468    -0.176589893203
    C                2.060504123108     0.379475756532     0.461891106797
    H                1.790314123108     0.064779756532     1.467544106797
    H                3.067915123108     0.801251756532     0.498862106797
    H                1.367964123108     1.165250756532     0.169233106797

""")

GEOS['%s-%s-%s' % (dbse, '10-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.084325360659     0.317153449818    -0.691217699918
    H               -1.255730360659    -0.267699550182    -0.741559699918
    H               -2.746153360659    -0.072505550182    -1.349453699918
    C               -2.644343360659     0.237636449818     0.656161300082
    H               -3.538422360659     0.855482449818     0.717381300082
    H               -2.908430360659    -0.769606550182     0.993666300082
    H               -1.925960360659     0.644912449818     1.365301300082
    --
    0 1
    N                2.255360639341    -0.810644550182    -0.419181699918
    H                2.551629639341    -0.536106550182    -1.347166699918
    H                2.878910639341    -1.558334550182    -0.143986699918
    C                2.415590639341     0.324565449818     0.494493300082
    H                2.145400639341     0.009870449818     1.500147300082
    H                3.423000639341     0.746342449818     0.531465300082
    H                1.723050639341     1.110341449818     0.201836300082

""")

GEOS['%s-%s-%s' % (dbse, '10-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.728937295726     0.416834256532    -0.750404183685
    H               -1.900342295726    -0.168018743468    -0.800746183685
    H               -3.390765295726     0.027175256532    -1.408640183685
    C               -3.288955295726     0.337317256532     0.596974816315
    H               -4.183034295726     0.955163256532     0.658194816315
    H               -3.553042295726    -0.669925743468     0.934479816315
    H               -2.570572295726     0.744593256532     1.306114816315
    --
    0 1
    N                2.899972704274    -0.910325743468    -0.359995183685
    H                3.196240704274    -0.635787743468    -1.287980183685
    H                3.523521704274    -1.658015743468    -0.084800183685
    C                3.060202704274     0.224885256532     0.553679816315
    H                2.790011704274    -0.089810743468     1.559332816315
    H                4.067612704274     0.646661256532     0.590651816315
    H                2.367661704274     1.010660256532     0.261022816315

""")

GEOS['%s-%s-%s' % (dbse, '11-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.480874806714    -0.021737855971    -0.180326614685
    H               -1.780288806714    -0.274698855971    -0.869039614685
    H               -3.372418806714    -0.309194855971    -0.561854614685
    C               -2.216680806714    -0.770399855971     1.047367385315
    H               -2.968315806714    -0.521839855971     1.794534385315
    H               -2.199190806714    -1.858401855971     0.933421385315
    H               -1.250708806714    -0.467600855971     1.448817385315
    --
    0 1
    C                0.161309193286     1.867101144029    -0.405469614685
    H                0.285928193286     2.278277144029    -1.404513614685
    H                0.667881193286     2.508093144029     0.312314385315
    H               -0.905102806714     1.840571144029    -0.187048614685
    C                0.673301193286     0.449303144029    -0.403175614685
    O                0.213321193286    -0.416432855971    -1.142350614685
    N                1.659784193286     0.176271144029     0.489155385315
    H                2.059545193286     0.931203144029     1.014507385315
    C                2.248229193286    -1.143319855971     0.566654385315
    H                1.460053193286    -1.884649855971     0.473634385315
    H                2.969489193286    -1.315314855971    -0.231889614685
    H                2.742883193286    -1.259187855971     1.526594385315

""")

GEOS['%s-%s-%s' % (dbse, '11-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.558598669458    -0.016211993600    -0.169671179418
    H               -1.858012669458    -0.269172993600    -0.858384179418
    H               -3.450142669458    -0.303668993600    -0.551199179418
    C               -2.294404669458    -0.764873993600     1.058022820582
    H               -3.046039669458    -0.516313993600     1.805189820582
    H               -2.276914669458    -1.852875993600     0.944076820582
    H               -1.328432669458    -0.462074993600     1.459472820582
    --
    0 1
    C                0.194336330542     1.864753006400    -0.409997179418
    H                0.318954330542     2.275930006400    -1.409042179418
    H                0.700907330542     2.505746006400     0.307786820582
    H               -0.872075669458     1.838223006400    -0.191576179418
    C                0.706328330542     0.446955006400    -0.407703179418
    O                0.246348330542    -0.418780993600    -1.146878179418
    N                1.692811330542     0.173923006400     0.484626820582
    H                2.092572330542     0.928856006400     1.009979820582
    C                2.281256330542    -1.145667993600     0.562126820582
    H                1.493079330542    -1.886997993600     0.469105820582
    H                3.002516330542    -1.317662993600    -0.236417179418
    H                2.775909330542    -1.261535993600     1.522065820582

""")

GEOS['%s-%s-%s' % (dbse, '11-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.636252670937    -0.010691274316    -0.159025204232
    H               -1.935666670937    -0.263652274316    -0.847738204232
    H               -3.527796670937    -0.298148274316    -0.540553204232
    C               -2.372058670937    -0.759353274316     1.068668795768
    H               -3.123693670937    -0.510793274316     1.815835795768
    H               -2.354568670937    -1.847355274316     0.954722795768
    H               -1.406086670937    -0.456554274316     1.470118795768
    --
    0 1
    C                0.227334329063     1.862406725684    -0.414521204232
    H                0.351952329063     2.273583725684    -1.413565204232
    H                0.733905329063     2.503399725684     0.303263795768
    H               -0.839077670937     1.835877725684    -0.196100204232
    C                0.739325329063     0.444609725684    -0.412227204232
    O                0.279345329063    -0.421127274316    -1.151402204232
    N                1.725808329063     0.171576725684     0.480103795768
    H                2.125569329063     0.926509725684     1.005455795768
    C                2.314254329063    -1.148013274316     0.557602795768
    H                1.526077329063    -1.889343274316     0.464582795768
    H                3.035513329063    -1.320009274316    -0.240941204232
    H                2.808907329063    -1.263881274316     1.517542795768

""")

GEOS['%s-%s-%s' % (dbse, '11-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.714045809340    -0.005160729856    -0.148360395777
    H               -2.013459809340    -0.258121729856    -0.837073395777
    H               -3.605589809340    -0.292617729856    -0.529888395777
    C               -2.449851809340    -0.753822729856     1.079333604223
    H               -3.201486809340    -0.505262729856     1.826500604223
    H               -2.432361809340    -1.841824729856     0.965387604223
    H               -1.483879809340    -0.451023729856     1.480783604223
    --
    0 1
    C                0.260390190660     1.860057270144    -0.419053395777
    H                0.385009190660     2.271233270144    -1.418097395777
    H                0.766962190660     2.501049270144     0.298731604223
    H               -0.806021809340     1.833527270144    -0.200631395777
    C                0.772382190660     0.442259270144    -0.416759395777
    O                0.312402190660    -0.423477729856    -1.155933395777
    N                1.758865190660     0.169227270144     0.475571604223
    H                2.158626190660     0.924159270144     1.000924604223
    C                2.347310190660    -1.150363729856     0.553071604223
    H                1.559134190660    -1.891693729856     0.460050604223
    H                3.068570190660    -1.322358729856    -0.245473395777
    H                2.841964190660    -1.266231729856     1.513010604223

""")

GEOS['%s-%s-%s' % (dbse, '11-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.791769412601     0.000364988539    -0.137704845459
    H               -2.091183412601    -0.252596011461    -0.826417845459
    H               -3.683313412601    -0.287092011461    -0.519232845459
    C               -2.527575412601    -0.748297011461     1.089989154541
    H               -3.279210412601    -0.499737011461     1.837156154541
    H               -2.510085412601    -1.836299011461     0.976043154541
    H               -1.561603412601    -0.445498011461     1.491439154541
    --
    0 1
    C                0.293417587399     1.857708988539    -0.423580845459
    H                0.418035587399     2.268885988539    -1.422624845459
    H                0.799988587399     2.498700988539     0.294204154541
    H               -0.772994412601     1.831178988539    -0.205159845459
    C                0.805408587399     0.439910988539    -0.421286845459
    O                0.345429587399    -0.425825011461    -1.160461845459
    N                1.791891587399     0.166878988539     0.471044154541
    H                2.191652587399     0.921811988539     0.996397154541
    C                2.380337587399    -1.152712011461     0.548543154541
    H                1.592160587399    -1.894042011461     0.455523154541
    H                3.101597587399    -1.324707011461    -0.250000845459
    H                2.874990587399    -1.268580011461     1.508483154541

""")

GEOS['%s-%s-%s' % (dbse, '11-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -3.152000828310     0.025975387219    -0.088319487194
    H               -2.451414828310    -0.226985612781    -0.777032487194
    H               -4.043544828310    -0.261481612781    -0.469847487194
    C               -2.887806828310    -0.722686612781     1.139374512806
    H               -3.639441828310    -0.474126612781     1.886541512806
    H               -2.870316828310    -1.810688612781     1.025428512806
    H               -1.921834828310    -0.419887612781     1.540824512806
    --
    0 1
    C                0.446490171690     1.846826387219    -0.444566487194
    H                0.571108171690     2.258003387219    -1.443610487194
    H                0.953061171690     2.487819387219     0.273218512806
    H               -0.619921828310     1.820296387219    -0.226144487194
    C                0.958481171690     0.429028387219    -0.442272487194
    O                0.498502171690    -0.436707612781    -1.181446487194
    N                1.944964171690     0.155996387219     0.450058512806
    H                2.344725171690     0.910929387219     0.975411512806
    C                2.533410171690    -1.163594612781     0.527558512806
    H                1.745233171690    -1.904924612781     0.434537512806
    H                3.254670171690    -1.335589612781    -0.270986487194
    H                3.028063171690    -1.279462612781     1.487497512806

""")

GEOS['%s-%s-%s' % (dbse, '11-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -3.745157620280     0.068145232979    -0.007001442494
    H               -3.044571620280    -0.184815767021    -0.695714442494
    H               -4.636701620280    -0.219311767021    -0.388529442494
    C               -3.480963620280    -0.680516767021     1.220692557506
    H               -4.232598620280    -0.431956767021     1.967859557506
    H               -3.463473620280    -1.768518767021     1.106746557506
    H               -2.514991620280    -0.377717767021     1.622142557506
    --
    0 1
    C                0.698539379720     1.828907232979    -0.479120442494
    H                0.823157379720     2.240084232979    -1.478164442494
    H                1.205110379720     2.469899232979     0.238664557506
    H               -0.367872620280     1.802377232979    -0.260699442494
    C                1.210530379720     0.411109232979    -0.476826442494
    O                0.750551379720    -0.454626767021    -1.216001442494
    N                2.197013379720     0.138077232979     0.415504557506
    H                2.596774379720     0.893010232979     0.940857557506
    C                2.785459379720    -1.181513767021     0.493003557506
    H                1.997282379720    -1.922843767021     0.399983557506
    H                3.506719379720    -1.353508767021    -0.305540442494
    H                3.280112379720    -1.297381767021     1.452943557506

""")

GEOS['%s-%s-%s' % (dbse, '11-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -4.714354340790     0.137049494457     0.125869106162
    H               -4.013768340790    -0.115911505543    -0.562843893838
    H               -5.605898340790    -0.150407505543    -0.255658893838
    C               -4.450160340790    -0.611612505543     1.353563106162
    H               -5.201795340790    -0.363052505543     2.100730106162
    H               -4.432670340790    -1.699614505543     1.239617106162
    H               -3.484188340790    -0.308813505543     1.755013106162
    --
    0 1
    C                1.110378659210     1.799627494457    -0.535580893838
    H                1.234996659210     2.210804494457    -1.534624893838
    H                1.616949659210     2.440620494457     0.182203106162
    H                0.043966659210     1.773098494457    -0.317159893838
    C                1.622369659210     0.381830494457    -0.533286893838
    O                1.162390659210    -0.483906505543    -1.272461893838
    N                2.608852659210     0.108797494457     0.359044106162
    H                3.008613659210     0.863730494457     0.884396106162
    C                3.197298659210    -1.210792505543     0.436543106162
    H                2.409121659210    -1.952123505543     0.343523106162
    H                3.918558659210    -1.382788505543    -0.362000893838
    H                3.691951659210    -1.326661505543     1.396483106162

""")

GEOS['%s-%s-%s' % (dbse, '12-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.736664839201    -0.665978177026    -0.359795039126
    H               -0.832351839201    -1.634809177026    -0.083942039126
    H               -1.325804839201    -0.552543177026    -1.174555039126
    C               -1.220109839201     0.202101822974     0.720581960874
    H               -1.115960839201     1.239167822974     0.412729960874
    H               -2.257937839201     0.028697822974     1.011712960874
    H               -0.588223839201     0.062416822974     1.593822960874
    --
    0 1
    O                1.741340160799     0.473331822974    -0.249100039126
    H                0.923823160799    -0.022941177026    -0.422252039126
    H                2.323242160799     0.214833822974    -0.964846039126

""")

GEOS['%s-%s-%s' % (dbse, '12-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.770533808008    -0.679094122819    -0.358521315653
    H               -0.866220808008    -1.647925122819    -0.082668315653
    H               -1.359673808008    -0.565659122819    -1.173281315653
    C               -1.253978808008     0.188985877181     0.721855684347
    H               -1.149829808008     1.226051877181     0.414003684347
    H               -2.291806808008     0.015581877181     1.012986684347
    H               -0.622092808008     0.049300877181     1.595096684347
    --
    0 1
    O                1.799715191992     0.495937877181    -0.251295315653
    H                0.982198191992    -0.000335122819    -0.424448315653
    H                2.381617191992     0.237439877181    -0.967041315653

""")

GEOS['%s-%s-%s' % (dbse, '12-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.804368630268    -0.692196871139    -0.357248714228
    H               -0.900055630268    -1.661027871139    -0.081395714228
    H               -1.393508630268    -0.578761871139    -1.172008714228
    C               -1.287813630268     0.175883128861     0.723128285772
    H               -1.183664630268     1.212949128861     0.415276285772
    H               -2.325641630268     0.002479128861     1.014259285772
    H               -0.655927630268     0.036198128861     1.596369285772
    --
    0 1
    O                1.858031369732     0.518521128861    -0.253488714228
    H                1.040514369732     0.022249128861    -0.426641714228
    H                2.439933369732     0.260023128861    -0.969234714228

""")

GEOS['%s-%s-%s' % (dbse, '12-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.838272071697    -0.705326055496    -0.355973196011
    H               -0.933959071697    -1.674157055496    -0.080120196011
    H               -1.427412071697    -0.591891055496    -1.170733196011
    C               -1.321717071697     0.162753944504     0.724403803989
    H               -1.217568071697     1.199819944504     0.416551803989
    H               -2.359545071697    -0.010650055496     1.015534803989
    H               -0.689831071697     0.023068944504     1.597644803989
    --
    0 1
    O                1.916465928303     0.541149944504    -0.255687196011
    H                1.098947928303     0.044877944504    -0.428839196011
    H                2.498366928303     0.282652944504    -0.971433196011

""")

GEOS['%s-%s-%s' % (dbse, '12-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.872141061050    -0.718442327365    -0.354699472538
    H               -0.967828061050    -1.687273327365    -0.078846472538
    H               -1.461281061050    -0.605007327365    -1.169459472538
    C               -1.355586061050     0.149637672635     0.725677527462
    H               -1.251437061050     1.186703672635     0.417825527462
    H               -2.393414061050    -0.023766327365     1.016808527462
    H               -0.723700061050     0.009952672635     1.598918527462
    --
    0 1
    O                1.974840938950     0.563756672635    -0.257882472538
    H                1.157323938950     0.067483672635    -0.431035472538
    H                2.556741938950     0.305258672635    -0.973628472538

""")

GEOS['%s-%s-%s' % (dbse, '12-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.973747988016    -0.757790205835    -0.350877629423
    H               -1.069434988016    -1.726621205835    -0.075024629423
    H               -1.562887988016    -0.644355205835    -1.165637629423
    C               -1.457192988016     0.110289794165     0.729499370577
    H               -1.353043988016     1.147355794165     0.421647370577
    H               -2.495020988016    -0.063114205835     1.020630370577
    H               -0.825306988016    -0.029395205835     1.602740370577
    --
    0 1
    O                2.149966011984     0.631574794165    -0.264469629423
    H                1.332449011984     0.135302794165    -0.437622629423
    H                2.731868011984     0.373077794165    -0.980215629423

""")

GEOS['%s-%s-%s' % (dbse, '12-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.143093199217    -0.823370628042    -0.344508013287
    H               -1.238780199217    -1.792201628042    -0.068655013287
    H               -1.732233199217    -0.709935628042    -1.159268013287
    C               -1.626538199217     0.044709371958     0.735868986713
    H               -1.522389199217     1.081775371958     0.428016986713
    H               -2.664366199217    -0.128694628042     1.026999986713
    H               -0.994652199217    -0.094975628042     1.609109986713
    --
    0 1
    O                2.441841800783     0.744606371958    -0.275448013287
    H                1.624324800783     0.248333371958    -0.448601013287
    H                3.023743800783     0.486108371958    -0.991194013287

""")

GEOS['%s-%s-%s' % (dbse, '12-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.481783580528    -0.954530840851    -0.331768413848
    H               -1.577470580528    -1.923361840851    -0.055915413848
    H               -2.070923580528    -0.841095840851    -1.146528413848
    C               -1.965228580528    -0.086450840851     0.748608586152
    H               -1.861079580528     0.950615159149     0.440756586152
    H               -3.003056580528    -0.259854840851     1.039739586152
    H               -1.333342580528    -0.226135840851     1.621849586152
    --
    0 1
    O                3.025593419472     0.970668159149    -0.297405413848
    H                2.208075419472     0.474396159149    -0.470558413848
    H                3.607494419472     0.712170159149    -1.013151413848

""")

GEOS['%s-%s-%s' % (dbse, '13-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.509834441292    -0.035405046464     1.836965639596
    H                0.523826558708     0.293046953536     1.761433639596
    H               -0.543015441292    -1.037957046464     2.257951639596
    H               -1.051320441292     0.620971953536     2.513677639596
    C               -1.224549441292    -0.049229046464     0.504723639596
    O               -2.410043441292    -0.347444046464     0.402009639596
    N               -0.462171441292     0.276738953536    -0.566409360404
    H                0.500670558708     0.544453953536    -0.436037360404
    C               -1.045867441292     0.301742953536    -1.888888360404
    H               -1.552134441292    -0.638707046464    -2.094847360404
    H               -1.780489441292     1.100305953536    -1.985631360404
    H               -0.255163441292     0.453664953536    -2.617879360404
    --
    0 1
    O                2.301369558708     0.555077953536    -0.221867360404
    H                3.022224558708     1.182377953536    -0.131453360404
    C                2.820453558708    -0.750288046464     0.012013639596
    H                1.987882558708    -1.439075046464    -0.093562360404
    H                3.586592558708    -1.019281046464    -0.714866360404
    H                3.228550558708    -0.851737046464     1.017648639596

""")

GEOS['%s-%s-%s' % (dbse, '13-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.540323016583    -0.035584877623     1.833339510968
    H                0.493337983417     0.292867122377     1.757807510968
    H               -0.573504016583    -1.038136877623     2.254325510968
    H               -1.081809016583     0.620792122377     2.510051510968
    C               -1.255038016583    -0.049408877623     0.501097510968
    O               -2.440532016583    -0.347623877623     0.398383510968
    N               -0.492660016583     0.276559122377    -0.570035489032
    H                0.470181983417     0.544274122377    -0.439663489032
    C               -1.076356016583     0.301563122377    -1.892514489032
    H               -1.582623016583    -0.638886877623    -2.098473489032
    H               -1.810978016583     1.100126122377    -1.989257489032
    H               -0.285652016583     0.453485122377    -2.621505489032
    --
    0 1
    O                2.370914983417     0.555488122377    -0.213596489032
    H                3.091768983417     1.182788122377    -0.123181489032
    C                2.889998983417    -0.749877877623     0.020285510968
    H                2.057426983417    -1.438664877623    -0.085291489032
    H                3.656137983417    -1.018869877623    -0.706595489032
    H                3.298095983417    -0.851326877623     1.025920510968

""")

GEOS['%s-%s-%s' % (dbse, '13-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.570781142415    -0.035764670418     1.829716877918
    H                0.462879857585     0.292687329582     1.754184877918
    H               -0.603962142415    -1.038316670418     2.250702877918
    H               -1.112267142415     0.620612329582     2.506428877918
    C               -1.285496142415    -0.049588670418     0.497474877918
    O               -2.470990142415    -0.347803670418     0.394760877918
    N               -0.523118142415     0.276379329582    -0.573658122082
    H                0.439723857585     0.544094329582    -0.443286122082
    C               -1.106814142415     0.301383329582    -1.896137122082
    H               -1.613081142415    -0.639066670418    -2.102096122082
    H               -1.841436142415     1.099946329582    -1.992880122082
    H               -0.316110142415     0.453305329582    -2.625128122082
    --
    0 1
    O                2.440390857585     0.555898329582    -0.205333122082
    H                3.161245857585     1.183197329582    -0.114918122082
    C                2.959474857585    -0.749467670418     0.028548877918
    H                2.126902857585    -1.438255670418    -0.077028122082
    H                3.725613857585    -1.018460670418    -0.698332122082
    H                3.367571857585    -0.850916670418     1.034182877918

""")

GEOS['%s-%s-%s' % (dbse, '13-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.601299738938    -0.035944644550     1.826087044293
    H                0.432361261062     0.292507355450     1.750555044293
    H               -0.634480738938    -1.038496644550     2.247073044293
    H               -1.142785738938     0.620432355450     2.502799044293
    C               -1.316014738938    -0.049768644550     0.493845044293
    O               -2.501508738938    -0.347983644550     0.391131044293
    N               -0.553636738938     0.276199355450    -0.577287955707
    H                0.409205261062     0.543914355450    -0.446915955707
    C               -1.137332738938     0.301203355450    -1.899766955707
    H               -1.643599738938    -0.639246644550    -2.105725955707
    H               -1.871954738938     1.099766355450    -1.996509955707
    H               -0.346628738938     0.453125355450    -2.628757955707
    --
    0 1
    O                2.510004261062     0.556308355450    -0.197052955707
    H                3.230859261062     1.183608355450    -0.106638955707
    C                3.029089261062    -0.749056644550     0.036828044293
    H                2.196517261062    -1.437844644550    -0.068747955707
    H                3.795227261062    -1.018049644550    -0.690051955707
    H                3.437186261062    -0.850505644550     1.042463044293

""")

GEOS['%s-%s-%s' % (dbse, '13-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.631788200029    -0.036124618336     1.822460906074
    H                0.401872799971     0.292327381664     1.746928906074
    H               -0.664969200029    -1.038676618336     2.243446906074
    H               -1.173274200029     0.620252381664     2.499172906074
    C               -1.346503200029    -0.049948618336     0.490218906074
    O               -2.531997200029    -0.348163618336     0.387504906074
    N               -0.584125200029     0.276019381664    -0.580914093926
    H                0.378716799971     0.543734381664    -0.450542093926
    C               -1.167821200029     0.301023381664    -1.903393093926
    H               -1.674088200029    -0.639426618336    -2.109352093926
    H               -1.902443200029     1.099586381664    -2.000136093926
    H               -0.377117200029     0.452945381664    -2.632384093926
    --
    0 1
    O                2.579549799971     0.556719381664    -0.188782093926
    H                3.300404799971     1.184018381664    -0.098367093926
    C                3.098633799971    -0.748646618336     0.045099906074
    H                2.266061799971    -1.437434618336    -0.060476093926
    H                3.864772799971    -1.017639618336    -0.681781093926
    H                3.506730799971    -0.850095618336     1.050734906074

""")

GEOS['%s-%s-%s' % (dbse, '13-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.723253373884    -0.036664111814     1.811582310771
    H                0.310407626116     0.291787888186     1.736050310771
    H               -0.756434373884    -1.039216111814     2.232568310771
    H               -1.264739373884     0.619712888186     2.488294310771
    C               -1.437968373884    -0.050488111814     0.479340310771
    O               -2.623462373884    -0.348703111814     0.376626310771
    N               -0.675590373884     0.275479888186    -0.591792689229
    H                0.287251626116     0.543194888186    -0.461420689229
    C               -1.259286373884     0.300483888186    -1.914271689229
    H               -1.765553373884    -0.639966111814    -2.120230689229
    H               -1.993908373884     1.099046888186    -2.011014689229
    H               -0.468582373884     0.452405888186    -2.643262689229
    --
    0 1
    O                2.788184626116     0.557949888186    -0.163967689229
    H                3.509039626116     1.185249888186    -0.073552689229
    C                3.307268626116    -0.747416111814     0.069914310771
    H                2.474696626116    -1.436203111814    -0.035661689229
    H                4.073407626116    -1.016408111814    -0.656966689229
    H                3.715365626116    -0.848865111814     1.075549310771

""")

GEOS['%s-%s-%s' % (dbse, '13-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.875695412720    -0.037563495663     1.793451248793
    H                0.157965587280     0.290888504337     1.717919248793
    H               -0.908876412720    -1.040115495663     2.214437248793
    H               -1.417181412720     0.618813504337     2.470163248793
    C               -1.590410412720    -0.051387495663     0.461209248793
    O               -2.775904412720    -0.349602495663     0.358495248793
    N               -0.828032412720     0.274580504337    -0.609923751207
    H                0.134809587280     0.542295504337    -0.479551751207
    C               -1.411728412720     0.299584504337    -1.932402751207
    H               -1.917995412720    -0.640865495663    -2.138361751207
    H               -2.146350412720     1.098147504337    -2.029145751207
    H               -0.621024412720     0.451506504337    -2.661393751207
    --
    0 1
    O                3.135909587280     0.560001504337    -0.122609751207
    H                3.856763587280     1.187300504337    -0.032194751207
    C                3.654993587280    -0.745364495663     0.111271248793
    H                2.822421587280    -1.434152495663     0.005695248793
    H                4.421131587280    -1.014357495663    -0.615608751207
    H                4.063090587280    -0.846813495663     1.116906248793

""")

GEOS['%s-%s-%s' % (dbse, '13-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.180579252749    -0.039362168690     1.757189334256
    H               -0.146918252749     0.289089831310     1.681657334256
    H               -1.213760252749    -1.041914168690     2.178175334256
    H               -1.722065252749     0.617014831310     2.433901334256
    C               -1.895294252749    -0.053186168690     0.424947334256
    O               -3.080788252749    -0.351401168690     0.322233334256
    N               -1.132916252749     0.272781831310    -0.646185665744
    H               -0.170074252749     0.540496831310    -0.515813665744
    C               -1.716612252749     0.297785831310    -1.968664665744
    H               -2.222879252749    -0.642664168690    -2.174623665744
    H               -2.451234252749     1.096348831310    -2.065407665744
    H               -0.925908252749     0.449707831310    -2.697655665744
    --
    0 1
    O                3.831358747251     0.564103831310    -0.039895665744
    H                4.552213747251     1.191403831310     0.050519334256
    C                4.350442747251    -0.741261168690     0.193986334256
    H                3.517871747251    -1.430049168690     0.088410334256
    H                5.116581747251    -1.010254168690    -0.532894665744
    H                4.758539747251    -0.842710168690     1.199621334256

""")

GEOS['%s-%s-%s' % (dbse, '14-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.444549658463    -0.074191914325     1.883302315567
    H                0.558768341537     0.338176085675     1.806398315567
    H               -0.388456658463    -1.097962914325     2.247499315567
    H               -1.017884658463     0.499946085675     2.606598315567
    C               -1.186479658463    -0.067498914325     0.565239315567
    O               -2.366811658463    -0.394456914325     0.482511315567
    N               -0.457930658463     0.319489085675    -0.506496684433
    H                0.528135341537     0.534837085675    -0.404160684433
    C               -1.063774658463     0.333047085675    -1.818403684433
    H               -1.980904658463     0.918020085675    -1.806795684433
    H               -0.364780658463     0.776390085675    -2.522451684433
    H               -1.318958658463    -0.672399914325    -2.152520684433
    --
    0 1
    N                2.383355341537     0.633640085675    -0.331923684433
    H                2.787021341537     1.173560085675     0.423276315567
    H                2.801359341537     1.001021085675    -1.177569684433
    C                2.755658341537    -0.777542914325    -0.178170684433
    H                2.329370341537    -1.348574914325    -0.999546684433
    H                3.831056341537    -0.964231914325    -0.152387684433
    H                2.321185341537    -1.161949914325     0.741665315567

""")

GEOS['%s-%s-%s' % (dbse, '14-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.475283836692    -0.075828771774     1.882105597285
    H                0.528034163308     0.336539228226     1.805201597285
    H               -0.419190836692    -1.099599771774     2.246302597285
    H               -1.048618836692     0.498309228226     2.605401597285
    C               -1.217213836692    -0.069135771774     0.564042597285
    O               -2.397545836692    -0.396093771774     0.481314597285
    N               -0.488664836692     0.317852228226    -0.507693402715
    H                0.497401163308     0.533200228226    -0.405357402715
    C               -1.094508836692     0.331410228226    -1.819600402715
    H               -2.011638836692     0.916383228226    -1.807992402715
    H               -0.395514836692     0.774753228226    -2.523648402715
    H               -1.349692836692    -0.674036771774    -2.153717402715
    --
    0 1
    N                2.455683163308     0.637492228226    -0.329107402715
    H                2.859349163308     1.177411228226     0.426092597285
    H                2.873687163308     1.004873228226    -1.174753402715
    C                2.827986163308    -0.773690771774    -0.175354402715
    H                2.401699163308    -1.344723771774    -0.996730402715
    H                3.903385163308    -0.960379771774    -0.149571402715
    H                2.393513163308    -1.158097771774     0.744481597285

""")

GEOS['%s-%s-%s' % (dbse, '14-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.505988328405    -0.077463888369     1.880910071845
    H                0.497329671595     0.334904111631     1.804006071845
    H               -0.449895328405    -1.101234888369     2.245107071845
    H               -1.079323328405     0.496674111631     2.604206071845
    C               -1.247918328405    -0.070770888369     0.562847071845
    O               -2.428250328405    -0.397728888369     0.480119071845
    N               -0.519369328405     0.316217111631    -0.508888928155
    H                0.466696671595     0.531565111631    -0.406552928155
    C               -1.125213328405     0.329775111631    -1.820795928155
    H               -2.042343328405     0.914748111631    -1.809187928155
    H               -0.426219328405     0.773118111631    -2.524843928155
    H               -1.380397328405    -0.675671888369    -2.154912928155
    --
    0 1
    N                2.527941671595     0.641340111631    -0.326293928155
    H                2.931607671595     1.181260111631     0.428906071845
    H                2.945944671595     1.008722111631    -1.171939928155
    C                2.900243671595    -0.769842888369    -0.172540928155
    H                2.473956671595    -1.340874888369    -0.993916928155
    H                3.975642671595    -0.956531888369    -0.146757928155
    H                2.465771671595    -1.154249888369     0.747295071845

""")

GEOS['%s-%s-%s' % (dbse, '14-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.536752308309    -0.079102381194     1.879712285683
    H                0.466565691691     0.333265618806     1.802808285683
    H               -0.480659308309    -1.102873381194     2.243909285683
    H               -1.110087308309     0.495035618806     2.603008285683
    C               -1.278682308309    -0.072409381194     0.561649285683
    O               -2.459014308309    -0.399367381194     0.478921285683
    N               -0.550133308309     0.314578618806    -0.510086714317
    H                0.435932691691     0.529926618806    -0.407750714317
    C               -1.155977308309     0.328136618806    -1.821993714317
    H               -2.073107308309     0.913109618806    -1.810385714317
    H               -0.456983308309     0.771479618806    -2.526041714317
    H               -1.411161308309    -0.677310381194    -2.156110714317
    --
    0 1
    N                2.600339691691     0.645195618806    -0.323474714317
    H                3.004005691691     1.185115618806     0.431725285683
    H                3.018342691691     1.012577618806    -1.169120714317
    C                2.972641691691    -0.765986381194    -0.169722714317
    H                2.546354691691    -1.337019381194    -0.991097714317
    H                4.048040691691    -0.952676381194    -0.143938714317
    H                2.538169691691    -1.150393381194     0.750113285683

""")

GEOS['%s-%s-%s' % (dbse, '14-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.567486592135    -0.080739123363     1.878515567402
    H                0.435831407865     0.331628876637     1.801611567402
    H               -0.511393592135    -1.104510123363     2.242712567402
    H               -1.140821592135     0.493398876637     2.601811567402
    C               -1.309416592135    -0.074046123363     0.560452567402
    O               -2.489748592135    -0.401004123363     0.477724567402
    N               -0.580867592135     0.312941876637    -0.511283432598
    H                0.405198407865     0.528289876637    -0.408947432598
    C               -1.186711592135     0.326499876637    -1.823190432598
    H               -2.103841592135     0.911472876637    -1.811582432598
    H               -0.487717592135     0.769842876637    -2.527238432598
    H               -1.441895592135    -0.678947123363    -2.157307432598
    --
    0 1
    N                2.672667407865     0.649047876637    -0.320658432598
    H                3.076333407865     1.188967876637     0.434541567402
    H                3.090671407865     1.016428876637    -1.166304432598
    C                3.044970407865    -0.762135123363    -0.166906432598
    H                2.618682407865    -1.333167123363    -0.988281432598
    H                4.120368407865    -0.948824123363    -0.141122432598
    H                2.610497407865    -1.146542123363     0.752929567402

""")

GEOS['%s-%s-%s' % (dbse, '14-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.659689241981    -0.085649590232     1.874925412558
    H                0.343628758019     0.326718409768     1.798021412558
    H               -0.603596241981    -1.109420590232     2.239122412558
    H               -1.233024241981     0.488488409768     2.598221412558
    C               -1.401619241981    -0.078956590232     0.556862412558
    O               -2.581951241981    -0.405914590232     0.474134412558
    N               -0.673070241981     0.308031409768    -0.514873587442
    H                0.312995758019     0.523379409768    -0.412537587442
    C               -1.278914241981     0.321589409768    -1.826780587442
    H               -2.196044241981     0.906562409768    -1.815172587442
    H               -0.579920241981     0.764932409768    -2.530828587442
    H               -1.534098241981    -0.683857590232    -2.160897587442
    --
    0 1
    N                2.889651758019     0.660603409768    -0.312209587442
    H                3.293317758019     1.200523409768     0.442990412558
    H                3.307654758019     1.027985409768    -1.157855587442
    C                3.261953758019    -0.750578590232    -0.158457587442
    H                2.835666758019    -1.321611590232    -0.979832587442
    H                4.337352758019    -0.937268590232    -0.132673587442
    H                2.827481758019    -1.134985590232     0.761378412558

""")

GEOS['%s-%s-%s' % (dbse, '14-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.813360459479    -0.093833551121     1.868941984718
    H                0.189957540521     0.318534448879     1.792037984718
    H               -0.757267459479    -1.117604551121     2.233138984718
    H               -1.386695459479     0.480304448879     2.592237984718
    C               -1.555290459479    -0.087140551121     0.550878984718
    O               -2.735622459479    -0.414098551121     0.468150984718
    N               -0.826741459479     0.299847448879    -0.520857015282
    H                0.159324540521     0.515195448879    -0.418521015282
    C               -1.432585459479     0.313405448879    -1.832764015282
    H               -2.349715459479     0.898378448879    -1.821156015282
    H               -0.733591459479     0.756748448879    -2.536812015282
    H               -1.687769459479    -0.692041551121    -2.166881015282
    --
    0 1
    N                3.251291540521     0.679863448879    -0.298129015282
    H                3.654957540521     1.219783448879     0.457070984718
    H                3.669295540521     1.047245448879    -1.143775015282
    C                3.623594540521    -0.731319551121    -0.144376015282
    H                3.197306540521    -1.302351551121    -0.965751015282
    H                4.698992540521    -0.918008551121    -0.118593015282
    H                3.189121540521    -1.115726551121     0.775459984718

""")

GEOS['%s-%s-%s' % (dbse, '14-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.120702673599    -0.110201713140     1.856974811584
    H               -0.117384673599     0.302166286860     1.780070811584
    H               -1.064609673599    -1.133972713140     2.221171811584
    H               -1.694037673599     0.463936286860     2.580270811584
    C               -1.862632673599    -0.103508713140     0.538911811584
    O               -3.042964673599    -0.430466713140     0.456183811584
    N               -1.134083673599     0.283479286860    -0.532824188416
    H               -0.148017673599     0.498827286860    -0.430488188416
    C               -1.739927673599     0.297037286860    -1.844731188416
    H               -2.657057673599     0.882010286860    -1.833123188416
    H               -1.040933673599     0.740380286860    -2.548779188416
    H               -1.995111673599    -0.708409713140    -2.178848188416
    --
    0 1
    N                3.974571326401     0.718383286860    -0.269966188416
    H                4.378238326401     1.258303286860     0.485233811584
    H                4.392575326401     1.085764286860    -1.115612188416
    C                4.346874326401    -0.692799713140    -0.116213188416
    H                3.920587326401    -1.263831713140    -0.937589188416
    H                5.422273326401    -0.879488713140    -0.090430188416
    H                3.912401326401    -1.077206713140     0.803622811584

""")

GEOS['%s-%s-%s' % (dbse, '15-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.576932869400     0.274842161549     1.966368435502
    H               -2.058725869400     0.961118161549     2.658358435502
    H               -0.525867869400     0.535148161549     1.867317435502
    H               -1.669486869400    -0.726134838451     2.383523435502
    C               -2.323975869400     0.320593161549     0.651299435502
    O               -3.535879869400     0.136991161549     0.593830435502
    N               -1.545486869400     0.571387161549    -0.430738564498
    H               -0.548754869400     0.688034161549    -0.298922564498
    C               -2.102484869400     0.655264161549    -1.760483564498
    H               -3.167453869400     0.458969161549    -1.681393564498
    H               -1.953231869400     1.645410161549    -2.189059564498
    H               -1.652057869400    -0.082528838451    -2.423888564498
    --
    0 1
    C                0.992622130600    -1.775610838451     0.232289435502
    H                1.550847130600    -2.685613838451     0.026381435502
    H                0.063930130600    -1.782139838451    -0.332035564498
    H                0.733403130600    -1.752163838451     1.289230435502
    C                1.760258130600    -0.521198838451    -0.103946564498
    O                1.215071130600     0.580238161549    -0.160919564498
    N                3.087016130600    -0.690012838451    -0.317694564498
    H                3.463337130600    -1.617257838451    -0.240937564498
    C                3.974254130600     0.416453161549    -0.606643564498
    H                3.360062130600     1.307263161549    -0.686467564498
    H                4.701669130600     0.555919161549     0.190759435502
    H                4.499244130600     0.256472161549    -1.545785564498

""")

GEOS['%s-%s-%s' % (dbse, '15-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.625925355604     0.277836189141     1.962535134170
    H               -2.107718355604     0.964112189141     2.654525134170
    H               -0.574860355604     0.538142189141     1.863484134170
    H               -1.718479355604    -0.723140810859     2.379690134170
    C               -2.372968355604     0.323587189141     0.647466134170
    O               -3.584872355604     0.139985189141     0.589997134170
    N               -1.594479355604     0.574381189141    -0.434571865830
    H               -0.597747355604     0.691028189141    -0.302755865830
    C               -2.151477355604     0.658258189141    -1.764316865830
    H               -3.216446355604     0.461963189141    -1.685226865830
    H               -2.002224355604     1.648404189141    -2.192892865830
    H               -1.701050355604    -0.079534810859    -2.427721865830
    --
    0 1
    C                1.041614644396    -1.778604810859     0.236123134170
    H                1.599838644396    -2.688607810859     0.030215134170
    H                0.112922644396    -1.785134810859    -0.328202865830
    H                0.782395644396    -1.755158810859     1.293064134170
    C                1.809250644396    -0.524192810859    -0.100113865830
    O                1.264063644396     0.577244189141    -0.157085865830
    N                3.136008644396    -0.693006810859    -0.313861865830
    H                3.512329644396    -1.620252810859    -0.237104865830
    C                4.023246644396     0.413459189141    -0.602809865830
    H                3.409054644396     1.304269189141    -0.682633865830
    H                4.750660644396     0.552925189141     0.194593134170
    H                4.548236644396     0.253477189141    -1.541952865830

""")

GEOS['%s-%s-%s' % (dbse, '15-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.674867876298     0.280827305514     1.958705935502
    H               -2.156660876298     0.967103305514     2.650695935502
    H               -0.623802876298     0.541133305514     1.859654935502
    H               -1.767421876298    -0.720149694486     2.375860935502
    C               -2.421910876298     0.326578305514     0.643636935502
    O               -3.633814876298     0.142976305514     0.586167935502
    N               -1.643421876298     0.577372305514    -0.438401064498
    H               -0.646689876298     0.694019305514    -0.306585064498
    C               -2.200419876298     0.661249305514    -1.768146064498
    H               -3.265388876298     0.464954305514    -1.689056064498
    H               -2.051166876298     1.651395305514    -2.196722064498
    H               -1.749992876298    -0.076543694486    -2.431551064498
    --
    0 1
    C                1.090557123702    -1.781595694486     0.239951935502
    H                1.648782123702    -2.691599694486     0.034043935502
    H                0.161865123702    -1.788125694486    -0.324373064498
    H                0.831338123702    -1.758149694486     1.296892935502
    C                1.858193123702    -0.527183694486    -0.096284064498
    O                1.313006123702     0.574252305514    -0.153257064498
    N                3.184951123702    -0.695997694486    -0.310032064498
    H                3.561272123702    -1.623243694486    -0.233275064498
    C                4.072189123702     0.410468305514    -0.598981064498
    H                3.457997123702     1.301278305514    -0.678805064498
    H                4.799604123702     0.549934305514     0.198421935502
    H                4.597180123702     0.250486305514    -1.538123064498

""")

GEOS['%s-%s-%s' % (dbse, '15-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.723909965328     0.283824661549     1.954868825778
    H               -2.205702965328     0.970100661549     2.646858825778
    H               -0.672844965328     0.544130661549     1.855817825778
    H               -1.816463965328    -0.717152338451     2.372023825778
    C               -2.470952965328     0.329575661549     0.639799825778
    O               -3.682856965328     0.145973661549     0.582330825778
    N               -1.692463965328     0.580369661549    -0.442238174222
    H               -0.695731965328     0.697016661549    -0.310422174222
    C               -2.249461965328     0.664246661549    -1.771983174222
    H               -3.314430965328     0.467951661549    -1.692893174222
    H               -2.100208965328     1.654392661549    -2.200559174222
    H               -1.799034965328    -0.073546338451    -2.435388174222
    --
    0 1
    C                1.139599034672    -1.784593338451     0.243789825778
    H                1.697824034672    -2.694596338451     0.037881825778
    H                0.210907034672    -1.791122338451    -0.320536174222
    H                0.880380034672    -1.761146338451     1.300730825778
    C                1.907235034672    -0.530181338451    -0.092447174222
    O                1.362048034672     0.571255661549    -0.149420174222
    N                3.233993034672    -0.698995338451    -0.306195174222
    H                3.610315034672    -1.626240338451    -0.229438174222
    C                4.121232034672     0.407470661549    -0.595144174222
    H                3.507039034672     1.298280661549    -0.674967174222
    H                4.848646034672     0.546936661549     0.202259825778
    H                4.646222034672     0.247489661549    -1.534286174222

""")

GEOS['%s-%s-%s' % (dbse, '15-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.772902376298     0.286818689141     1.951035620374
    H               -2.254695376298     0.973094689141     2.643025620374
    H               -0.721837376298     0.547124689141     1.851984620374
    H               -1.865456376298    -0.714158310859     2.368190620374
    C               -2.519945376298     0.332569689141     0.635966620374
    O               -3.731849376298     0.148967689141     0.578497620374
    N               -1.741456376298     0.583363689141    -0.446071379626
    H               -0.744724376298     0.700010689141    -0.314255379626
    C               -2.298454376298     0.667240689141    -1.775816379626
    H               -3.363423376298     0.470945689141    -1.696726379626
    H               -2.149201376298     1.657386689141    -2.204392379626
    H               -1.848027376298    -0.070552310859    -2.439221379626
    --
    0 1
    C                1.188591623702    -1.787587310859     0.247622620374
    H                1.746816623702    -2.697590310859     0.041714620374
    H                0.259899623702    -1.794117310859    -0.316702379626
    H                0.929372623702    -1.764141310859     1.304563620374
    C                1.956227623702    -0.533175310859    -0.088614379626
    O                1.411040623702     0.568261689141    -0.145586379626
    N                3.282985623702    -0.701989310859    -0.302362379626
    H                3.659306623702    -1.629235310859    -0.225604379626
    C                4.170223623702     0.404476689141    -0.591310379626
    H                3.556031623702     1.295286689141    -0.671134379626
    H                4.897638623702     0.543942689141     0.206092620374
    H                4.695214623702     0.244494689141    -1.530453379626

""")

GEOS['%s-%s-%s' % (dbse, '15-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.919879574803     0.295801189141     1.939535935502
    H               -2.401672574803     0.982077189141     2.631525935502
    H               -0.868814574803     0.556107189141     1.840484935502
    H               -2.012433574803    -0.705175810859     2.356690935502
    C               -2.666922574803     0.341552189141     0.624466935502
    O               -3.878826574803     0.157950189141     0.566997935502
    N               -1.888433574803     0.592346189141    -0.457571064498
    H               -0.891701574803     0.708993189141    -0.325755064498
    C               -2.445431574803     0.676223189141    -1.787316064498
    H               -3.510400574803     0.479928189141    -1.708226064498
    H               -2.296178574803     1.666369189141    -2.215892064498
    H               -1.995004574803    -0.061569810859    -2.450721064498
    --
    0 1
    C                1.335568425197    -1.796569810859     0.259121935502
    H                1.893793425197    -2.706572810859     0.053213935502
    H                0.406876425197    -1.803099810859    -0.305203064498
    H                1.076349425197    -1.773123810859     1.316062935502
    C                2.103204425197    -0.542157810859    -0.077114064498
    O                1.558018425197     0.559279189141    -0.134087064498
    N                3.429962425197    -0.710971810859    -0.290862064498
    H                3.806284425197    -1.638217810859    -0.214105064498
    C                4.317201425197     0.395494189141    -0.579811064498
    H                3.703008425197     1.286304189141    -0.659635064498
    H                5.044615425197     0.534960189141     0.217591935502
    H                4.842191425197     0.235512189141    -1.518953064498

""")

GEOS['%s-%s-%s' % (dbse, '15-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.164841574803     0.310772161549     1.920369935502
    H               -2.646634574803     0.997048161549     2.612359935502
    H               -1.113776574803     0.571078161549     1.821318935502
    H               -2.257395574803    -0.690204838451     2.337524935502
    C               -2.911884574803     0.356523161549     0.605300935502
    O               -4.123788574803     0.172921161549     0.547831935502
    N               -2.133395574803     0.607317161549    -0.476737064498
    H               -1.136663574803     0.723964161549    -0.344921064498
    C               -2.690393574803     0.691194161549    -1.806482064498
    H               -3.755362574803     0.494899161549    -1.727392064498
    H               -2.541140574803     1.681340161549    -2.235058064498
    H               -2.239966574803    -0.046598838451    -2.469887064498
    --
    0 1
    C                1.580530425197    -1.811540838451     0.278287935502
    H                2.138755425197    -2.721543838451     0.072379935502
    H                0.651838425197    -1.818069838451    -0.286037064498
    H                1.321311425197    -1.788093838451     1.335228935502
    C                2.348166425197    -0.557128838451    -0.057948064498
    O                1.802980425197     0.544308161549    -0.114921064498
    N                3.674924425197    -0.725942838451    -0.271696064498
    H                4.051246425197    -1.653187838451    -0.194939064498
    C                4.562163425197     0.380523161549    -0.560645064498
    H                3.947970425197     1.271333161549    -0.640469064498
    H                5.289577425197     0.519989161549     0.236757935502
    H                5.087153425197     0.220542161549    -1.499787064498

""")

GEOS['%s-%s-%s' % (dbse, '15-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.742453362502     0.346072682243     1.875177113476
    H               -3.224246362502     1.032348682243     2.567167113476
    H               -1.691388362502     0.606378682243     1.776126113476
    H               -2.835007362502    -0.654904317757     2.292332113476
    C               -3.489496362502     0.391823682243     0.560108113476
    O               -4.701400362502     0.208221682243     0.502639113476
    N               -2.711007362502     0.642617682243    -0.521929886524
    H               -1.714275362502     0.759264682243    -0.390113886524
    C               -3.268005362502     0.726494682243    -1.851674886524
    H               -4.332974362502     0.530199682243    -1.772584886524
    H               -3.118752362502     1.716640682243    -2.280250886524
    H               -2.817578362502    -0.011298317757    -2.515079886524
    --
    0 1
    C                2.158142637498    -1.846841317757     0.323481113476
    H                2.716366637498    -2.756844317757     0.117573113476
    H                1.229450637498    -1.853371317757    -0.240843886524
    H                1.898923637498    -1.823394317757     1.380422113476
    C                2.925778637498    -0.592429317757    -0.012755886524
    O                2.380591637498     0.509007682243    -0.069727886524
    N                4.252536637498    -0.761243317757    -0.226503886524
    H                4.628857637498    -1.688489317757    -0.149745886524
    C                5.139774637498     0.345222682243    -0.515451886524
    H                4.525582637498     1.236032682243    -0.595275886524
    H                5.867189637498     0.484688682243     0.281951113476
    H                5.664764637498     0.185240682243    -1.454593886524

""")

GEOS['%s-%s-%s' % (dbse, '16-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.132791956027    -0.482035617798     1.797858063170
    H                1.192414956027    -0.407042617798     1.567172063170
    H               -0.094600043973    -1.489089617798     2.139641063170
    H               -0.112531043973     0.200820382202     2.607755063170
    C               -0.757838043973    -0.164528617798     0.618059063170
    O               -1.980227043973    -0.192121617798     0.708123063170
    N               -0.096323043973     0.146496382202    -0.526829936830
    H                0.910905956027     0.147333382202    -0.522649936830
    C               -0.801027043973     0.472994382202    -1.744562936830
    H               -1.863275043973     0.399844382202    -1.532662936830
    H               -0.573208043973     1.485863382202    -2.074103936830
    H               -0.548529043973    -0.221434617798    -2.544593936830
    --
    0 1
    O                2.780529956027     0.162991382202    -0.630719936830
    H                3.345506956027    -0.549967617798    -0.935918936830
    H                3.360281956027     0.927185382202    -0.606821936830

""")

GEOS['%s-%s-%s' % (dbse, '16-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.112234832618    -0.482207873680     1.799046329340
    H                1.171857832618    -0.407214873680     1.568360329340
    H               -0.115157167382    -1.489261873680     2.140829329340
    H               -0.133088167382     0.200648126320     2.608943329340
    C               -0.778395167382    -0.164700873680     0.619247329340
    O               -2.000784167382    -0.192293873680     0.709311329340
    N               -0.116880167382     0.146324126320    -0.525641670660
    H                0.890348832618     0.147161126320    -0.521461670660
    C               -0.821584167382     0.472822126320    -1.743374670660
    H               -1.883832167382     0.399672126320    -1.531474670660
    H               -0.593765167382     1.485691126320    -2.072915670660
    H               -0.569086167382    -0.221606873680    -2.543405670660
    --
    0 1
    O                2.863911832618     0.163690126320    -0.635539670660
    H                3.428888832618    -0.549268873680    -0.940738670660
    H                3.443663832618     0.927883126320    -0.611641670660

""")

GEOS['%s-%s-%s' % (dbse, '16-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.091711738549    -0.482379745068     1.800232617703
    H                1.151334738549    -0.407386745068     1.569546617703
    H               -0.135680261451    -1.489433745068     2.142015617703
    H               -0.153611261451     0.200476254932     2.610129617703
    C               -0.798918261451    -0.164872745068     0.620433617703
    O               -2.021307261451    -0.192465745068     0.710497617703
    N               -0.137403261451     0.146152254932    -0.524455382297
    H                0.869825738549     0.146989254932    -0.520275382297
    C               -0.842107261451     0.472650254932    -1.742188382297
    H               -1.904355261451     0.399500254932    -1.530288382297
    H               -0.614288261451     1.485519254932    -2.071729382297
    H               -0.589609261451    -0.221778745068    -2.542219382297
    --
    0 1
    O                2.947155738549     0.164387254932    -0.640351382297
    H                3.512132738549    -0.548571745068    -0.945550382297
    H                3.526906738549     0.928580254932    -0.616453382297

""")

GEOS['%s-%s-%s' % (dbse, '16-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.071149263995    -0.482551836372     1.801421268367
    H                1.130772263995    -0.407558836372     1.570735268367
    H               -0.156242736005    -1.489605836372     2.143204268367
    H               -0.174173736005     0.200304163628     2.611318268367
    C               -0.819480736005    -0.165044836372     0.621622268367
    O               -2.041869736005    -0.192637836372     0.711686268367
    N               -0.157965736005     0.145980163628    -0.523266731633
    H                0.849263263995     0.146817163628    -0.519086731633
    C               -0.862669736005     0.472478163628    -1.740999731633
    H               -1.924917736005     0.399328163628    -1.529099731633
    H               -0.634850736005     1.485347163628    -2.070540731633
    H               -0.610171736005    -0.221950836372    -2.541030731633
    --
    0 1
    O                3.030559263995     0.165085163628    -0.645172731633
    H                3.595536263995    -0.547872836372    -0.950371731633
    H                3.610311263995     0.929279163628    -0.621273731633

""")

GEOS['%s-%s-%s' % (dbse, '16-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.050606391861    -0.482723894474     1.802608567768
    H                1.110229391861    -0.407730894474     1.571922567768
    H               -0.176785608139    -1.489777894474     2.144391567768
    H               -0.194716608139     0.200132105526     2.612505567768
    C               -0.840023608139    -0.165216894474     0.622809567768
    O               -2.062412608139    -0.192809894474     0.712873567768
    N               -0.178508608139     0.145808105526    -0.522079432232
    H                0.828720391861     0.146645105526    -0.517899432232
    C               -0.883212608139     0.472306105526    -1.739812432232
    H               -1.945460608139     0.399156105526    -1.527912432232
    H               -0.655393608139     1.485175105526    -2.069353432232
    H               -0.630714608139    -0.222122894474    -2.539843432232
    --
    0 1
    O                3.113883391861     0.165783105526    -0.649988432232
    H                3.678860391861    -0.547175894474    -0.955188432232
    H                3.693634391861     0.929977105526    -0.626090432232

""")

GEOS['%s-%s-%s' % (dbse, '16-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.011021873250    -0.483240090914     1.806170970776
    H                1.048601126750    -0.408247090914     1.575484970776
    H               -0.238413873250    -1.490294090914     2.147953970776
    H               -0.256344873250     0.199615909086     2.616067970776
    C               -0.901651873250    -0.165733090914     0.626371970776
    O               -2.124040873250    -0.193326090914     0.716435970776
    N               -0.240136873250     0.145291909086    -0.518517029224
    H                0.767092126750     0.146128909086    -0.514337029224
    C               -0.944840873250     0.471789909086    -1.736250029224
    H               -2.007088873250     0.398639909086    -1.524350029224
    H               -0.717021873250     1.484658909086    -2.065791029224
    H               -0.692342873250    -0.222639090914    -2.536281029224
    --
    0 1
    O                3.363854126750     0.167876909086    -0.664438029224
    H                3.928831126750    -0.545082090914    -0.969637029224
    H                3.943606126750     0.932069909086    -0.640539029224

""")

GEOS['%s-%s-%s' % (dbse, '16-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.113735882630    -0.484100238979     1.812108148232
    H                0.945887117370    -0.409107238979     1.581422148232
    H               -0.341127882630    -1.491154238979     2.153891148232
    H               -0.359058882630     0.198755761021     2.622005148232
    C               -1.004365882630    -0.166593238979     0.632309148232
    O               -2.226754882630    -0.194186238979     0.722373148232
    N               -0.342850882630     0.144431761021    -0.512579851768
    H                0.664378117370     0.145268761021    -0.508399851768
    C               -1.047554882630     0.470929761021    -1.730312851768
    H               -2.109802882630     0.397779761021    -1.518412851768
    H               -0.819735882630     1.483798761021    -2.059853851768
    H               -0.795056882630    -0.223499238979    -2.530343851768
    --
    0 1
    O                3.780473117370     0.171365761021    -0.688519851768
    H                4.345450117370    -0.541593238979    -0.993718851768
    H                4.360224117370     0.935558761021    -0.664620851768

""")

GEOS['%s-%s-%s' % (dbse, '16-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.319163725744    -0.485820557243     1.823982503144
    H                0.740459274256    -0.410827557243     1.593296503144
    H               -0.546555725744    -1.492874557243     2.165765503144
    H               -0.564486725744     0.197035442757     2.633879503144
    C               -1.209793725744    -0.168313557243     0.644183503144
    O               -2.432182725744    -0.195906557243     0.734247503144
    N               -0.548278725744     0.142711442757    -0.500705496856
    H                0.458950274256     0.143548442757    -0.496525496856
    C               -1.252982725744     0.469209442757    -1.718438496856
    H               -2.315230725744     0.396059442757    -1.506538496856
    H               -1.025163725744     1.482078442757    -2.047979496856
    H               -1.000484725744    -0.225219557243    -2.518469496856
    --
    0 1
    O                4.613710274256     0.178343442757    -0.736683496856
    H                5.178687274256    -0.534614557243    -1.041882496856
    H                5.193461274256     0.942537442757    -0.712784496856

""")

GEOS['%s-%s-%s' % (dbse, '17-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.654103572342     0.975143232358     0.000744703790
    H               -0.625681572342     1.026379232358     0.001452703790
    C               -2.220929572342    -0.288051767642     0.001331703790
    O               -1.518203572342    -1.301142767642     0.002463703790
    C               -3.667726572342    -0.309956767642     0.000308703790
    H               -4.173694572342    -1.259459767642     0.000662703790
    C               -4.346124572342     0.856466232358    -0.001093296210
    H               -5.425001572342     0.903159232358    -0.001915296210
    N               -3.698941572342     2.057784232358    -0.001588296210
    H               -4.207942572342     2.926252232358    -0.002656296210
    C               -2.315583572342     2.189395232358    -0.000695296210
    O               -1.763948572342     3.269410232358    -0.001171296210
    --
    0 1
    N                3.051347427658    -0.124554767642     0.000327703790
    H                3.498591427658     0.782019232358     0.000714703790
    C                3.907395427658    -1.239171767642    -0.001171296210
    O                5.116847427658    -1.087467767642    -0.001921296210
    C                3.193176427658    -2.501345767642    -0.001676296210
    H                3.759761427658    -3.416376767642    -0.002814296210
    C                1.843008427658    -2.505163767642    -0.000716296210
    H                1.266044427658    -3.418127767642    -0.001034296210
    N                1.105016427658    -1.354282767642     0.000734703790
    H                0.078680427658    -1.375725767642     0.001411703790
    C                1.677787427658    -0.109617767642     0.001284703790
    O                1.027466427658     0.931600232358     0.002521703790

""")

GEOS['%s-%s-%s' % (dbse, '17-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.698447518784     0.977453866301     0.000744404730
    H               -0.670025518784     1.028689866301     0.001452404730
    C               -2.265273518784    -0.285741133699     0.001331404730
    O               -1.562547518784    -1.298832133699     0.002463404730
    C               -3.712070518784    -0.307646133699     0.000308404730
    H               -4.218038518784    -1.257149133699     0.000662404730
    C               -4.390468518784     0.858776866301    -0.001093595270
    H               -5.469345518784     0.905469866301    -0.001915595270
    N               -3.743285518784     2.060094866301    -0.001588595270
    H               -4.252286518784     2.928562866301    -0.002656595270
    C               -2.359927518784     2.191705866301    -0.000695595270
    O               -1.808292518784     3.271720866301    -0.001171595270
    --
    0 1
    N                3.095691481216    -0.126865133699     0.000328404730
    H                3.542935481216     0.779707866301     0.000715404730
    C                3.951739481216    -1.241483133699    -0.001171595270
    O                5.161191481216    -1.089779133699    -0.001920595270
    C                3.237520481216    -2.503656133699    -0.001675595270
    H                3.804105481216    -3.418687133699    -0.002814595270
    C                1.887351481216    -2.507474133699    -0.000715595270
    H                1.310388481216    -3.420439133699    -0.001034595270
    N                1.149360481216    -1.356593133699     0.000734404730
    H                0.123024481216    -1.378036133699     0.001411404730
    C                1.722131481216    -0.111928133699     0.001285404730
    O                1.071810481216     0.929289866301     0.002521404730

""")

GEOS['%s-%s-%s' % (dbse, '17-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.742741518784     0.979762232358     0.000744266344
    H               -0.714319518784     1.030998232358     0.001452266344
    C               -2.309567518784    -0.283432767642     0.001331266344
    O               -1.606841518784    -1.296523767642     0.002463266344
    C               -3.756364518784    -0.305337767642     0.000308266344
    H               -4.262332518784    -1.254840767642     0.000662266344
    C               -4.434762518784     0.861085232358    -0.001093733656
    H               -5.513639518784     0.907778232358    -0.001915733656
    N               -3.787579518784     2.062403232358    -0.001588733656
    H               -4.296580518784     2.930871232358    -0.002656733656
    C               -2.404221518784     2.194014232358    -0.000695733656
    O               -1.852586518784     3.274029232358    -0.001171733656
    --
    0 1
    N                3.139985481216    -0.129173767642     0.000328266344
    H                3.587229481216     0.777400232358     0.000715266344
    C                3.996033481216    -1.243790767642    -0.001171733656
    O                5.205485481216    -1.092086767642    -0.001920733656
    C                3.281814481216    -2.505964767642    -0.001675733656
    H                3.848399481216    -3.420995767642    -0.002814733656
    C                1.931645481216    -2.509782767642    -0.000715733656
    H                1.354682481216    -3.422746767642    -0.001034733656
    N                1.193654481216    -1.358901767642     0.000735266344
    H                0.167318481216    -1.380344767642     0.001412266344
    C                1.766425481216    -0.114236767642     0.001285266344
    O                1.116104481216     0.926981232358     0.002522266344

""")

GEOS['%s-%s-%s' % (dbse, '17-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.787135125901     0.982075589475     0.000743904730
    H               -0.758713125901     1.033311589475     0.001451904730
    C               -2.353961125901    -0.281119410525     0.001330904730
    O               -1.651235125901    -1.294210410525     0.002462904730
    C               -3.800758125901    -0.303024410525     0.000307904730
    H               -4.306726125901    -1.252527410525     0.000661904730
    C               -4.479156125901     0.863398589475    -0.001094095270
    H               -5.558033125901     0.910091589475    -0.001916095270
    N               -3.831973125901     2.064716589475    -0.001589095270
    H               -4.340974125901     2.933184589475    -0.002657095270
    C               -2.448615125901     2.196327589475    -0.000696095270
    O               -1.896980125901     3.276342589475    -0.001172095270
    --
    0 1
    N                3.184378874099    -0.131486410525     0.000328904730
    H                3.631622874099     0.775086589475     0.000715904730
    C                4.040426874099    -1.246104410525    -0.001171095270
    O                5.249878874099    -1.094400410525    -0.001920095270
    C                3.326208874099    -2.508278410525    -0.001675095270
    H                3.892792874099    -3.423308410525    -0.002814095270
    C                1.976039874099    -2.512096410525    -0.000715095270
    H                1.399075874099    -3.425060410525    -0.001034095270
    N                1.238047874099    -1.361215410525     0.000734904730
    H                0.211711874099    -1.382657410525     0.001411904730
    C                1.810818874099    -0.116550410525     0.001285904730
    O                1.160497874099     0.924668589475     0.002521904730

""")

GEOS['%s-%s-%s' % (dbse, '17-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.831479018784     0.984386294913     0.000743828843
    H               -0.803057018784     1.035622294913     0.001451828843
    C               -2.398305018784    -0.278808705087     0.001330828843
    O               -1.695579018784    -1.291899705087     0.002462828843
    C               -3.845102018784    -0.300713705087     0.000307828843
    H               -4.351070018784    -1.250216705087     0.000661828843
    C               -4.523500018784     0.865709294913    -0.001094171157
    H               -5.602377018784     0.912402294913    -0.001916171157
    N               -3.876317018784     2.067027294913    -0.001589171157
    H               -4.385318018784     2.935495294913    -0.002657171157
    C               -2.492959018784     2.198638294913    -0.000696171157
    O               -1.941324018784     3.278653294913    -0.001172171157
    --
    0 1
    N                3.228722981216    -0.133797705087     0.000328828843
    H                3.675966981216     0.772775294913     0.000715828843
    C                4.084770981216    -1.248415705087    -0.001171171157
    O                5.294222981216    -1.096710705087    -0.001920171157
    C                3.370551981216    -2.510588705087    -0.001675171157
    H                3.937136981216    -3.425619705087    -0.002814171157
    C                2.020382981216    -2.514406705087    -0.000715171157
    H                1.443419981216    -3.427371705087    -0.001034171157
    N                1.282391981216    -1.363525705087     0.000734828843
    H                0.256055981216    -1.384968705087     0.001412828843
    C                1.855162981216    -0.118860705087     0.001285828843
    O                1.204841981216     0.922357294913     0.002522828843

""")

GEOS['%s-%s-%s' % (dbse, '17-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -1.964423518784     0.991314236856     0.000743069903
    H               -0.936001518784     1.042550236856     0.001451069903
    C               -2.531249518784    -0.271880763144     0.001330069903
    O               -1.828523518784    -1.284971763144     0.002462069903
    C               -3.978046518784    -0.293785763144     0.000307069903
    H               -4.484014518784    -1.243288763144     0.000661069903
    C               -4.656444518784     0.872637236856    -0.001094930097
    H               -5.735321518784     0.919330236856    -0.001916930097
    N               -4.009261518784     2.073955236856    -0.001589930097
    H               -4.518262518784     2.942423236856    -0.002657930097
    C               -2.625903518784     2.205566236856    -0.000696930097
    O               -2.074268518784     3.285581236856    -0.001172930097
    --
    0 1
    N                3.361667481216    -0.140725763144     0.000330069903
    H                3.808911481216     0.765848236856     0.000716069903
    C                4.217715481216    -1.255342763144    -0.001169930097
    O                5.427167481216    -1.103638763144    -0.001918930097
    C                3.503496481216    -2.517516763144    -0.001674930097
    H                4.070081481216    -3.432547763144    -0.002812930097
    C                2.153327481216    -2.521334763144    -0.000714930097
    H                1.576364481216    -3.434299763144    -0.001032930097
    N                1.415336481216    -1.370453763144     0.000736069903
    H                0.389000481216    -1.391896763144     0.001413069903
    C                1.988107481216    -0.125788763144     0.001286069903
    O                1.337786481216     0.915429236856     0.002523069903

""")

GEOS['%s-%s-%s' % (dbse, '17-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.186228014286     1.002872669860     0.000741904730
    H               -1.157806014286     1.054108669860     0.001449904730
    C               -2.753054014286    -0.260322330140     0.001328904730
    O               -2.050328014286    -1.273413330140     0.002460904730
    C               -4.199851014286    -0.282227330140     0.000305904730
    H               -4.705819014286    -1.231730330140     0.000659904730
    C               -4.878249014286     0.884195669860    -0.001096095270
    H               -5.957126014286     0.930888669860    -0.001918095270
    N               -4.231066014286     2.085513669860    -0.001591095270
    H               -4.740067014286     2.953981669860    -0.002659095270
    C               -2.847708014286     2.217124669860    -0.000698095270
    O               -2.296073014286     3.297139669860    -0.001174095270
    --
    0 1
    N                3.583471985714    -0.152283330140     0.000330904730
    H                4.030715985714     0.754289669860     0.000717904730
    C                4.439519985714    -1.266901330140    -0.001169095270
    O                5.648971985714    -1.115197330140    -0.001918095270
    C                3.725300985714    -2.529075330140    -0.001673095270
    H                4.291885985714    -3.444106330140    -0.002812095270
    C                2.375131985714    -2.532893330140    -0.000713095270
    H                1.798168985714    -3.445857330140    -0.001032095270
    N                1.637140985714    -1.382012330140     0.000736904730
    H                0.610803985714    -1.403455330140     0.001413904730
    C                2.209911985714    -0.137347330140     0.001287904730
    O                1.559590985714     0.903870669860     0.002523904730

""")

GEOS['%s-%s-%s' % (dbse, '17-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -2.629662625901     1.025980294913     0.000739703790
    H               -1.601240625901     1.077216294913     0.001447703790
    C               -3.196488625901    -0.237214705087     0.001326703790
    O               -2.493762625901    -1.250305705087     0.002458703790
    C               -4.643285625901    -0.259119705087     0.000303703790
    H               -5.149253625901    -1.208622705087     0.000657703790
    C               -5.321683625901     0.907303294913    -0.001098296210
    H               -6.400560625901     0.953996294913    -0.001920296210
    N               -4.674500625901     2.108621294913    -0.001593296210
    H               -5.183501625901     2.977089294913    -0.002661296210
    C               -3.291142625901     2.240232294913    -0.000700296210
    O               -2.739507625901     3.320247294913    -0.001176296210
    --
    0 1
    N                4.026906374099    -0.175391705087     0.000332703790
    H                4.474150374099     0.731181294913     0.000719703790
    C                4.882954374099    -1.290009705087    -0.001166296210
    O                6.092406374099    -1.138304705087    -0.001916296210
    C                4.168736374099    -2.552182705087    -0.001671296210
    H                4.735320374099    -3.467213705087    -0.002809296210
    C                2.818567374099    -2.556000705087    -0.000711296210
    H                2.241603374099    -3.468965705087    -0.001029296210
    N                2.080575374099    -1.405119705087     0.000739703790
    H                1.054239374099    -1.426562705087     0.001416703790
    C                2.653346374099    -0.160454705087     0.001289703790
    O                2.003025374099     0.880763294913     0.002526703790

""")

GEOS['%s-%s-%s' % (dbse, '18-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.383651752551    -0.111466639613    -0.000445671240
    H               -3.702580752551     0.792022360387     0.000197328760
    H               -2.418161752551    -0.011601639613    -0.000198671240
    --
    0 1
    N               -0.624374752551     0.078902360387     0.000168328760
    C                0.071531247449     0.044932360387     1.144915328760
    H               -0.513140752551     0.068542360387     2.055783328760
    C                1.460555247449    -0.020250639613     1.194912328760
    H                1.966461247449    -0.045104639613     2.149241328760
    C                2.171718247449    -0.053376639613     0.000000328760
    H                3.251558247449    -0.105149639613    -0.000079671240
    C                1.460410247449    -0.020317639613    -1.194800671240
    H                1.966207247449    -0.045190639613    -2.149199671240
    C                0.071380247449     0.044877360387    -1.144654671240
    H               -0.513388752551     0.068478360387    -2.055446671240

""")

GEOS['%s-%s-%s' % (dbse, '18-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.464808515040    -0.115561434586    -0.000462351441
    H               -3.783737515040     0.787927565414     0.000180648559
    H               -2.499318515040    -0.015696434586    -0.000215351441
    --
    0 1
    N               -0.605882515040     0.079835565414     0.000171648559
    C                0.090024484960     0.045864565414     1.144919648559
    H               -0.494648515040     0.069475565414     2.055787648559
    C                1.479047484960    -0.019317434586     1.194915648559
    H                1.984953484960    -0.044171434586     2.149244648559
    C                2.190210484960    -0.052443434586     0.000004648559
    H                3.270050484960    -0.104217434586    -0.000076351441
    C                1.478902484960    -0.019384434586    -1.194797351441
    H                1.984699484960    -0.044257434586    -2.149195351441
    C                0.089872484960     0.045810565414    -1.144650351441
    H               -0.494896515040     0.069411565414    -2.055443351441

""")

GEOS['%s-%s-%s' % (dbse, '18-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.545883855796    -0.119652281079    -0.000478773970
    H               -3.864812855796     0.783836718921     0.000164226030
    H               -2.580393855796    -0.019787281079    -0.000231773970
    --
    0 1
    N               -0.587408855796     0.080767718921     0.000175226030
    C                0.108498144204     0.046796718921     1.144923226030
    H               -0.476174855796     0.070407718921     2.055791226030
    C                1.497521144204    -0.018385281079     1.194919226030
    H                2.003428144204    -0.043240281079     2.149248226030
    C                2.208685144204    -0.051511281079     0.000008226030
    H                3.288525144204    -0.103285281079    -0.000071773970
    C                1.497376144204    -0.018452281079    -1.194792773970
    H                2.003173144204    -0.043325281079    -2.149191773970
    C                0.108346144204     0.046742718921    -1.144646773970
    H               -0.476422855796     0.070343718921    -2.055439773970

""")

GEOS['%s-%s-%s' % (dbse, '18-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.627122071171    -0.123751148178    -0.000495495579
    H               -3.946051071171     0.779737851822     0.000147504421
    H               -2.661632071171    -0.023886148178    -0.000248495579
    --
    0 1
    N               -0.568898071171     0.081701851822     0.000179504421
    C                0.127008928829     0.047730851822     1.144927504421
    H               -0.457664071171     0.071341851822     2.055795504421
    C                1.516032928829    -0.017452148178     1.194923504421
    H                2.021938928829    -0.042306148178     2.149252504421
    C                2.227195928829    -0.050577148178     0.000011504421
    H                3.307035928829    -0.102351148178    -0.000068495579
    C                1.515886928829    -0.017518148178    -1.194789495579
    H                2.021683928829    -0.042391148178    -2.149187495579
    C                0.126856928829     0.047676851822    -1.144643495579
    H               -0.457911071171     0.071276851822    -2.055435495579

""")

GEOS['%s-%s-%s' % (dbse, '18-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.708279122357    -0.127845541196    -0.000512155011
    H               -4.027208122357     0.775643458804     0.000130844989
    H               -2.742789122357    -0.027980541196    -0.000265155011
    --
    0 1
    N               -0.550405122357     0.082634458804     0.000182844989
    C                0.145500877643     0.048664458804     1.144930844989
    H               -0.439171122357     0.072274458804     2.055798844989
    C                1.534524877643    -0.016518541196     1.194926844989
    H                2.040430877643    -0.041372541196     2.149255844989
    C                2.245687877643    -0.049644541196     0.000015844989
    H                3.325527877643    -0.101417541196    -0.000065155011
    C                1.534379877643    -0.016585541196    -1.194785155011
    H                2.040176877643    -0.041458541196    -2.149184155011
    C                0.145349877643     0.048609458804    -1.144639155011
    H               -0.439419122357     0.072210458804    -2.055432155011

""")

GEOS['%s-%s-%s' % (dbse, '18-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.951711833014    -0.140127761282    -0.000561834937
    H               -4.270640833014     0.763361238718     0.000081165063
    H               -2.986221833014    -0.040262761282    -0.000314834937
    --
    0 1
    N               -0.494936833014     0.085433238718     0.000194165063
    C                0.200970166986     0.051462238718     1.144942165063
    H               -0.383702833014     0.075073238718     2.055810165063
    C                1.589993166986    -0.013719761282     1.194938165063
    H                2.095900166986    -0.038573761282     2.149267165063
    C                2.301157166986    -0.046845761282     0.000027165063
    H                3.380996166986    -0.098618761282    -0.000053834937
    C                1.589848166986    -0.013786761282    -1.194773834937
    H                2.095645166986    -0.038659761282    -2.149172834937
    C                0.200818166986     0.051408238718    -1.144627834937
    H               -0.383950833014     0.075009238718    -2.055420834937

""")

GEOS['%s-%s-%s' % (dbse, '18-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -4.357533717685    -0.160603344361    -0.000644782645
    H               -4.676462717685     0.742885655639    -0.000001782645
    H               -3.392043717685    -0.060738344361    -0.000397782645
    --
    0 1
    N               -0.402466717685     0.090098655639     0.000213217355
    C                0.293440282315     0.056128655639     1.144961217355
    H               -0.291231717685     0.079738655639     2.055829217355
    C                1.682464282315    -0.009054344361     1.194957217355
    H                2.188370282315    -0.033908344361     2.149286217355
    C                2.393627282315    -0.042180344361     0.000046217355
    H                3.473467282315    -0.093953344361    -0.000034782645
    C                1.682319282315    -0.009121344361    -1.194755782645
    H                2.188116282315    -0.033994344361    -2.149153782645
    C                0.293289282315     0.056073655639    -1.144608782645
    H               -0.291479717685     0.079674655639    -2.055401782645

""")

GEOS['%s-%s-%s' % (dbse, '18-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -5.169102002914    -0.201550861118    -0.000810255337
    H               -5.488031002914     0.701938138882    -0.000167255337
    H               -4.203612002914    -0.101685861118    -0.000563255337
    --
    0 1
    N               -0.217542002914     0.099429138882     0.000250744663
    C                0.478364997086     0.065458138882     1.144998744663
    H               -0.106308002914     0.089069138882     2.055866744663
    C                1.867387997086     0.000276138882     1.194994744663
    H                2.373293997086    -0.024578861118     2.149323744663
    C                2.578550997086    -0.032849861118     0.000083744663
    H                3.658390997086    -0.084623861118     0.000003744663
    C                1.867242997086     0.000209138882    -1.194717255337
    H                2.373039997086    -0.024663861118    -2.149116255337
    C                0.478212997086     0.065404138882    -1.144571255337
    H               -0.106556002914     0.089005138882    -2.055363255337

""")

GEOS['%s-%s-%s' % (dbse, '19-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.885517700681     0.273595530865     0.001405877239
    H               -1.914263700681     0.308431530865    -0.000665122761
    C               -3.235796700681    -1.092427469135     0.001172877239
    H               -4.321257700681    -1.155914469135     0.004943877239
    H               -2.872749700681    -1.620247469135    -0.885449122761
    H               -2.866505700681    -1.622108469135     0.884107877239
    --
    0 1
    N               -0.159294700681     0.212099530865    -0.001599122761
    C                0.537074299319     0.187663530865    -1.146319122761
    H               -0.047404700681     0.210887530865    -2.057294122761
    C                1.926711299319     0.137920530865    -1.195004122761
    H                2.434063299319     0.120628530865    -2.148787122761
    C                2.637084299319     0.112208530865     0.000608877239
    H                3.717420299319     0.074443530865     0.001457877239
    C                1.924837299319     0.137229530865     1.195148877239
    H                2.430635299319     0.119404530865     2.149633877239
    C                0.535283299319     0.186979530865     1.144194877239
    H               -0.050722700681     0.209653530865     2.054312877239

""")

GEOS['%s-%s-%s' % (dbse, '19-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -2.954898804199     0.277404064897     0.001442883209
    H               -1.983644804199     0.312240064897    -0.000628116791
    C               -3.305177804199    -1.088618935103     0.001209883209
    H               -4.390638804199    -1.152105935103     0.004980883209
    H               -2.942130804199    -1.616438935103    -0.885412116791
    H               -2.935886804199    -1.618299935103     0.884144883209
    --
    0 1
    N               -0.131182804199     0.210556064897    -0.001614116791
    C                0.565186195801     0.186121064897    -1.146334116791
    H               -0.019292804199     0.209344064897    -2.057309116791
    C                1.954823195801     0.136378064897    -1.195019116791
    H                2.462175195801     0.119086064897    -2.148802116791
    C                2.665196195801     0.110665064897     0.000593883209
    H                3.745532195801     0.072900064897     0.001442883209
    C                1.952948195801     0.135686064897     1.195133883209
    H                2.458747195801     0.117862064897     2.149618883209
    C                0.563395195801     0.185436064897     1.144179883209
    H               -0.022610804199     0.208110064897     2.054297883209

""")

GEOS['%s-%s-%s' % (dbse, '19-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.024208742390     0.281208563241     0.001479871031
    H               -2.052954742390     0.316044563241    -0.000591128969
    C               -3.374487742390    -1.084814436759     0.001246871031
    H               -4.459948742390    -1.148301436759     0.005017871031
    H               -3.011440742390    -1.612634436759    -0.885375128969
    H               -3.005196742390    -1.614495436759     0.884181871031
    --
    0 1
    N               -0.103099742390     0.209014563241    -0.001629128969
    C                0.593268257610     0.184579563241    -1.146349128969
    H                0.008790257610     0.207802563241    -2.057324128969
    C                1.982906257610     0.134836563241    -1.195034128969
    H                2.490258257610     0.117544563241    -2.148817128969
    C                2.693279257610     0.109123563241     0.000578871031
    H                3.773615257610     0.071358563241     0.001427871031
    C                1.981031257610     0.134144563241     1.195118871031
    H                2.486830257610     0.116320563241     2.149604871031
    C                0.591478257610     0.183894563241     1.144164871031
    H                0.005472257610     0.206568563241     2.054283871031

""")

GEOS['%s-%s-%s' % (dbse, '19-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.093660758855     0.285020889770     0.001516877000
    H               -2.122406758855     0.319856889770    -0.000554123000
    C               -3.443939758855    -1.081002110230     0.001283877000
    H               -4.529400758855    -1.144489110230     0.005054877000
    H               -3.080892758855    -1.608822110230    -0.885338123000
    H               -3.074648758855    -1.610683110230     0.884218877000
    --
    0 1
    N               -0.074958758855     0.207469889770    -0.001644123000
    C                0.621409241145     0.183034889770    -1.146364123000
    H                0.036930241145     0.206257889770    -2.057339123000
    C                2.011046241145     0.133291889770    -1.195049123000
    H                2.518398241145     0.115999889770    -2.148832123000
    C                2.721419241145     0.107578889770     0.000563877000
    H                3.801755241145     0.069813889770     0.001412877000
    C                2.009172241145     0.132599889770     1.195103877000
    H                2.514971241145     0.114775889770     2.149589877000
    C                0.619618241145     0.182349889770     1.144149877000
    H                0.033612241145     0.205023889770     2.054268877000

""")

GEOS['%s-%s-%s' % (dbse, '19-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.163041835264     0.288829414728     0.001553756781
    H               -2.191787835264     0.323665414728    -0.000517243219
    C               -3.513320835264    -1.077193585272     0.001320756781
    H               -4.598781835264    -1.140680585272     0.005091756781
    H               -3.150273835264    -1.605013585272    -0.885301243219
    H               -3.144029835264    -1.606874585272     0.884255756781
    --
    0 1
    N               -0.046847835264     0.205926414728    -0.001659243219
    C                0.649521164736     0.181491414728    -1.146378243219
    H                0.065042164736     0.204715414728    -2.057354243219
    C                2.039158164736     0.131748414728    -1.195064243219
    H                2.546510164736     0.114456414728    -2.148846243219
    C                2.749531164736     0.106036414728     0.000548756781
    H                3.829867164736     0.068271414728     0.001398756781
    C                2.037284164736     0.131057414728     1.195088756781
    H                2.543082164736     0.113232414728     2.149574756781
    C                0.647730164736     0.180806414728     1.144134756781
    H                0.061724164736     0.203481414728     2.054253756781

""")

GEOS['%s-%s-%s' % (dbse, '19-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.371184893437     0.300254539515     0.001664558607
    H               -2.399930893437     0.335090539515    -0.000406441393
    C               -3.721463893437    -1.065768460485     0.001431558607
    H               -4.806924893437    -1.129255460485     0.005202558607
    H               -3.358416893437    -1.593588460485    -0.885190441393
    H               -3.352172893437    -1.595449460485     0.884366558607
    --
    0 1
    N                0.037488106563     0.201297539515    -0.001704441393
    C                0.733856106563     0.176862539515    -1.146423441393
    H                0.149377106563     0.200085539515    -2.057399441393
    C                2.123493106563     0.127119539515    -1.195108441393
    H                2.630845106563     0.109827539515    -2.148891441393
    C                2.833866106563     0.101406539515     0.000504558607
    H                3.914202106563     0.063641539515     0.001353558607
    C                2.121619106563     0.126427539515     1.195043558607
    H                2.627418106563     0.108603539515     2.149529558607
    C                0.732065106563     0.176177539515     1.144089558607
    H                0.146059106563     0.198851539515     2.054208558607

""")

GEOS['%s-%s-%s' % (dbse, '19-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -3.718090104428     0.319296822261     0.001849210000
    H               -2.746836104428     0.354132822261    -0.000221790000
    C               -4.068369104428    -1.046726177739     0.001616210000
    H               -5.153830104428    -1.110213177739     0.005387210000
    H               -3.705322104428    -1.574546177739    -0.885005790000
    H               -3.699078104428    -1.576407177739     0.884551210000
    --
    0 1
    N                0.178045895572     0.193581822261    -0.001778790000
    C                0.874414895572     0.169146822261    -1.146498790000
    H                0.289935895572     0.192370822261    -2.057473790000
    C                2.264051895572     0.119403822261    -1.195183790000
    H                2.771403895572     0.102111822261    -2.148966790000
    C                2.974424895572     0.093690822261     0.000429210000
    H                4.054760895572     0.055926822261     0.001279210000
    C                2.262177895572     0.118712822261     1.194969210000
    H                2.767975895572     0.100887822261     2.149455210000
    C                0.872623895572     0.168461822261     1.144015210000
    H                0.286617895572     0.191136822261     2.054134210000

""")

GEOS['%s-%s-%s' % (dbse, '19-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    O               -4.411900472586     0.357381063740     0.002218440928
    H               -3.440646472586     0.392217063740     0.000147440928
    C               -4.762179472586    -1.008641936260     0.001985440928
    H               -5.847640472586    -1.072128936260     0.005756440928
    H               -4.399132472586    -1.536461936260    -0.884636559072
    H               -4.392888472586    -1.538322936260     0.884920440928
    --
    0 1
    N                0.459163527414     0.178151063740    -0.001928559072
    C                1.155531527414     0.153716063740    -1.146647559072
    H                0.571053527414     0.176939063740    -2.057623559072
    C                2.545168527414     0.103973063740    -1.195333559072
    H                3.052521527414     0.086681063740    -2.149115559072
    C                3.255542527414     0.078260063740     0.000279440928
    H                4.335878527414     0.040496063740     0.001129440928
    C                2.543294527414     0.103281063740     1.194819440928
    H                3.049093527414     0.085457063740     2.149305440928
    C                1.153741527414     0.153031063740     1.143865440928
    H                0.567735527414     0.175706063740     2.053984440928

""")

GEOS['%s-%s-%s' % (dbse, '20-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.846096072408     0.094754008774     0.010795518157
    O               -1.142548072408     1.068072008774     0.250548518157
    O               -1.373691072408    -1.107468991226    -0.277475481843
    H               -0.379951072408    -1.074663991226    -0.262852481843
    C               -3.342815072408     0.140163008774     0.014992518157
    H               -3.680385072408     1.145077008774     0.237051518157
    H               -3.717276072408    -0.179996991226    -0.954260481843
    H               -3.721599072408    -0.557476991226     0.758292518157
    --
    0 1
    C                1.846126927592    -0.094670991226    -0.011296481843
    O                1.142508927592    -1.067984991226    -0.251083481843
    O                1.373700927592     1.107765008774     0.276482518157
    H                0.379911927592     1.074867008774     0.262081518157
    C                3.342838927592    -0.140572991226    -0.013260481843
    H                3.680418927592    -1.141892991226    -0.250786481843
    H                3.724337927592     0.569780008774    -0.742729481843
    H                3.714355927592     0.162121008774     0.962794518157

""")

GEOS['%s-%s-%s' % (dbse, '20-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.888383680768     0.094566900413     0.010472009762
    O               -1.184835680768     1.067884900413     0.250225009762
    O               -1.415978680768    -1.107656099587    -0.277798990238
    H               -0.422238680768    -1.074851099587    -0.263175990238
    C               -3.385102680768     0.139975900413     0.014669009762
    H               -3.722672680768     1.144889900413     0.236728009762
    H               -3.759563680768    -0.180184099587    -0.954583990238
    H               -3.763886680768    -0.557664099587     0.757969009762
    --
    0 1
    C                1.888414319232    -0.094483099587    -0.010972990238
    O                1.184796319232    -1.067798099587    -0.250759990238
    O                1.415988319232     1.107951900413     0.276806009762
    H                0.422200319232     1.075053900413     0.262405009762
    C                3.385127319232    -0.140386099587    -0.012936990238
    H                3.722706319232    -1.141705099587    -0.250461990238
    H                3.766625319232     0.569966900413    -0.742405990238
    H                3.756643319232     0.162307900413     0.963118009762

""")

GEOS['%s-%s-%s' % (dbse, '20-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.930621572408     0.094380000378     0.010148784645
    O               -1.227073572408     1.067698000378     0.249901784645
    O               -1.458216572408    -1.107842999622    -0.278122215355
    H               -0.464476572408    -1.075037999622    -0.263499215355
    C               -3.427340572408     0.139789000378     0.014345784645
    H               -3.764910572408     1.144703000378     0.236404784645
    H               -3.801801572408    -0.180370999622    -0.954907215355
    H               -3.806124572408    -0.557850999622     0.757645784645
    --
    0 1
    C                1.930652427592    -0.094296999622    -0.010649215355
    O                1.227034427592    -1.067610999622    -0.250437215355
    O                1.458226427592     1.108139000378     0.277128784645
    H                0.464437427592     1.075241000378     0.262728784645
    C                3.427364427592    -0.140198999622    -0.012613215355
    H                3.764944427592    -1.141517999622    -0.250139215355
    H                3.808863427592     0.570154000378    -0.742082215355
    H                3.798881427592     0.162495000378     0.963441784645

""")

GEOS['%s-%s-%s' % (dbse, '20-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.972959189164     0.094192625530     0.009824784645
    O               -1.269411189164     1.067510625530     0.249577784645
    O               -1.500554189164    -1.108030374470    -0.278446215355
    H               -0.506814189164    -1.075225374470    -0.263823215355
    C               -3.469678189164     0.139601625530     0.014021784645
    H               -3.807248189164     1.144515625530     0.236080784645
    H               -3.844139189164    -0.180558374470    -0.955231215355
    H               -3.848462189164    -0.558038374470     0.757321784645
    --
    0 1
    C                1.972989810836    -0.094109374470    -0.010325215355
    O                1.269371810836    -1.067423374470    -0.250113215355
    O                1.500563810836     1.108326625530     0.277452784645
    H                0.506775810836     1.075427625530     0.263052784645
    C                3.469702810836    -0.140012374470    -0.012289215355
    H                3.807282810836    -1.141331374470    -0.249815215355
    H                3.851200810836     0.570340625530    -0.741758215355
    H                3.841218810836     0.162682625530     0.963765784645

""")

GEOS['%s-%s-%s' % (dbse, '20-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.015247072408     0.094005508774     0.009501018157
    O               -1.311699072408     1.067323508774     0.249254018157
    O               -1.542842072408    -1.108217491226    -0.278769981843
    H               -0.549102072408    -1.075412491226    -0.264146981843
    C               -3.511966072408     0.139414508774     0.013698018157
    H               -3.849536072408     1.144328508774     0.235757018157
    H               -3.886427072408    -0.180745491226    -0.955554981843
    H               -3.890750072408    -0.558225491226     0.756998018157
    --
    0 1
    C                2.015277927592    -0.093922491226    -0.010001981843
    O                1.311659927592    -1.067236491226    -0.249788981843
    O                1.542851927592     1.108513508774     0.277777018157
    H                0.549062927592     1.075615508774     0.263376018157
    C                3.511989927592    -0.139824491226    -0.011965981843
    H                3.849569927592    -1.141144491226    -0.249491981843
    H                3.893488927592     0.570528508774    -0.741434981843
    H                3.883506927592     0.162869508774     0.964089018157

""")

GEOS['%s-%s-%s' % (dbse, '20-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.142110297524     0.093444125530     0.008530284645
    O               -1.438562297524     1.066762125530     0.248283284645
    O               -1.669705297524    -1.108778874470    -0.279740715355
    H               -0.675965297524    -1.075973874470    -0.265117715355
    C               -3.638829297524     0.138853125530     0.012727284645
    H               -3.976399297524     1.143767125530     0.234786284645
    H               -4.013290297524    -0.181306874470    -0.956525715355
    H               -4.017613297524    -0.558786874470     0.756027284645
    --
    0 1
    C                2.142141702476    -0.093360874470    -0.009030715355
    O                1.438522702476    -1.066674874470    -0.248818715355
    O                1.669714702476     1.109075125530     0.278747284645
    H                0.675926702476     1.076176125530     0.264347284645
    C                3.638853702476    -0.139263874470    -0.010994715355
    H                3.976433702476    -1.140582874470    -0.248520715355
    H                4.020352702476     0.571089125530    -0.740463715355
    H                4.010369702476     0.163431125530     0.965060284645

""")

GEOS['%s-%s-%s' % (dbse, '20-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.353549172373     0.092508500378     0.006912018157
    O               -1.650001172373     1.065826500378     0.246665018157
    O               -1.881144172373    -1.109714499622    -0.281358981843
    H               -0.887404172373    -1.076909499622    -0.266735981843
    C               -3.850268172373     0.137917500378     0.011109018157
    H               -4.187838172373     1.142831500378     0.233168018157
    H               -4.224729172373    -0.182242499622    -0.958143981843
    H               -4.229052172373    -0.559722499622     0.754409018157
    --
    0 1
    C                2.353579827627    -0.092425499622    -0.007412981843
    O                1.649961827627    -1.065739499622    -0.247199981843
    O                1.881153827627     1.110010500378     0.280366018157
    H                0.887364827627     1.077112500378     0.265965018157
    C                3.850292827627    -0.138327499622    -0.009376981843
    H                4.187871827627    -1.139646499622    -0.246902981843
    H                4.231790827627     0.572025500378    -0.738845981843
    H                4.221808827627     0.164366500378     0.966678018157

""")

GEOS['%s-%s-%s' % (dbse, '20-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.776426797524     0.090637258774     0.003675893006
    O               -2.072878797524     1.063955258774     0.243428893006
    O               -2.304021797524    -1.111585741226    -0.284595106994
    H               -1.310281797524    -1.078780741226    -0.269972106994
    C               -4.273145797524     0.136046258774     0.007872893006
    H               -4.610715797524     1.140960258774     0.229931893006
    H               -4.647606797524    -0.184113741226    -0.961380106994
    H               -4.651929797524    -0.561593741226     0.751172893006
    --
    0 1
    C                2.776458202476    -0.090553741226    -0.004176106994
    O                2.072839202476    -1.063867741226    -0.243964106994
    O                2.304031202476     1.111881258774     0.283601893006
    H                1.310243202476     1.078983258774     0.269201893006
    C                4.273170202476    -0.136456741226    -0.006141106994
    H                4.610750202476    -1.137775741226    -0.243666106994
    H                4.654669202476     0.573896258774    -0.735610106994
    H                4.644686202476     0.166238258774     0.969914893006

""")

GEOS['%s-%s-%s' % (dbse, '21-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -1.959141934386     0.059815540726    -0.074278681395
    O               -1.374691934386     1.034787540726     0.403604318605
    N               -1.315012934386    -1.025305459274    -0.541102681395
    H               -0.294808934386    -1.068912459274    -0.508417681395
    H               -1.833018934386    -1.793960459274    -0.919864681395
    C               -3.466110934386     0.035638540726    -0.159714681395
    H               -3.870020934386     0.142180540726     0.843973318605
    H               -3.858819934386    -0.871726459274    -0.611008681395
    H               -3.792549934386     0.896235540726    -0.738001681395
    --
    0 1
    C                1.959111065614    -0.060022459274     0.074599318605
    O                1.374689065614    -1.034043459274    -0.405131681395
    N                1.315023065614     1.024720540726     0.542446318605
    H                0.294918065614     1.069158540726     0.508144318605
    H                1.833047065614     1.792681540726     0.922566318605
    C                3.466110065614    -0.035738459274     0.159824318605
    H                3.857981065614     0.863911540726     0.627008318605
    H                3.869666065614    -0.123283459274    -0.845955681395
    H                3.793889065614    -0.906313459274     0.722004318605

""")

GEOS['%s-%s-%s' % (dbse, '21-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.005513425850     0.058854040726    -0.077164707002
    O               -1.421063425850     1.033826040726     0.400718292998
    N               -1.361384425850    -1.026266959274    -0.543988707002
    H               -0.341180425850    -1.069873959274    -0.511303707002
    H               -1.879390425850    -1.794921959274    -0.922750707002
    C               -3.512482425850     0.034677040726    -0.162600707002
    H               -3.916392425850     0.141219040726     0.841087292998
    H               -3.905191425850    -0.872687959274    -0.613894707002
    H               -3.838921425850     0.895274040726    -0.740887707002
    --
    0 1
    C                2.005482574150    -0.059060959274     0.077485292998
    O                1.421060574150    -1.033081959274    -0.402245707002
    N                1.361394574150     1.025682040726     0.545332292998
    H                0.341289574150     1.070120040726     0.511031292998
    H                1.879418574150     1.793643040726     0.925453292998
    C                3.512481574150    -0.034776959274     0.162710292998
    H                3.904352574150     0.864873040726     0.629894292998
    H                3.916037574150    -0.122321959274    -0.843069707002
    H                3.840259574150    -0.905351959274     0.724891292998

""")

GEOS['%s-%s-%s' % (dbse, '21-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.051834714053     0.057893540726    -0.080047817168
    O               -1.467384714053     1.032865540726     0.397835182832
    N               -1.407705714053    -1.027227459274    -0.546871817168
    H               -0.387501714053    -1.070834459274    -0.514186817168
    H               -1.925711714053    -1.795882459274    -0.925633817168
    C               -3.558803714053     0.033716540726    -0.165483817168
    H               -3.962713714053     0.140258540726     0.838204182832
    H               -3.951512714053    -0.873648459274    -0.616777817168
    H               -3.885242714053     0.894313540726    -0.743770817168
    --
    0 1
    C                2.051803285947    -0.058100459274     0.080368182832
    O                1.467382285947    -1.032121459274    -0.399362817168
    N                1.407716285947     1.026642540726     0.548215182832
    H                0.387610285947     1.071080540726     0.513914182832
    H                1.925740285947     1.794603540726     0.928336182832
    C                3.558802285947    -0.033816459274     0.165594182832
    H                3.950674285947     0.865833540726     0.632778182832
    H                3.962359285947    -0.121361459274    -0.840186817168
    H                3.886581285947    -0.904391459274     0.727774182832

""")

GEOS['%s-%s-%s' % (dbse, '21-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.098255934386     0.056930786559    -0.082937062800
    O               -1.513805934386     1.031902786559     0.394945937200
    N               -1.454126934386    -1.028190213441    -0.549761062800
    H               -0.433922934386    -1.071797213441    -0.517076062800
    H               -1.972132934386    -1.796845213441    -0.928523062800
    C               -3.605224934386     0.032753786559    -0.168373062800
    H               -4.009134934386     0.139295786559     0.835314937200
    H               -3.997933934386    -0.874611213441    -0.619667062800
    H               -3.931663934386     0.893350786559    -0.746660062800
    --
    0 1
    C                2.098225065614    -0.057138213441     0.083257937200
    O                1.513803065614    -1.031158213441    -0.396473062800
    N                1.454137065614     1.027604786559     0.551103937200
    H                0.434032065614     1.072042786559     0.516802937200
    H                1.972161065614     1.795566786559     0.931224937200
    C                3.605224065614    -0.032853213441     0.168482937200
    H                3.997095065614     0.866795786559     0.635666937200
    H                4.008780065614    -0.120399213441    -0.837297062800
    H                3.933003065614    -0.903428213441     0.730662937200

""")

GEOS['%s-%s-%s' % (dbse, '21-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.144627061517     0.055969278024    -0.085823189931
    O               -1.560177061517     1.030941278024     0.392059810069
    N               -1.500498061517    -1.029151721976    -0.552647189931
    H               -0.480294061517    -1.072758721976    -0.519962189931
    H               -2.018504061517    -1.797806721976    -0.931409189931
    C               -3.651596061517     0.031792278024    -0.171259189931
    H               -4.055506061517     0.138334278024     0.832428810069
    H               -4.044305061517    -0.875572721976    -0.622553189931
    H               -3.978035061517     0.892389278024    -0.749546189931
    --
    0 1
    C                2.144595938483    -0.056176721976     0.086143810069
    O                1.560173938483    -1.030196721976    -0.393587189931
    N                1.500508938483     1.028566278024     0.553990810069
    H                0.480402938483     1.073004278024     0.519688810069
    H                2.018532938483     1.796528278024     0.934110810069
    C                3.651594938483    -0.031891721976     0.171368810069
    H                4.043465938483     0.867757278024     0.638552810069
    H                4.055150938483    -0.119436721976    -0.834411189931
    H                3.979373938483    -0.902466721976     0.733549810069

""")

GEOS['%s-%s-%s' % (dbse, '21-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.283740934386     0.053084540726    -0.094481689931
    O               -1.699290934386     1.028056540726     0.383401310069
    N               -1.639611934386    -1.032036459274    -0.561305689931
    H               -0.619407934386    -1.075643459274    -0.528620689931
    H               -2.157617934386    -1.800691459274    -0.940067689931
    C               -3.790709934386     0.028907540726    -0.179917689931
    H               -4.194619934386     0.135449540726     0.823770310069
    H               -4.183418934386    -0.878457459274    -0.631211689931
    H               -4.117148934386     0.889504540726    -0.758204689931
    --
    0 1
    C                2.283710065614    -0.053291459274     0.094802310069
    O                1.699288065614    -1.027312459274    -0.384928689931
    N                1.639622065614     1.031451540726     0.562649310069
    H                0.619517065614     1.075889540726     0.528347310069
    H                2.157646065614     1.799412540726     0.942769310069
    C                3.790709065614    -0.029007459274     0.180027310069
    H                4.182580065614     0.870642540726     0.647211310069
    H                4.194265065614    -0.116552459274    -0.825752689931
    H                4.118488065614    -0.899582459274     0.742208310069

""")

GEOS['%s-%s-%s' % (dbse, '21-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.515597315684     0.048276778024    -0.108912554264
    O               -1.931147315684     1.023248778024     0.368970445736
    N               -1.871468315684    -1.036844221976    -0.575736554264
    H               -0.851264315684    -1.080451221976    -0.543051554264
    H               -2.389474315684    -1.805499221976    -0.954498554264
    C               -4.022566315684     0.024099778024    -0.194348554264
    H               -4.426476315684     0.130641778024     0.809339445736
    H               -4.415275315684    -0.883265221976    -0.645642554264
    H               -4.349005315684     0.884696778024    -0.772635554264
    --
    0 1
    C                2.515565684316    -0.048484221976     0.109233445736
    O                1.931144684316    -1.022504221976    -0.370497554264
    N                1.871478684316     1.036258778024     0.577079445736
    H                0.851372684316     1.080696778024     0.542778445736
    H                2.389502684316     1.804220778024     0.957200445736
    C                4.022565684316    -0.024199221976     0.194458445736
    H                4.414436684316     0.875449778024     0.661642445736
    H                4.426121684316    -0.111744221976    -0.811322554264
    H                4.350343684316    -0.894774221976     0.756638445736

""")

GEOS['%s-%s-%s' % (dbse, '21-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.053978214053     0.037112888190    -0.142421207002
    O               -2.469528214053     1.012084888190     0.335461792998
    N               -2.409849214053    -1.048008111810    -0.609245207002
    H               -1.389645214053    -1.091615111810    -0.576560207002
    H               -2.927855214053    -1.816663111810    -0.988007207002
    C               -4.560947214053     0.012935888190    -0.227857207002
    H               -4.964857214053     0.119477888190     0.775830792998
    H               -4.953656214053    -0.894429111810    -0.679151207002
    H               -4.887386214053     0.873532888190    -0.806144207002
    --
    0 1
    C                3.053946785947    -0.037320111810     0.142741792998
    O                2.469525785947    -1.011340111810    -0.336989207002
    N                2.409859785947     1.047422888190     0.610588792998
    H                1.389753785947     1.091860888190     0.576287792998
    H                2.927883785947     1.815384888190     0.990709792998
    C                4.560945785947    -0.013036111810     0.227966792998
    H                4.952817785947     0.886613888190     0.695150792998
    H                4.964502785947    -0.100581111810    -0.777813207002
    H                4.888724785947    -0.883610111810     0.790147792998

""")

GEOS['%s-%s-%s' % (dbse, '22-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.075730686323    -0.363060967885    -0.038245726667
    O               -2.429187686323     0.659307032115     0.148124273333
    O               -2.540193686323    -1.553160967885    -0.263795726667
    H               -1.548784686323    -1.486823967885    -0.257925726667
    C               -4.573529686323    -0.403890967885    -0.036051726667
    H               -4.968753686323     0.586804032115     0.152377273333
    H               -4.926360686323    -0.774821967885    -0.995413726667
    H               -4.915218686323    -1.098282967885     0.727827273333
    --
    0 1
    N                2.064431313677    -0.606218967885    -0.129919726667
    H                2.457637313677    -1.521608967885    -0.301638726667
    C                2.985959313677     0.441897032115     0.051533273333
    O                4.183106313677     0.223601032115    -0.001346726667
    C                2.346893313677     1.721359032115     0.292140273333
    H                2.966428313677     2.588284032115     0.443618273333
    C                0.999809313677     1.801349032115     0.321522273333
    H                0.476969313677     2.729796032115     0.496820273333
    N                0.196078313677     0.710667032115     0.131116273333
    H               -0.827973686323     0.780615032115     0.154906273333
    C                0.696011313677    -0.539555967885    -0.103056726667
    O               -0.016021686323    -1.529060967885    -0.276526726667

""")

GEOS['%s-%s-%s' % (dbse, '22-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.131255926241    -0.364461885708    -0.038036536759
    O               -2.484712926241     0.657906114292     0.148333463241
    O               -2.595718926241    -1.554561885708    -0.263586536759
    H               -1.604309926241    -1.488224885708    -0.257716536759
    C               -4.629054926241    -0.405291885708    -0.035842536759
    H               -5.024278926241     0.585403114292     0.152586463241
    H               -4.981885926241    -0.776222885708    -0.995204536759
    H               -4.970743926241    -1.099683885708     0.728036463241
    --
    0 1
    N                2.094181073759    -0.605468885708    -0.130031536759
    H                2.487386073759    -1.520858885708    -0.301751536759
    C                3.015708073759     0.442647114292     0.051421463241
    O                4.212855073759     0.224351114292    -0.001458536759
    C                2.376642073759     1.722109114292     0.292028463241
    H                2.996177073759     2.589034114292     0.443505463241
    C                1.029558073759     1.802100114292     0.321409463241
    H                0.506718073759     2.730547114292     0.496707463241
    N                0.225827073759     0.711418114292     0.131003463241
    H               -0.798223926241     0.781365114292     0.154793463241
    C                0.725760073759    -0.538804885708    -0.103168536759
    O                0.013727073759    -1.528309885708    -0.276638536759

""")

GEOS['%s-%s-%s' % (dbse, '22-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.186708616505    -0.365861024548    -0.037827277315
    O               -2.540165616505     0.656506975452     0.148542722685
    O               -2.651171616505    -1.555961024548    -0.263377277315
    H               -1.659762616505    -1.489624024548    -0.257507277315
    C               -4.684507616505    -0.406691024548    -0.035633277315
    H               -5.079731616505     0.584003975452     0.152795722685
    H               -5.037338616505    -0.777622024548    -0.994995277315
    H               -5.026196616505    -1.101083024548     0.728245722685
    --
    0 1
    N                2.123890383495    -0.604719024548    -0.130144277315
    H                2.517095383495    -1.520109024548    -0.301863277315
    C                3.045418383495     0.443396975452     0.051308722685
    O                4.242564383495     0.225100975452    -0.001570277315
    C                2.406352383495     1.722858975452     0.291916722685
    H                3.025887383495     2.589783975452     0.443393722685
    C                1.059268383495     1.802848975452     0.321297722685
    H                0.536428383495     2.731295975452     0.496595722685
    N                0.255537383495     0.712166975452     0.130891722685
    H               -0.768514616505     0.782114975452     0.154681722685
    C                0.755470383495    -0.538055024548    -0.103280277315
    O                0.043437383495    -1.527560024548    -0.276751277315

""")

GEOS['%s-%s-%s' % (dbse, '22-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.242285057875    -0.367263244786    -0.037617773553
    O               -2.595742057875     0.655104755214     0.148752226447
    O               -2.706748057875    -1.557363244786    -0.263167773553
    H               -1.715339057875    -1.491026244786    -0.257297773553
    C               -4.740084057875    -0.408093244786    -0.035423773553
    H               -5.135308057875     0.582601755214     0.153005226447
    H               -5.092915057875    -0.779024244786    -0.994785773553
    H               -5.081773057875    -1.102485244786     0.728455226447
    --
    0 1
    N                2.153666942125    -0.603967244786    -0.130256773553
    H                2.546871942125    -1.519358244786    -0.301975773553
    C                3.075194942125     0.444147755214     0.051197226447
    O                4.272340942125     0.225851755214    -0.001682773553
    C                2.436128942125     1.723610755214     0.291804226447
    H                3.055662942125     2.590535755214     0.443281226447
    C                1.089043942125     1.803600755214     0.321185226447
    H                0.566203942125     2.732047755214     0.496483226447
    N                0.285313942125     0.712918755214     0.130779226447
    H               -0.738738057875     0.782866755214     0.154569226447
    C                0.785245942125    -0.537304244786    -0.103392773553
    O                0.073213942125    -1.526809244786    -0.276862773553

""")

GEOS['%s-%s-%s' % (dbse, '22-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.297790752007    -0.368663680044    -0.037408211897
    O               -2.651247752007     0.653704319956     0.148961788103
    O               -2.762253752007    -1.558763680044    -0.262958211897
    H               -1.770844752007    -1.492426680044    -0.257088211897
    C               -4.795589752007    -0.409493680044    -0.035214211897
    H               -5.190813752007     0.581201319956     0.153214788103
    H               -5.148420752007    -0.780424680044    -0.994576211897
    H               -5.137278752007    -1.103885680044     0.728664788103
    --
    0 1
    N                2.183405247993    -0.603217680044    -0.130368211897
    H                2.576610247993    -1.518607680044    -0.302087211897
    C                3.104933247993     0.444898319956     0.051084788103
    O                4.302079247993     0.226602319956    -0.001795211897
    C                2.465867247993     1.724360319956     0.291691788103
    H                3.085402247993     2.591285319956     0.443169788103
    C                1.118783247993     1.804351319956     0.321072788103
    H                0.595942247993     2.732798319956     0.496371788103
    N                0.315052247993     0.713669319956     0.130666788103
    H               -0.708999752007     0.783616319956     0.154457788103
    C                0.814984247993    -0.536553680044    -0.103505211897
    O                0.102952247993    -1.526058680044    -0.276975211897

""")

GEOS['%s-%s-%s' % (dbse, '22-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.464281079494    -0.372864334957    -0.036780491312
    O               -2.817738079494     0.649503665043     0.149589508688
    O               -2.928744079494    -1.562964334957    -0.262330491312
    H               -1.937335079494    -1.496627334957    -0.256460491312
    C               -4.962080079494    -0.413694334957    -0.034586491312
    H               -5.357304079494     0.577000665043     0.153842508688
    H               -5.314911079494    -0.784625334957    -0.993948491312
    H               -5.303769079494    -1.108086334957     0.729292508688
    --
    0 1
    N                2.272605920506    -0.600966334957    -0.130704491312
    H                2.665810920506    -1.516357334957    -0.302424491312
    C                3.194133920506     0.447148665043     0.050748508688
    O                4.391280920506     0.228852665043    -0.002131491312
    C                2.555067920506     1.726611665043     0.291355508688
    H                3.174602920506     2.593536665043     0.442832508688
    C                1.207983920506     1.806601665043     0.320736508688
    H                0.685143920506     2.735048665043     0.496034508688
    N                0.404252920506     0.715919665043     0.130330508688
    H               -0.619799079494     0.785867665043     0.154120508688
    C                0.904185920506    -0.534303334957    -0.103841491312
    O                0.192152920506    -1.523808334957    -0.277311491312

""")

GEOS['%s-%s-%s' % (dbse, '22-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.741702873893    -0.379863749976    -0.035734124057
    O               -3.095159873893     0.642504250024     0.150635875943
    O               -3.206165873893    -1.569963749976    -0.261284124057
    H               -2.214756873893    -1.503626749976    -0.255414124057
    C               -5.239501873893    -0.420693749976    -0.033540124057
    H               -5.634725873893     0.570001250024     0.154888875943
    H               -5.592332873893    -0.791624749976    -0.992902124057
    H               -5.581190873893    -1.115085749976     0.730338875943
    --
    0 1
    N                2.421241126107    -0.597216749976    -0.131265124057
    H                2.814446126107    -1.512606749976    -0.302985124057
    C                3.342769126107     0.450899250024     0.050187875943
    O                4.539915126107     0.232603250024    -0.002692124057
    C                2.703703126107     1.730361250024     0.290794875943
    H                3.323238126107     2.597286250024     0.442271875943
    C                1.356619126107     1.810351250024     0.320175875943
    H                0.833779126107     2.738798250024     0.495474875943
    N                0.552888126107     0.719669250024     0.129769875943
    H               -0.471163873893     0.789617250024     0.153560875943
    C                1.052820126107    -0.530552749976    -0.104402124057
    O                0.340788126107    -1.520057749976    -0.277872124057

""")

GEOS['%s-%s-%s' % (dbse, '22-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -4.296418211561    -0.393859621019    -0.033642017253
    O               -3.649875211561     0.628508378981     0.152727982747
    O               -3.760881211561    -1.583959621019    -0.259192017253
    H               -2.769472211561    -1.517622621019    -0.253322017253
    C               -5.794217211561    -0.434689621019    -0.031448017253
    H               -6.189441211561     0.556005378981     0.156980982747
    H               -6.147048211561    -0.805620621019    -0.990810017253
    H               -6.135906211561    -1.129081621019     0.732430982747
    --
    0 1
    N                2.718442788439    -0.589717621019    -0.132386017253
    H                3.111647788439    -1.505108621019    -0.304106017253
    C                3.639969788439     0.458397378981     0.049066982747
    O                4.837116788439     0.240101378981    -0.003813017253
    C                3.000903788439     1.737860378981     0.289673982747
    H                3.620438788439     2.604784378981     0.441150982747
    C                1.653819788439     1.817850378981     0.319054982747
    H                1.130979788439     2.746297378981     0.494353982747
    N                0.850088788439     0.727168378981     0.128648982747
    H               -0.173962211561     0.797115378981     0.152439982747
    C                1.350021788439    -0.523054621019    -0.105523017253
    O                0.637989788439    -1.512559621019    -0.278993017253

""")

GEOS['%s-%s-%s' % (dbse, '23-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.110215995244    -0.443650069534    -0.080527814256
    O               -2.448765995244     0.586726930466     0.086713185744
    N               -2.548069995244    -1.651400069534    -0.253184814256
    H               -1.533641995244    -1.745849069534    -0.247230814256
    H               -3.121702995244    -2.461613069534    -0.385586814256
    C               -4.615733995244    -0.388531069534    -0.094790814256
    H               -4.955463995244    -0.008684069534     0.865598185744
    H               -5.079603995244    -1.352424069534    -0.285114814256
    H               -4.927475995244     0.320200930466    -0.857500814256
    --
    0 1
    N                2.149653004756    -0.540390069534    -0.060561814256
    H                2.639682004756    -1.416347069534    -0.178500814256
    C                2.953279004756     0.602454930466     0.089919185744
    O                4.168841004756     0.506832930466     0.074735185744
    C                2.180687004756     1.817771930466     0.253063185744
    H                2.702450004756     2.750964930466     0.376098185744
    C                0.831526004756     1.756095930466     0.247387185744
    H                0.211122004756     2.632456930466     0.365063185744
    N                0.146444004756     0.583814930466     0.092821185744
    H               -0.886345995244     0.574753930466     0.091983185744
    C                0.777334004756    -0.620655069534    -0.069438814256
    O                0.180145004756    -1.684571069534    -0.211246814256

""")

GEOS['%s-%s-%s' % (dbse, '23-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.167064400605    -0.444505733423    -0.081243619789
    O               -2.505614400605     0.585871266577     0.085997380211
    N               -2.604918400605    -1.652255733423    -0.253900619789
    H               -1.590490400605    -1.746704733423    -0.247946619789
    H               -3.178551400605    -2.462468733423    -0.386302619789
    C               -4.672582400605    -0.389386733423    -0.095506619789
    H               -5.012312400605    -0.009539733423     0.864882380211
    H               -5.136452400605    -1.353279733423    -0.285830619789
    H               -4.984324400605     0.319345266577    -0.858216619789
    --
    0 1
    N                2.179611599395    -0.539938733423    -0.060184619789
    H                2.669640599395    -1.415896733423    -0.178123619789
    C                2.983237599395     0.602906266577     0.090296380211
    O                4.198799599395     0.507283266577     0.075112380211
    C                2.210644599395     1.818222266577     0.253440380211
    H                2.732407599395     2.751415266577     0.376476380211
    C                0.861484599395     1.756547266577     0.247764380211
    H                0.241080599395     2.632907266577     0.365441380211
    N                0.176402599395     0.584265266577     0.093198380211
    H               -0.856388400605     0.575204266577     0.092361380211
    C                0.807292599395    -0.620203733423    -0.069061619789
    O                0.210103599395    -1.684119733423    -0.210869619789

""")

GEOS['%s-%s-%s' % (dbse, '23-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.223845510920    -0.445360052671    -0.081958571438
    O               -2.562395510920     0.585016947329     0.085282428562
    N               -2.661699510920    -1.653110052671    -0.254615571438
    H               -1.647271510920    -1.747559052671    -0.248661571438
    H               -3.235332510920    -2.463323052671    -0.387017571438
    C               -4.729363510920    -0.390241052671    -0.096221571438
    H               -5.069093510920    -0.010394052671     0.864167428562
    H               -5.193233510920    -1.354134052671    -0.286545571438
    H               -5.041105510920     0.318490947329    -0.858931571438
    --
    0 1
    N                2.209534489080    -0.539488052671    -0.059807571438
    H                2.699563489080    -1.415446052671    -0.177746571438
    C                3.013160489080     0.603355947329     0.090673428562
    O                4.228722489080     0.507733947329     0.075488428562
    C                2.240568489080     1.818672947329     0.253817428562
    H                2.762331489080     2.751865947329     0.376852428562
    C                0.891407489080     1.756996947329     0.248140428562
    H                0.271003489080     2.633357947329     0.365817428562
    N                0.206325489080     0.584715947329     0.093575428562
    H               -0.826465510920     0.575654947329     0.092737428562
    C                0.837215489080    -0.619754052671    -0.068684571438
    O                0.240026489080    -1.683670052671    -0.210492571438

""")

GEOS['%s-%s-%s' % (dbse, '23-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.280756136122    -0.446216558662    -0.082675265655
    O               -2.619306136122     0.584160441338     0.084565734345
    N               -2.718610136122    -1.653966558662    -0.255332265655
    H               -1.704182136122    -1.748415558662    -0.249378265655
    H               -3.292243136122    -2.464179558662    -0.387734265655
    C               -4.786274136122    -0.391097558662    -0.096938265655
    H               -5.126004136122    -0.011250558662     0.863450734345
    H               -5.250144136122    -1.354990558662    -0.287262265655
    H               -5.098016136122     0.317634441338    -0.859648265655
    --
    0 1
    N                2.239525863878    -0.539037558662    -0.059430265655
    H                2.729554863878    -1.414994558662    -0.177369265655
    C                3.043151863878     0.603807441338     0.091051734345
    O                4.258713863878     0.508185441338     0.075866734345
    C                2.270558863878     1.819124441338     0.254194734345
    H                2.792321863878     2.752317441338     0.377230734345
    C                0.921398863878     1.757448441338     0.248518734345
    H                0.300994863878     2.633809441338     0.366195734345
    N                0.236316863878     0.585167441338     0.093952734345
    H               -0.796474136122     0.576106441338     0.093115734345
    C                0.867206863878    -0.619302558662    -0.068307265655
    O                0.270017863878    -1.683218558662    -0.210115265655

""")

GEOS['%s-%s-%s' % (dbse, '23-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.337600115270    -0.447072222551    -0.083391053513
    O               -2.676150115270     0.583304777449     0.083849946487
    N               -2.775454115270    -1.654822222551    -0.256048053513
    H               -1.761026115270    -1.749271222551    -0.250094053513
    H               -3.349087115270    -2.465035222551    -0.388450053513
    C               -4.843118115270    -0.391953222551    -0.097654053513
    H               -5.182848115270    -0.012106222551     0.862734946487
    H               -5.306988115270    -1.355846222551    -0.287978053513
    H               -5.154860115270     0.316778777449    -0.860364053513
    --
    0 1
    N                2.269481884730    -0.538586222551    -0.059053053513
    H                2.759510884730    -1.414544222551    -0.176992053513
    C                3.073107884730     0.604258777449     0.091428946487
    O                4.288669884730     0.508635777449     0.076243946487
    C                2.300515884730     1.819574777449     0.254571946487
    H                2.822278884730     2.752767777449     0.377607946487
    C                0.951354884730     1.757899777449     0.248895946487
    H                0.330950884730     2.634259777449     0.366572946487
    N                0.266272884730     0.585617777449     0.094329946487
    H               -0.766518115270     0.576556777449     0.093492946487
    C                0.897162884730    -0.618851222551    -0.067930053513
    O                0.299973884730    -1.682767222551    -0.209738053513

""")

GEOS['%s-%s-%s' % (dbse, '23-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.508126106073    -0.449638325871    -0.085538346939
    O               -2.846676106073     0.580738674129     0.081702653061
    N               -2.945980106073    -1.657388325871    -0.258195346939
    H               -1.931552106073    -1.751837325871    -0.252241346939
    H               -3.519613106073    -2.467601325871    -0.390597346939
    C               -5.013644106073    -0.394519325871    -0.099801346939
    H               -5.353374106073    -0.014672325871     0.860587653061
    H               -5.477514106073    -1.358412325871    -0.290125346939
    H               -5.325386106073     0.314212674129    -0.862511346939
    --
    0 1
    N                2.359346893927    -0.537234325871    -0.057921346939
    H                2.849376893927    -1.413191325871    -0.175860346939
    C                3.162972893927     0.605610674129     0.092559653061
    O                4.378534893927     0.509988674129     0.077375653061
    C                2.390380893927     1.820927674129     0.255703653061
    H                2.912143893927     2.754120674129     0.378739653061
    C                1.041220893927     1.759251674129     0.250027653061
    H                0.420815893927     2.635612674129     0.367704653061
    N                0.356137893927     0.586970674129     0.095461653061
    H               -0.676652106073     0.577909674129     0.094624653061
    C                0.987027893927    -0.617499325871    -0.066798346939
    O                0.389839893927    -1.681415325871    -0.208606346939

""")

GEOS['%s-%s-%s' % (dbse, '23-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -3.792323180131    -0.453915371161    -0.089116871038
    O               -3.130873180131     0.576461628839     0.078124128962
    N               -3.230177180131    -1.661665371161    -0.261773871038
    H               -2.215749180131    -1.756114371161    -0.255819871038
    H               -3.803810180131    -2.471878371161    -0.394175871038
    C               -5.297841180131    -0.398796371161    -0.103379871038
    H               -5.637571180131    -0.018949371161     0.857009128962
    H               -5.761711180131    -1.362689371161    -0.293703871038
    H               -5.609583180131     0.309935628839    -0.866089871038
    --
    0 1
    N                2.509115819869    -0.534980371161    -0.056035871038
    H                2.999144819869    -1.410937371161    -0.173973871038
    C                3.312741819869     0.607864628839     0.094446128962
    O                4.528303819869     0.512242628839     0.079261128962
    C                2.540149819869     1.823181628839     0.257590128962
    H                3.061912819869     2.756374628839     0.380625128962
    C                1.190988819869     1.761505628839     0.251913128962
    H                0.570584819869     2.637866628839     0.369590128962
    N                0.505906819869     0.589224628839     0.097347128962
    H               -0.526884180131     0.580163628839     0.096510128962
    C                1.136796819869    -0.615245371161    -0.064912871038
    O                0.539607819869    -1.679161371161    -0.206719871038

""")

GEOS['%s-%s-%s' % (dbse, '23-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -4.360689519117    -0.462469052367    -0.096274094737
    O               -3.699239519117     0.567907947633     0.070966905263
    N               -3.798543519117    -1.670219052367    -0.268931094737
    H               -2.784115519117    -1.764668052367    -0.262977094737
    H               -4.372176519117    -2.480432052367    -0.401333094737
    C               -5.866207519117    -0.407350052367    -0.110537094737
    H               -6.205937519117    -0.027503052367     0.849851905263
    H               -6.330077519117    -1.371243052367    -0.300861094737
    H               -6.177949519117     0.301381947633    -0.873247094737
    --
    0 1
    N                2.808638480883    -0.530472052367    -0.052264094737
    H                3.298667480883    -1.406430052367    -0.170202094737
    C                3.612264480883     0.612372947633     0.098217905263
    O                4.827826480883     0.516749947633     0.083032905263
    C                2.839672480883     1.827688947633     0.261361905263
    H                3.361435480883     2.760881947633     0.384396905263
    C                1.490511480883     1.766012947633     0.255684905263
    H                0.870107480883     2.642373947633     0.373361905263
    N                0.805429480883     0.593731947633     0.101118905263
    H               -0.227361519117     0.584670947633     0.100281905263
    C                1.436319480883    -0.610738052367    -0.061141094737
    O                0.839130480883    -1.674653052367    -0.202948094737

""")

GEOS['%s-%s-%s' % (dbse, '24-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.273643980470     1.120421269971    -1.514993763193
    H                0.918840980470     1.985824269971    -1.447797763193
    C                0.819234980470    -0.159826730029    -1.451300763193
    H                1.885952980470    -0.287674730029    -1.328791763193
    C               -0.012116019530    -1.275101730029    -1.532884763193
    H                0.411443980470    -2.269007730029    -1.480784763193
    C               -1.388579019530    -1.110648730029    -1.675848763193
    H               -2.033457019530    -1.976848730029    -1.739247763193
    C               -1.934527019530     0.170476269971    -1.737080763193
    H               -3.002784019530     0.298646269971    -1.849237763193
    C               -1.102829019530     1.286068269971    -1.658935763193
    H               -1.525902019530     2.280425269971    -1.708420763193
    --
    0 1
    C                1.388506980470     1.110686269971     1.676346236807
    H                2.033351980470     1.976870269971     1.740097236807
    C                1.934459980470    -0.170464730029     1.737866236807
    H                3.002666980470    -0.298611730029     1.850610236807
    C                1.102809980470    -1.286064730029     1.659216236807
    H                1.525891980470    -2.280417730029     1.708869236807
    C               -0.273591019530    -1.120429730029     1.514499236807
    H               -0.918694019530    -1.985840730029     1.446866236807
    C               -0.819151019530     0.159802269971     1.450533236807
    H               -1.885779019530     0.287632269971     1.327354236807
    C                0.012125980470     1.275083269971     1.532604236807
    H               -0.411399019530     2.268988269971     1.480235236807

""")

GEOS['%s-%s-%s' % (dbse, '24-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.242995480470     1.120708936638    -1.602670763193
    H                0.888192480470     1.986111936638    -1.535474763193
    C                0.788586480470    -0.159539063362    -1.538977763193
    H                1.855304480470    -0.287387063362    -1.416468763193
    C               -0.042764519530    -1.274814063362    -1.620561763193
    H                0.380795480470    -2.268720063362    -1.568461763193
    C               -1.419227519530    -1.110361063362    -1.763525763193
    H               -2.064105519530    -1.976561063362    -1.826924763193
    C               -1.965175519530     0.170763936638    -1.824757763193
    H               -3.033432519530     0.298933936638    -1.936914763193
    C               -1.133477519530     1.286355936638    -1.746612763193
    H               -1.556550519530     2.280712936638    -1.796097763193
    --
    0 1
    C                1.419155480470     1.110398936638     1.764023236807
    H                2.064000480470     1.976581936638     1.827774236807
    C                1.965108480470    -0.170752063362     1.825543236807
    H                3.033315480470    -0.298899063362     1.938287236807
    C                1.133458480470    -1.286353063362     1.746893236807
    H                1.556540480470    -2.280705063362     1.796546236807
    C               -0.242942519530    -1.120718063362     1.602176236807
    H               -0.888045519530    -1.986128063362     1.534543236807
    C               -0.788502519530     0.159514936638     1.538210236807
    H               -1.855130519530     0.287343936638     1.415031236807
    C                0.042774480470     1.274795936638     1.620281236807
    H               -0.380750519530     2.268700936638     1.567912236807

""")

GEOS['%s-%s-%s' % (dbse, '24-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.212653890680     1.120993596848    -1.689469859440
    H                0.857850890680     1.986396596848    -1.622273859440
    C                0.758244890680    -0.159254403152    -1.625776859440
    H                1.824962890680    -0.287102403152    -1.503267859440
    C               -0.073106109320    -1.274529403152    -1.707360859440
    H                0.350453890680    -2.268435403152    -1.655260859440
    C               -1.449569109320    -1.110076403152    -1.850324859440
    H               -2.094447109320    -1.976276403152    -1.913723859440
    C               -1.995517109320     0.171048596848    -1.911556859440
    H               -3.063774109320     0.299218596848    -2.023713859440
    C               -1.163819109320     1.286640596848    -1.833411859440
    H               -1.586892109320     2.280997596848    -1.882896859440
    --
    0 1
    C                1.449496890680     1.110113596848     1.850823140560
    H                2.094341890680     1.976297596848     1.914573140560
    C                1.995449890680    -0.171037403152     1.912342140560
    H                3.063656890680    -0.299184403152     2.025087140560
    C                1.163800890680    -1.286637403152     1.833692140560
    H                1.586882890680    -2.280989403152     1.883346140560
    C               -0.212601109320    -1.121002403152     1.688975140560
    H               -0.857703109320    -1.986412403152     1.621342140560
    C               -0.758161109320     0.159230596848     1.625009140560
    H               -1.824789109320     0.287059596848     1.501831140560
    C                0.073115890680     1.274511596848     1.707080140560
    H               -0.350409109320     2.268416596848     1.654711140560

""")

GEOS['%s-%s-%s' % (dbse, '24-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.182521403593     1.121276269971    -1.775670109440
    H                0.827718403593     1.986679269971    -1.708474109440
    C                0.728112403593    -0.158971730029    -1.711977109440
    H                1.794830403593    -0.286819730029    -1.589468109440
    C               -0.103238596407    -1.274246730029    -1.793561109440
    H                0.320321403593    -2.268152730029    -1.741461109440
    C               -1.479701596407    -1.109793730029    -1.936525109440
    H               -2.124579596407    -1.975993730029    -1.999924109440
    C               -2.025649596407     0.171331269971    -1.997757109440
    H               -3.093906596407     0.299501269971    -2.109914109440
    C               -1.193951596407     1.286923269971    -1.919612109440
    H               -1.617024596407     2.281280269971    -1.969097109440
    --
    0 1
    C                1.479629403593     1.109831269971     1.937022890560
    H                2.124474403593     1.976015269971     2.000773890560
    C                2.025582403593    -0.171319730029     1.998542890560
    H                3.093789403593    -0.299466730029     2.111286890560
    C                1.193933403593    -1.286919730029     1.919891890560
    H                1.617014403593    -2.281272730029     1.969545890560
    C               -0.182468596407    -1.121284730029     1.775174890560
    H               -0.827571596407    -1.986695730029     1.707542890560
    C               -0.728028596407     0.158947269971     1.711209890560
    H               -1.794656596407     0.286777269971     1.588030890560
    C                0.103248403593     1.274228269971     1.793280890560
    H               -0.320276596407     2.268133269971     1.740911890560

""")

GEOS['%s-%s-%s' % (dbse, '24-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.152610300890     1.121557019971    -1.861237365896
    H                0.797807300890     1.986960019971    -1.794041365896
    C                0.698201300890    -0.158690980029    -1.797544365896
    H                1.764919300890    -0.286538980029    -1.675035365896
    C               -0.133149699110    -1.273965980029    -1.879128365896
    H                0.290410300890    -2.267871980029    -1.827028365896
    C               -1.509612699110    -1.109512980029    -2.022092365896
    H               -2.154490699110    -1.975712980029    -2.085491365896
    C               -2.055560699110     0.171612019971    -2.083324365896
    H               -3.123817699110     0.299782019971    -2.195481365896
    C               -1.223862699110     1.287204019971    -2.005179365896
    H               -1.646935699110     2.281561019971    -2.054664365896
    --
    0 1
    C                1.509540300890     1.109551019971     2.022590634104
    H                2.154385300890     1.975734019971     2.086340634104
    C                2.055493300890    -0.171600980029     2.084109634104
    H                3.123700300890    -0.299747980029     2.196854634104
    C                1.223844300890    -1.287200980029     2.005459634104
    H                1.646926300890    -2.281552980029     2.055113634104
    C               -0.152557699110    -1.121565980029     1.860742634104
    H               -0.797659699110    -1.986975980029     1.793110634104
    C               -0.698116699110     0.158667019971     1.796776634104
    H               -1.764744699110     0.286496019971     1.673598634104
    C                0.133159300890     1.273948019971     1.878847634104
    H               -0.290364699110     2.267853019971     1.826478634104

""")

GEOS['%s-%s-%s' % (dbse, '24-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.060548307347     1.122420686638    -2.124600763193
    H                0.705745307347     1.987823686638    -2.057404763193
    C                0.606139307347    -0.157827313362    -2.060907763193
    H                1.672857307347    -0.285675313362    -1.938398763193
    C               -0.225211692653    -1.273102313362    -2.142491763193
    H                0.198348307347    -2.267008313362    -2.090391763193
    C               -1.601674692653    -1.108649313362    -2.285455763193
    H               -2.246552692653    -1.974849313362    -2.348854763193
    C               -2.147622692653     0.172475686638    -2.346687763193
    H               -3.215879692653     0.300645686638    -2.458844763193
    C               -1.315924692653     1.288067686638    -2.268542763193
    H               -1.738997692653     2.282424686638    -2.318027763193
    --
    0 1
    C                1.601602307347     1.108686686638     2.285953236807
    H                2.246447307347     1.974870686638     2.349704236807
    C                2.147555307347    -0.172464313362     2.347473236807
    H                3.215762307347    -0.300611313362     2.460217236807
    C                1.315906307347    -1.288064313362     2.268823236807
    H                1.738988307347    -2.282417313362     2.318476236807
    C               -0.060495692653    -1.122429313362     2.124106236807
    H               -0.705597692653    -1.987839313362     2.056473236807
    C               -0.606054692653     0.157803686638     2.060140236807
    H               -1.672683692653     0.285632686638     1.936961236807
    C                0.225221307347     1.273083686638     2.142211236807
    H               -0.198302692653     2.266988686638     2.089842236807

""")

GEOS['%s-%s-%s' % (dbse, '24-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.099588615777     1.123923186638    -2.582707186316
    H                0.545608384223     1.989326186638    -2.515511186316
    C                0.446002384223    -0.156324813362    -2.519014186316
    H                1.512720384223    -0.284172813362    -2.396505186316
    C               -0.385348615777    -1.271599813362    -2.600598186316
    H                0.038211384223    -2.265505813362    -2.548498186316
    C               -1.761811615777    -1.107146813362    -2.743562186316
    H               -2.406689615777    -1.973346813362    -2.806961186316
    C               -2.307759615777     0.173978186638    -2.804794186316
    H               -3.376016615777     0.302148186638    -2.916951186316
    C               -1.476061615777     1.289570186638    -2.726649186316
    H               -1.899134615777     2.283927186638    -2.776134186316
    --
    0 1
    C                1.761739384223     1.107184186638     2.744059813684
    H                2.406584384223     1.973368186638     2.807810813684
    C                2.307692384223    -0.173966813362     2.805579813684
    H                3.375899384223    -0.302113813362     2.918323813684
    C                1.476043384223    -1.289566813362     2.726928813684
    H                1.899125384223    -2.283919813362     2.776582813684
    C                0.099641384223    -1.123931813362     2.582212813684
    H               -0.545460615777    -1.989341813362     2.514579813684
    C               -0.445918615777     0.156301186638     2.518246813684
    H               -1.512546615777     0.284130186638     2.395067813684
    C                0.385358384223     1.271581186638     2.600317813684
    H               -0.038165615777     2.265486186638     2.547948813684

""")

GEOS['%s-%s-%s' % (dbse, '24-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.406424859320     1.126802173725    -3.460478186316
    H                0.238772140680     1.992205173725    -3.393282186316
    C                0.139166140680    -0.153445826275    -3.396785186316
    H                1.205884140680    -0.281293826275    -3.274276186316
    C               -0.692184859320    -1.268720826275    -3.478369186316
    H               -0.268624859320    -2.262626826275    -3.426269186316
    C               -2.068647859320    -1.104267826275    -3.621333186316
    H               -2.713525859320    -1.970467826275    -3.684732186316
    C               -2.614595859320     0.176857173725    -3.682565186316
    H               -3.682852859320     0.305027173725    -3.794722186316
    C               -1.782897859320     1.292449173725    -3.604420186316
    H               -2.205970859320     2.286806173725    -3.653905186316
    --
    0 1
    C                2.068575140680     1.104305173725     3.621830813684
    H                2.713421140680     1.970489173725     3.685581813684
    C                2.614529140680    -0.176845826275     3.683350813684
    H                3.682735140680    -0.304992826275     3.796094813684
    C                1.782879140680    -1.292445826275     3.604699813684
    H                2.205961140680    -2.286797826275     3.654353813684
    C                0.406477140680    -1.126810826275     3.459983813684
    H               -0.238624859320    -1.992220826275     3.392350813684
    C               -0.139081859320     0.153422173725     3.396017813684
    H               -1.205709859320     0.281251173725     3.272838813684
    C                0.692195140680     1.268702173725     3.478088813684
    H                0.268670140680     2.262608173725     3.425719813684

""")

GEOS['%s-%s-%s' % (dbse, '25-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.559771199415     0.229787560320    -1.834629267813
    C                0.956650199415    -0.927921439680    -1.533622267813
    H                1.600929199415    -1.796942439680    -1.475803267813
    C               -0.410867800585    -1.053580439680    -1.297188267813
    H               -0.831134800585    -2.016498439680    -1.044584267813
    C               -1.208514800585     0.081796560320    -1.382751267813
    H               -2.273399800585     0.024777560320    -1.204705267813
    C               -0.599837800585     1.292650560320    -1.698253267813
    H               -1.174521800585     2.204738560320    -1.778611267813
    C                0.775837199415     1.314944560320    -1.910389267813
    H                1.275722199415     2.244032560320    -2.156675267813
    --
    0 1
    N               -0.496665800585    -1.498394439680     1.831590732187
    C               -1.429144800585    -0.537765439680     1.905441732187
    H               -2.431933800585    -0.868720439680     2.147634732187
    C               -1.167141800585     0.813302560320     1.696062732187
    H               -1.965698800585     1.537808560320     1.774333732187
    C                0.132286199415     1.202544560320     1.385793732187
    H                0.374270199415     2.241542560320     1.209922732187
    C                1.111845199415     0.219646560320     1.302432732187
    H                2.134146199415     0.466391560320     1.053995732187
    C                0.749365199415    -1.105491439680     1.535625732187
    H                1.493226199415    -1.891046439680     1.479176732187

""")

GEOS['%s-%s-%s' % (dbse, '25-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.569613370278     0.238138958910    -1.920539767813
    C                0.966492370278    -0.919570041090    -1.619532767813
    H                1.610771370278    -1.788591041090    -1.561713767813
    C               -0.401025629722    -1.045229041090    -1.383098767813
    H               -0.821292629722    -2.008147041090    -1.130494767813
    C               -1.198672629722     0.090147958910    -1.468661767813
    H               -2.263557629722     0.033128958910    -1.290615767813
    C               -0.589995629722     1.301001958910    -1.784163767813
    H               -1.164679629722     2.213089958910    -1.864521767813
    C                0.785679370278     1.323295958910    -1.996299767813
    H                1.285564370278     2.252383958910    -2.242585767813
    --
    0 1
    N               -0.506508629722    -1.506746041090     1.917501232187
    C               -1.438986629722    -0.546117041090     1.991352232187
    H               -2.441775629722    -0.877071041090     2.233545232187
    C               -1.176983629722     0.804951958910     1.781973232187
    H               -1.975541629722     1.529456958910     1.860244232187
    C                0.122443370278     1.194192958910     1.471704232187
    H                0.364428370278     2.233191958910     1.295833232187
    C                1.102003370278     0.211294958910     1.388343232187
    H                2.124304370278     0.458040958910     1.139906232187
    C                0.739523370278    -1.113843041090     1.621536232187
    H                1.483384370278    -1.899397041090     1.565087232187

""")

GEOS['%s-%s-%s' % (dbse, '25-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.579386287994     0.246431458910    -2.005845350097
    C                0.976265287994    -0.911277541090    -1.704838350097
    H                1.620544287994    -1.780298541090    -1.647019350097
    C               -0.391252712006    -1.036936541090    -1.468404350097
    H               -0.811519712006    -1.999854541090    -1.215800350097
    C               -1.188899712006     0.098440458910    -1.553967350097
    H               -2.253784712006     0.041421458910    -1.375921350097
    C               -0.580222712006     1.309294458910    -1.869469350097
    H               -1.154906712006     2.221382458910    -1.949827350097
    C                0.795452287994     1.331588458910    -2.081605350097
    H                1.295337287994     2.260676458910    -2.327891350097
    --
    0 1
    N               -0.516281712006    -1.515038541090     2.002806649903
    C               -1.448759712006    -0.554409541090     2.076657649903
    H               -2.451548712006    -0.885363541090     2.318850649903
    C               -1.186756712006     0.796659458910     1.867278649903
    H               -1.985313712006     1.521164458910     1.945549649903
    C                0.112671287994     1.185900458910     1.557010649903
    H                0.354655287994     2.224899458910     1.381138649903
    C                1.092230287994     0.203002458910     1.473648649903
    H                2.114531287994     0.449748458910     1.225211649903
    C                0.729750287994    -1.122135541090     1.706841649903
    H                1.473611287994    -1.907689541090     1.650393649903

""")

GEOS['%s-%s-%s' % (dbse, '25-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.589116870278     0.254688060320    -2.090782767813
    C                0.985995870278    -0.903020939680    -1.789775767813
    H                1.630274870278    -1.772041939680    -1.731956767813
    C               -0.381522129722    -1.028679939680    -1.553341767813
    H               -0.801789129722    -1.991597939680    -1.300737767813
    C               -1.179169129722     0.106697060320    -1.638904767813
    H               -2.244054129722     0.049678060320    -1.460858767813
    C               -0.570492129722     1.317551060320    -1.954406767813
    H               -1.145176129722     2.229639060320    -2.034764767813
    C                0.805182870278     1.339845060320    -2.166542767813
    H                1.305067870278     2.268933060320    -2.412828767813
    --
    0 1
    N               -0.526012129722    -1.523294939680     2.087744232187
    C               -1.458490129722    -0.562665939680     2.161595232187
    H               -2.461279129722    -0.893620939680     2.403788232187
    C               -1.196487129722     0.788402060320     1.952216232187
    H               -1.995045129722     1.512908060320     2.030487232187
    C                0.102939870278     1.177644060320     1.641947232187
    H                0.344924870278     2.216642060320     1.466076232187
    C                1.082499870278     0.194746060320     1.558586232187
    H                2.104800870278     0.441491060320     1.310149232187
    C                0.720019870278    -1.130391939680     1.791779232187
    H                1.463880870278    -1.915946939680     1.735330232187

""")

GEOS['%s-%s-%s' % (dbse, '25-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.598796370278     0.262901383002    -2.175272850097
    C                0.995675370278    -0.894807616998    -1.874265850097
    H                1.639954370278    -1.763828616998    -1.816446850097
    C               -0.371842629722    -1.020466616998    -1.637831850097
    H               -0.792109629722    -1.983384616998    -1.385227850097
    C               -1.169489629722     0.114910383002    -1.723394850097
    H               -2.234374629722     0.057891383002    -1.545348850097
    C               -0.560812629722     1.325764383002    -2.038896850097
    H               -1.135496629722     2.237852383002    -2.119254850097
    C                0.814862370278     1.348058383002    -2.251032850097
    H                1.314747370278     2.277146383002    -2.497318850097
    --
    0 1
    N               -0.535691629722    -1.531508616998     2.172234149903
    C               -1.468169629722    -0.570878616998     2.246085149903
    H               -2.470958629722    -0.901833616998     2.488278149903
    C               -1.206166629722     0.780189383002     2.036706149903
    H               -2.004724629722     1.504694383002     2.114977149903
    C                0.093260370278     1.169430383002     1.726438149903
    H                0.335245370278     2.208429383002     1.550566149903
    C                1.072820370278     0.186532383002     1.643076149903
    H                2.095121370278     0.433278383002     1.394639149903
    C                0.710340370278    -1.138605616998     1.876269149903
    H                1.454201370278    -1.924159616998     1.819821149903

""")

GEOS['%s-%s-%s' % (dbse, '25-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.627620370278     0.287359300718    -2.426872767813
    C                1.024499370278    -0.870349699282    -2.125865767813
    H                1.668778370278    -1.739370699282    -2.068046767813
    C               -0.343018629722    -0.996008699282    -1.889431767813
    H               -0.763285629722    -1.958926699282    -1.636827767813
    C               -1.140665629722     0.139368300718    -1.974994767813
    H               -2.205550629722     0.082349300718    -1.796948767813
    C               -0.531988629722     1.350222300718    -2.290496767813
    H               -1.106672629722     2.262310300718    -2.370854767813
    C                0.843686370278     1.372516300718    -2.502632767813
    H                1.343571370278     2.301604300718    -2.748918767813
    --
    0 1
    N               -0.564515629722    -1.555966699282     2.423834232187
    C               -1.496993629722    -0.595336699282     2.497685232187
    H               -2.499782629722    -0.926291699282     2.739878232187
    C               -1.234990629722     0.755731300718     2.288306232187
    H               -2.033548629722     1.480237300718     2.366577232187
    C                0.064436370278     1.144972300718     1.978037232187
    H                0.306421370278     2.183971300718     1.802166232187
    C                1.043996370278     0.162074300718     1.894676232187
    H                2.066297370278     0.408820300718     1.646239232187
    C                0.681516370278    -1.163062699282     2.127869232187
    H                1.425377370278    -1.948617699282     2.071420232187

""")

GEOS['%s-%s-%s' % (dbse, '25-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.679001370278     0.330956984411    -2.875366350097
    C                1.075880370278    -0.826752015589    -2.574359350097
    H                1.720159370278    -1.695773015589    -2.516540350097
    C               -0.291637629722    -0.952411015589    -2.337925350097
    H               -0.711904629722    -1.915329015589    -2.085321350097
    C               -1.089284629722     0.182965984411    -2.423488350097
    H               -2.154169629722     0.125946984411    -2.245442350097
    C               -0.480607629722     1.393819984411    -2.738990350097
    H               -1.055291629722     2.305907984411    -2.819348350097
    C                0.895067370278     1.416113984411    -2.951126350097
    H                1.394952370278     2.345201984411    -3.197412350097
    --
    0 1
    N               -0.615896629722    -1.599564015589     2.872327649903
    C               -1.548374629722    -0.638935015589     2.946178649903
    H               -2.551163629722    -0.969890015589     3.188371649903
    C               -1.286371629722     0.712133984411     2.736799649903
    H               -2.084929629722     1.436638984411     2.815070649903
    C                0.013055370278     1.101374984411     2.426531649903
    H                0.255040370278     2.140372984411     2.250659649903
    C                0.992615370278     0.118476984411     2.343169649903
    H                2.014916370278     0.365221984411     2.094732649903
    C                0.630135370278    -1.206661015589     2.576362649903
    H                1.373996370278    -1.992216015589     2.519914649903

""")

GEOS['%s-%s-%s' % (dbse, '25-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.779358794370     0.416112560320    -3.751367267813
    C                1.176237794370    -0.741596439680    -3.450360267813
    H                1.820516794370    -1.610617439680    -3.392541267813
    C               -0.191280205630    -0.867255439680    -3.213926267813
    H               -0.611547205630    -1.830173439680    -2.961322267813
    C               -0.988927205630     0.268121560320    -3.299489267813
    H               -2.053812205630     0.211102560320    -3.121443267813
    C               -0.380250205630     1.478975560320    -3.614991267813
    H               -0.954934205630     2.391063560320    -3.695349267813
    C                0.995424794370     1.501269560320    -3.827127267813
    H                1.495309794370     2.430357560320    -4.073413267813
    --
    0 1
    N               -0.716254205630    -1.684719439680     3.748328732187
    C               -1.648732205630    -0.724090439680     3.822179732187
    H               -2.651521205630    -1.055045439680     4.064372732187
    C               -1.386729205630     0.626977560320     3.612800732187
    H               -2.185287205630     1.351483560320     3.691071732187
    C               -0.087301205630     1.016219560320     3.302531732187
    H                0.154682794370     2.055217560320     3.126660732187
    C                0.892257794370     0.033321560320     3.219170732187
    H                1.914558794370     0.280066560320     2.970733732187
    C                0.529777794370    -1.291816439680     3.452363732187
    H                1.273638794370    -2.077371439680     3.395914732187

""")

GEOS['%s-%s-%s' % (dbse, '26-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.239332441943     0.815996562499     0.734453688919
    H                0.914243441943     1.362473562499     1.523184688919
    C                1.171414441943     1.433779562499    -0.520826311081
    O                0.782992441943     2.587327562499    -0.626148311081
    N                1.873854441943    -1.236958437501    -0.098243311081
    H                1.589717441943     0.967092562499    -2.612167311081
    C                1.888168441943    -0.720921437501    -1.364568311081
    H                2.159948441943    -1.414810437501    -2.145816311081
    C                1.576943441943     0.568187562499    -1.612658311081
    H                1.991885441943    -2.225271437501     0.056657688919
    C                1.508376441943    -0.508956437501     1.018546688919
    O                1.423547441943    -0.995566437501     2.129637688919
    --
    0 1
    N               -1.239332558057    -0.815997437501     0.734453688919
    H               -0.914243558057    -1.362473437501     1.523184688919
    C               -1.171413558057    -1.433779437501    -0.520826311081
    O               -0.782992558057    -2.587327437501    -0.626148311081
    N               -1.873854558057     1.236958562499    -0.098243311081
    H               -1.589716558057    -0.967092437501    -2.612167311081
    C               -1.888168558057     0.720921562499    -1.364568311081
    H               -2.159948558057     1.414810562499    -2.145816311081
    C               -1.576943558057    -0.568187437501    -1.612658311081
    H               -1.991885558057     2.225271562499     0.056657688919
    C               -1.508376558057     0.508956562499     1.018546688919
    O               -1.423547558057     0.995566562499     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.308248129445     0.827894500000     0.734453688919
    H                0.983159129445     1.374371500000     1.523184688919
    C                1.240330129445     1.445677500000    -0.520826311081
    O                0.851908129445     2.599225500000    -0.626148311081
    N                1.942770129445    -1.225060500000    -0.098243311081
    H                1.658633129445     0.978990500000    -2.612167311081
    C                1.957084129445    -0.709023500000    -1.364568311081
    H                2.228864129445    -1.402912500000    -2.145816311081
    C                1.645859129445     0.580085500000    -1.612658311081
    H                2.060801129445    -2.213373500000     0.056657688919
    C                1.577292129445    -0.497058500000     1.018546688919
    O                1.492463129445    -0.983668500000     2.129637688919
    --
    0 1
    N               -1.308247870555    -0.827894500000     0.734453688919
    H               -0.983159870555    -1.374371500000     1.523184688919
    C               -1.240329870555    -1.445677500000    -0.520826311081
    O               -0.851908870555    -2.599225500000    -0.626148311081
    N               -1.942769870555     1.225060500000    -0.098243311081
    H               -1.658632870555    -0.978990500000    -2.612167311081
    C               -1.957083870555     0.709023500000    -1.364568311081
    H               -2.228863870555     1.402912500000    -2.145816311081
    C               -1.645859870555    -0.580085500000    -1.612658311081
    H               -2.060800870555     2.213373500000     0.056657688919
    C               -1.577291870555     0.497058500000     1.018546688919
    O               -1.492462870555     0.983668500000     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.376901000000     0.839747000000     0.734453688919
    H                1.051812000000     1.386224000000     1.523184688919
    C                1.308983000000     1.457530000000    -0.520826311081
    O                0.920561000000     2.611078000000    -0.626148311081
    N                2.011423000000    -1.213208000000    -0.098243311081
    H                1.727286000000     0.990843000000    -2.612167311081
    C                2.025737000000    -0.697171000000    -1.364568311081
    H                2.297517000000    -1.391060000000    -2.145816311081
    C                1.714512000000     0.591938000000    -1.612658311081
    H                2.129454000000    -2.201521000000     0.056657688919
    C                1.645945000000    -0.485206000000     1.018546688919
    O                1.561116000000    -0.971816000000     2.129637688919
    --
    0 1
    N               -1.376901000000    -0.839747000000     0.734453688919
    H               -1.051812000000    -1.386224000000     1.523184688919
    C               -1.308983000000    -1.457530000000    -0.520826311081
    O               -0.920561000000    -2.611078000000    -0.626148311081
    N               -2.011423000000     1.213208000000    -0.098243311081
    H               -1.727286000000    -0.990843000000    -2.612167311081
    C               -2.025737000000     0.697171000000    -1.364568311081
    H               -2.297517000000     1.391060000000    -2.145816311081
    C               -1.714512000000    -0.591938000000    -1.612658311081
    H               -2.129454000000     2.201521000000     0.056657688919
    C               -1.645945000000     0.485206000000     1.018546688919
    O               -1.561116000000     0.971816000000     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.445471000000     0.851585066997     0.734453688919
    H                1.120382000000     1.398062066997     1.523184688919
    C                1.377553000000     1.469368066997    -0.520826311081
    O                0.989131000000     2.622916066997    -0.626148311081
    N                2.079993000000    -1.201369933003    -0.098243311081
    H                1.795856000000     1.002681066997    -2.612167311081
    C                2.094307000000    -0.685332933003    -1.364568311081
    H                2.366087000000    -1.379221933003    -2.145816311081
    C                1.783082000000     0.603776066997    -1.612658311081
    H                2.198024000000    -2.189682933003     0.056657688919
    C                1.714515000000    -0.473367933003     1.018546688919
    O                1.629686000000    -0.959977933003     2.129637688919
    --
    0 1
    N               -1.445471000000    -0.851585933003     0.734453688919
    H               -1.120382000000    -1.398061933003     1.523184688919
    C               -1.377553000000    -1.469367933003    -0.520826311081
    O               -0.989131000000    -2.622915933003    -0.626148311081
    N               -2.079993000000     1.201370066997    -0.098243311081
    H               -1.795856000000    -1.002680933003    -2.612167311081
    C               -2.094307000000     0.685333066997    -1.364568311081
    H               -2.366087000000     1.379222066997    -2.145816311081
    C               -1.783082000000    -0.603775933003    -1.612658311081
    H               -2.198024000000     2.189682066997     0.056657688919
    C               -1.714515000000     0.473368066997     1.018546688919
    O               -1.629686000000     0.959978066997     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.513836196442     0.863388062499     0.734453688919
    H                1.188747196442     1.409865062499     1.523184688919
    C                1.445918196442     1.481171062499    -0.520826311081
    O                1.057496196442     2.634719062499    -0.626148311081
    N                2.148358196442    -1.189566937501    -0.098243311081
    H                1.864221196442     1.014484062499    -2.612167311081
    C                2.162672196442    -0.673529937501    -1.364568311081
    H                2.434452196442    -1.367418937501    -2.145816311081
    C                1.851447196442     0.615579062499    -1.612658311081
    H                2.266389196442    -2.177879937501     0.056657688919
    C                1.782880196442    -0.461564937501     1.018546688919
    O                1.698051196442    -0.948174937501     2.129637688919
    --
    0 1
    N               -1.513836803558    -0.863388937501     0.734453688919
    H               -1.188747803558    -1.409864937501     1.523184688919
    C               -1.445917803558    -1.481170937501    -0.520826311081
    O               -1.057496803558    -2.634718937501    -0.626148311081
    N               -2.148357803558     1.189567062499    -0.098243311081
    H               -1.864220803558    -1.014483937501    -2.612167311081
    C               -2.162671803558     0.673530062499    -1.364568311081
    H               -2.434451803558     1.367419062499    -2.145816311081
    C               -1.851447803558    -0.615578937501    -1.612658311081
    H               -2.266389803558     2.177880062499     0.056657688919
    C               -1.782879803558     0.461565062499     1.018546688919
    O               -1.698050803558     0.948175062499     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.718192441943     0.898669000000     0.734453688919
    H                1.393103441943     1.445146000000     1.523184688919
    C                1.650274441943     1.516452000000    -0.520826311081
    O                1.261852441943     2.670000000000    -0.626148311081
    N                2.352714441943    -1.154286000000    -0.098243311081
    H                2.068577441943     1.049765000000    -2.612167311081
    C                2.367028441943    -0.638249000000    -1.364568311081
    H                2.638808441943    -1.332138000000    -2.145816311081
    C                2.055803441943     0.650860000000    -1.612658311081
    H                2.470745441943    -2.142599000000     0.056657688919
    C                1.987236441943    -0.426284000000     1.018546688919
    O                1.902407441943    -0.912894000000     2.129637688919
    --
    0 1
    N               -1.718192558057    -0.898669000000     0.734453688919
    H               -1.393103558057    -1.445146000000     1.523184688919
    C               -1.650273558057    -1.516452000000    -0.520826311081
    O               -1.261852558057    -2.670000000000    -0.626148311081
    N               -2.352714558057     1.154286000000    -0.098243311081
    H               -2.068576558057    -1.049765000000    -2.612167311081
    C               -2.367028558057     0.638249000000    -1.364568311081
    H               -2.638808558057     1.332138000000    -2.145816311081
    C               -2.055803558057    -0.650860000000    -1.612658311081
    H               -2.470745558057     2.142599000000     0.056657688919
    C               -1.987236558057     0.426284000000     1.018546688919
    O               -1.902407558057     0.912894000000     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                2.057081629445     0.957176120555     0.734453688919
    H                1.731992629445     1.503653120555     1.523184688919
    C                1.989163629445     1.574959120555    -0.520826311081
    O                1.600741629445     2.728507120555    -0.626148311081
    N                2.691603629445    -1.095778879445    -0.098243311081
    H                2.407466629445     1.108272120555    -2.612167311081
    C                2.705917629445    -0.579741879445    -1.364568311081
    H                2.977697629445    -1.273630879445    -2.145816311081
    C                2.394692629445     0.709367120555    -1.612658311081
    H                2.809634629445    -2.084091879445     0.056657688919
    C                2.326125629445    -0.367776879445     1.018546688919
    O                2.241296629445    -0.854386879445     2.129637688919
    --
    0 1
    N               -2.057081370555    -0.957176879445     0.734453688919
    H               -1.731993370555    -1.503652879445     1.523184688919
    C               -1.989163370555    -1.574958879445    -0.520826311081
    O               -1.600742370555    -2.728506879445    -0.626148311081
    N               -2.691603370555     1.095779120555    -0.098243311081
    H               -2.407466370555    -1.108271879445    -2.612167311081
    C               -2.705917370555     0.579742120555    -1.364568311081
    H               -2.977697370555     1.273631120555    -2.145816311081
    C               -2.394693370555    -0.709366879445    -1.612658311081
    H               -2.809634370555     2.084091120555     0.056657688919
    C               -2.326125370555     0.367776120555     1.018546688919
    O               -2.241296370555     0.854387120555     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '26-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                2.731397629445     1.073593062499     0.734453688919
    H                2.406308629445     1.620070062499     1.523184688919
    C                2.663479629445     1.691376062499    -0.520826311081
    O                2.275057629445     2.844924062499    -0.626148311081
    N                3.365919629445    -0.979361937501    -0.098243311081
    H                3.081782629445     1.224689062499    -2.612167311081
    C                3.380233629445    -0.463324937501    -1.364568311081
    H                3.652013629445    -1.157213937501    -2.145816311081
    C                3.069008629445     0.825784062499    -1.612658311081
    H                3.483950629445    -1.967674937501     0.056657688919
    C                3.000441629445    -0.251359937501     1.018546688919
    O                2.915612629445    -0.737969937501     2.129637688919
    --
    0 1
    N               -2.731397370555    -1.073593937501     0.734453688919
    H               -2.406309370555    -1.620069937501     1.523184688919
    C               -2.663479370555    -1.691375937501    -0.520826311081
    O               -2.275058370555    -2.844923937501    -0.626148311081
    N               -3.365919370555     0.979362062499    -0.098243311081
    H               -3.081782370555    -1.224688937501    -2.612167311081
    C               -3.380233370555     0.463325062499    -1.364568311081
    H               -3.652013370555     1.157214062499    -2.145816311081
    C               -3.069009370555    -0.825783937501    -1.612658311081
    H               -3.483950370555     1.967675062499     0.056657688919
    C               -3.000441370555     0.251360062499     1.018546688919
    O               -2.915612370555     0.737970062499     2.129637688919

""")

GEOS['%s-%s-%s' % (dbse, '27-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.272098569594     0.883120533206    -1.390392705236
    H                1.919465569594     1.735616533206    -1.233957705236
    C                1.822348569594    -0.371575466794    -1.645376705236
    H                2.896387569594    -0.492913466794    -1.689252705236
    C                0.987730569594    -1.469544466794    -1.850566705236
    H                1.414199569594    -2.442615466794    -2.054185705236
    C               -0.395764430406    -1.311558466794    -1.798574705236
    H               -1.043717430406    -2.162911466794    -1.958231705236
    C               -0.946133430406    -0.057081466794    -1.538244705236
    H               -2.019335430406     0.063856533206    -1.485296705236
    C               -0.111940430406     1.040351533206    -1.336399705236
    H               -0.539205430406     2.012609533206    -1.132420705236
    --
    0 1
    N               -1.822555430406     0.121115533206     1.496042294764
    C               -1.207669430406     1.278782533206     1.776106294764
    H               -1.855273430406     2.131954533206     1.938618294764
    C                0.174545569594     1.419630533206     1.866277294764
    H                0.606394569594     2.383280533206     2.097493294764
    C                0.975496569594     0.302651533206     1.652063294764
    H                2.053065569594     0.373035533206     1.706898294764
    C                0.354944569594    -0.906003466794     1.358404294764
    H                0.930719569594    -1.801625466794     1.174329294764
    C               -1.035569430406    -0.944041466794     1.292954294764
    H               -1.544422430406    -1.872494466794     1.064112294764

""")

GEOS['%s-%s-%s' % (dbse, '27-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.296656348484     0.871099778458    -1.479723094512
    H                1.944023348484     1.723595778458    -1.323288094512
    C                1.846906348484    -0.383596221542    -1.734707094512
    H                2.920945348484    -0.504934221542    -1.778583094512
    C                1.012288348484    -1.481565221542    -1.939897094512
    H                1.438757348484    -2.454636221542    -2.143516094512
    C               -0.371206651516    -1.323579221542    -1.887905094512
    H               -1.019159651516    -2.174932221542    -2.047562094512
    C               -0.921575651516    -0.069102221542    -1.627575094512
    H               -1.994777651516     0.051835778458    -1.574627094512
    C               -0.087382651516     1.028330778458    -1.425730094512
    H               -0.514647651516     2.000588778458    -1.221751094512
    --
    0 1
    N               -1.846803651516     0.132984778458     1.584247905488
    C               -1.231918651516     1.290652778458     1.864311905488
    H               -1.879522651516     2.143823778458     2.026823905488
    C                0.150296348484     1.431499778458     1.954482905488
    H                0.582145348484     2.395149778458     2.185698905488
    C                0.951248348484     0.314520778458     1.740268905488
    H                2.028817348484     0.384904778458     1.795103905488
    C                0.330696348484    -0.894134221542     1.446609905488
    H                0.906470348484    -1.789756221542     1.262534905488
    C               -1.059817651516    -0.932172221542     1.381159905488
    H               -1.568670651516    -1.860625221542     1.152316905488

""")

GEOS['%s-%s-%s' % (dbse, '27-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.320878667162     0.859243381133    -1.567832308408
    H                1.968245667162     1.711739381133    -1.411397308408
    C                1.871128667162    -0.395452618867    -1.822816308408
    H                2.945167667162    -0.516790618867    -1.866692308408
    C                1.036510667162    -1.493421618867    -2.028006308408
    H                1.462979667162    -2.466492618867    -2.231625308408
    C               -0.346984332838    -1.335435618867    -1.976014308408
    H               -0.994937332838    -2.186788618867    -2.135671308408
    C               -0.897353332838    -0.080958618867    -1.715684308408
    H               -1.970555332838     0.039979381133    -1.662736308408
    C               -0.063160332838     1.016474381133    -1.513839308408
    H               -0.490425332838     1.988732381133    -1.309860308408
    --
    0 1
    N               -1.870721332838     0.144691381133     1.671247691592
    C               -1.255835332838     1.302359381133     1.951311691592
    H               -1.903439332838     2.155531381133     2.113823691592
    C                0.126379667162     1.443207381133     2.041482691592
    H                0.558228667162     2.406857381133     2.272698691592
    C                0.927330667162     0.326228381133     1.827268691592
    H                2.004900667162     0.396612381133     1.882103691592
    C                0.306778667162    -0.882427618867     1.533609691592
    H                0.882553667162    -1.778048618867     1.349534691592
    C               -1.083735332838    -0.920464618867     1.468159691592
    H               -1.592588332838    -1.848918618867     1.239317691592

""")

GEOS['%s-%s-%s' % (dbse, '27-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.344860662912     0.847504400344    -1.655068519783
    H                1.992227662912     1.700000400344    -1.498633519783
    C                1.895110662912    -0.407191599656    -1.910052519783
    H                2.969149662912    -0.528529599656    -1.953928519783
    C                1.060492662912    -1.505160599656    -2.115242519783
    H                1.486961662912    -2.478231599656    -2.318861519783
    C               -0.323002337088    -1.347174599656    -2.063250519783
    H               -0.970955337088    -2.198527599656    -2.222907519783
    C               -0.873371337088    -0.092697599656    -1.802920519783
    H               -1.946573337088     0.028240400344    -1.749972519783
    C               -0.039178337088     1.004735400344    -1.601075519783
    H               -0.466443337088     1.976993400344    -1.397096519783
    --
    0 1
    N               -1.894401337088     0.156282400344     1.757385480217
    C               -1.279515337088     1.313950400344     2.037449480217
    H               -1.927119337088     2.167122400344     2.199961480217
    C                0.102699662912     1.454798400344     2.127620480217
    H                0.534547662912     2.418448400344     2.358836480217
    C                0.903650662912     0.337819400344     1.913406480217
    H                1.981219662912     0.408203400344     1.968241480217
    C                0.283098662912    -0.870835599656     1.619747480217
    H                0.858873662912    -1.766457599656     1.435672480217
    C               -1.107415337088    -0.908873599656     1.554297480217
    H               -1.616268337088    -1.837327599656     1.325455480217

""")

GEOS['%s-%s-%s' % (dbse, '27-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.368606570919     0.835881224534    -1.741444817415
    H                2.015973570919     1.688377224534    -1.585009817415
    C                1.918856570919    -0.418814775466    -1.996428817415
    H                2.992895570919    -0.540152775466    -2.040304817415
    C                1.084238570919    -1.516783775466    -2.201618817415
    H                1.510707570919    -2.489854775466    -2.405237817415
    C               -0.299256429081    -1.358797775466    -2.149626817415
    H               -0.947209429081    -2.210150775466    -2.309283817415
    C               -0.849625429081    -0.104320775466    -1.889296817415
    H               -1.922827429081     0.016617224534    -1.836348817415
    C               -0.015432429081     0.993112224534    -1.687451817415
    H               -0.442697429081     1.965370224534    -1.483472817415
    --
    0 1
    N               -1.917848429081     0.167759224534     1.842674182585
    C               -1.302962429081     1.325427224534     2.122738182585
    H               -1.950566429081     2.178599224534     2.285250182585
    C                0.079252570919     1.466275224534     2.212909182585
    H                0.511101570919     2.429925224534     2.444125182585
    C                0.880203570919     0.349296224534     1.998695182585
    H                1.957773570919     0.419680224534     2.053530182585
    C                0.259652570919    -0.859358775466     1.705036182585
    H                0.835426570919    -1.754980775466     1.520961182585
    C               -1.130862429081    -0.897396775466     1.639586182585
    H               -1.639715429081    -1.825850775466     1.410744182585

""")

GEOS['%s-%s-%s' % (dbse, '27-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.438813495424     0.801515947616    -1.996825613489
    H                2.086180495424     1.654011947616    -1.840390613489
    C                1.989063495424    -0.453180052384    -2.251809613489
    H                3.063102495424    -0.574518052384    -2.295685613489
    C                1.154445495424    -1.551149052384    -2.456999613489
    H                1.580914495424    -2.524220052384    -2.660618613489
    C               -0.229049504576    -1.393163052384    -2.405007613489
    H               -0.877002504576    -2.244516052384    -2.564664613489
    C               -0.779418504576    -0.138686052384    -2.144677613489
    H               -1.852620504576    -0.017748052384    -2.091729613489
    C                0.054774495424     0.958746947616    -1.942832613489
    H               -0.372490504576     1.931004947616    -1.738853613489
    --
    0 1
    N               -1.987171504576     0.201691947616     2.094839386511
    C               -1.372285504576     1.359359947616     2.374903386511
    H               -2.019889504576     2.212531947616     2.537415386511
    C                0.009929495424     1.500207947616     2.465074386511
    H                0.441778495424     2.463857947616     2.696290386511
    C                0.810881495424     0.383228947616     2.250860386511
    H                1.888450495424     0.453612947616     2.305695386511
    C                0.190329495424    -0.825427052384     1.957201386511
    H                0.766103495424    -1.721048052384     1.773126386511
    C               -1.200185504576    -0.863464052384     1.891751386511
    H               -1.709038504576    -1.791918052384     1.662909386511

""")

GEOS['%s-%s-%s' % (dbse, '27-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.564292974218     0.740095517396    -2.453262695842
    H                2.211659974218     1.592591517396    -2.296827695842
    C                2.114542974218    -0.514600482604    -2.708246695842
    H                3.188581974218    -0.635938482604    -2.752122695842
    C                1.279924974218    -1.612569482604    -2.913436695842
    H                1.706393974218    -2.585640482604    -3.117055695842
    C               -0.103570025782    -1.454583482604    -2.861444695842
    H               -0.751523025782    -2.305936482604    -3.021101695842
    C               -0.653939025782    -0.200106482604    -2.601114695842
    H               -1.727141025782    -0.079168482604    -2.548166695842
    C                0.180253974218     0.897326517396    -2.399269695842
    H               -0.247011025782     1.869584517396    -2.195290695842
    --
    0 1
    N               -2.111071025782     0.262339517396     2.545529304158
    C               -1.496185025782     1.420007517396     2.825593304158
    H               -2.143789025782     2.273178517396     2.988105304158
    C               -0.113970025782     1.560854517396     2.915764304158
    H                0.317878974218     2.524504517396     3.146980304158
    C                0.686981974218     0.443875517396     2.701550304158
    H                1.764550974218     0.514259517396     2.756385304158
    C                0.066429974218    -0.764779482604     2.407891304158
    H                0.642203974218    -1.660401482604     2.223816304158
    C               -1.324085025782    -0.802817482604     2.342441304158
    H               -1.832938025782    -1.731270482604     2.113599304158

""")

GEOS['%s-%s-%s' % (dbse, '27-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.806848603036     0.621368047095    -3.335569322036
    H                2.454215603036     1.473864047095    -3.179134322036
    C                2.357098603036    -0.633327952905    -3.590553322036
    H                3.431137603036    -0.754665952905    -3.634429322036
    C                1.522480603036    -1.731296952905    -3.795743322036
    H                1.948949603036    -2.704367952905    -3.999362322036
    C                0.138985603036    -1.573310952905    -3.743751322036
    H               -0.508967396964    -2.424663952905    -3.903408322036
    C               -0.411383396964    -0.318833952905    -3.483421322036
    H               -1.484585396964    -0.197895952905    -3.430473322036
    C                0.422809603036     0.778599047095    -3.281576322036
    H               -0.004455396964     1.750857047095    -3.077597322036
    --
    0 1
    N               -2.350572396964     0.379572047095     3.416726677964
    C               -1.735686396964     1.537240047095     3.696790677964
    H               -2.383290396964     2.390411047095     3.859302677964
    C               -0.353471396964     1.678087047095     3.786961677964
    H                0.078377603036     2.641737047095     4.018177677964
    C                0.447479603036     0.561108047095     3.572747677964
    H                1.525049603036     0.631492047095     3.627581677964
    C               -0.173071396964    -0.647546952905     3.279087677964
    H                0.402702603036    -1.543168952905     3.095012677964
    C               -1.563586396964    -0.685584952905     3.213638677964
    H               -2.072439396964    -1.614037952905     2.984795677964

""")

GEOS['%s-%s-%s' % (dbse, '28-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.829098800101     1.257133923746    -1.868159638452
    H                1.524342800101     2.083733923746    -1.910380638452
    C                1.303489800101    -0.042332076254    -1.700653638452
    H                2.366987800101    -0.221652076254    -1.620235638452
    C                0.406852800101    -1.107943076254    -1.629664638452
    H                0.777082800101    -2.116818076254    -1.503788638452
    C               -0.964479199899    -0.874581076254    -1.724769638452
    H               -1.661879199899    -1.699374076254    -1.667482638452
    C               -1.440178199899     0.425091923746    -1.890350638452
    H               -2.504181199899     0.606109923746    -1.952059638452
    C               -0.542425199899     1.489377923746    -1.964156638452
    H               -0.910892199899     2.498030923746    -2.095764638452
    --
    0 1
    N               -0.284193199899     0.727730923746     1.056989361548
    H               -0.657500199899     1.629492923746     0.794096361548
    C                1.107953800101     0.654743923746     1.189002361548
    O                1.798863800101     1.649167923746     1.044168361548
    C                1.586853800101    -0.679333076254     1.507113361548
    H                2.643905800101    -0.831549076254     1.641234361548
    C                0.707144800101    -1.693714076254     1.633329361548
    H                1.013071800101    -2.702135076254     1.869853361548
    N               -0.638101199899    -1.508683076254     1.463237361548
    H               -1.286849199899    -2.271241076254     1.562106361548
    C               -1.212549199899    -0.283881076254     1.163822361548
    O               -2.410338199899    -0.133058076254     1.021710361548

""")

GEOS['%s-%s-%s' % (dbse, '28-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.825641724375     1.266851278664    -1.959425035497
    H                1.520885724375     2.093451278664    -2.001646035497
    C                1.300032724375    -0.032614721336    -1.791919035497
    H                2.363530724375    -0.211934721336    -1.711501035497
    C                0.403395724375    -1.098225721336    -1.720930035497
    H                0.773625724375    -2.107100721336    -1.595054035497
    C               -0.967936275625    -0.864863721336    -1.816035035497
    H               -1.665336275625    -1.689656721336    -1.758748035497
    C               -1.443635275625     0.434809278664    -1.981616035497
    H               -2.507638275625     0.615827278664    -2.043325035497
    C               -0.545882275625     1.499095278664    -2.055422035497
    H               -0.914349275625     2.507748278664    -2.187030035497
    --
    0 1
    N               -0.281784275625     0.720961278664     1.120571964503
    H               -0.655092275625     1.622723278664     0.857678964503
    C                1.110361724375     0.647974278664     1.252584964503
    O                1.801272724375     1.642398278664     1.107750964503
    C                1.589262724375    -0.686102721336     1.570695964503
    H                2.646314724375    -0.838318721336     1.704816964503
    C                0.709552724375    -1.700484721336     1.696911964503
    H                1.015480724375    -2.708904721336     1.933435964503
    N               -0.635693275625    -1.515452721336     1.526819964503
    H               -1.284440275625    -2.278011721336     1.625688964503
    C               -1.210141275625    -0.290651721336     1.227404964503
    O               -2.407929275625    -0.139827721336     1.085292964503

""")

GEOS['%s-%s-%s' % (dbse, '28-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.822191905502     1.276547873468    -2.050496150555
    H                1.517435905502     2.103147873468    -2.092717150555
    C                1.296582905502    -0.022918126532    -1.882990150555
    H                2.360080905502    -0.202238126532    -1.802572150555
    C                0.399945905502    -1.088529126532    -1.812001150555
    H                0.770175905502    -2.097404126532    -1.686125150555
    C               -0.971386094498    -0.855167126532    -1.907106150555
    H               -1.668786094498    -1.679960126532    -1.849819150555
    C               -1.447085094498     0.444505873468    -2.072687150555
    H               -2.511088094498     0.625523873468    -2.134396150555
    C               -0.549332094498     1.508791873468    -2.146493150555
    H               -0.917799094498     2.517444873468    -2.278101150555
    --
    0 1
    N               -0.279381094498     0.714205873468     1.184018849445
    H               -0.652689094498     1.615967873468     0.921125849445
    C                1.112765905502     0.641218873468     1.316032849445
    O                1.803675905502     1.635642873468     1.171198849445
    C                1.591665905502    -0.692858126532     1.634142849445
    H                2.648717905502    -0.845074126532     1.768263849445
    C                0.711955905502    -1.707240126532     1.760358849445
    H                1.017883905502    -2.715660126532     1.996883849445
    N               -0.633289094498    -1.522208126532     1.590266849445
    H               -1.282037094498    -2.284767126532     1.689135849445
    C               -1.207738094498    -0.297407126532     1.290852849445
    O               -2.405526094498    -0.146583126532     1.148739849445

""")

GEOS['%s-%s-%s' % (dbse, '28-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.818742154998     1.286244620858    -2.141568533899
    H                1.513986154998     2.112844620858    -2.183789533899
    C                1.293133154998    -0.013221379142    -1.974062533899
    H                2.356631154998    -0.192541379142    -1.893644533899
    C                0.396496154998    -1.078832379142    -1.903073533899
    H                0.766726154998    -2.087707379142    -1.777197533899
    C               -0.974835845002    -0.845470379142    -1.998178533899
    H               -1.672235845002    -1.670263379142    -1.940891533899
    C               -1.450534845002     0.454202620858    -2.163759533899
    H               -2.514537845002     0.635220620858    -2.225468533899
    C               -0.552781845002     1.518488620858    -2.237565533899
    H               -0.921248845002     2.527141620858    -2.369173533899
    --
    0 1
    N               -0.276977845002     0.707450620858     1.247467466101
    H               -0.650284845002     1.609211620858     0.984574466101
    C                1.115169154998     0.634463620858     1.379480466101
    O                1.806079154998     1.628887620858     1.234646466101
    C                1.594069154998    -0.699614379142     1.697590466101
    H                2.651121154998    -0.851829379142     1.831711466101
    C                0.714360154998    -1.713995379142     1.823807466101
    H                1.020287154998    -2.722415379142     2.060331466101
    N               -0.630885845002    -1.528963379142     1.653715466101
    H               -1.279633845002    -2.291522379142     1.752584466101
    C               -1.205334845002    -0.304162379142     1.354300466101
    O               -2.403122845002    -0.153339379142     1.212188466101

""")

GEOS['%s-%s-%s' % (dbse, '28-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.815298429933     1.295924291866    -2.232482582625
    H                1.510542429933     2.122524291866    -2.274703582625
    C                1.289689429933    -0.003541708134    -2.064976582625
    H                2.353187429933    -0.182861708134    -1.984558582625
    C                0.393052429933    -1.069152708134    -1.993987582625
    H                0.763282429933    -2.078027708134    -1.868111582625
    C               -0.978279570067    -0.835790708134    -2.089092582625
    H               -1.675679570067    -1.660583708134    -2.031805582625
    C               -1.453978570067     0.463882291866    -2.254673582625
    H               -2.517981570067     0.644900291866    -2.316382582625
    C               -0.556225570067     1.528168291866    -2.328479582625
    H               -0.924692570067     2.536821291866    -2.460087582625
    --
    0 1
    N               -0.274578570067     0.700706291866     1.310805417375
    H               -0.647885570067     1.602468291866     1.047912417375
    C                1.117568429933     0.627720291866     1.442818417375
    O                1.808478429933     1.622143291866     1.297984417375
    C                1.596468429933    -0.706357708134     1.760928417375
    H                2.653520429933    -0.858573708134     1.895049417375
    C                0.716758429933    -1.720738708134     1.887145417375
    H                1.022686429933    -2.729159708134     2.123669417375
    N               -0.628486570067    -1.535706708134     1.717052417375
    H               -1.277234570067    -2.298265708134     1.815922417375
    C               -1.202935570067    -0.310905708134     1.417638417375
    O               -2.400723570067    -0.160082708134     1.275526417375

""")

GEOS['%s-%s-%s' % (dbse, '28-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.804985273202     1.324913217431    -2.504748214716
    H                1.500229273202     2.151513217431    -2.546969214716
    C                1.279376273202     0.025447217431    -2.337242214716
    H                2.342874273202    -0.153872782569    -2.256824214716
    C                0.382739273202    -1.040163782569    -2.266253214716
    H                0.752969273202    -2.049038782569    -2.140377214716
    C               -0.988592726798    -0.806801782569    -2.361358214716
    H               -1.685992726798    -1.631594782569    -2.304071214716
    C               -1.464291726798     0.492871217431    -2.526939214716
    H               -2.528294726798     0.673889217431    -2.588648214716
    C               -0.566538726798     1.557157217431    -2.600745214716
    H               -0.935005726798     2.565810217431    -2.732353214716
    --
    0 1
    N               -0.267393726798     0.680511217431     1.500486785284
    H               -0.640700726798     1.582272217431     1.237593785284
    C                1.124753273202     0.607524217431     1.632499785284
    O                1.815663273202     1.601947217431     1.487665785284
    C                1.603653273202    -0.726553782569     1.950609785284
    H                2.660705273202    -0.878769782569     2.084731785284
    C                0.723944273202    -1.740934782569     2.076826785284
    H                1.029871273202    -2.749355782569     2.313350785284
    N               -0.621301726798    -1.555902782569     1.906734785284
    H               -1.270049726798    -2.318461782569     2.005603785284
    C               -1.195750726798    -0.331101782569     1.607319785284
    O               -2.393538726798    -0.180278782569     1.465207785284

""")

GEOS['%s-%s-%s' % (dbse, '28-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.787838442664     1.373110338759    -2.957419742842
    H                1.483082442664     2.199710338759    -2.999640742842
    C                1.262229442664     0.073644338759    -2.789913742842
    H                2.325727442664    -0.105675661241    -2.709495742842
    C                0.365592442664    -0.991966661241    -2.718924742842
    H                0.735822442664    -2.000841661241    -2.593048742842
    C               -1.005739557336    -0.758604661241    -2.814029742842
    H               -1.703139557336    -1.583397661241    -2.756742742842
    C               -1.481438557336     0.541068338759    -2.979610742842
    H               -2.545441557336     0.722086338759    -3.041319742842
    C               -0.583685557336     1.605354338759    -3.053416742842
    H               -0.952152557336     2.614007338759    -3.185024742842
    --
    0 1
    N               -0.255447557336     0.646933338759     1.815853257158
    H               -0.628755557336     1.548694338759     1.552960257158
    C                1.136698442664     0.573946338759     1.947866257158
    O                1.827609442664     1.568369338759     1.803032257158
    C                1.615599442664    -0.760131661241     2.265976257158
    H                2.672651442664    -0.912347661241     2.400097257158
    C                0.735889442664    -1.774512661241     2.392193257158
    H                1.041817442664    -2.782933661241     2.628717257158
    N               -0.609356557336    -1.589480661241     2.222100257158
    H               -1.258103557336    -2.352039661241     2.320970257158
    C               -1.183804557336    -0.364679661241     1.922686257158
    O               -2.381592557336    -0.213856661241     1.780574257158

""")

GEOS['%s-%s-%s' % (dbse, '28-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.753630421607     1.469264022069    -3.860500374048
    H                1.448874421607     2.295864022069    -3.902721374048
    C                1.228021421607     0.169798022069    -3.692994374048
    H                2.291519421607    -0.009521977931    -3.612576374048
    C                0.331384421607    -0.895812977931    -3.622005374048
    H                0.701614421607    -1.904687977931    -3.496129374048
    C               -1.039947578393    -0.662450977931    -3.717110374048
    H               -1.737347578393    -1.487243977931    -3.659823374048
    C               -1.515646578393     0.637222022069    -3.882691374048
    H               -2.579649578393     0.818240022069    -3.944400374048
    C               -0.617893578393     1.701508022069    -3.956497374048
    H               -0.986360578393     2.710161022069    -4.088105374048
    --
    0 1
    N               -0.231615578393     0.579945022069     2.445009625952
    H               -0.604923578393     1.481706022069     2.182116625952
    C                1.160530421607     0.506958022069     2.577022625952
    O                1.851441421607     1.501382022069     2.432188625952
    C                1.639431421607    -0.827119977931     2.895132625952
    H                2.696483421607    -0.979335977931     3.029253625952
    C                0.759721421607    -1.841500977931     3.021349625952
    H                1.065649421607    -2.849920977931     3.257873625952
    N               -0.585524578393    -1.656468977931     2.851257625952
    H               -1.234271578393    -2.419027977931     2.950126625952
    C               -1.159972578393    -0.431667977931     2.551842625952
    O               -2.357760578393    -0.280844977931     2.409730625952

""")

GEOS['%s-%s-%s' % (dbse, '29-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.215726254715     0.009922004209    -1.494608455985
    C                0.616906254715    -1.177202995791    -1.670867455985
    H                1.256449254715    -2.048276995791    -1.605026455985
    C               -0.746342745285    -1.327403995791    -1.913153455985
    H               -1.165444745285    -2.315622995791    -2.037702455985
    C               -1.542896745285    -0.188693995791    -1.980786455985
    H               -2.606041745285    -0.264708995791    -2.163075455985
    C               -0.938650745285     1.051882004209    -1.804496455985
    H               -1.513842745285     1.965746004209    -1.835963455985
    C                0.432690254715     1.097355004209    -1.566970455985
    H                0.929666254715     2.049298004209    -1.426771455985
    --
    0 1
    N               -0.741141745285    -0.321570995791     1.103902544015
    H               -1.636030745285    -0.753433995791     0.919299544015
    C               -0.740886745285     1.076239004209     1.196913544015
    O               -1.780769745285     1.704439004209     1.082063544015
    C                0.578608254715     1.635320004209     1.424476544015
    H                0.679924254715     2.702542004209     1.520344544015
    C                1.648482254715     0.816628004209     1.495996544015
    H                2.652361254715     1.185293004209     1.644423544015
    N                1.530420254715    -0.540664995791     1.383441544015
    H                2.343856254715    -1.132852995791     1.375274544015
    C                0.329973254715    -1.187897995791     1.157060544015
    O                0.237836254715    -2.393315995791     1.023481544015

""")

GEOS['%s-%s-%s' % (dbse, '29-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.206494175716     0.004634890893    -1.598094914856
    C                0.607674175716    -1.182490109107    -1.774353914856
    H                1.247217175716    -2.053564109107    -1.708512914856
    C               -0.755574824284    -1.332691109107    -2.016639914856
    H               -1.174676824284    -2.320910109107    -2.141188914856
    C               -1.552128824284    -0.193981109107    -2.084272914856
    H               -2.615273824284    -0.269996109107    -2.266561914856
    C               -0.947882824284     1.046594890893    -1.907982914856
    H               -1.523074824284     1.960458890893    -1.939449914856
    C                0.423458175716     1.092067890893    -1.670456914856
    H                0.920434175716     2.044010890893    -1.530257914856
    --
    0 1
    N               -0.734627824284    -0.317840109107     1.176919085144
    H               -1.629516824284    -0.749704109107     0.992315085144
    C               -0.734372824284     1.079968890893     1.269930085144
    O               -1.774255824284     1.708168890893     1.155079085144
    C                0.585122175716     1.639050890893     1.497493085144
    H                0.686438175716     2.706272890893     1.593361085144
    C                1.654995175716     0.820357890893     1.569013085144
    H                2.658874175716     1.189022890893     1.717439085144
    N                1.536934175716    -0.536935109107     1.456458085144
    H                2.350370175716    -1.129122109107     1.448290085144
    C                0.336487175716    -1.184167109107     1.230076085144
    O                0.244350175716    -2.389585109107     1.096497085144

""")

GEOS['%s-%s-%s' % (dbse, '29-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.197387432391    -0.000580681203    -1.700179301368
    C                0.598567432391    -1.187705681203    -1.876438301368
    H                1.238110432391    -2.058779681203    -1.810597301368
    C               -0.764681567609    -1.337906681203    -2.118724301368
    H               -1.183783567609    -2.326125681203    -2.243273301368
    C               -1.561235567609    -0.199196681203    -2.186357301368
    H               -2.624380567609    -0.275211681203    -2.368646301368
    C               -0.956989567609     1.041379318797    -2.010067301368
    H               -1.532181567609     1.955243318797    -2.041534301368
    C                0.414351432391     1.086852318797    -1.772541301368
    H                0.911327432391     2.038795318797    -1.632342301368
    --
    0 1
    N               -0.728202567609    -0.314160681203     1.248945698632
    H               -1.623091567609    -0.746024681203     1.064341698632
    C               -0.727947567609     1.083649318797     1.341956698632
    O               -1.767830567609     1.711849318797     1.227106698632
    C                0.591547432391     1.642730318797     1.569519698632
    H                0.692863432391     2.709952318797     1.665387698632
    C                1.661421432391     0.824038318797     1.641039698632
    H                2.665300432391     1.192702318797     1.789465698632
    N                1.543359432391    -0.533254681203     1.528484698632
    H                2.356795432391    -1.125442681203     1.520317698632
    C                0.342912432391    -1.180487681203     1.302103698632
    O                0.250775432391    -2.385905681203     1.168523698632

""")

GEOS['%s-%s-%s' % (dbse, '29-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.188369228204    -0.005745264821    -1.801269378130
    C                0.589549228204    -1.192870264821    -1.977528378130
    H                1.229092228204    -2.063944264821    -1.911687378130
    C               -0.773699771796    -1.343071264821    -2.219814378130
    H               -1.192801771796    -2.331290264821    -2.344363378130
    C               -1.570253771796    -0.204361264821    -2.287447378130
    H               -2.633398771796    -0.280376264821    -2.469736378130
    C               -0.966007771796     1.036214735179    -2.111157378130
    H               -1.541199771796     1.950078735179    -2.142624378130
    C                0.405333228204     1.081687735179    -1.873631378130
    H                0.902309228204     2.033630735179    -1.733432378130
    --
    0 1
    N               -0.721839771796    -0.310516264821     1.320270621870
    H               -1.616728771796    -0.742380264821     1.135667621870
    C               -0.721584771796     1.087292735179     1.413281621870
    O               -1.761467771796     1.715492735179     1.298431621870
    C                0.597910228204     1.646374735179     1.640844621870
    H                0.699226228204     2.713596735179     1.736713621870
    C                1.667784228204     0.827681735179     1.712365621870
    H                2.671663228204     1.196346735179     1.860791621870
    N                1.549722228204    -0.529611264821     1.599810621870
    H                2.363158228204    -1.121798264821     1.591642621870
    C                0.349276228204    -1.176843264821     1.373428621870
    O                0.257138228204    -2.382261264821     1.239849621870

""")

GEOS['%s-%s-%s' % (dbse, '29-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.179439165251    -0.010859414630    -1.901370620765
    C                0.580619165251    -1.197984414630    -2.077629620765
    H                1.220162165251    -2.069058414630    -2.011788620765
    C               -0.782629834749    -1.348185414630    -2.319915620765
    H               -1.201731834749    -2.336404414630    -2.444464620765
    C               -1.579183834749    -0.209475414630    -2.387548620765
    H               -2.642328834749    -0.285490414630    -2.569837620765
    C               -0.974937834749     1.031100585370    -2.211258620765
    H               -1.550129834749     1.944964585370    -2.242725620765
    C                0.396403165251     1.076573585370    -1.973732620765
    H                0.893379165251     2.028516585370    -1.833533620765
    --
    0 1
    N               -0.715538834749    -0.306908414630     1.390898379235
    H               -1.610427834749    -0.738771414630     1.206295379235
    C               -0.715283834749     1.090901585370     1.483909379235
    O               -1.755166834749     1.719101585370     1.369059379235
    C                0.604211165251     1.649982585370     1.711472379235
    H                0.705527165251     2.717204585370     1.807340379235
    C                1.674084165251     0.831290585370     1.782992379235
    H                2.677963165251     1.199955585370     1.931419379235
    N                1.556023165251    -0.526002414630     1.670438379235
    H                2.369459165251    -1.118190414630     1.662270379235
    C                0.355576165251    -1.173235414630     1.444056379235
    O                0.263439165251    -2.378653414630     1.310477379235

""")

GEOS['%s-%s-%s' % (dbse, '29-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.153032684165    -0.025982288349    -2.197375446497
    C                0.554212684165    -1.213107288349    -2.373634446497
    H                1.193755684165    -2.084181288349    -2.307793446497
    C               -0.809036315835    -1.363308288349    -2.615920446497
    H               -1.228138315835    -2.351527288349    -2.740469446497
    C               -1.605590315835    -0.224598288349    -2.683553446497
    H               -2.668735315835    -0.300613288349    -2.865842446497
    C               -1.001344315835     1.015977711651    -2.507263446497
    H               -1.576536315835     1.929841711651    -2.538730446497
    C                0.369996684165     1.061450711651    -2.269737446497
    H                0.866972684165     2.013393711651    -2.129538446497
    --
    0 1
    N               -0.696907315835    -0.296238288349     1.599748553503
    H               -1.591796315835    -0.728101288349     1.415144553503
    C               -0.696652315835     1.101571711651     1.692759553503
    O               -1.736535315835     1.729771711651     1.577909553503
    C                0.622841684165     1.660652711651     1.920322553503
    H                0.724158684165     2.727874711651     2.016190553503
    C                1.692715684165     0.841960711651     1.991842553503
    H                2.696594684165     1.210625711651     2.140268553503
    N                1.574654684165    -0.515332288349     1.879287553503
    H                2.388090684165    -1.107520288349     1.871120553503
    C                0.374207684165    -1.162565288349     1.652905553503
    O                0.282070684165    -2.367983288349     1.519326553503

""")

GEOS['%s-%s-%s' % (dbse, '29-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.109874478000    -0.050698899170    -2.681161475396
    C                0.511054478000    -1.237823899170    -2.857420475396
    H                1.150597478000    -2.108897899170    -2.791579475396
    C               -0.852194522000    -1.388024899170    -3.099706475396
    H               -1.271296522000    -2.376243899170    -3.224255475396
    C               -1.648748522000    -0.249314899170    -3.167339475396
    H               -2.711893522000    -0.325329899170    -3.349628475396
    C               -1.044502522000     0.991261100830    -2.991049475396
    H               -1.619694522000     1.905125100830    -3.022516475396
    C                0.326838478000     1.036734100830    -2.753523475396
    H                0.823814478000     1.988677100830    -2.613324475396
    --
    0 1
    N               -0.666456522000    -0.278798899170     1.941089524604
    H               -1.561345522000    -0.710662899170     1.756485524604
    C               -0.666201522000     1.119011100830     2.034100524604
    O               -1.706084522000     1.747210100830     1.919250524604
    C                0.653292478000     1.678092100830     2.261663524604
    H                0.754609478000     2.745314100830     2.357531524604
    C                1.723166478000     0.859400100830     2.333183524604
    H                2.727045478000     1.228064100830     2.481609524604
    N                1.605105478000    -0.497893899170     2.220628524604
    H                2.418541478000    -1.090080899170     2.212461524604
    C                0.404658478000    -1.145125899170     1.994247524604
    O                0.312521478000    -2.350543899170     1.860667524604

""")

GEOS['%s-%s-%s' % (dbse, '29-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.025216840103    -0.099182036096    -3.630139474575
    C                0.426396840103    -1.286307036096    -3.806398474575
    H                1.065939840103    -2.157381036096    -3.740557474575
    C               -0.936852159897    -1.436508036096    -4.048684474575
    H               -1.355954159897    -2.424727036096    -4.173233474575
    C               -1.733406159897    -0.297798036096    -4.116317474575
    H               -2.796551159897    -0.373813036096    -4.298606474575
    C               -1.129160159897     0.942777963904    -3.940027474575
    H               -1.704352159897     1.856641963904    -3.971494474575
    C                0.242180840103     0.988250963904    -3.702501474575
    H                0.739156840103     1.940193963904    -3.562302474575
    --
    0 1
    N               -0.606725159897    -0.244591036096     2.610652525425
    H               -1.501614159897    -0.676455036096     2.426048525425
    C               -0.606470159897     1.153218963904     2.703663525425
    O               -1.646354159897     1.781417963904     2.588813525425
    C                0.713023840103     1.712299963904     2.931226525425
    H                0.814340840103     2.779521963904     3.027094525425
    C                1.782897840103     0.893607963904     3.002746525425
    H                2.786776840103     1.262271963904     3.151172525425
    N                1.664836840103    -0.463686036096     2.890191525425
    H                2.478272840103    -1.055873036096     2.882024525425
    C                0.464389840103    -1.110918036096     2.663809525425
    O                0.372252840103    -2.316336036096     2.530230525425

""")

GEOS['%s-%s-%s' % (dbse, '30-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835519772867     1.115164519887    -0.815758985804
    H                1.484326772867     1.980606519887    -0.817625985804
    C                1.383277772867    -0.166149480113    -0.813394985804
    H                2.457151772867    -0.295207480113    -0.814388985804
    C                0.547557772867    -1.281318480113    -0.815473985804
    H                0.972938772867    -2.275807480113    -0.817380985804
    C               -0.835520227133    -1.115164480113    -0.815760985804
    H               -1.484331227133    -1.980608480113    -0.817629985804
    C               -1.383281227133     0.166151519887    -0.813401985804
    H               -2.457153227133     0.295206519887    -0.814402985804
    C               -0.547563227133     1.281320519887    -0.815479985804
    H               -0.972940227133     2.275802519887    -0.817390985804
    --
    0 1
    C                0.655782772867    -0.116792480113     2.269832014196
    H                1.047243772867    -1.123911480113     2.265364014196
    H                1.370856772867     0.693271519887     2.265330014196
    C               -0.655774227133     0.116789519887     2.269841014196
    H               -1.370846227133    -0.693274480113     2.265345014196
    H               -1.047237227133     1.123908519887     2.265383014196

""")

GEOS['%s-%s-%s' % (dbse, '30-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835519640741     1.115164519887    -0.862258713981
    H                1.484326640741     1.980606519887    -0.864125713981
    C                1.383277640741    -0.166149480113    -0.859894713981
    H                2.457151640741    -0.295207480113    -0.860888713981
    C                0.547557640741    -1.281318480113    -0.861973713981
    H                0.972938640741    -2.275807480113    -0.863880713981
    C               -0.835520359259    -1.115164480113    -0.862260713981
    H               -1.484331359259    -1.980608480113    -0.864129713981
    C               -1.383281359259     0.166151519887    -0.859901713981
    H               -2.457153359259     0.295206519887    -0.860902713981
    C               -0.547563359259     1.281320519887    -0.861979713981
    H               -0.972940359259     2.275802519887    -0.863890713981
    --
    0 1
    C                0.655782640741    -0.116792480113     2.399300286019
    H                1.047243640741    -1.123911480113     2.394831286019
    H                1.370856640741     0.693271519887     2.394798286019
    C               -0.655773359259     0.116789519887     2.399309286019
    H               -1.370845359259    -0.693274480113     2.394813286019
    H               -1.047236359259     1.123908519887     2.394850286019

""")

GEOS['%s-%s-%s' % (dbse, '30-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835519640741     1.115164519887    -0.908668380017
    H                1.484326640741     1.980606519887    -0.910535380017
    C                1.383277640741    -0.166149480113    -0.906304380017
    H                2.457151640741    -0.295207480113    -0.907298380017
    C                0.547557640741    -1.281318480113    -0.908383380017
    H                0.972938640741    -2.275807480113    -0.910290380017
    C               -0.835520359259    -1.115164480113    -0.908670380017
    H               -1.484331359259    -1.980608480113    -0.910539380017
    C               -1.383281359259     0.166151519887    -0.906311380017
    H               -2.457153359259     0.295206519887    -0.907312380017
    C               -0.547563359259     1.281320519887    -0.908389380017
    H               -0.972940359259     2.275802519887    -0.910300380017
    --
    0 1
    C                0.655782640741    -0.116792480113     2.528517619983
    H                1.047243640741    -1.123911480113     2.524049619983
    H                1.370856640741     0.693271519887     2.524016619983
    C               -0.655773359259     0.116789519887     2.528526619983
    H               -1.370845359259    -0.693274480113     2.524030619983
    H               -1.047236359259     1.123908519887     2.524068619983

""")

GEOS['%s-%s-%s' % (dbse, '30-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835519508616     1.115164642512    -0.955076922543
    H                1.484326508616     1.980606642512    -0.956943922543
    C                1.383277508616    -0.166149357488    -0.952712922543
    H                2.457151508616    -0.295207357488    -0.953706922543
    C                0.547557508616    -1.281318357488    -0.954791922543
    H                0.972938508616    -2.275807357488    -0.956698922543
    C               -0.835520491384    -1.115164357488    -0.955078922543
    H               -1.484331491384    -1.980608357488    -0.956947922543
    C               -1.383281491384     0.166151642512    -0.952719922543
    H               -2.457153491384     0.295206642512    -0.953720922543
    C               -0.547563491384     1.281320642512    -0.954797922543
    H               -0.972940491384     2.275802642512    -0.956708922543
    --
    0 1
    C                0.655783508616    -0.116793357488     2.657732077457
    H                1.047244508616    -1.123911357488     2.653263077457
    H                1.370857508616     0.693270642512     2.653230077457
    C               -0.655773491384     0.116789642512     2.657741077457
    H               -1.370845491384    -0.693274357488     2.653244077457
    H               -1.047236491384     1.123908642512     2.653282077457

""")

GEOS['%s-%s-%s' % (dbse, '30-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835519385991     1.115164652012    -1.001412192379
    H                1.484326385991     1.980606652012    -1.003279192379
    C                1.383277385991    -0.166149347988    -0.999048192379
    H                2.457151385991    -0.295207347988    -1.000042192379
    C                0.547557385991    -1.281318347988    -1.001127192379
    H                0.972938385991    -2.275807347988    -1.003034192379
    C               -0.835520614009    -1.115164347988    -1.001414192379
    H               -1.484331614009    -1.980608347988    -1.003283192379
    C               -1.383281614009     0.166151652012    -0.999055192379
    H               -2.457153614009     0.295206652012    -1.000056192379
    C               -0.547563614009     1.281320652012    -1.001133192379
    H               -0.972940614009     2.275802652012    -1.003044192379
    --
    0 1
    C                0.655783385991    -0.116793347988     2.786742807621
    H                1.047244385991    -1.123911347988     2.782273807621
    H                1.370857385991     0.693270652012     2.782240807621
    C               -0.655772614009     0.116789652012     2.786750807621
    H               -1.370844614009    -0.693275347988     2.782254807621
    H               -1.047236614009     1.123908652012     2.782292807621

""")

GEOS['%s-%s-%s' % (dbse, '30-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835519121740     1.115164661513    -1.140195200159
    H                1.484326121740     1.980606661513    -1.142062200159
    C                1.383277121740    -0.166149338487    -1.137831200159
    H                2.457151121740    -0.295207338487    -1.138825200159
    C                0.547557121740    -1.281318338487    -1.139910200159
    H                0.972938121740    -2.275807338487    -1.141817200159
    C               -0.835520878260    -1.115164338487    -1.140197200159
    H               -1.484331878260    -1.980608338487    -1.142066200159
    C               -1.383281878260     0.166151661513    -1.137838200159
    H               -2.457153878260     0.295206661513    -1.138839200159
    C               -0.547563878260     1.281320661513    -1.139916200159
    H               -0.972940878260     2.275802661513    -1.141827200159
    --
    0 1
    C                0.655784121740    -0.116793338487     3.173152799841
    H                1.047245121740    -1.123912338487     3.168684799841
    H                1.370858121740     0.693270661513     3.168651799841
    C               -0.655771878260     0.116789661513     3.173161799841
    H               -1.370843878260    -0.693275338487     3.168665799841
    H               -1.047235878260     1.123908661513     3.168703799841

""")

GEOS['%s-%s-%s' % (dbse, '30-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835518715862     1.115164906763    -1.370982900321
    H                1.484325715862     1.980606906763    -1.372849900321
    C                1.383276715862    -0.166149093237    -1.368618900321
    H                2.457150715862    -0.295207093237    -1.369612900321
    C                0.547556715862    -1.281318093237    -1.370697900321
    H                0.972937715862    -2.275807093237    -1.372604900321
    C               -0.835521284138    -1.115164093237    -1.370984900321
    H               -1.484332284138    -1.980608093237    -1.372853900321
    C               -1.383282284138     0.166151906763    -1.368625900321
    H               -2.457154284138     0.295206906763    -1.369626900321
    C               -0.547564284138     1.281320906763    -1.370703900321
    H               -0.972941284138     2.275802906763    -1.372614900321
    --
    0 1
    C                0.655785715862    -0.116794093237     3.815730099679
    H                1.047246715862    -1.123912093237     3.811262099679
    H                1.370859715862     0.693269906763     3.811228099679
    C               -0.655771284138     0.116788906763     3.815739099679
    H               -1.370843284138    -0.693275093237     3.811243099679
    H               -1.047234284138     1.123907906763     3.811281099679

""")

GEOS['%s-%s-%s' % (dbse, '30-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.835517913608     1.115165190015    -1.831498144014
    H                1.484324913608     1.980607190015    -1.833365144014
    C                1.383275913608    -0.166148809985    -1.829134144014
    H                2.457149913608    -0.295206809985    -1.830128144014
    C                0.547555913608    -1.281317809985    -1.831213144014
    H                0.972936913608    -2.275806809985    -1.833120144014
    C               -0.835522086392    -1.115163809985    -1.831500144014
    H               -1.484333086392    -1.980607809985    -1.833369144014
    C               -1.383283086392     0.166152190015    -1.829141144014
    H               -2.457155086392     0.295207190015    -1.830142144014
    C               -0.547565086392     1.281321190015    -1.831219144014
    H               -0.972942086392     2.275803190015    -1.833130144014
    --
    0 1
    C                0.655787913608    -0.116794809985     5.097932855986
    H                1.047248913608    -1.123913809985     5.093464855986
    H                1.370861913608     0.693269190015     5.093430855986
    C               -0.655769086392     0.116788190015     5.097941855986
    H               -1.370840086392    -0.693276809985     5.093445855986
    H               -1.047232086392     1.123907190015     5.093483855986

""")

GEOS['%s-%s-%s' % (dbse, '31-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.039950622830    -0.962945048259    -0.631979153238
    H               -0.042298622830    -1.973558048259    -0.616119153238
    C               -1.297889622830    -0.344740048259    -0.661915153238
    O               -2.316296622830    -1.015419048259    -0.671770153238
    C               -1.225894622830     1.105183951741    -0.678165153238
    H               -2.141815622830     1.668595951741    -0.720715153238
    C               -0.024270622830     1.714971951741    -0.632236153238
    H                0.081289377170     2.789611951741    -0.634059153238
    N                1.145452377170     1.007418951741    -0.574096153238
    H                2.034643377170     1.473907951741    -0.510248153238
    C                1.204109377170    -0.374697048259    -0.550166153238
    O                2.247321377170    -0.994052048259    -0.471755153238
    --
    0 1
    C                0.729315377170     0.032497951741     2.448070846762
    H                1.246433377170    -0.838490048259     2.070843846762
    H                1.334926377170     0.819491951741     2.875384846762
    C               -0.599460622830     0.133087951741     2.398823846762
    H               -1.211463622830    -0.654391048259     1.981922846762
    H               -1.111285622830     1.004157951741     2.782822846762

""")

GEOS['%s-%s-%s' % (dbse, '31-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.040759788170    -0.963974853372    -0.662174278427
    H               -0.043107788170    -1.974587853372    -0.646314278427
    C               -1.298698788170    -0.345769853372    -0.692110278427
    O               -2.317105788170    -1.016448853372    -0.701965278427
    C               -1.226703788170     1.104154146628    -0.708360278427
    H               -2.142624788170     1.667566146628    -0.750910278427
    C               -0.025079788170     1.713942146628    -0.662431278427
    H                0.080480211830     2.788582146628    -0.664254278427
    N                1.144643211830     1.006389146628    -0.604291278427
    H                2.033834211830     1.472878146628    -0.540443278427
    C                1.203300211830    -0.375726853372    -0.580361278427
    O                2.246512211830    -0.995081853372    -0.501950278427
    --
    0 1
    C                0.732549211830     0.036613146628     2.568745721573
    H                1.249667211830    -0.834374853372     2.191519721573
    H                1.338160211830     0.823607146628     2.996060721573
    C               -0.596226788170     0.137204146628     2.519498721573
    H               -1.208229788170    -0.650275853372     2.102598721573
    H               -1.108051788170     1.008273146628     2.903498721573

""")

GEOS['%s-%s-%s' % (dbse, '31-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.041566644705    -0.965001770848    -0.692285294518
    H               -0.043914644705    -1.975614770848    -0.676425294518
    C               -1.299505644705    -0.346796770848    -0.722221294518
    O               -2.317912644705    -1.017475770848    -0.732076294518
    C               -1.227510644705     1.103127229152    -0.738471294518
    H               -2.143431644705     1.666539229152    -0.781021294518
    C               -0.025886644705     1.712915229152    -0.692542294518
    H                0.079673355295     2.787555229152    -0.694365294518
    N                1.143836355295     1.005362229152    -0.634402294518
    H                2.033027355295     1.471851229152    -0.570554294518
    C                1.202493355295    -0.376753770848    -0.610472294518
    O                2.245705355295    -0.996108770848    -0.532061294518
    --
    0 1
    C                0.735774355295     0.040717229152     2.689084705482
    H                1.252891355295    -0.830270770848     2.311857705482
    H                1.341385355295     0.827711229152     3.116398705482
    C               -0.593002644705     0.141308229152     2.639837705482
    H               -1.205005644705    -0.646171770848     2.222936705482
    H               -1.104827644705     1.012377229152     3.023837705482

""")

GEOS['%s-%s-%s' % (dbse, '31-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.042373115353    -0.966028095101    -0.722381514652
    H               -0.044721115353    -1.976641095101    -0.706521514652
    C               -1.300312115353    -0.347823095101    -0.752317514652
    O               -2.318719115353    -1.018502095101    -0.762172514652
    C               -1.228317115353     1.102100904899    -0.768567514652
    H               -2.144238115353     1.665512904899    -0.811117514652
    C               -0.026693115353     1.711888904899    -0.722638514652
    H                0.078866884647     2.786528904899    -0.724461514652
    N                1.143029884647     1.004335904899    -0.664498514652
    H                2.032220884647     1.470824904899    -0.600650514652
    C                1.201686884647    -0.377780095101    -0.640568514652
    O                2.244898884647    -0.997135095101    -0.562157514652
    --
    0 1
    C                0.738996884647     0.044818904899     2.809364485348
    H                1.256114884647    -0.826168095101     2.432137485348
    H                1.344607884647     0.831812904899     3.236678485348
    C               -0.589779115353     0.145409904899     2.760117485348
    H               -1.201782115353    -0.642070095101     2.343216485348
    H               -1.101604115353     1.016478904899     3.144116485348

""")

GEOS['%s-%s-%s' % (dbse, '31-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.043177770350    -0.967052203424    -0.752410509390
    H               -0.045525770350    -1.977665203424    -0.736550509390
    C               -1.301116770350    -0.348847203424    -0.782346509390
    O               -2.319523770350    -1.019526203424    -0.792201509390
    C               -1.229121770350     1.101076796576    -0.798596509390
    H               -2.145042770350     1.664488796576    -0.841146509390
    C               -0.027497770350     1.710864796576    -0.752667509390
    H                0.078062229650     2.785504796576    -0.754490509390
    N                1.142225229650     1.003311796576    -0.694527509390
    H                2.031416229650     1.469800796576    -0.630679509390
    C                1.200882229650    -0.378804203424    -0.670597509390
    O                2.244094229650    -0.998159203424    -0.592186509390
    --
    0 1
    C                0.742213229650     0.048911796576     2.929375490610
    H                1.259330229650    -0.822076203424     2.552149490610
    H                1.347824229650     0.835905796576     3.356689490610
    C               -0.586563770350     0.149502796576     2.880128490610
    H               -1.198566770350    -0.637977203424     2.463227490610
    H               -1.098388770350     1.020571796576     3.264128490610

""")

GEOS['%s-%s-%s' % (dbse, '31-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.045585759931    -0.970116858988    -0.842273479566
    H               -0.047933759931    -1.980729858988    -0.826413479566
    C               -1.303524759931    -0.351911858988    -0.872209479566
    O               -2.321931759931    -1.022590858988    -0.882064479566
    C               -1.231529759931     1.098012141012    -0.888459479566
    H               -2.147450759931     1.661424141012    -0.931009479566
    C               -0.029905759931     1.707800141012    -0.842530479566
    H                0.075654240069     2.782440141012    -0.844353479566
    N                1.139817240069     1.000247141012    -0.784390479566
    H                2.029008240069     1.466736141012    -0.720542479566
    C                1.198474240069    -0.381868858988    -0.760460479566
    O                2.241686240069    -1.001223858988    -0.682049479566
    --
    0 1
    C                0.751836240069     0.061160141012     3.288513520434
    H                1.268954240069    -0.809827858988     2.911286520434
    H                1.357447240069     0.848154141012     3.715827520434
    C               -0.576939759931     0.161750141012     3.239266520434
    H               -1.188942759931    -0.625728858988     2.822365520434
    H               -1.088764759931     1.032820141012     3.623265520434

""")

GEOS['%s-%s-%s' % (dbse, '31-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.049585254258    -0.975206900866    -0.991525586291
    H               -0.051933254258    -1.985819900866    -0.975665586291
    C               -1.307524254258    -0.357001900866    -1.021461586291
    O               -2.325931254258    -1.027680900866    -1.031316586291
    C               -1.235529254258     1.092922099134    -1.037711586291
    H               -2.151450254258     1.656334099134    -1.080261586291
    C               -0.033905254258     1.702710099134    -0.991782586291
    H                0.071654745742     2.777350099134    -0.993605586291
    N                1.135817745742     0.995157099134    -0.933642586291
    H                2.025008745742     1.461646099134    -0.869794586291
    C                1.194474745742    -0.386958900866    -0.909712586291
    O                2.237686745742    -1.006313900866    -0.831301586291
    --
    0 1
    C                0.767820745742     0.081502099134     3.885000413709
    H                1.284938745742    -0.789485900866     3.507774413709
    H                1.373431745742     0.868496099134     4.312315413709
    C               -0.560956254258     0.182093099134     3.835753413709
    H               -1.172959254258    -0.605386900866     3.418853413709
    H               -1.072781254258     1.053162099134     4.219753413709

""")

GEOS['%s-%s-%s' % (dbse, '31-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.057627465919    -0.985442051858    -1.291644448752
    H               -0.059975465919    -1.996055051858    -1.275784448752
    C               -1.315566465919    -0.367237051858    -1.321580448752
    O               -2.333973465919    -1.037916051858    -1.331435448752
    C               -1.243571465919     1.082686948142    -1.337830448752
    H               -2.159492465919     1.646098948142    -1.380380448752
    C               -0.041947465919     1.692474948142    -1.291901448752
    H                0.063612534081     2.767114948142    -1.293724448752
    N                1.127775534081     0.984921948142    -1.233761448752
    H                2.016966534081     1.451410948142    -1.169913448752
    C                1.186432534081    -0.397194051858    -1.209831448752
    O                2.229644534081    -1.016549051858    -1.131420448752
    --
    0 1
    C                0.799961534081     0.122406948142     5.084427551248
    H                1.317078534081    -0.748581051858     4.707200551248
    H                1.405572534081     0.909400948142     5.511741551248
    C               -0.528815465919     0.222997948142     5.035180551248
    H               -1.140818465919    -0.564482051858     4.618279551248
    H               -1.040640465919     1.094066948142     5.419180551248

""")

GEOS['%s-%s-%s' % (dbse, '32-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.045842063002    -0.947644394962    -0.609916590543
    H               -0.047704063002    -1.957366394962    -0.564873590543
    C               -1.304348063002    -0.334798394962    -0.684512590543
    O               -2.319285063002    -1.007554394962    -0.743029590543
    C               -1.238747063002     1.116398605038    -0.686435590543
    H               -2.154737063002     1.675679605038    -0.767028590543
    C               -0.043468063002     1.731773605038    -0.585731590543
    H                0.057722936998     2.806776605038    -0.576506590543
    N                1.125537936998     1.027937605038    -0.484757590543
    H                2.006266936998     1.497626605038    -0.358306590543
    C                1.184958936998    -0.353458394962    -0.443760590543
    O                2.224242936998    -0.966118394962    -0.284754590543
    --
    0 1
    C                0.701802936998    -0.174137394962     2.363596409457
    H                1.697620936998    -0.528123394962     2.249377409457
    C               -0.442801063002     0.212320605038     2.469788409457
    H               -1.447684063002     0.541020605038     2.570152409457

""")

GEOS['%s-%s-%s' % (dbse, '32-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.047375769797    -0.947858109259    -0.638595492258
    H               -0.049237769797    -1.957580109259    -0.593552492258
    C               -1.305881769797    -0.335012109259    -0.713191492258
    O               -2.320818769797    -1.007768109259    -0.771708492258
    C               -1.240280769797     1.116184890741    -0.715114492258
    H               -2.156270769797     1.675465890741    -0.795707492258
    C               -0.045001769797     1.731559890741    -0.614410492258
    H                0.056189230203     2.806562890741    -0.605185492258
    N                1.124004230203     1.027723890741    -0.513436492258
    H                2.004733230203     1.497412890741    -0.386985492258
    C                1.183425230203    -0.353672109259    -0.472439492258
    O                2.222709230203    -0.966332109259    -0.313433492258
    --
    0 1
    C                0.708407230203    -0.173217109259     2.487092507742
    H                1.704226230203    -0.527203109259     2.372873507742
    C               -0.436196769797     0.213240890741     2.593283507742
    H               -1.441078769797     0.541940890741     2.693648507742

""")

GEOS['%s-%s-%s' % (dbse, '32-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.048901546646    -0.948070605863    -0.667124553210
    H               -0.050763546646    -1.957792605863    -0.622081553210
    C               -1.307407546646    -0.335224605863    -0.741720553210
    O               -2.322344546646    -1.007980605863    -0.800237553210
    C               -1.241806546646     1.115972394137    -0.743643553210
    H               -2.157796546646     1.675253394137    -0.824236553210
    C               -0.046527546646     1.731347394137    -0.642939553210
    H                0.054663453354     2.806350394137    -0.633714553210
    N                1.122478453354     1.027511394137    -0.541965553210
    H                2.003207453354     1.497200394137    -0.415514553210
    C                1.181899453354    -0.353884605863    -0.500968553210
    O                2.221183453354    -0.966544605863    -0.341962553210
    --
    0 1
    C                0.714977453354    -0.172301605863     2.609942446790
    H                1.710796453354    -0.526287605863     2.495723446790
    C               -0.429626546646     0.214155394137     2.716134446790
    H               -1.434508546646     0.542856394137     2.816498446790

""")

GEOS['%s-%s-%s' % (dbse, '32-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.050423539683    -0.948282624015    -0.695580216068
    H               -0.052285539683    -1.958004624015    -0.650537216068
    C               -1.308929539683    -0.335436624015    -0.770176216068
    O               -2.323866539683    -1.008192624015    -0.828693216068
    C               -1.243328539683     1.115760375985    -0.772099216068
    H               -2.159318539683     1.675041375985    -0.852692216068
    C               -0.048049539683     1.731135375985    -0.671395216068
    H                0.053141460317     2.806138375985    -0.662170216068
    N                1.120956460317     1.027299375985    -0.570421216068
    H                2.001685460317     1.496988375985    -0.443970216068
    C                1.180377460317    -0.354096624015    -0.529424216068
    O                2.219661460317    -0.966756624015    -0.370418216068
    --
    0 1
    C                0.721531460317    -0.171388624015     2.732476783932
    H                1.717349460317    -0.525374624015     2.618257783932
    C               -0.423072539683     0.215068375985     2.838668783932
    H               -1.427955539683     0.543769375985     2.939032783932

""")

GEOS['%s-%s-%s' % (dbse, '32-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.051939241192    -0.948493786793    -0.723922237197
    H               -0.053801241192    -1.958215786793    -0.678879237197
    C               -1.310445241192    -0.335647786793    -0.798518237197
    O               -2.325382241192    -1.008403786793    -0.857035237197
    C               -1.244844241192     1.115549213207    -0.800441237197
    H               -2.160834241192     1.674830213207    -0.881034237197
    C               -0.049565241192     1.730924213207    -0.699737237197
    H                0.051625758808     2.805927213207    -0.690512237197
    N                1.119440758808     1.027088213207    -0.598763237197
    H                2.000169758808     1.496777213207    -0.472312237197
    C                1.178861758808    -0.354307786793    -0.557766237197
    O                2.218145758808    -0.966967786793    -0.398760237197
    --
    0 1
    C                0.728058758808    -0.170479786793     2.854521762803
    H                1.723876758808    -0.524465786793     2.740302762803
    C               -0.416546241192     0.215978213207     2.960713762803
    H               -1.421428241192     0.544678213207     3.061077762803

""")

GEOS['%s-%s-%s' % (dbse, '32-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.056462570483    -0.949123998970    -0.808499771705
    H               -0.058324570483    -1.958845998970    -0.763456771705
    C               -1.314968570483    -0.336277998970    -0.883095771705
    O               -2.329905570483    -1.009033998970    -0.941612771705
    C               -1.249367570483     1.114919001030    -0.885018771705
    H               -2.165357570483     1.674200001030    -0.965611771705
    C               -0.054088570483     1.730294001030    -0.784314771705
    H                0.047102429517     2.805297001030    -0.775089771705
    N                1.114917429517     1.026458001030    -0.683340771705
    H                1.995646429517     1.496147001030    -0.556889771705
    C                1.174338429517    -0.354937998970    -0.642343771705
    O                2.213622429517    -0.967597998970    -0.483337771705
    --
    0 1
    C                0.747536429517    -0.167765998970     3.218725228295
    H                1.743354429517    -0.521751998970     3.104507228295
    C               -0.397067570483     0.218692001030     3.324917228295
    H               -1.401950570483     0.547392001030     3.425281228295

""")

GEOS['%s-%s-%s' % (dbse, '32-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.063946528545    -0.950166562894    -0.948434382931
    H               -0.065808528545    -1.959888562894    -0.903391382931
    C               -1.322452528545    -0.337320562894    -1.023030382931
    O               -2.337389528545    -1.010076562894    -1.081547382931
    C               -1.256851528545     1.113876437106    -1.024953382931
    H               -2.172841528545     1.673157437106    -1.105546382931
    C               -0.061572528545     1.729251437106    -0.924249382931
    H                0.039618471455     2.804254437106    -0.915024382931
    N                1.107433471455     1.025415437106    -0.823275382931
    H                1.988162471455     1.495104437106    -0.696824382931
    C                1.166854471455    -0.355980562894    -0.782278382931
    O                2.206138471455    -0.968640562894    -0.623272382931
    --
    0 1
    C                0.779763471455    -0.163276562894     3.821304617069
    H                1.775581471455    -0.517262562894     3.707085617069
    C               -0.364840528545     0.223181437106     3.927496617069
    H               -1.369723528545     0.551881437106     4.027860617069

""")

GEOS['%s-%s-%s' % (dbse, '32-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.078803267529    -0.952236244276    -1.226228857466
    H               -0.080665267529    -1.961958244276    -1.181185857466
    C               -1.337309267529    -0.339390244276    -1.300824857466
    O               -2.352246267529    -1.012146244276    -1.359341857466
    C               -1.271708267529     1.111806755724    -1.302747857466
    H               -2.187698267529     1.671087755724    -1.383340857466
    C               -0.076429267529     1.727181755724    -1.202043857466
    H                0.024761732471     2.802184755724    -1.192818857466
    N                1.092576732471     1.023345755724    -1.101069857466
    H                1.973305732471     1.493034755724    -0.974618857466
    C                1.151997732471    -0.358050244276    -1.060072857466
    O                2.191281732471    -0.970710244276    -0.901066857466
    --
    0 1
    C                0.843738732471    -0.154364244276     5.017529142534
    H                1.839557732471    -0.508349244276     4.903310142534
    C               -0.300865267529     0.232093755724     5.123721142534
    H               -1.305747267529     0.560793755724     5.224085142534

""")

GEOS['%s-%s-%s' % (dbse, '33-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.385148594215    -0.000213131185    -0.719999629348
    C                0.683117594215    -1.140219131185    -0.759383629348
    H                1.262486594215    -2.054942131185    -0.725579629348
    C               -0.705955405785    -1.193094131185    -0.844799629348
    H               -1.210321405785    -2.148542131185    -0.876772629348
    C               -1.417847405785     0.000152868815    -0.892280629348
    H               -2.496929405785     0.000277868815    -0.960633629348
    C               -0.705634405785     1.193194868815    -0.844941629348
    H               -1.209745405785     2.148767868815    -0.876991629348
    C                0.683418594215     1.139975868815    -0.759570629348
    H                1.263027594215     2.054530868815    -0.725961629348
    --
    0 1
    C                0.020035594215     0.666442868815     2.265136370652
    H                0.930509594215     1.225928868815     2.106021370652
    H               -0.889512405785     1.228846868815     2.422096370652
    C                0.020367594215    -0.666244131185     2.265268370652
    H                0.931103594215    -1.225327131185     2.106258370652
    H               -0.888920405785    -1.229069131185     2.422357370652

""")

GEOS['%s-%s-%s' % (dbse, '33-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.384770702635    -0.000214963750    -0.761982917024
    C                0.682739702635    -1.140220963750    -0.801366917024
    H                1.262108702635    -2.054943963750    -0.767562917024
    C               -0.706333297365    -1.193095963750    -0.886782917024
    H               -1.210699297365    -2.148543963750    -0.918755917024
    C               -1.418225297365     0.000151036250    -0.934263917024
    H               -2.497307297365     0.000276036250    -1.002616917024
    C               -0.706012297365     1.193193036250    -0.886924917024
    H               -1.210123297365     2.148766036250    -0.918974917024
    C                0.683040702635     1.139974036250    -0.801553917024
    H                1.262649702635     2.054529036250    -0.767944917024
    --
    0 1
    C                0.021100702635     0.666448036250     2.383520082976
    H                0.931574702635     1.225934036250     2.224405082976
    H               -0.888447297365     1.228852036250     2.540481082976
    C                0.021433702635    -0.666238963750     2.383652082976
    H                0.932168702635    -1.225321963750     2.224642082976
    H               -0.887854297365    -1.229063963750     2.540742082976

""")

GEOS['%s-%s-%s' % (dbse, '33-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.384393315819    -0.000216796315    -0.803926626602
    C                0.682362315819    -1.140222796315    -0.843310626602
    H                1.261731315819    -2.054945796315    -0.809506626602
    C               -0.706710684181    -1.193097796315    -0.928726626602
    H               -1.211076684181    -2.148545796315    -0.960699626602
    C               -1.418602684181     0.000149203685    -0.976207626602
    H               -2.497684684181     0.000274203685    -1.044560626602
    C               -0.706389684181     1.193191203685    -0.928868626602
    H               -1.210500684181     2.148764203685    -0.960918626602
    C                0.682663315819     1.139972203685    -0.843497626602
    H                1.262272315819     2.054527203685    -0.809888626602
    --
    0 1
    C                0.022165315819     0.666453203685     2.501792373398
    H                0.932639315819     1.225939203685     2.342677373398
    H               -0.887382684181     1.228857203685     2.658752373398
    C                0.022497315819    -0.666233796315     2.501924373398
    H                0.933233315819    -1.225316796315     2.342913373398
    H               -0.886790684181    -1.229058796315     2.659013373398

""")

GEOS['%s-%s-%s' % (dbse, '33-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.384015414826    -0.000218628879    -0.845910830561
    C                0.681984414826    -1.140224628879    -0.885294830561
    H                1.261353414826    -2.054947628879    -0.851490830561
    C               -0.707088585174    -1.193099628879    -0.970710830561
    H               -1.211454585174    -2.148547628879    -1.002683830561
    C               -1.418980585174     0.000147371121    -1.018191830561
    H               -2.498062585174     0.000272371121    -1.086544830561
    C               -0.706767585174     1.193189371121    -0.970852830561
    H               -1.210878585174     2.148762371121    -1.002902830561
    C                0.682285414826     1.139970371121    -0.885481830561
    H                1.261894414826     2.054525371121    -0.851872830561
    --
    0 1
    C                0.023230414826     0.666458371121     2.620178169439
    H                0.933704414826     1.225944371121     2.461064169439
    H               -0.886316585174     1.228862371121     2.777139169439
    C                0.023563414826    -0.666228628879     2.620311169439
    H                0.934298414826    -1.225311628879     2.461300169439
    H               -0.885724585174    -1.229053628879     2.777400169439

""")

GEOS['%s-%s-%s' % (dbse, '33-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.383637906525    -0.000220461444    -0.887858878580
    C                0.681606906525    -1.140226461444    -0.927242878580
    H                1.260975906525    -2.054949461444    -0.893438878580
    C               -0.707466093475    -1.193101461444    -1.012658878580
    H               -1.211832093475    -2.148549461444    -1.044631878580
    C               -1.419358093475     0.000145538556    -1.060139878580
    H               -2.498440093475     0.000270538556    -1.128492878580
    C               -0.707145093475     1.193187538556    -1.012800878580
    H               -1.211256093475     2.148760538556    -1.044850878580
    C                0.681907906525     1.139968538556    -0.927429878580
    H                1.261516906525     2.054523538556    -0.893820878580
    --
    0 1
    C                0.024294906525     0.666463538556     2.738463121420
    H                0.934768906525     1.225949538556     2.579348121420
    H               -0.885252093475     1.228867538556     2.895423121420
    C                0.024627906525    -0.666223461444     2.738595121420
    H                0.935362906525    -1.225306461444     2.579585121420
    H               -0.884660093475    -1.229048461444     2.895684121420

""")

GEOS['%s-%s-%s' % (dbse, '33-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.382505895800    -0.000225959137    -1.013654740297
    C                0.680474895800    -1.140231959137    -1.053038740297
    H                1.259843895800    -2.054954959137    -1.019234740297
    C               -0.708598104200    -1.193106959137    -1.138454740297
    H               -1.212964104200    -2.148554959137    -1.170427740297
    C               -1.420490104200     0.000140040863    -1.185935740297
    H               -2.499572104200     0.000265040863    -1.254288740297
    C               -0.708277104200     1.193182040863    -1.138596740297
    H               -1.212388104200     2.148755040863    -1.170646740297
    C                0.680775895800     1.139963040863    -1.053225740297
    H                1.260384895800     2.054518040863    -1.019616740297
    --
    0 1
    C                0.027486895800     0.666479040863     3.093180259703
    H                0.937960895800     1.225965040863     2.934065259703
    H               -0.882060104200     1.228883040863     3.250140259703
    C                0.027819895800    -0.666207959137     3.093312259703
    H                0.938555895800    -1.225290959137     2.934301259703
    H               -0.881468104200    -1.229032959137     3.250401259703

""")

GEOS['%s-%s-%s' % (dbse, '33-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.380619953302    -0.000235121960    -1.223201972515
    C                0.678588953302    -1.140241121960    -1.262585972515
    H                1.257957953302    -2.054964121960    -1.228781972515
    C               -0.710484046698    -1.193116121960    -1.348001972515
    H               -1.214850046698    -2.148564121960    -1.379974972515
    C               -1.422376046698     0.000130878040    -1.395482972515
    H               -2.501458046698     0.000255878040    -1.463835972515
    C               -0.710163046698     1.193172878040    -1.348143972515
    H               -1.214274046698     2.148745878040    -1.380193972515
    C                0.678889953302     1.139953878040    -1.262772972515
    H                1.258498953302     2.054508878040    -1.229163972515
    --
    0 1
    C                0.032804953302     0.666504878040     3.684058027485
    H                0.943278953302     1.225990878040     3.524943027485
    H               -0.876743046698     1.228908878040     3.841019027485
    C                0.033137953302    -0.666182121960     3.684190027485
    H                0.943872953302    -1.225265121960     3.525180027485
    H               -0.876151046698    -1.229007121960     3.841279027485

""")

GEOS['%s-%s-%s' % (dbse, '33-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.376850480060    -0.000253326121    -1.642065215546
    C                0.674819480060    -1.140259326121    -1.681449215546
    H                1.254188480060    -2.054982326121    -1.647645215546
    C               -0.714253519940    -1.193134326121    -1.766865215546
    H               -1.218619519940    -2.148582326121    -1.798838215546
    C               -1.426145519940     0.000112673879    -1.814346215546
    H               -2.505227519940     0.000237673879    -1.882699215546
    C               -0.713932519940     1.193154673879    -1.767007215546
    H               -1.218043519940     2.148727673879    -1.799057215546
    C                0.675120480060     1.139935673879    -1.681636215546
    H                1.254729480060     2.054490673879    -1.648027215546
    --
    0 1
    C                0.043434480060     0.666556673879     4.865161784454
    H                0.953908480060     1.226041673879     4.706046784454
    H               -0.866113519940     1.228960673879     5.022121784454
    C                0.043766480060    -0.666131326121     4.865293784454
    H                0.954502480060    -1.225213326121     4.706282784454
    H               -0.865521519940    -1.228955326121     5.022382784454

""")

GEOS['%s-%s-%s' % (dbse, '34-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533300732961    -0.297959782232    -1.760127182209
    H               -2.563618732961    -0.980162782232    -2.609699182209
    H               -2.566969732961    -0.898956782232    -0.851544182209
    H               -3.434417732961     0.312876217768    -1.789172182209
    C               -1.271876732961     0.554574217768    -1.798921182209
    H               -1.271017732961     1.253485217768    -0.958957182209
    H               -1.266624732961     1.164815217768    -2.705103182209
    C               -0.000126732961    -0.281498782232    -1.753673182209
    H               -0.000150732961    -0.890310782232    -0.844639182209
    H               -0.000356732961    -0.983794782232    -2.592680182209
    C                1.271903267039     0.554301217768    -1.799215182209
    H                1.270980267039     1.253552217768    -0.959571182209
    H                1.266644267039     1.164102217768    -2.705663182209
    C                2.533412267039    -0.298022782232    -1.759996182209
    H                2.563927267039    -0.980854782232    -2.609054182209
    H                3.434318267039     0.313173217768    -1.789677182209
    H                2.567566267039    -0.898289782232    -0.850947182209
    --
    0 1
    C                2.533566267039     0.297641217768     1.760001817791
    H                2.568151267039     0.897448217768     0.850675817791
    H                2.564070267039     0.980847217768     2.608746817791
    H                3.434247267039    -0.313855782232     1.790210817791
    C                1.271740267039    -0.554245782232     1.799305817791
    H                1.266296267039    -1.163973782232     2.705802817791
    H                1.270609267039    -1.253599782232     0.959724817791
    C               -0.000034732961     0.281853217768     1.753689817791
    H               -0.000189732961     0.984163217768     2.592673817791
    H                0.000042267039     0.890629217768     0.844619817791
    C               -1.271795732961    -0.554288782232     1.798960817791
    H               -1.266413732961    -1.164398782232     2.705210817791
    H               -1.270688732961    -1.253311782232     0.959096817791
    C               -2.533514732961     0.297757217768     1.759982817791
    H               -2.567707732961     0.898291217768     0.851106817791
    H               -3.434316732961    -0.313540782232     1.789496817791
    H               -2.564054732961     0.980343217768     2.609244817791

""")

GEOS['%s-%s-%s' % (dbse, '34-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533300600808    -0.296418685028    -1.835773272423
    H               -2.563618600808    -0.978621685028    -2.685345272423
    H               -2.566969600808    -0.897415685028    -0.927190272423
    H               -3.434417600808     0.314417314972    -1.864818272423
    C               -1.271876600808     0.556115314972    -1.874567272423
    H               -1.271017600808     1.255026314972    -1.034603272423
    H               -1.266624600808     1.166356314972    -2.780749272423
    C               -0.000126600808    -0.279957685028    -1.829319272423
    H               -0.000150600808    -0.888769685028    -0.920285272423
    H               -0.000356600808    -0.982253685028    -2.668326272423
    C                1.271903399192     0.555842314972    -1.874861272423
    H                1.270980399192     1.255093314972    -1.035217272423
    H                1.266644399192     1.165643314972    -2.781309272423
    C                2.533412399192    -0.296481685028    -1.835642272423
    H                2.563927399192    -0.979313685028    -2.684700272423
    H                3.434318399192     0.314714314972    -1.865323272423
    H                2.567566399192    -0.896748685028    -0.926593272423
    --
    0 1
    C                2.533566399192     0.296100314972     1.835647727577
    H                2.568150399192     0.895906314972     0.926321727577
    H                2.564069399192     0.979306314972     2.684392727577
    H                3.434246399192    -0.315397685028     1.865856727577
    C                1.271740399192    -0.555787685028     1.874951727577
    H                1.266296399192    -1.165514685028     2.781448727577
    H                1.270609399192    -1.255140685028     1.035370727577
    C               -0.000034600808     0.280312314972     1.829335727577
    H               -0.000189600808     0.982622314972     2.668319727577
    H                0.000041399192     0.889088314972     0.920265727577
    C               -1.271795600808    -0.555829685028     1.874606727577
    H               -1.266413600808    -1.165939685028     2.780856727577
    H               -1.270689600808    -1.254852685028     1.034743727577
    C               -2.533515600808     0.296216314972     1.835629727577
    H               -2.567708600808     0.896750314972     0.926752727577
    H               -3.434317600808    -0.315081685028     1.865142727577
    H               -2.564054600808     0.978802314972     2.684890727577

""")

GEOS['%s-%s-%s' % (dbse, '34-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533300239950    -0.294914685028    -1.909600966831
    H               -2.563618239950    -0.977117685028    -2.759172966831
    H               -2.566969239950    -0.895911685028    -1.001017966831
    H               -3.434417239950     0.315921314972    -1.938645966831
    C               -1.271876239950     0.557619314972    -1.948394966831
    H               -1.271017239950     1.256530314972    -1.108430966831
    H               -1.266624239950     1.167860314972    -2.854576966831
    C               -0.000126239950    -0.278453685028    -1.903146966831
    H               -0.000150239950    -0.887265685028    -0.994112966831
    H               -0.000356239950    -0.980749685028    -2.742153966831
    C                1.271903760050     0.557346314972    -1.948688966831
    H                1.270980760050     1.256597314972    -1.109044966831
    H                1.266644760050     1.167147314972    -2.855136966831
    C                2.533412760050    -0.294977685028    -1.909469966831
    H                2.563927760050    -0.977809685028    -2.758527966831
    H                3.434318760050     0.316218314972    -1.939150966831
    H                2.567566760050    -0.895244685028    -1.000420966831
    --
    0 1
    C                2.533565760050     0.294596314972     1.909475033169
    H                2.568150760050     0.894402314972     1.000150033169
    H                2.564069760050     0.977802314972     2.758220033169
    H                3.434246760050    -0.316901685028     1.939684033169
    C                1.271739760050    -0.557291685028     1.948780033169
    H                1.266295760050    -1.167018685028     2.855277033169
    H                1.270609760050    -1.256644685028     1.109198033169
    C               -0.000035239950     0.278808314972     1.903163033169
    H               -0.000190239950     0.981118314972     2.742148033169
    H                0.000041760050     0.887584314972     0.994093033169
    C               -1.271796239950    -0.557333685028     1.948435033169
    H               -1.266414239950    -1.167443685028     2.854685033169
    H               -1.270689239950    -1.256356685028     1.108571033169
    C               -2.533515239950     0.294712314972     1.909457033169
    H               -2.567708239950     0.895246314972     1.000580033169
    H               -3.434317239950    -0.316585685028     1.938970033169
    H               -2.564055239950     0.977298314972     2.758718033169

""")

GEOS['%s-%s-%s' % (dbse, '34-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533300191023    -0.293437393416    -1.982108078015
    H               -2.563618191023    -0.975640393416    -2.831680078015
    H               -2.566969191023    -0.894434393416    -1.073525078015
    H               -3.434417191023     0.317398606584    -2.011153078015
    C               -1.271876191023     0.559096606584    -2.020902078015
    H               -1.271017191023     1.258007606584    -1.180938078015
    H               -1.266624191023     1.169337606584    -2.927084078015
    C               -0.000126191023    -0.276976393416    -1.975654078015
    H               -0.000150191023    -0.885788393416    -1.066620078015
    H               -0.000356191023    -0.979272393416    -2.814661078015
    C                1.271903808977     0.558823606584    -2.021196078015
    H                1.270980808977     1.258074606584    -1.181552078015
    H                1.266644808977     1.168624606584    -2.927644078015
    C                2.533412808977    -0.293500393416    -1.981977078015
    H                2.563927808977    -0.976332393416    -2.831035078015
    H                3.434318808977     0.317695606584    -2.011658078015
    H                2.567566808977    -0.893767393416    -1.072928078015
    --
    0 1
    C                2.533565808977     0.293118606584     1.981982921985
    H                2.568149808977     0.892925606584     1.072656921985
    H                2.564068808977     0.976325606584     2.830726921985
    H                3.434245808977    -0.318378393416     2.012191921985
    C                1.271739808977    -0.558768393416     2.021286921985
    H                1.266295808977    -1.168496393416     2.927783921985
    H                1.270608808977    -1.258122393416     1.181705921985
    C               -0.000035191023     0.277330606584     1.975669921985
    H               -0.000190191023     0.979640606584     2.814654921985
    H                0.000040808977     0.886107606584     1.066600921985
    C               -1.271796191023    -0.558811393416     2.020941921985
    H               -1.266414191023    -1.168920393416     2.927191921985
    H               -1.270690191023    -1.257834393416     1.181077921985
    C               -2.533515191023     0.293235606584     1.981963921985
    H               -2.567709191023     0.893768606584     1.073086921985
    H               -3.434317191023    -0.318063393416     2.011476921985
    H               -2.564055191023     0.975821606584     2.831225921985

""")

GEOS['%s-%s-%s' % (dbse, '34-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533299753930    -0.291984997609    -2.053402286403
    H               -2.563617753930    -0.974187997609    -2.902974286403
    H               -2.566968753930    -0.892981997609    -1.144819286403
    H               -3.434416753930     0.318851002391    -2.082447286403
    C               -1.271875753930     0.560549002391    -2.092196286403
    H               -1.271016753930     1.259460002391    -1.252232286403
    H               -1.266623753930     1.170790002391    -2.998378286403
    C               -0.000125753930    -0.275523997609    -2.046948286403
    H               -0.000149753930    -0.884335997609    -1.137914286403
    H               -0.000355753930    -0.977819997609    -2.885955286403
    C                1.271904246070     0.560276002391    -2.092490286403
    H                1.270981246070     1.259527002391    -1.252846286403
    H                1.266645246070     1.170077002391    -2.998938286403
    C                2.533413246070    -0.292047997609    -2.053271286403
    H                2.563928246070    -0.974879997609    -2.902329286403
    H                3.434319246070     0.319148002391    -2.082952286403
    H                2.567567246070    -0.892314997609    -1.144222286403
    --
    0 1
    C                2.533565246070     0.291667002391     2.053276713597
    H                2.568150246070     0.891473002391     1.143951713597
    H                2.564069246070     0.974873002391     2.902021713597
    H                3.434246246070    -0.319830997609     2.083485713597
    C                1.271739246070    -0.560220997609     2.092580713597
    H                1.266295246070    -1.169947997609     2.999077713597
    H                1.270609246070    -1.259573997609     1.252999713597
    C               -0.000035753930     0.275878002391     2.046964713597
    H               -0.000190753930     0.978188002391     2.885948713597
    H                0.000041246070     0.884655002391     1.137894713597
    C               -1.271796753930    -0.560263997609     2.092235713597
    H               -1.266413753930    -1.170372997609     2.998486713597
    H               -1.270689753930    -1.259285997609     1.252372713597
    C               -2.533515753930     0.291783002391     2.053258713597
    H               -2.567708753930     0.892316002391     1.144381713597
    H               -3.434317753930    -0.319515997609     2.082771713597
    H               -2.564054753930     0.974369002391     2.902519713597

""")

GEOS['%s-%s-%s' % (dbse, '34-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533299205002    -0.287734289222    -2.262062272423
    H               -2.563617205002    -0.969937289222    -3.111634272423
    H               -2.566968205002    -0.888731289222    -1.353479272423
    H               -3.434416205002     0.323101710778    -2.291107272423
    C               -1.271875205002     0.564799710778    -2.300856272423
    H               -1.271016205002     1.263710710778    -1.460892272423
    H               -1.266623205002     1.175040710778    -3.207038272423
    C               -0.000125205002    -0.271273289222    -2.255608272423
    H               -0.000149205002    -0.880085289222    -1.346574272423
    H               -0.000355205002    -0.973569289222    -3.094615272423
    C                1.271904794998     0.564526710778    -2.301150272423
    H                1.270981794998     1.263777710778    -1.461506272423
    H                1.266645794998     1.174327710778    -3.207598272423
    C                2.533413794998    -0.287797289222    -2.261931272423
    H                2.563928794998    -0.970629289222    -3.110989272423
    H                3.434319794998     0.323398710778    -2.291612272423
    H                2.567567794998    -0.888064289222    -1.352882272423
    --
    0 1
    C                2.533564794998     0.287415710778     2.261936727577
    H                2.568148794998     0.887222710778     1.352610727577
    H                2.564067794998     0.970622710778     3.110681727577
    H                3.434245794998    -0.324081289222     2.292145727577
    C                1.271738794998    -0.564471289222     2.301240727577
    H                1.266294794998    -1.174199289222     3.207737727577
    H                1.270607794998    -1.263825289222     1.461659727577
    C               -0.000036205002     0.271627710778     2.255624727577
    H               -0.000191205002     0.973937710778     3.094608727577
    H                0.000040794998     0.880403710778     1.346554727577
    C               -1.271797205002    -0.564514289222     2.300895727577
    H               -1.266415205002    -1.174624289222     3.207145727577
    H               -1.270691205002    -1.263537289222     1.461032727577
    C               -2.533516205002     0.287531710778     2.261918727577
    H               -2.567710205002     0.888065710778     1.353041727577
    H               -3.434318205002    -0.323766289222     2.291431727577
    H               -2.564056205002     0.970117710778     3.111179727577

""")

GEOS['%s-%s-%s' % (dbse, '34-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533298191023    -0.280878289222    -2.598602182209
    H               -2.563616191023    -0.963081289222    -3.448174182209
    H               -2.566967191023    -0.881875289222    -1.690019182209
    H               -3.434415191023     0.329957710778    -2.627647182209
    C               -1.271874191023     0.571655710778    -2.637396182209
    H               -1.271015191023     1.270566710778    -1.797432182209
    H               -1.266622191023     1.181896710778    -3.543578182209
    C               -0.000124191023    -0.264417289222    -2.592148182209
    H               -0.000148191023    -0.873229289222    -1.683114182209
    H               -0.000354191023    -0.966713289222    -3.431155182209
    C                1.271905808977     0.571382710778    -2.637690182209
    H                1.270982808977     1.270633710778    -1.798046182209
    H                1.266646808977     1.181183710778    -3.544138182209
    C                2.533414808977    -0.280941289222    -2.598471182209
    H                2.563929808977    -0.963773289222    -3.447529182209
    H                3.434320808977     0.330254710778    -2.628152182209
    H                2.567568808977    -0.881208289222    -1.689422182209
    --
    0 1
    C                2.533563808977     0.280559710778     2.598476817791
    H                2.568147808977     0.880366710778     1.689150817791
    H                2.564066808977     0.963766710778     3.447221817791
    H                3.434243808977    -0.330937289222     2.628685817791
    C                1.271737808977    -0.571327289222     2.637780817791
    H                1.266293808977    -1.181055289222     3.544277817791
    H                1.270606808977    -1.270681289222     1.798199817791
    C               -0.000037191023     0.264771710778     2.592164817791
    H               -0.000192191023     0.967081710778     3.431148817791
    H                0.000038808977     0.873547710778     1.683094817791
    C               -1.271798191023    -0.571370289222     2.637435817791
    H               -1.266416191023    -1.181480289222     3.543685817791
    H               -1.270692191023    -1.270393289222     1.797571817791
    C               -2.533517191023     0.280675710778     2.598457817791
    H               -2.567711191023     0.881209710778     1.689581817791
    H               -3.434319191023    -0.330622289222     2.627971817791
    H               -2.564057191023     0.963261710778     3.447719817791

""")

GEOS['%s-%s-%s' % (dbse, '34-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.533295844144    -0.267594282232    -3.250674286403
    H               -2.563613844144    -0.949797282232    -4.100246286403
    H               -2.566964844144    -0.868591282232    -2.342091286403
    H               -3.434412844144     0.343241717768    -3.279719286403
    C               -1.271871844144     0.584939717768    -3.289468286403
    H               -1.271012844144     1.283850717768    -2.449504286403
    H               -1.266619844144     1.195180717768    -4.195650286403
    C               -0.000121844144    -0.251133282232    -3.244220286403
    H               -0.000145844144    -0.859945282232    -2.335186286403
    H               -0.000351844144    -0.953429282232    -4.083227286403
    C                1.271908155856     0.584666717768    -3.289762286403
    H                1.270985155856     1.283917717768    -2.450118286403
    H                1.266649155856     1.194467717768    -4.196210286403
    C                2.533417155856    -0.267657282232    -3.250543286403
    H                2.563932155856    -0.950489282232    -4.099601286403
    H                3.434323155856     0.343538717768    -3.280224286403
    H                2.567571155856    -0.867924282232    -2.341494286403
    --
    0 1
    C                2.533562155856     0.267275717768     3.250548713597
    H                2.568146155856     0.867082717768     2.341223713597
    H                2.564065155856     0.950481717768     4.099293713597
    H                3.434242155856    -0.344221282232     3.280757713597
    C                1.271735155856    -0.584611282232     3.289852713597
    H                1.266292155856    -1.194339282232     4.196349713597
    H                1.270605155856    -1.283965282232     2.450271713597
    C               -0.000039844144     0.251487717768     3.244236713597
    H               -0.000194844144     0.953797717768     4.083220713597
    H                0.000037155856     0.860263717768     2.335166713597
    C               -1.271800844144    -0.584654282232     3.289507713597
    H               -1.266417844144    -1.194764282232     4.195758713597
    H               -1.270693844144    -1.283677282232     2.449644713597
    C               -2.533519844144     0.267391717768     3.250530713597
    H               -2.567712844144     0.867925717768     2.341653713597
    H               -3.434321844144    -0.343906282232     3.280043713597
    H               -2.564058844144     0.949977717768     4.099791713597

""")

GEOS['%s-%s-%s' % (dbse, '35-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.530066326613    -0.412775697678    -2.135766645884
    H               -2.559569326613    -0.977990697678    -3.067228645884
    H               -2.553719326613    -1.129065697678    -1.314414645884
    H               -3.435897326613     0.188943302322    -2.080301645884
    C               -1.275840326613     0.448434302322    -2.067047645884
    H               -1.277767326613     1.030017302322    -1.141987645884
    H               -1.280022326613     1.173355302322    -2.884230645884
    C                0.002521673387    -0.375916697678    -2.138079645884
    H                0.008145673387    -1.106613697678    -1.323241645884
    H                0.006565673387    -0.955723697678    -3.065892645884
    C                1.268649673387     0.467195302322    -2.067703645884
    H                1.262336673387     1.049049302322    -1.142826645884
    H                1.261951673387     1.191856302322    -2.885107645884
    C                2.535282673387    -0.375625697678    -2.136386645884
    H                2.572756673387    -0.940917697678    -3.067524645884
    H                3.432297673387     0.239214302322    -2.081494645884
    H                2.569524673387    -1.091010697678    -1.314616645884
    --
    0 1
    C               -0.000177326613     0.059850302322     2.110375354116
    C                0.000894673387    -0.080281697678     3.630108354116
    H               -0.886141326613     0.383795302322     4.063478354116
    H                0.010145673387    -1.131061697678     3.923118354116
    H                0.879554673387     0.399383302322     4.063751354116
    C               -1.239632326613    -0.621801697678     1.536477354116
    H               -1.262931326613    -0.532844697678     0.449648354116
    H               -1.251718326613    -1.683078697678     1.789490354116
    H               -2.150576326613    -0.169510697678     1.931562354116
    C                1.252427673387    -0.597690697678     1.536905354116
    H                1.273754673387    -0.509405697678     0.449934354116
    H                1.285558673387    -1.658251697678     1.790997354116
    H                2.154239673387    -0.127047697678     1.931326354116
    C               -0.014425326613     1.539642302322     1.735754354116
    H                0.863340673387     2.050229302322     2.134709354116
    H               -0.014949326613     1.666097302322     0.652102354116
    H               -0.902531326613     2.032976302322     2.133542354116

""")

GEOS['%s-%s-%s' % (dbse, '35-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.530058819624    -0.415239801871    -2.222464131905
    H               -2.559561819624    -0.980454801871    -3.153926131905
    H               -2.553711819624    -1.131529801871    -1.401112131905
    H               -3.435889819624     0.186479198129    -2.166999131905
    C               -1.275832819624     0.445970198129    -2.153745131905
    H               -1.277759819624     1.027553198129    -1.228685131905
    H               -1.280014819624     1.170891198129    -2.970928131905
    C                0.002529180376    -0.378380801871    -2.224777131905
    H                0.008153180376    -1.109077801871    -1.409939131905
    H                0.006573180376    -0.958187801871    -3.152590131905
    C                1.268657180376     0.464731198129    -2.154401131905
    H                1.262344180376     1.046585198129    -1.229524131905
    H                1.261959180376     1.189392198129    -2.971805131905
    C                2.535290180376    -0.378089801871    -2.223084131905
    H                2.572764180376    -0.943381801871    -3.154222131905
    H                3.432305180376     0.236750198129    -2.168192131905
    H                2.569532180376    -1.093474801871    -1.401314131905
    --
    0 1
    C               -0.000184819624     0.062314198129     2.197072868095
    C                0.000887180376    -0.077816801871     3.716805868095
    H               -0.886148819624     0.386259198129     4.150174868095
    H                0.010138180376    -1.128596801871     4.009815868095
    H                0.879547180376     0.401847198129     4.150448868095
    C               -1.239639819624    -0.619337801871     1.623174868095
    H               -1.262939819624    -0.530380801871     0.536344868095
    H               -1.251725819624    -1.680613801871     1.876187868095
    H               -2.150583819624    -0.167046801871     2.018259868095
    C                1.252420180376    -0.595226801871     1.623602868095
    H                1.273747180376    -0.506940801871     0.536631868095
    H                1.285551180376    -1.655787801871     1.877694868095
    H                2.154232180376    -0.124583801871     2.018023868095
    C               -0.014432819624     1.542106198129     1.822451868095
    H                0.863333180376     2.052693198129     2.221406868095
    H               -0.014956819624     1.668561198129     0.738799868095
    H               -0.902538819624     2.035440198129     2.220239868095

""")

GEOS['%s-%s-%s' % (dbse, '35-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.530051639195    -0.417591697678    -2.305208645884
    H               -2.559554639195    -0.982806697678    -3.236670645884
    H               -2.553704639195    -1.133881697678    -1.483856645884
    H               -3.435882639195     0.184127302322    -2.249743645884
    C               -1.275825639195     0.443618302322    -2.236489645884
    H               -1.277752639195     1.025201302322    -1.311429645884
    H               -1.280007639195     1.168539302322    -3.053672645884
    C                0.002536360805    -0.380732697678    -2.307521645884
    H                0.008160360805    -1.111429697678    -1.492683645884
    H                0.006580360805    -0.960539697678    -3.235334645884
    C                1.268664360805     0.462379302322    -2.237145645884
    H                1.262351360805     1.044233302322    -1.312268645884
    H                1.261966360805     1.187040302322    -3.054549645884
    C                2.535297360805    -0.380441697678    -2.305828645884
    H                2.572771360805    -0.945733697678    -3.236966645884
    H                3.432312360805     0.234398302322    -2.250936645884
    H                2.569539360805    -1.095826697678    -1.484058645884
    --
    0 1
    C               -0.000191639195     0.064666302322     2.279817354116
    C                0.000879360805    -0.075465697678     3.799550354116
    H               -0.886156639195     0.388611302322     4.232920354116
    H                0.010131360805    -1.126245697678     4.092560354116
    H                0.879540360805     0.404199302322     4.233193354116
    C               -1.239647639195    -0.616985697678     1.705919354116
    H               -1.262946639195    -0.528028697678     0.619090354116
    H               -1.251732639195    -1.678262697678     1.958932354116
    H               -2.150590639195    -0.164694697678     2.101004354116
    C                1.252413360805    -0.592874697678     1.706347354116
    H                1.273740360805    -0.504589697678     0.619376354116
    H                1.285543360805    -1.653435697678     1.960439354116
    H                2.154225360805    -0.122231697678     2.100768354116
    C               -0.014439639195     1.544458302322     1.905196354116
    H                0.863326360805     2.055045302322     2.304151354116
    H               -0.014963639195     1.670913302322     0.821544354116
    H               -0.902545639195     2.037792302322     2.302984354116

""")

GEOS['%s-%s-%s' % (dbse, '35-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.530044639195    -0.419862301871    -2.385098638895
    H               -2.559547639195    -0.985077301871    -3.316560638895
    H               -2.553697639195    -1.136152301871    -1.563746638895
    H               -3.435875639195     0.181856698129    -2.329633638895
    C               -1.275818639195     0.441347698129    -2.316379638895
    H               -1.277745639195     1.022930698129    -1.391319638895
    H               -1.280000639195     1.166268698129    -3.133562638895
    C                0.002543360805    -0.383003301871    -2.387411638895
    H                0.008167360805    -1.113700301871    -1.572573638895
    H                0.006587360805    -0.962810301871    -3.315224638895
    C                1.268671360805     0.460108698129    -2.317035638895
    H                1.262358360805     1.041962698129    -1.392158638895
    H                1.261973360805     1.184769698129    -3.134439638895
    C                2.535304360805    -0.382712301871    -2.385718638895
    H                2.572778360805    -0.948004301871    -3.316856638895
    H                3.432319360805     0.232127698129    -2.330826638895
    H                2.569546360805    -1.098097301871    -1.563948638895
    --
    0 1
    C               -0.000198639195     0.066936698129     2.359707361105
    C                0.000872360805    -0.073194301871     3.879440361105
    H               -0.886163639195     0.390881698129     4.312810361105
    H                0.010124360805    -1.123974301871     4.172450361105
    H                0.879533360805     0.406469698129     4.313083361105
    C               -1.239654639195    -0.614715301871     1.785809361105
    H               -1.262953639195    -0.525758301871     0.698979361105
    H               -1.251739639195    -1.675991301871     2.038822361105
    H               -2.150597639195    -0.162424301871     2.180894361105
    C                1.252406360805    -0.590604301871     1.786237361105
    H                1.273733360805    -0.502318301871     0.699266361105
    H                1.285536360805    -1.651165301871     2.040329361105
    H                2.154218360805    -0.119961301871     2.180658361105
    C               -0.014446639195     1.546728698129     1.985086361105
    H                0.863319360805     2.057315698129     2.384041361105
    H               -0.014970639195     1.673183698129     0.901434361105
    H               -0.902552639195     2.040062698129     2.382874361105

""")

GEOS['%s-%s-%s' % (dbse, '35-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.530037833603    -0.422065913055    -2.462627145884
    H               -2.559540833603    -0.987280913055    -3.394089145884
    H               -2.553690833603    -1.138355913055    -1.641275145884
    H               -3.435868833603     0.179653086945    -2.407162145884
    C               -1.275811833603     0.439144086945    -2.393908145884
    H               -1.277738833603     1.020727086945    -1.468848145884
    H               -1.279993833603     1.164065086945    -3.211091145884
    C                0.002550166397    -0.385206913055    -2.464940145884
    H                0.008174166397    -1.115903913055    -1.650102145884
    H                0.006594166397    -0.965013913055    -3.392753145884
    C                1.268678166397     0.457905086945    -2.394564145884
    H                1.262365166397     1.039759086945    -1.469687145884
    H                1.261980166397     1.182566086945    -3.211968145884
    C                2.535311166397    -0.384915913055    -2.463247145884
    H                2.572785166397    -0.950207913055    -3.394385145884
    H                3.432326166397     0.229924086945    -2.408355145884
    H                2.569553166397    -1.100300913055    -1.641477145884
    --
    0 1
    C               -0.000205833603     0.069140086945     2.437235854116
    C                0.000866166397    -0.070990913055     3.956968854116
    H               -0.886169833603     0.393085086945     4.390338854116
    H                0.010117166397    -1.121770913055     4.249978854116
    H                0.879526166397     0.408674086945     4.390611854116
    C               -1.239660833603    -0.612511913055     1.863337854116
    H               -1.262959833603    -0.523554913055     0.776508854116
    H               -1.251746833603    -1.673787913055     2.116350854116
    H               -2.150604833603    -0.160220913055     2.258422854116
    C                1.252399166397    -0.588400913055     1.863765854116
    H                1.273726166397    -0.500114913055     0.776794854116
    H                1.285530166397    -1.648961913055     2.117857854116
    H                2.154212166397    -0.117756913055     2.258186854116
    C               -0.014453833603     1.548933086945     2.062614854116
    H                0.863312166397     2.059520086945     2.461569854116
    H               -0.014977833603     1.675388086945     0.978962854116
    H               -0.902559833603     2.042266086945     2.460402854116

""")

GEOS['%s-%s-%s' % (dbse, '35-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.530018611236    -0.428396413055    -2.685353458466
    H               -2.559521611236    -0.993611413055    -3.616815458466
    H               -2.553671611236    -1.144686413055    -1.864001458466
    H               -3.435849611236     0.173322586945    -2.629888458466
    C               -1.275792611236     0.432813586945    -2.616634458466
    H               -1.277719611236     1.014396586945    -1.691574458466
    H               -1.279974611236     1.157734586945    -3.433817458466
    C                0.002569388764    -0.391537413055    -2.687666458466
    H                0.008193388764    -1.122234413055    -1.872828458466
    H                0.006613388764    -0.971344413055    -3.615479458466
    C                1.268697388764     0.451574586945    -2.617290458466
    H                1.262384388764     1.033428586945    -1.692413458466
    H                1.261999388764     1.176235586945    -3.434694458466
    C                2.535330388764    -0.391246413055    -2.685973458466
    H                2.572804388764    -0.956538413055    -3.617111458466
    H                3.432345388764     0.223593586945    -2.631081458466
    H                2.569572388764    -1.106631413055    -1.864203458466
    --
    0 1
    C               -0.000224611236     0.075470586945     2.659962541534
    C                0.000846388764    -0.064660413055     4.179695541534
    H               -0.886189611236     0.399415586945     4.613064541534
    H                0.010098388764    -1.115440413055     4.472705541534
    H                0.879506388764     0.415004586945     4.613338541534
    C               -1.239680611236    -0.606181413055     2.086063541534
    H               -1.262979611236    -0.517224413055     0.999234541534
    H               -1.251765611236    -1.667457413055     2.339077541534
    H               -2.150624611236    -0.153890413055     2.481148541534
    C                1.252380388764    -0.582070413055     2.086491541534
    H                1.273706388764    -0.493784413055     0.999521541534
    H                1.285510388764    -1.642631413055     2.340584541534
    H                2.154192388764    -0.111426413055     2.480913541534
    C               -0.014472611236     1.555263586945     2.285341541534
    H                0.863293388764     2.065850586945     2.684296541534
    H               -0.014997611236     1.681718586945     1.201689541534
    H               -0.902578611236     2.048596586945     2.683129541534

""")

GEOS['%s-%s-%s' % (dbse, '35-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.529988021022    -0.438379017249    -3.036573048680
    H               -2.559491021022    -1.003594017249    -3.968035048680
    H               -2.553641021022    -1.154669017249    -2.215221048680
    H               -3.435819021022     0.163339982751    -2.981108048680
    C               -1.275762021022     0.422830982751    -2.967854048680
    H               -1.277689021022     1.004413982751    -2.042794048680
    H               -1.279944021022     1.147751982751    -3.785037048680
    C                0.002599978978    -0.401520017249    -3.038886048680
    H                0.008223978978    -1.132217017249    -2.224048048680
    H                0.006643978978    -0.981327017249    -3.966699048680
    C                1.268727978978     0.441591982751    -2.968510048680
    H                1.262414978978     1.023445982751    -2.043633048680
    H                1.262029978978     1.166252982751    -3.785914048680
    C                2.535360978978    -0.401229017249    -3.037193048680
    H                2.572834978978    -0.966521017249    -3.968331048680
    H                3.432375978978     0.213610982751    -2.982301048680
    H                2.569602978978    -1.116614017249    -2.215423048680
    --
    0 1
    C               -0.000256021022     0.085452982751     3.011181951320
    C                0.000815978978    -0.054678017249     4.530914951320
    H               -0.886220021022     0.409397982751     4.964283951320
    H                0.010067978978    -1.105458017249     4.823924951320
    H                0.879475978978     0.424986982751     4.964557951320
    C               -1.239711021022    -0.596199017249     2.437282951320
    H               -1.263010021022    -0.507242017249     1.350453951320
    H               -1.251797021022    -1.657475017249     2.690296951320
    H               -2.150655021022    -0.143907017249     2.832368951320
    C                1.252349978978    -0.572087017249     2.437711951320
    H                1.273675978978    -0.483802017249     1.350740951320
    H                1.285479978978    -1.632648017249     2.691803951320
    H                2.154161978978    -0.101444017249     2.832132951320
    C               -0.014503021022     1.565245982751     2.636560951320
    H                0.863262978978     2.075832982751     3.035515951320
    H               -0.015028021022     1.691700982751     1.552908951320
    H               -0.902609021022     2.058579982751     3.034348951320

""")

GEOS['%s-%s-%s' % (dbse, '35-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -2.529930021022    -0.457345017249    -3.703855333303
    H               -2.559433021022    -1.022560017249    -4.635317333303
    H               -2.553583021022    -1.173635017249    -2.882503333303
    H               -3.435761021022     0.144373982751    -3.648390333303
    C               -1.275704021022     0.403864982751    -3.635136333303
    H               -1.277631021022     0.985447982751    -2.710076333303
    H               -1.279886021022     1.128785982751    -4.452319333303
    C                0.002657978978    -0.420486017249    -3.706168333303
    H                0.008281978978    -1.151183017249    -2.891330333303
    H                0.006701978978    -1.000293017249    -4.633981333303
    C                1.268785978978     0.422625982751    -3.635792333303
    H                1.262472978978     1.004479982751    -2.710915333303
    H                1.262087978978     1.147286982751    -4.453196333303
    C                2.535418978978    -0.420195017249    -3.704475333303
    H                2.572892978978    -0.985487017249    -4.635613333303
    H                3.432433978978     0.194644982751    -3.649583333303
    H                2.569660978978    -1.135580017249    -2.882705333303
    --
    0 1
    C               -0.000314021022     0.104418982751     3.678464666697
    C                0.000757978978    -0.035712017249     5.198197666697
    H               -0.886278021022     0.428363982751     5.631566666697
    H                0.010009978978    -1.086492017249     5.491206666697
    H                0.879417978978     0.443952982751     5.631840666697
    C               -1.239769021022    -0.577233017249     3.104565666697
    H               -1.263068021022    -0.488276017249     2.017736666697
    H               -1.251855021022    -1.638509017249     3.357578666697
    H               -2.150713021022    -0.124941017249     3.499650666697
    C                1.252291978978    -0.553121017249     3.104993666697
    H                1.273617978978    -0.464836017249     2.018022666697
    H                1.285421978978    -1.613682017249     3.359086666697
    H                2.154103978978    -0.082478017249     3.499414666697
    C               -0.014561021022     1.584211982751     3.303842666697
    H                0.863204978978     2.094798982751     3.702797666697
    H               -0.015086021022     1.710666982751     2.220191666697
    H               -0.902667021022     2.077545982751     3.701630666697

""")

GEOS['%s-%s-%s' % (dbse, '36-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.359019651819    -0.066292119302    -2.464189184640
    C               -1.064141348181     0.401125880698    -2.170149184640
    H               -1.795077348181    -0.277185119302    -2.612755184640
    H               -1.248220348181     0.440049880698    -1.095566184640
    H               -1.237568348181     1.398038880698    -2.577994184640
    C                0.567344651819    -1.462503119302    -1.883107184640
    H                1.579411651819    -1.818635119302    -2.080982184640
    H                0.415466651819    -1.462431119302    -0.802798184640
    H               -0.133421348181    -2.174369119302    -2.321771184640
    C                1.354766651819     0.902749880698    -1.831936184640
    H                2.380891651819     0.589755880698    -2.030351184640
    H                1.220278651819     0.950288880698    -0.750375184640
    H                1.224870651819     1.909337880698    -2.232134184640
    C                0.578458651819    -0.106719119302    -3.974182184640
    H                0.435709651819     0.881353880698    -4.413983184640
    H                1.590191651819    -0.439144119302    -4.210757184640
    H               -0.123033348181    -0.793235119302    -4.450306184640
    --
    0 1
    C               -0.359021348181     0.066330880698     2.464220815360
    C                1.063790651819    -0.402635119302     2.170820815360
    H                1.248229651819    -0.441965119302     1.096300815360
    H                1.795224651819     0.275036880698     2.613556815360
    H                1.236054651819    -1.399641119302     2.578941815360
    C               -0.565418348181     1.462891880698     1.883389815360
    H               -1.577381348181     1.819877880698     2.080237815360
    H                0.135500651819     2.173963880698     2.323111815360
    H               -0.412309348181     1.463088880698     0.803247815360
    C               -0.579316348181     0.106506880698     3.974073815360
    H               -1.590532348181     0.440783880698     4.210146815360
    H                0.123196651819     0.791455880698     4.450911815360
    H               -0.438753348181    -0.882014119302     4.413525815360
    C               -1.355484348181    -0.901447119302     1.831073815360
    H               -1.226558348181    -1.908422119302     2.230652815360
    H               -2.381373348181    -0.587674119302     2.029461815360
    H               -1.220780348181    -0.948421119302     0.749497815360

""")

GEOS['%s-%s-%s' % (dbse, '36-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.370779540636    -0.068464223495    -2.544906545498
    C               -1.052381459364     0.398953776505    -2.250866545498
    H               -1.783317459364    -0.279357223495    -2.693472545498
    H               -1.236460459364     0.437877776505    -1.176283545498
    H               -1.225808459364     1.395866776505    -2.658711545498
    C                0.579104540636    -1.464675223495    -1.963824545498
    H                1.591171540636    -1.820807223495    -2.161699545498
    H                0.427226540636    -1.464603223495    -0.883515545498
    H               -0.121661459364    -2.176541223495    -2.402488545498
    C                1.366526540636     0.900577776505    -1.912653545498
    H                2.392651540636     0.587583776505    -2.111068545498
    H                1.232038540636     0.948116776505    -0.831092545498
    H                1.236630540636     1.907165776505    -2.312851545498
    C                0.590218540636    -0.108891223495    -4.054899545498
    H                0.447469540636     0.879181776505    -4.494700545498
    H                1.601951540636    -0.441316223495    -4.291474545498
    H               -0.111273459364    -0.795407223495    -4.531023545498
    --
    0 1
    C               -0.370781459364     0.068502776505     2.544937454502
    C                1.052030540636    -0.400463223495     2.251538454502
    H                1.236470540636    -0.439792223495     1.177017454502
    H                1.783465540636     0.277209776505     2.694273454502
    H                1.224294540636    -1.397469223495     2.659658454502
    C               -0.577178459364     1.465064776505     1.964107454502
    H               -1.589141459364     1.822049776505     2.160954454502
    H                0.123740540636     2.176135776505     2.403828454502
    H               -0.424069459364     1.465260776505     0.883964454502
    C               -0.591075459364     0.108678776505     4.054791454502
    H               -1.602292459364     0.442955776505     4.290864454502
    H                0.111436540636     0.793627776505     4.531629454502
    H               -0.450512459364    -0.879841223495     4.494242454502
    C               -1.367244459364    -0.899275223495     1.911791454502
    H               -1.238318459364    -1.906250223495     2.311370454502
    H               -2.393133459364    -0.585502223495     2.110179454502
    H               -1.232539459364    -0.946249223495     0.830214454502

""")

GEOS['%s-%s-%s' % (dbse, '36-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.382184353217    -0.070570917904    -2.623187066467
    C               -1.040976646783     0.396847082096    -2.329147066467
    H               -1.771912646783    -0.281463917904    -2.771753066467
    H               -1.225055646783     0.435771082096    -1.254564066467
    H               -1.214403646783     1.393760082096    -2.736992066467
    C                0.590509353217    -1.466781917904    -2.042105066467
    H                1.602576353217    -1.822913917904    -2.239980066467
    H                0.438631353217    -1.466709917904    -0.961796066467
    H               -0.110256646783    -2.178647917904    -2.480769066467
    C                1.377931353217     0.898471082096    -1.990934066467
    H                2.404056353217     0.585477082096    -2.189349066467
    H                1.243443353217     0.946010082096    -0.909373066467
    H                1.248035353217     1.905059082096    -2.391132066467
    C                0.601623353217    -0.110997917904    -4.133180066467
    H                0.458874353217     0.877075082096    -4.572981066467
    H                1.613356353217    -0.443422917904    -4.369755066467
    H               -0.099868646783    -0.797513917904    -4.609304066467
    --
    0 1
    C               -0.382186646783     0.070609082096     2.623217933533
    C                1.040626353217    -0.398356917904     2.329818933533
    H                1.225065353217    -0.437685917904     1.255297933533
    H                1.772060353217     0.279316082096     2.772553933533
    H                1.212889353217    -1.395362917904     2.737938933533
    C               -0.588583646783     1.467171082096     2.042387933533
    H               -1.600546646783     1.824156082096     2.239234933533
    H                0.112335353217     2.178242082096     2.482109933533
    H               -0.435473646783     1.467368082096     0.962245933533
    C               -0.602480646783     0.110786082096     4.133071933533
    H               -1.613696646783     0.445063082096     4.369144933533
    H                0.100031353217     0.795735082096     4.609909933533
    H               -0.461917646783    -0.877734917904     4.572523933533
    C               -1.378648646783    -0.897167917904     1.990071933533
    H               -1.249722646783    -1.904143917904     2.389650933533
    H               -2.404538646783    -0.583394917904     2.188459933533
    H               -1.243944646783    -0.944142917904     0.908494933533

""")

GEOS['%s-%s-%s' % (dbse, '36-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.393328957411    -0.072629626291    -2.699681198619
    C               -1.029832042589     0.394788373709    -2.405641198619
    H               -1.760768042589    -0.283522626291    -2.848247198619
    H               -1.213911042589     0.433712373709    -1.331058198619
    H               -1.203259042589     1.391701373709    -2.813486198619
    C                0.601653957411    -1.468840626291    -2.118599198619
    H                1.613720957411    -1.824972626291    -2.316474198619
    H                0.449775957411    -1.468768626291    -1.038290198619
    H               -0.099112042589    -2.180706626291    -2.557263198619
    C                1.389075957411     0.896412373709    -2.067428198619
    H                2.415200957411     0.583418373709    -2.265843198619
    H                1.254587957411     0.943951373709    -0.985867198619
    H                1.259179957411     1.903000373709    -2.467626198619
    C                0.612767957411    -0.113056626291    -4.209674198619
    H                0.470018957411     0.875016373709    -4.649475198619
    H                1.624500957411    -0.445481626291    -4.446249198619
    H               -0.088724042589    -0.799572626291    -4.685798198619
    --
    0 1
    C               -0.393331042589     0.072668373709     2.699712801381
    C                1.029481957411    -0.396297626291     2.406312801381
    H                1.213920957411    -0.435627626291     1.331792801381
    H                1.760915957411     0.281374373709     2.849048801381
    H                1.201744957411    -1.393303626291     2.814433801381
    C               -0.599728042589     1.469229373709     2.118881801381
    H               -1.611691042589     1.826215373709     2.315729801381
    H                0.101190957411     2.180301373709     2.558603801381
    H               -0.446619042589     1.469426373709     1.038739801381
    C               -0.613625042589     0.112844373709     4.209565801381
    H               -1.624842042589     0.447121373709     4.445639801381
    H                0.088886957411     0.797793373709     4.686404801381
    H               -0.473062042589    -0.875675626291     4.649017801381
    C               -1.389794042589    -0.895109626291     2.066565801381
    H               -1.260868042589    -1.902084626291     2.466144801381
    H               -2.415683042589    -0.581336626291     2.264953801381
    H               -1.255089042589    -0.942083626291     0.984989801381

""")

GEOS['%s-%s-%s' % (dbse, '36-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.404244540636    -0.074645806720    -2.774603198619
    C               -1.018916459364     0.392772193280    -2.480563198619
    H               -1.749852459364    -0.285538806720    -2.923169198619
    H               -1.202995459364     0.431696193280    -1.405980198619
    H               -1.192343459364     1.389685193280    -2.888408198619
    C                0.612569540636    -1.470856806720    -2.193521198619
    H                1.624636540636    -1.826988806720    -2.391396198619
    H                0.460691540636    -1.470784806720    -1.113212198619
    H               -0.088196459364    -2.182722806720    -2.632185198619
    C                1.399991540636     0.894396193280    -2.142350198619
    H                2.426116540636     0.581402193280    -2.340765198619
    H                1.265503540636     0.941935193280    -1.060789198619
    H                1.270095540636     1.900984193280    -2.542548198619
    C                0.623683540636    -0.115072806720    -4.284596198619
    H                0.480934540636     0.873000193280    -4.724397198619
    H                1.635416540636    -0.447497806720    -4.521171198619
    H               -0.077808459364    -0.801588806720    -4.760720198619
    --
    0 1
    C               -0.404246459364     0.074684193280     2.774634801381
    C                1.018565540636    -0.394281806720     2.481234801381
    H                1.203005540636    -0.433610806720     1.406714801381
    H                1.750000540636     0.283391193280     2.923970801381
    H                1.190829540636    -1.391287806720     2.889355801381
    C               -0.610643459364     1.471246193280     2.193803801381
    H               -1.622606459364     1.828231193280     2.390651801381
    H                0.090275540636     2.182317193280     2.633525801381
    H               -0.457534459364     1.471442193280     1.113661801381
    C               -0.624540459364     0.114860193280     4.284487801381
    H               -1.635757459364     0.449137193280     4.520561801381
    H                0.077971540636     0.799809193280     4.761326801381
    H               -0.483977459364    -0.873659806720     4.723939801381
    C               -1.400709459364    -0.893092806720     2.141487801381
    H               -1.271783459364    -1.900068806720     2.541066801381
    H               -2.426598459364    -0.579320806720     2.339875801381
    H               -1.266004459364    -0.940067806720     1.059911801381

""")

GEOS['%s-%s-%s' % (dbse, '36-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.436010151819    -0.080513591343    -2.992636052487
    C               -0.987150848181     0.386904408657    -2.698596052487
    H               -1.718086848181    -0.291406591343    -3.141202052487
    H               -1.171229848181     0.425828408657    -1.624013052487
    H               -1.160577848181     1.383817408657    -3.106441052487
    C                0.644335151819    -1.476724591343    -2.411554052487
    H                1.656402151819    -1.832856591343    -2.609429052487
    H                0.492457151819    -1.476652591343    -1.331245052487
    H               -0.056430848181    -2.188590591343    -2.850218052487
    C                1.431757151819     0.888528408657    -2.360383052487
    H                2.457882151819     0.575534408657    -2.558798052487
    H                1.297269151819     0.936067408657    -1.278822052487
    H                1.301861151819     1.895116408657    -2.760581052487
    C                0.655449151819    -0.120940591343    -4.502629052487
    H                0.512700151819     0.867132408657    -4.942430052487
    H                1.667182151819    -0.453365591343    -4.739204052487
    H               -0.046042848181    -0.807456591343    -4.978753052487
    --
    0 1
    C               -0.436011848181     0.080552408657     2.992666947513
    C                0.986800151819    -0.388413591343     2.699267947513
    H                1.171239151819    -0.427743591343     1.624746947513
    H                1.718234151819     0.289258408657     3.142002947513
    H                1.159064151819    -1.385420591343     3.107387947513
    C               -0.642408848181     1.477113408657     2.411836947513
    H               -1.654371848181     1.834098408657     2.608683947513
    H                0.058510151819     2.188185408657     2.851557947513
    H               -0.489299848181     1.477310408657     1.331693947513
    C               -0.656306848181     0.120728408657     4.502520947513
    H               -1.667522848181     0.455005408657     4.738593947513
    H                0.046206151819     0.805677408657     4.979358947513
    H               -0.515743848181    -0.867792591343     4.941972947513
    C               -1.432474848181    -0.887225591343     2.359520947513
    H               -1.303548848181    -1.894201591343     2.759099947513
    H               -2.458363848181    -0.573452591343     2.557908947513
    H               -1.297770848181    -0.934200591343     1.277943947513

""")

GEOS['%s-%s-%s' % (dbse, '36-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.486882151819    -0.089910410914    -3.341811052487
    C               -0.936278848181     0.377507589086    -3.047771052487
    H               -1.667214848181    -0.300803410914    -3.490377052487
    H               -1.120357848181     0.416431589086    -1.973188052487
    H               -1.109705848181     1.374420589086    -3.455616052487
    C                0.695207151819    -1.486121410914    -2.760729052487
    H                1.707274151819    -1.842253410914    -2.958604052487
    H                0.543329151819    -1.486049410914    -1.680420052487
    H               -0.005558848181    -2.197987410914    -3.199393052487
    C                1.482629151819     0.879131589086    -2.709558052487
    H                2.508754151819     0.566137589086    -2.907973052487
    H                1.348141151819     0.926670589086    -1.627997052487
    H                1.352733151819     1.885719589086    -3.109756052487
    C                0.706321151819    -0.130337410914    -4.851804052487
    H                0.563572151819     0.857735589086    -5.291605052487
    H                1.718054151819    -0.462762410914    -5.088379052487
    H                0.004829151819    -0.816853410914    -5.327928052487
    --
    0 1
    C               -0.486883848181     0.089948589086     3.341841947513
    C                0.935928151819    -0.379017410914     3.048442947513
    H                1.120367151819    -0.418346410914     1.973921947513
    H                1.667362151819     0.298655589086     3.491177947513
    H                1.108192151819    -1.376023410914     3.456562947513
    C               -0.693280848181     1.486510589086     2.761011947513
    H               -1.705243848181     1.843495589086     2.957858947513
    H                0.007638151819     2.197581589086     3.200732947513
    H               -0.540171848181     1.486706589086     1.680868947513
    C               -0.707178848181     0.130125589086     4.851695947513
    H               -1.718394848181     0.464402589086     5.087768947513
    H               -0.004665848181     0.815074589086     5.328533947513
    H               -0.566615848181    -0.858395410914     5.291147947513
    C               -1.483346848181    -0.877828410914     2.708695947513
    H               -1.354420848181    -1.884804410914     3.108274947513
    H               -2.509235848181    -0.564055410914     2.907083947513
    H               -1.348642848181    -0.924803410914     1.627118947513

""")

GEOS['%s-%s-%s' % (dbse, '36-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.584824756013    -0.108002133281    -4.014069198619
    C               -0.838336243987     0.359415866719    -3.720029198619
    H               -1.569272243987    -0.318895133281    -4.162635198619
    H               -1.022415243987     0.398339866719    -2.645446198619
    H               -1.011763243987     1.356328866719    -4.127874198619
    C                0.793149756013    -1.504213133281    -3.432987198619
    H                1.805216756013    -1.860345133281    -3.630862198619
    H                0.641271756013    -1.504141133281    -2.352678198619
    H                0.092383756013    -2.216079133281    -3.871651198619
    C                1.580571756013     0.861039866719    -3.381816198619
    H                2.606696756013     0.548045866719    -3.580231198619
    H                1.446083756013     0.908578866719    -2.300255198619
    H                1.450675756013     1.867627866719    -3.782014198619
    C                0.804263756013    -0.148429133281    -5.524062198619
    H                0.661514756013     0.839643866719    -5.963863198619
    H                1.815996756013    -0.480854133281    -5.760637198619
    H                0.102771756013    -0.834945133281    -6.000186198619
    --
    0 1
    C               -0.584827243987     0.108040866719     4.014100801381
    C                0.837985756013    -0.360925133281     3.720700801381
    H                1.022424756013    -0.400254133281     2.646180801381
    H                1.569419756013     0.316746866719     4.163436801381
    H                1.010249756013    -1.357931133281     4.128821801381
    C               -0.791223243987     1.504601866719     3.433269801381
    H               -1.803187243987     1.861587866719     3.630117801381
    H               -0.090305243987     2.215673866719     3.872991801381
    H               -0.638114243987     1.504798866719     2.353127801381
    C               -0.805121243987     0.148216866719     5.523953801381
    H               -1.816337243987     0.482493866719     5.760027801381
    H               -0.102609243987     0.833165866719     6.000792801381
    H               -0.664558243987    -0.840303133281     5.963405801381
    C               -1.581289243987    -0.859737133281     3.380953801381
    H               -1.452363243987    -1.866712133281     3.780532801381
    H               -2.607178243987    -0.545964133281     3.579341801381
    H               -1.446585243987    -0.906711133281     2.299377801381

""")

GEOS['%s-%s-%s' % (dbse, '37-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.805120492664    -1.009045857411    -2.457157159209
    H                0.858762492664    -1.109044857411    -3.542908159209
    H                1.496608492664    -1.731162857411    -2.025174159209
    C                1.122093492664     0.437959142589    -2.045232159209
    H                1.843348492664     0.903151142589    -2.714441159209
    H                1.560776492664     0.452831142589    -1.047810159209
    C               -0.239352507336     1.178696142589    -2.041917159209
    H               -0.252871507336     2.013869142589    -2.739571159209
    H               -0.443597507336     1.590002142589    -1.053910159209
    C               -1.293507507336     0.116818142589    -2.405585159209
    H               -1.468354507336     0.118249142589    -3.482899159209
    H               -2.251527507336     0.291047142589    -1.917736159209
    C               -0.641673507336    -1.207061857411    -2.008587159209
    H               -1.119232507336    -2.074620857411    -2.461901159209
    H               -0.680812507336    -1.332276857411    -0.924674159209
    --
    0 1
    C                0.047412492664     0.092335142589     2.129122840791
    C               -0.048931507336    -0.059750857411     3.644763840791
    H                0.943614492664    -0.075216857411     4.097120840791
    H               -0.607856507336     0.768134142589     4.083198840791
    H               -0.556096507336    -0.988416857411     3.910112840791
    C                0.761465492664     1.399240142589     1.793498840791
    H                0.842985492664     1.532739142589     0.713816840791
    H                0.217993492664     2.253929142589     2.198999840791
    H                1.769070492664     1.409891142589     2.211731840791
    C               -1.357599507336     0.112160142589     1.531803840791
    H               -1.320667507336     0.229350142589     0.447954840791
    H               -1.939898507336     0.939583142589     1.940149840791
    H               -1.887502507336    -0.815626857411     1.753132840791
    C                0.835312492664    -1.081149857411     1.551784840791
    H                0.340092492664    -2.028139857411     1.772027840791
    H                0.930155492664    -0.993966857411     0.468649840791
    H                1.840030492664    -1.118553857411     1.975533840791

""")

GEOS['%s-%s-%s' % (dbse, '37-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.803156869066    -1.012869806703    -2.545121037430
    H                0.856798869066    -1.112868806703    -3.630872037430
    H                1.494644869066    -1.734986806703    -2.113138037430
    C                1.120129869066     0.434135193297    -2.133196037430
    H                1.841384869066     0.899327193297    -2.802405037430
    H                1.558812869066     0.449007193297    -1.135774037430
    C               -0.241316130934     1.174872193297    -2.129881037430
    H               -0.254835130934     2.010045193297    -2.827535037430
    H               -0.445561130934     1.586178193297    -1.141874037430
    C               -1.295471130934     0.112994193297    -2.493549037430
    H               -1.470318130934     0.114425193297    -3.570863037430
    H               -2.253491130934     0.287223193297    -2.005700037430
    C               -0.643637130934    -1.210885806703    -2.096551037430
    H               -1.121196130934    -2.078444806703    -2.549865037430
    H               -0.682776130934    -1.336100806703    -1.012638037430
    --
    0 1
    C                0.049320869066     0.096052193297     2.214627962570
    C               -0.047023130934    -0.056033806703     3.730267962570
    H                0.945523869066    -0.071499806703     4.182625962570
    H               -0.605948130934     0.771851193297     4.168702962570
    H               -0.554188130934    -0.984699806703     3.995617962570
    C                0.763374869066     1.402957193297     1.879002962570
    H                0.844893869066     1.536456193297     0.799320962570
    H                0.219901869066     2.257646193297     2.284503962570
    H                1.770978869066     1.413608193297     2.297235962570
    C               -1.355690130934     0.115877193297     1.617307962570
    H               -1.318759130934     0.233067193297     0.533458962570
    H               -1.937990130934     0.943300193297     2.025653962570
    H               -1.885594130934    -0.811909806703     1.838637962570
    C                0.837220869066    -1.077432806703     1.637289962570
    H                0.342000869066    -2.024423806703     1.857532962570
    H                0.932063869066    -0.990249806703     0.554154962570
    H                1.841938869066    -1.114836806703     2.061037962570

""")

GEOS['%s-%s-%s' % (dbse, '37-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.801294515017    -1.016496505553    -2.628548802390
    H                0.854936515017    -1.116495505553    -3.714299802390
    H                1.492782515017    -1.738613505553    -2.196565802390
    C                1.118267515017     0.430508494447    -2.216623802390
    H                1.839522515017     0.895700494447    -2.885832802390
    H                1.556950515017     0.445380494447    -1.219201802390
    C               -0.243178484983     1.171245494447    -2.213308802390
    H               -0.256697484983     2.006418494447    -2.910962802390
    H               -0.447423484983     1.582551494447    -1.225301802390
    C               -1.297333484983     0.109367494447    -2.576976802390
    H               -1.472180484983     0.110798494447    -3.654290802390
    H               -2.255353484983     0.283596494447    -2.089127802390
    C               -0.645499484983    -1.214512505553    -2.179978802390
    H               -1.123058484983    -2.082071505553    -2.633292802390
    H               -0.684638484983    -1.339727505553    -1.096065802390
    --
    0 1
    C                0.051131515017     0.099577494447     2.295723197610
    C               -0.045212484983    -0.052508505553     3.811363197610
    H                0.947333515017    -0.067974505553     4.263721197610
    H               -0.604137484983     0.775376494447     4.249798197610
    H               -0.552377484983    -0.981174505553     4.076713197610
    C                0.765184515017     1.406482494447     1.960098197610
    H                0.846704515017     1.539981494447     0.880416197610
    H                0.221712515017     2.261171494447     2.365599197610
    H                1.772789515017     1.417133494447     2.378331197610
    C               -1.353880484983     0.119402494447     1.698403197610
    H               -1.316948484983     0.236592494447     0.614554197610
    H               -1.936179484983     0.946825494447     2.106749197610
    H               -1.883783484983    -0.808384505553     1.919733197610
    C                0.839031515017    -1.073907505553     1.718385197610
    H                0.343810515017    -2.020898505553     1.938628197610
    H                0.933874515017    -0.986724505553     0.635250197610
    H                1.843749515017    -1.111311505553     2.142133197610

""")

GEOS['%s-%s-%s' % (dbse, '37-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.799504498764    -1.019982254993    -2.708736988570
    H                0.853146498764    -1.119981254993    -3.794487988570
    H                1.490992498764    -1.742099254993    -2.276753988570
    C                1.116477498764     0.427022745007    -2.296811988570
    H                1.837732498764     0.892214745007    -2.966020988570
    H                1.555160498764     0.441894745007    -1.299389988570
    C               -0.244968501236     1.167759745007    -2.293496988570
    H               -0.258487501236     2.002932745007    -2.991150988570
    H               -0.449213501236     1.579065745007    -1.305489988570
    C               -1.299123501236     0.105881745007    -2.657164988570
    H               -1.473970501236     0.107312745007    -3.734478988570
    H               -2.257143501236     0.280110745007    -2.169315988570
    C               -0.647289501236    -1.217998254993    -2.260166988570
    H               -1.124848501236    -2.085557254993    -2.713480988570
    H               -0.686428501236    -1.343213254993    -1.176253988570
    --
    0 1
    C                0.052871498764     0.102965745007     2.373669011430
    C               -0.043472501236    -0.049120254993     3.889310011430
    H                0.949073498764    -0.064586254993     4.341667011430
    H               -0.602397501236     0.778764745007     4.327745011430
    H               -0.550637501236    -0.977786254993     4.154659011430
    C                0.766924498764     1.409870745007     2.038044011430
    H                0.848444498764     1.543370745007     0.958363011430
    H                0.223452498764     2.264559745007     2.443546011430
    H                1.774529498764     1.420521745007     2.456277011430
    C               -1.352140501236     0.122790745007     1.776350011430
    H               -1.315208501236     0.239980745007     0.692501011430
    H               -1.934440501236     0.950214745007     2.184696011430
    H               -1.882043501236    -0.804996254993     1.997679011430
    C                0.840771498764    -1.070519254993     1.796331011430
    H                0.345550498764    -2.017509254993     2.016574011430
    H                0.935614498764    -0.983336254993     0.713196011430
    H                1.845489498764    -1.107923254993     2.220080011430

""")

GEOS['%s-%s-%s' % (dbse, '37-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.797773297718    -1.023353831485    -2.786291933019
    H                0.851415297718    -1.123352831485    -3.872042933019
    H                1.489261297718    -1.745470831485    -2.354308933019
    C                1.114746297718     0.423651168515    -2.374366933019
    H                1.836001297718     0.888843168515    -3.043575933019
    H                1.553429297718     0.438523168515    -1.376944933019
    C               -0.246699702282     1.164388168515    -2.371051933019
    H               -0.260218702282     1.999561168515    -3.068705933019
    H               -0.450944702282     1.575694168515    -1.383044933019
    C               -1.300854702282     0.102510168515    -2.734719933019
    H               -1.475701702282     0.103941168515    -3.812033933019
    H               -2.258874702282     0.276739168515    -2.246870933019
    C               -0.649020702282    -1.221369831485    -2.337721933019
    H               -1.126579702282    -2.088928831485    -2.791035933019
    H               -0.688159702282    -1.346584831485    -1.253808933019
    --
    0 1
    C                0.054554297718     0.106243168515     2.449056066981
    C               -0.041789702282    -0.045842831485     3.964696066981
    H                0.950756297718    -0.061308831485     4.417053066981
    H               -0.600714702282     0.782041168515     4.403131066981
    H               -0.548954702282    -0.974508831485     4.230046066981
    C                0.768607297718     1.413148168515     2.113431066981
    H                0.850127297718     1.546647168515     1.033749066981
    H                0.225135297718     2.267836168515     2.518932066981
    H                1.776212297718     1.423798168515     2.531664066981
    C               -1.350457702282     0.126068168515     1.851736066981
    H               -1.313525702282     0.243257168515     0.767887066981
    H               -1.932757702282     0.953491168515     2.260082066981
    H               -1.880360702282    -0.801718831485     2.073065066981
    C                0.842454297718    -1.067241831485     1.871718066981
    H                0.347233297718    -2.014232831485     2.091961066981
    H                0.937297297718    -0.980058831485     0.788583066981
    H                1.847172297718    -1.104646831485     2.295466066981

""")

GEOS['%s-%s-%s' % (dbse, '37-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.792822583012    -1.032995145627    -3.008080403080
    H                0.846464583012    -1.132994145627    -4.093831403080
    H                1.484310583012    -1.755112145627    -2.576097403080
    C                1.109795583012     0.414009854373    -2.596155403080
    H                1.831050583012     0.879201854373    -3.265364403080
    H                1.548478583012     0.428881854373    -1.598733403080
    C               -0.251650416988     1.154746854373    -2.592840403080
    H               -0.265169416988     1.989919854373    -3.290494403080
    H               -0.455895416988     1.566052854373    -1.604833403080
    C               -1.305805416988     0.092868854373    -2.956508403080
    H               -1.480652416988     0.094299854373    -4.033822403080
    H               -2.263825416988     0.267097854373    -2.468659403080
    C               -0.653971416988    -1.231011145627    -2.559510403080
    H               -1.131530416988    -2.098570145627    -3.012824403080
    H               -0.693110416988    -1.356226145627    -1.475597403080
    --
    0 1
    C                0.059366583012     0.115614854373     2.664643596920
    C               -0.036977416988    -0.036471145627     4.180283596920
    H                0.955568583012    -0.051937145627     4.632641596920
    H               -0.595902416988     0.791413854373     4.618718596920
    H               -0.544142416988    -0.965137145627     4.445633596920
    C                0.773419583012     1.422519854373     2.329018596920
    H                0.854939583012     1.556018854373     1.249336596920
    H                0.229947583012     2.277208854373     2.734519596920
    H                1.781024583012     1.433170854373     2.747251596920
    C               -1.345645416988     0.135439854373     2.067323596920
    H               -1.308713416988     0.252629854373     0.983474596920
    H               -1.927944416988     0.962862854373     2.475669596920
    H               -1.875548416988    -0.792347145627     2.288653596920
    C                0.847266583012    -1.057870145627     2.087305596920
    H                0.352045583012    -2.004861145627     2.307548596920
    H                0.942109583012    -0.970687145627     1.004170596920
    H                1.851984583012    -1.095274145627     2.511053596920

""")

GEOS['%s-%s-%s' % (dbse, '37-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.785056729911    -1.048118919853    -3.355981532100
    H                0.838698729911    -1.148117919853    -4.441732532100
    H                1.476544729911    -1.770235919853    -2.923998532100
    C                1.102029729911     0.398886080147    -2.944056532100
    H                1.823284729911     0.864078080147    -3.613265532100
    H                1.540712729911     0.413758080147    -1.946634532100
    C               -0.259416270089     1.139623080147    -2.940741532100
    H               -0.272935270089     1.974796080147    -3.638395532100
    H               -0.463661270089     1.550929080147    -1.952734532100
    C               -1.313571270089     0.077745080147    -3.304409532100
    H               -1.488418270089     0.079176080147    -4.381723532100
    H               -2.271591270089     0.251974080147    -2.816560532100
    C               -0.661737270089    -1.246134919853    -2.907411532100
    H               -1.139296270089    -2.113693919853    -3.360725532100
    H               -0.700876270089    -1.371349919853    -1.823498532100
    --
    0 1
    C                0.066915729911     0.130316080147     3.002817467900
    C               -0.029429270089    -0.021769919853     4.518458467900
    H                0.963117729911    -0.037236919853     4.970815467900
    H               -0.588353270089     0.806114080147     4.956893467900
    H               -0.536593270089    -0.950436919853     4.783807467900
    C                0.780968729911     1.437221080147     2.667192467900
    H                0.862487729911     1.570720080147     1.587510467900
    H                0.237495729911     2.291909080147     3.072694467900
    H                1.788572729911     1.447871080147     3.085425467900
    C               -1.338096270089     0.150140080147     2.405498467900
    H               -1.301165270089     0.267330080147     1.321649467900
    H               -1.920396270089     0.977564080147     2.813844467900
    H               -1.867999270089    -0.777646919853     2.626827467900
    C                0.854814729911    -1.043168919853     2.425479467900
    H                0.359594729911    -1.990159919853     2.645722467900
    H                0.949658729911    -0.955986919853     1.342344467900
    H                1.859532729911    -1.080573919853     2.849228467900

""")

GEOS['%s-%s-%s' % (dbse, '37-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.770331386340    -1.076795451921    -4.015654321272
    H                0.823973386340    -1.176794451921    -5.101405321272
    H                1.461819386340    -1.798912451921    -3.583671321272
    C                1.087304386340     0.370209548079    -3.603729321272
    H                1.808559386340     0.835401548079    -4.272938321272
    H                1.525987386340     0.385081548079    -2.606307321272
    C               -0.274141613660     1.110946548079    -3.600414321272
    H               -0.287660613660     1.946119548079    -4.298068321272
    H               -0.478386613660     1.522252548079    -2.612407321272
    C               -1.328296613660     0.049068548079    -3.964082321272
    H               -1.503143613660     0.050499548079    -5.041396321272
    H               -2.286316613660     0.223297548079    -3.476233321272
    C               -0.676462613660    -1.274811451921    -3.567084321272
    H               -1.154021613660    -2.142370451921    -4.020398321272
    H               -0.715601613660    -1.400026451921    -2.483171321272
    --
    0 1
    C                0.081229386340     0.158190548079     3.644046678728
    C               -0.015115613660     0.006104548079     5.159687678728
    H                0.977431386340    -0.009361451921     5.612044678728
    H               -0.574039613660     0.833989548079     5.598122678728
    H               -0.522279613660    -0.922561451921     5.425036678728
    C                0.795282386340     1.465095548079     3.308421678728
    H                0.876801386340     1.598594548079     2.228739678728
    H                0.251809386340     2.319784548079     3.713923678728
    H                1.802886386340     1.475746548079     3.726654678728
    C               -1.323782613660     0.178015548079     3.046727678728
    H               -1.286851613660     0.295205548079     1.962878678728
    H               -1.906082613660     1.005438548079     3.455073678728
    H               -1.853685613660    -0.749771451921     3.268056678728
    C                0.869128386340    -1.015294451921     3.066708678728
    H                0.373908386340    -1.962284451921     3.286951678728
    H                0.963971386340    -0.928111451921     1.983573678728
    H                1.873846386340    -1.052698451921     3.490457678728

""")

GEOS['%s-%s-%s' % (dbse, '38-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.961246388746    -0.886671709554    -1.793131081911
    H                1.508932388746    -1.273183709554    -2.651661081911
    H                1.425750388746    -1.309603709554    -0.904056081911
    C                0.995315388746     0.663682290446    -1.789577081911
    H                1.514960388746     1.028270290446    -2.675133081911
    H                1.520624388746     1.064992290446    -0.924544081911
    C               -0.475085611254     1.107493290446    -1.831202081911
    H               -0.611902611254     2.070051290446    -2.321514081911
    H               -0.870375611254     1.194245290446    -0.817011081911
    C               -1.177740611254    -0.047622709554    -2.541735081911
    H               -0.944515611254    -0.021655709554    -3.608886081911
    H               -2.261298611254    -0.028390709554    -2.433807081911
    C               -0.526293611254    -1.269715709554    -1.895771081911
    H               -0.686024611254    -2.191846709554    -2.452089081911
    H               -0.946482611254    -1.410237709554    -0.898333081911
    --
    0 1
    C               -1.133879611254    -0.391329709554     1.835030918089
    H               -1.467013611254    -0.154630709554     0.822445918089
    H               -1.938667611254    -0.934427709554     2.328016918089
    C                0.179726388746    -1.186874709554     1.787269918089
    H                0.229671388746    -1.844616709554     0.921135918089
    H                0.260014388746    -1.818231709554     2.671654918089
    C                1.309042388746    -0.124168709554     1.787865918089
    H                1.930307388746    -0.184847709554     0.895991918089
    H                1.966451388746    -0.274051709554     2.643270918089
    C                0.598868388746     1.237500290446     1.895732918089
    H                0.419005388746     1.645992290446     0.899732918089
    H                1.179252388746     1.971903290446     2.451705918089
    C               -0.743183611254     0.903027290446     2.545543918089
    H               -1.482852611254     1.695539290446     2.440662918089
    H               -0.603143611254     0.711387290446     3.612092918089

""")

GEOS['%s-%s-%s' % (dbse, '38-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.959704195937    -0.889771702363    -1.863541581911
    H                1.507390195937    -1.276283702363    -2.722071581911
    H                1.424208195937    -1.312703702363    -0.974466581911
    C                0.993773195937     0.660582297637    -1.859987581911
    H                1.513418195937     1.025170297637    -2.745543581911
    H                1.519082195937     1.061892297637    -0.994954581911
    C               -0.476627804063     1.104393297637    -1.901612581911
    H               -0.613444804063     2.066951297637    -2.391924581911
    H               -0.871917804063     1.191145297637    -0.887421581911
    C               -1.179282804063    -0.050722702363    -2.612145581911
    H               -0.946057804063    -0.024755702363    -3.679296581911
    H               -2.262840804063    -0.031490702363    -2.504217581911
    C               -0.527835804063    -1.272815702363    -1.966181581911
    H               -0.687566804063    -2.194946702363    -2.522499581911
    H               -0.948024804063    -1.413337702363    -0.968743581911
    --
    0 1
    C               -1.132337804063    -0.388229702363     1.905441418089
    H               -1.465471804063    -0.151530702363     0.892856418089
    H               -1.937125804063    -0.931327702363     2.398427418089
    C                0.181269195937    -1.183774702363     1.857680418089
    H                0.231214195937    -1.841517702363     0.991546418089
    H                0.261556195937    -1.815131702363     2.742065418089
    C                1.310584195937    -0.121068702363     1.858276418089
    H                1.931849195937    -0.181747702363     0.966402418089
    H                1.967993195937    -0.270951702363     2.713681418089
    C                0.600411195937     1.240600297637     1.966143418089
    H                0.420547195937     1.649092297637     0.970143418089
    H                1.180795195937     1.975003297637     2.522116418089
    C               -0.741641804063     0.906127297637     2.615954418089
    H               -1.481309804063     1.698639297637     2.511073418089
    H               -0.601601804063     0.714487297637     3.682503418089

""")

GEOS['%s-%s-%s' % (dbse, '38-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.958013388746    -0.893170680791    -1.940742717864
    H                1.505699388746    -1.279682680791    -2.799272717864
    H                1.422517388746    -1.316102680791    -1.051667717864
    C                0.992082388746     0.657183319209    -1.937188717864
    H                1.511727388746     1.021771319209    -2.822744717864
    H                1.517391388746     1.058493319209    -1.072155717864
    C               -0.478318611254     1.100994319209    -1.978813717864
    H               -0.615135611254     2.063552319209    -2.469125717864
    H               -0.873608611254     1.187746319209    -0.964622717864
    C               -1.180973611254    -0.054121680791    -2.689346717864
    H               -0.947748611254    -0.028154680791    -3.756497717864
    H               -2.264531611254    -0.034889680791    -2.581418717864
    C               -0.529526611254    -1.276214680791    -2.043382717864
    H               -0.689257611254    -2.198345680791    -2.599700717864
    H               -0.949715611254    -1.416736680791    -1.045944717864
    --
    0 1
    C               -1.130646611254    -0.384830680791     1.982642282136
    H               -1.463780611254    -0.148131680791     0.970058282136
    H               -1.935434611254    -0.927929680791     2.475629282136
    C                0.182959388746    -1.180375680791     1.934881282136
    H                0.232904388746    -1.838118680791     1.068747282136
    H                0.263247388746    -1.811733680791     2.819267282136
    C                1.312275388746    -0.117669680791     1.935477282136
    H                1.933540388746    -0.178348680791     1.043604282136
    H                1.969684388746    -0.267552680791     2.790882282136
    C                0.602101388746     1.243999319209     2.043344282136
    H                0.422238388746     1.652490319209     1.047345282136
    H                1.182485388746     1.978402319209     2.599317282136
    C               -0.739950611254     0.909526319209     2.693156282136
    H               -1.479619611254     1.702038319209     2.588275282136
    H               -0.599910611254     0.717886319209     3.759705282136

""")

GEOS['%s-%s-%s' % (dbse, '38-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.956312995937    -0.896588680791    -2.018377196292
    H                1.503998995937    -1.283100680791    -2.876907196292
    H                1.420816995937    -1.319520680791    -1.129302196292
    C                0.990381995937     0.653765319209    -2.014823196292
    H                1.510026995937     1.018353319209    -2.900379196292
    H                1.515690995937     1.055075319209    -1.149790196292
    C               -0.480019004063     1.097576319209    -2.056448196292
    H               -0.616836004063     2.060134319209    -2.546760196292
    H               -0.875309004063     1.184328319209    -1.042257196292
    C               -1.182674004063    -0.057539680791    -2.766981196292
    H               -0.949449004063    -0.031572680791    -3.834132196292
    H               -2.266232004063    -0.038307680791    -2.659053196292
    C               -0.531227004063    -1.279632680791    -2.121017196292
    H               -0.690958004063    -2.201763680791    -2.677335196292
    H               -0.951416004063    -1.420154680791    -1.123579196292
    --
    0 1
    C               -1.128946004063    -0.381412680791     2.060276803708
    H               -1.462081004063    -0.144713680791     1.047692803708
    H               -1.933735004063    -0.924511680791     2.553263803708
    C                0.184659995937    -1.176957680791     2.012515803708
    H                0.234604995937    -1.834700680791     1.146381803708
    H                0.264947995937    -1.808315680791     2.896900803708
    C                1.313974995937    -0.114251680791     2.013111803708
    H                1.935239995937    -0.174930680791     1.121237803708
    H                1.971384995937    -0.264134680791     2.868516803708
    C                0.603801995937     1.247417319209     2.120978803708
    H                0.423938995937     1.655908319209     1.124979803708
    H                1.184185995937     1.981820319209     2.676951803708
    C               -0.738250004063     0.912944319209     2.770790803708
    H               -1.477919004063     1.705456319209     2.665908803708
    H               -0.598210004063     0.721304319209     3.837339803708

""")

GEOS['%s-%s-%s' % (dbse, '38-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.954661388746    -0.899908687981    -2.093786996292
    H                1.502347388746    -1.286420687981    -2.952316996292
    H                1.419165388746    -1.322840687981    -1.204711996292
    C                0.988730388746     0.650445312019    -2.090232996292
    H                1.508375388746     1.015033312019    -2.975788996292
    H                1.514039388746     1.051755312019    -1.225199996292
    C               -0.481670611254     1.094256312019    -2.131857996292
    H               -0.618487611254     2.056814312019    -2.622169996292
    H               -0.876960611254     1.181008312019    -1.117666996292
    C               -1.184325611254    -0.060859687981    -2.842390996292
    H               -0.951100611254    -0.034892687981    -3.909541996292
    H               -2.267883611254    -0.041627687981    -2.734462996292
    C               -0.532878611254    -1.282952687981    -2.196426996292
    H               -0.692609611254    -2.205083687981    -2.752744996292
    H               -0.953067611254    -1.423474687981    -1.198988996292
    --
    0 1
    C               -1.127294611254    -0.378092687981     2.135687003708
    H               -1.460428611254    -0.141393687981     1.123102003708
    H               -1.932082611254    -0.921190687981     2.628673003708
    C                0.186311388746    -1.173637687981     2.087926003708
    H                0.236256388746    -1.831380687981     1.221792003708
    H                0.266599388746    -1.804995687981     2.972311003708
    C                1.315627388746    -0.110931687981     2.088522003708
    H                1.936892388746    -0.171610687981     1.196648003708
    H                1.973036388746    -0.260814687981     2.943927003708
    C                0.605453388746     1.250737312019     2.196388003708
    H                0.425590388746     1.659228312019     1.200389003708
    H                1.185837388746     1.985140312019     2.752362003708
    C               -0.736598611254     0.916264312019     2.846200003708
    H               -1.476267611254     1.708776312019     2.741319003708
    H               -0.596558611254     0.724624312019     3.912749003708

""")

GEOS['%s-%s-%s' % (dbse, '38-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.949910388746    -0.909458995172    -2.310706210674
    H                1.497596388746    -1.295970995172    -3.169236210674
    H                1.414414388746    -1.332390995172    -1.421631210674
    C                0.983979388746     0.640895004828    -2.307152210674
    H                1.503624388746     1.005483004828    -3.192708210674
    H                1.509288388746     1.042205004828    -1.442119210674
    C               -0.486421611254     1.084706004828    -2.348777210674
    H               -0.623238611254     2.047264004828    -2.839089210674
    H               -0.881711611254     1.171458004828    -1.334586210674
    C               -1.189076611254    -0.070409995172    -3.059310210674
    H               -0.955851611254    -0.044442995172    -4.126461210674
    H               -2.272634611254    -0.051177995172    -2.951382210674
    C               -0.537629611254    -1.292502995172    -2.413346210674
    H               -0.697360611254    -2.214633995172    -2.969664210674
    H               -0.957818611254    -1.433024995172    -1.415908210674
    --
    0 1
    C               -1.122543611254    -0.368541995172     2.352605789326
    H               -1.455677611254    -0.131843995172     1.340021789326
    H               -1.927331611254    -0.911640995172     2.845592789326
    C                0.191062388746    -1.164086995172     2.304844789326
    H                0.241007388746    -1.821829995172     1.438710789326
    H                0.271350388746    -1.795444995172     3.189229789326
    C                1.320378388746    -0.101381995172     2.305440789326
    H                1.941643388746    -0.162059995172     1.413567789326
    H                1.977787388746    -0.251264995172     3.160845789326
    C                0.610204388746     1.260287004828     2.413307789326
    H                0.430341388746     1.668779004828     1.417308789326
    H                1.190588388746     1.994691004828     2.969280789326
    C               -0.731847611254     0.925815004828     3.063119789326
    H               -1.471516611254     1.718327004828     2.958238789326
    H               -0.591807611254     0.734175004828     4.129668789326

""")

GEOS['%s-%s-%s' % (dbse, '38-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.942407088746    -0.924541680791    -2.653286581911
    H                1.490093088746    -1.311053680791    -3.511816581911
    H                1.406911088746    -1.347473680791    -1.764211581911
    C                0.976476088746     0.625812319209    -2.649732581911
    H                1.496121088746     0.990400319209    -3.535288581911
    H                1.501785088746     1.027122319209    -1.784699581911
    C               -0.493924911254     1.069623319209    -2.691357581911
    H               -0.630741911254     2.032181319209    -3.181669581911
    H               -0.889214911254     1.156375319209    -1.677166581911
    C               -1.196579911254    -0.085492680791    -3.401890581911
    H               -0.963354911254    -0.059525680791    -4.469041581911
    H               -2.280137911254    -0.066260680791    -3.293962581911
    C               -0.545132911254    -1.307585680791    -2.755926581911
    H               -0.704863911254    -2.229716680791    -3.312244581911
    H               -0.965321911254    -1.448107680791    -1.758488581911
    --
    0 1
    C               -1.115039911254    -0.353459680791     2.695186418089
    H               -1.448174911254    -0.116760680791     1.682601418089
    H               -1.919828911254    -0.896558680791     3.188172418089
    C                0.198566088746    -1.149004680791     2.647425418089
    H                0.248511088746    -1.806747680791     1.781291418089
    H                0.278854088746    -1.780362680791     3.531810418089
    C                1.327881088746    -0.086298680791     2.648021418089
    H                1.949146088746    -0.146977680791     1.756147418089
    H                1.985291088746    -0.236181680791     3.503426418089
    C                0.617708088746     1.275370319209     2.755888418089
    H                0.437845088746     1.683861319209     1.759888418089
    H                1.198092088746     2.009773319209     3.311861418089
    C               -0.724344911254     0.940897319209     3.405699418089
    H               -1.464012911254     1.733409319209     3.300818418089
    H               -0.584304911254     0.749257319209     4.472248418089

""")

GEOS['%s-%s-%s' % (dbse, '38-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.928133674365    -0.953233587981    -3.304978696292
    H                1.475819674365    -1.339745587981    -4.163508696292
    H                1.392637674365    -1.376165587981    -2.415903696292
    C                0.962202674365     0.597120412019    -3.301424696292
    H                1.481847674365     0.961708412019    -4.186980696292
    H                1.487511674365     0.998430412019    -2.436391696292
    C               -0.508198325635     1.040931412019    -3.343049696292
    H               -0.645015325635     2.003489412019    -3.833361696292
    H               -0.903488325635     1.127683412019    -2.328858696292
    C               -1.210853325635    -0.114184587981    -4.053582696292
    H               -0.977628325635    -0.088217587981    -5.120733696292
    H               -2.294411325635    -0.094952587981    -3.945654696292
    C               -0.559406325635    -1.336277587981    -3.407618696292
    H               -0.719137325635    -2.258408587981    -3.963936696292
    H               -0.979595325635    -1.476799587981    -2.410180696292
    --
    0 1
    C               -1.100767325635    -0.324767587981     3.346878303708
    H               -1.433901325635    -0.088069587981     2.334294303708
    H               -1.905555325635    -0.867866587981     3.839865303708
    C                0.212839674365    -1.120312587981     3.299117303708
    H                0.262784674365    -1.778055587981     2.432983303708
    H                0.293127674365    -1.751670587981     4.183502303708
    C                1.342154674365    -0.057606587981     3.299713303708
    H                1.963419674365    -0.118285587981     2.407839303708
    H                1.999564674365    -0.207489587981     4.155118303708
    C                0.631981674365     1.304061412019     3.407580303708
    H                0.452118674365     1.712553412019     2.411581303708
    H                1.212365674365     2.038465412019     3.963553303708
    C               -0.710071325635     0.969589412019     4.057392303708
    H               -1.449739325635     1.762101412019     3.952510303708
    H               -0.570031325635     0.777949412019     5.123941303708

""")

GEOS['%s-%s-%s' % (dbse, '39-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.788034451023     0.908905989857    -1.389755073360
    H                1.459965451023     1.720668989857    -1.145643073360
    C                1.260142451023    -0.402176010143    -1.416858073360
    H                2.298248451023    -0.607876010143    -1.193035073360
    C                0.394726451023    -1.447875010143    -1.733477073360
    H                0.760677451023    -2.465418010143    -1.753690073360
    C               -0.942443548977    -1.182310010143    -2.023868073360
    H               -1.613969548977    -1.993899010143    -2.268520073360
    C               -1.414575548977     0.129067989857    -1.997469073360
    H               -2.452194548977     0.334963989857    -2.222213073360
    C               -0.549416548977     1.174685989857    -1.679933073360
    H               -0.915209548977     2.192372989857    -1.659668073360
    --
    0 1
    C               -0.756694548977    -0.713934010143     1.501449926640
    H               -0.862948548977    -0.538985010143     0.429142926640
    H               -1.408332548977    -1.543198010143     1.774599926640
    C                0.712846451023    -0.970997010143     1.845578926640
    H                1.208077451023    -1.608582010143     1.114994926640
    H                0.776125451023    -1.471211010143     2.813045926640
    C                1.352983451023     0.434399989857     1.936531926640
    H                2.003251451023     0.627069989857     1.084678926640
    H                1.965850451023     0.523568989857     2.832257926640
    C                0.173937451023     1.433576989857     1.963403926640
    H                0.056785451023     1.892546989857     0.981118926640
    H                0.319530451023     2.235080989857     2.685802926640
    C               -1.059834548977     0.576480989857     2.259052926640
    H               -1.990822548977     1.052218989857     1.953139926640
    H               -1.125260548977     0.367159989857     3.329587926640

""")

GEOS['%s-%s-%s' % (dbse, '39-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.784665927723     0.902948121603    -1.464197022170
    H                1.456596927723     1.714711121603    -1.220085022170
    C                1.256773927723    -0.408133878397    -1.491300022170
    H                2.294879927723    -0.613833878397    -1.267477022170
    C                0.391357927723    -1.453832878397    -1.807919022170
    H                0.757308927723    -2.471375878397    -1.828132022170
    C               -0.945812072277    -1.188267878397    -2.098310022170
    H               -1.617338072277    -1.999856878397    -2.342962022170
    C               -1.417944072277     0.123110121603    -2.071911022170
    H               -2.455563072277     0.329006121603    -2.296655022170
    C               -0.552785072277     1.168728121603    -1.754375022170
    H               -0.918578072277     2.186415121603    -1.734110022170
    --
    0 1
    C               -0.752943072277    -0.707298878397     1.584355977830
    H               -0.859197072277    -0.532349878397     0.512048977830
    H               -1.404580072277    -1.536562878397     1.857506977830
    C                0.716597927723    -0.964361878397     1.928485977830
    H                1.211828927723    -1.601946878397     1.197901977830
    H                0.779876927723    -1.464575878397     2.895952977830
    C                1.356734927723     0.441035121603     2.019438977830
    H                2.007003927723     0.633706121603     1.167585977830
    H                1.969602927723     0.530204121603     2.915163977830
    C                0.177688927723     1.440212121603     2.046310977830
    H                0.060537927723     1.899182121603     1.064025977830
    H                0.323282927723     2.241717121603     2.768709977830
    C               -1.056083072277     0.583117121603     2.341959977830
    H               -1.987071072277     1.058854121603     2.036046977830
    H               -1.121508072277     0.373796121603     3.412494977830

""")

GEOS['%s-%s-%s' % (dbse, '39-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.781216112616     0.896847005027    -1.540429267960
    H                1.453147112616     1.708610005027    -1.296317267960
    C                1.253324112616    -0.414234994973    -1.567532267960
    H                2.291430112616    -0.619934994973    -1.343709267960
    C                0.387908112616    -1.459933994973    -1.884151267960
    H                0.753859112616    -2.477476994973    -1.904364267960
    C               -0.949261887384    -1.194368994973    -2.174542267960
    H               -1.620787887384    -2.005957994973    -2.419194267960
    C               -1.421393887384     0.117009005027    -2.148143267960
    H               -2.459012887384     0.322905005027    -2.372887267960
    C               -0.556234887384     1.162627005027    -1.830607267960
    H               -0.922027887384     2.180314005027    -1.810342267960
    --
    0 1
    C               -0.749100887384    -0.700503994973     1.669256732040
    H               -0.855354887384    -0.525554994973     0.596949732040
    H               -1.400738887384    -1.529767994973     1.942407732040
    C                0.720440112616    -0.957566994973     2.013386732040
    H                1.215671112616    -1.595151994973     1.282802732040
    H                0.783719112616    -1.457780994973     2.980853732040
    C                1.360577112616     0.447830005027     2.104339732040
    H                2.010845112616     0.640501005027     1.252486732040
    H                1.973444112616     0.536999005027     3.000064732040
    C                0.181531112616     1.447007005027     2.131211732040
    H                0.064379112616     1.905977005027     1.148925732040
    H                0.327124112616     2.248512005027     2.853609732040
    C               -1.052240887384     0.589912005027     2.426860732040
    H               -1.983228887384     1.065649005027     2.120947732040
    H               -1.117666887384     0.380591005027     3.497395732040

""")

GEOS['%s-%s-%s' % (dbse, '39-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.777699549379     0.890627592672    -1.618140638603
    H                1.449630549379     1.702390592672    -1.374028638603
    C                1.249807549379    -0.420454407328    -1.645243638603
    H                2.287913549379    -0.626154407328    -1.421420638603
    C                0.384391549379    -1.466153407328    -1.961862638603
    H                0.750342549379    -2.483696407328    -1.982075638603
    C               -0.952778450621    -1.200588407328    -2.252253638603
    H               -1.624304450621    -2.012177407328    -2.496905638603
    C               -1.424910450621     0.110789592672    -2.225854638603
    H               -2.462529450621     0.316685592672    -2.450598638603
    C               -0.559751450621     1.156407592672    -1.908318638603
    H               -0.925544450621     2.174094592672    -1.888053638603
    --
    0 1
    C               -0.745184450621    -0.693577407328     1.755805361397
    H               -0.851438450621    -0.518627407328     0.683498361397
    H               -1.396822450621    -1.522840407328     2.028955361397
    C                0.724356549379    -0.950640407328     2.099935361397
    H                1.219587549379    -1.588225407328     1.369350361397
    H                0.787635549379    -1.450853407328     3.067401361397
    C                1.364493549379     0.454756592672     2.190887361397
    H                2.014761549379     0.647427592672     1.339034361397
    H                1.977360549379     0.543925592672     3.086613361397
    C                0.185447549379     1.453933592672     2.217759361397
    H                0.068295549379     1.912903592672     1.235474361397
    H                0.331040549379     2.255438592672     2.940158361397
    C               -1.048324450621     0.596838592672     2.513408361397
    H               -1.979312450621     1.072575592672     2.207495361397
    H               -1.113750450621     0.387517592672     3.583944361397

""")

GEOS['%s-%s-%s' % (dbse, '39-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.774242556104     0.884513249667    -1.694537354876
    H                1.446173556104     1.696276249667    -1.450425354876
    C                1.246350556104    -0.426568750333    -1.721640354876
    H                2.284456556104    -0.632268750333    -1.497817354876
    C                0.380934556104    -1.472267750333    -2.038259354876
    H                0.746885556104    -2.489810750333    -2.058472354876
    C               -0.956235443896    -1.206702750333    -2.328650354876
    H               -1.627761443896    -2.018291750333    -2.573302354876
    C               -1.428367443896     0.104675249667    -2.302251354876
    H               -2.465986443896     0.310571249667    -2.526995354876
    C               -0.563208443896     1.150293249667    -1.984715354876
    H               -0.929001443896     2.167980249667    -1.964450354876
    --
    0 1
    C               -0.741334443896    -0.686767750333     1.840888645124
    H               -0.847588443896    -0.511818750333     0.768581645124
    H               -1.392971443896    -1.516031750333     2.114039645124
    C                0.728206556104    -0.943830750333     2.185018645124
    H                1.223437556104    -1.581415750333     1.454434645124
    H                0.791485556104    -1.444044750333     3.152485645124
    C                1.368343556104     0.461566249667     2.275971645124
    H                2.018612556104     0.654237249667     1.424118645124
    H                1.981211556104     0.550735249667     3.171696645124
    C                0.189297556104     1.460743249667     2.302843645124
    H                0.072146556104     1.919713249667     1.320558645124
    H                0.334891556104     2.262248249667     3.025242645124
    C               -1.044474443896     0.603648249667     2.598492645124
    H               -1.975462443896     1.079385249667     2.292579645124
    H               -1.109899443896     0.394327249667     3.669027645124

""")

GEOS['%s-%s-%s' % (dbse, '39-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.764128505031     0.866625546480    -1.918040431921
    H                1.436059505031     1.678388546480    -1.673928431921
    C                1.236236505031    -0.444456453520    -1.945143431921
    H                2.274342505031    -0.650156453520    -1.721320431921
    C                0.370820505031    -1.490155453520    -2.261762431921
    H                0.736771505031    -2.507698453520    -2.281975431921
    C               -0.966349494969    -1.224590453520    -2.552153431921
    H               -1.637875494969    -2.036179453520    -2.796805431921
    C               -1.438481494969     0.086787546480    -2.525754431921
    H               -2.476100494969     0.292683546480    -2.750498431921
    C               -0.573322494969     1.132405546480    -2.208218431921
    H               -0.939115494969     2.150092546480    -2.187953431921
    --
    0 1
    C               -0.730070494969    -0.666845453520     2.089806568079
    H               -0.836324494969    -0.491896453520     1.017500568079
    H               -1.381707494969    -1.496109453520     2.362957568079
    C                0.739470505031    -0.923908453520     2.433936568079
    H                1.234701505031    -1.561493453520     1.703352568079
    H                0.802750505031    -1.424122453520     3.401403568079
    C                1.379607505031     0.481487546480     2.524889568079
    H                2.029876505031     0.674158546480     1.673036568079
    H                1.992475505031     0.570657546480     3.420614568079
    C                0.200561505031     1.480664546480     2.551761568079
    H                0.083410505031     1.939635546480     1.569476568079
    H                0.346155505031     2.282169546480     3.274160568079
    C               -1.033209494969     0.623569546480     2.847410568079
    H               -1.964198494969     1.099307546480     2.541497568079
    H               -1.098635494969     0.414248546480     3.917945568079

""")

GEOS['%s-%s-%s' % (dbse, '39-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.747592108287     0.837379096269    -2.283467759257
    H                1.419523108287     1.649142096269    -2.039355759257
    C                1.219700108287    -0.473702903731    -2.310570759257
    H                2.257806108287    -0.679402903731    -2.086747759257
    C                0.354284108287    -1.519401903731    -2.627189759257
    H                0.720235108287    -2.536944903731    -2.647402759257
    C               -0.982885891713    -1.253836903731    -2.917580759257
    H               -1.654411891713    -2.065425903731    -3.162232759257
    C               -1.455017891713     0.057541096269    -2.891181759257
    H               -2.492636891713     0.263437096269    -3.115925759257
    C               -0.589858891713     1.103159096269    -2.573645759257
    H               -0.955651891713     2.120846096269    -2.553380759257
    --
    0 1
    C               -0.711653891713    -0.634273903731     2.496787240743
    H               -0.817907891713    -0.459324903731     1.424480240743
    H               -1.363290891713    -1.463537903731     2.769938240743
    C                0.757887108287    -0.891336903731     2.840917240743
    H                1.253118108287    -1.528921903731     2.110333240743
    H                0.821167108287    -1.391550903731     3.808383240743
    C                1.398025108287     0.514060096269     2.931870240743
    H                2.048293108287     0.706731096269     2.080017240743
    H                2.010892108287     0.603229096269     3.827595240743
    C                0.218978108287     1.513237096269     2.958742240743
    H                0.101827108287     1.972207096269     1.976456240743
    H                0.364572108287     2.314742096269     3.681140240743
    C               -1.014792891713     0.656142096269     3.254391240743
    H               -1.945781891713     1.131879096269     2.948477240743
    H               -1.080218891713     0.446821096269     4.324926240743

""")

GEOS['%s-%s-%s' % (dbse, '39-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.713298401927     0.776726899497    -3.041303972808
    H                1.385229401927     1.588489899497    -2.797191972808
    C                1.185406401927    -0.534355100503    -3.068406972808
    H                2.223512401927    -0.740055100503    -2.844583972808
    C                0.319990401927    -1.580054100503    -3.385025972808
    H                0.685941401927    -2.597597100503    -3.405238972808
    C               -1.017179598073    -1.314489100503    -3.675416972808
    H               -1.688705598073    -2.126078100503    -3.920068972808
    C               -1.489311598073    -0.003111100503    -3.649017972808
    H               -2.526930598073     0.202784899497    -3.873761972808
    C               -0.624152598073     1.042506899497    -3.331481972808
    H               -0.989945598073     2.060193899497    -3.311216972808
    --
    0 1
    C               -0.673460598073    -0.566724100503     3.340798027192
    H               -0.779714598073    -0.391775100503     2.268492027192
    H               -1.325097598073    -1.395988100503     3.613949027192
    C                0.796080401927    -0.823788100503     3.684928027192
    H                1.291311401927    -1.461373100503     2.954344027192
    H                0.859360401927    -1.324001100503     4.652395027192
    C                1.436218401927     0.581608899497     3.775881027192
    H                2.086486401927     0.774279899497     2.924028027192
    H                2.049085401927     0.670778899497     4.671607027192
    C                0.257171401927     1.580785899497     3.802753027192
    H                0.140020401927     2.039756899497     2.820468027192
    H                0.402765401927     2.382290899497     4.525152027192
    C               -0.976599598073     0.723690899497     4.098402027192
    H               -1.907588598073     1.199427899497     3.792489027192
    H               -1.042025598073     0.514369899497     5.168937027192

""")

GEOS['%s-%s-%s' % (dbse, '40-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.349160948137     0.685425231462    -1.922143004777
    H                0.779342948137     1.677771231462    -1.937276004777
    C                1.179388948137    -0.433672768538    -1.882780004777
    H                2.253717948137    -0.309847768538    -1.866492004777
    C                0.625010948137    -1.712284768538    -1.864698004777
    H                1.269118948137    -2.580439768538    -1.833620004777
    C               -0.759451051863    -1.871973768538    -1.885879004777
    H               -1.189567051863    -2.864041768538    -1.870205004777
    C               -1.589686051863    -0.753071768538    -1.924674004777
    H               -2.663915051863    -0.876947768538    -1.939027004777
    C               -1.035458051863     0.525608231462    -1.942868004777
    H               -1.679767051863     1.393803231462    -1.972922004777
    --
    0 1
    C                0.221946948137     0.641936231462     2.062335995223
    C                1.668195948137     0.820723231462     1.607374995223
    H                2.084051948137     1.754001231462     1.990797995223
    H                2.291594948137     0.000108231462     1.966227995223
    H                1.728804948137     0.840115231462     0.518439995223
    C                0.167554948137     0.610565231462     3.587200995223
    H               -0.858235051863     0.483145231462     3.936288995223
    H                0.762960948137    -0.215121768538     3.979754995223
    H                0.555849948137     1.538450231462     4.010106995223
    C               -0.327433051863    -0.667555768538     1.502039995223
    H               -1.362307051863    -0.817830768538     1.814711995223
    H                0.259022948137    -1.516529768538     1.857850995223
    H               -0.298653051863    -0.668008768538     0.411725995223
    C               -0.619426051863     1.806508231462     1.546130995223
    H               -0.597885051863     1.842538231462     0.456131995223
    H               -1.659215051863     1.702738231462     1.860612995223
    H               -0.243773051863     2.757202231462     1.928237995223

""")

GEOS['%s-%s-%s' % (dbse, '40-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.352464439449     0.688064373963    -2.003969639630
    H                0.782646439449     1.680410373963    -2.019102639630
    C                1.182692439449    -0.431033626037    -1.964606639630
    H                2.257021439449    -0.307208626037    -1.948318639630
    C                0.628314439449    -1.709645626037    -1.946524639630
    H                1.272422439449    -2.577800626037    -1.915446639630
    C               -0.756147560551    -1.869334626037    -1.967705639630
    H               -1.186263560551    -2.861402626037    -1.952031639630
    C               -1.586382560551    -0.750432626037    -2.006500639630
    H               -2.660611560551    -0.874308626037    -2.020853639630
    C               -1.032154560551     0.528247373963    -2.024694639630
    H               -1.676463560551     1.396442373963    -2.054748639630
    --
    0 1
    C                0.218370439449     0.639079373963     2.150919360370
    C                1.664619439449     0.817866373963     1.695958360370
    H                2.080476439449     1.751144373963     2.079381360370
    H                2.288018439449    -0.002748626037     2.054811360370
    H                1.725228439449     0.837257373963     0.607023360370
    C                0.163979439449     0.607708373963     3.675784360370
    H               -0.861810560551     0.480287373963     4.024872360370
    H                0.759385439449    -0.217978626037     4.068338360370
    H                0.552273439449     1.535592373963     4.098690360370
    C               -0.331009560551    -0.670412626037     1.590623360370
    H               -1.365882560551    -0.820687626037     1.903295360370
    H                0.255447439449    -1.519386626037     1.946434360370
    H               -0.302229560551    -0.670865626037     0.500309360370
    C               -0.623002560551     1.803650373963     1.634714360370
    H               -0.601461560551     1.839681373963     0.544715360370
    H               -1.662791560551     1.699881373963     1.949196360370
    H               -0.247349560551     2.754345373963     2.016820360370

""")

GEOS['%s-%s-%s' % (dbse, '40-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.355708035785     0.690655792429    -2.084316021512
    H                0.785890035785     1.683001792429    -2.099449021512
    C                1.185936035785    -0.428442207571    -2.044953021512
    H                2.260265035785    -0.304617207571    -2.028665021512
    C                0.631558035785    -1.707054207571    -2.026871021512
    H                1.275666035785    -2.575209207571    -1.995793021512
    C               -0.752903964215    -1.866743207571    -2.048052021512
    H               -1.183019964215    -2.858811207571    -2.032378021512
    C               -1.583138964215    -0.747841207571    -2.086847021512
    H               -2.657367964215    -0.871717207571    -2.101200021512
    C               -1.028910964215     0.530838792429    -2.105041021512
    H               -1.673219964215     1.399033792429    -2.135095021512
    --
    0 1
    C                0.214859035785     0.636273792429     2.237899978488
    C                1.661108035785     0.815060792429     1.782938978488
    H                2.076964035785     1.748338792429     2.166361978488
    H                2.284507035785    -0.005554207571     2.141792978488
    H                1.721717035785     0.834452792429     0.694004978488
    C                0.160468035785     0.604902792429     3.762765978488
    H               -0.865321964215     0.477481792429     4.111853978488
    H                0.755874035785    -0.220784207571     4.155319978488
    H                0.548762035785     1.532786792429     4.185670978488
    C               -0.334520964215    -0.673218207571     1.677603978488
    H               -1.369394964215    -0.823493207571     1.990275978488
    H                0.251936035785    -1.522192207571     2.033415978488
    H               -0.305740964215    -0.673671207571     0.587289978488
    C               -0.626513964215     1.800845792429     1.721694978488
    H               -0.604972964215     1.836875792429     0.631695978488
    H               -1.666302964215     1.697075792429     2.036176978488
    H               -0.250860964215     2.751539792429     2.103801978488

""")

GEOS['%s-%s-%s' % (dbse, '40-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.358908883110     0.693213091610    -2.163600509238
    H                0.789090883110     1.685559091610    -2.178733509238
    C                1.189136883110    -0.425884908390    -2.124237509238
    H                2.263465883110    -0.302059908390    -2.107949509238
    C                0.634758883110    -1.704496908390    -2.106155509238
    H                1.278866883110    -2.572651908390    -2.075077509238
    C               -0.749703116890    -1.864185908390    -2.127336509238
    H               -1.179819116890    -2.856253908390    -2.111662509238
    C               -1.579938116890    -0.745283908390    -2.166131509238
    H               -2.654167116890    -0.869159908390    -2.180484509238
    C               -1.025710116890     0.533396091610    -2.184325509238
    H               -1.670019116890     1.401591091610    -2.214379509238
    --
    0 1
    C                0.211393883110     0.633505091610     2.323731490762
    C                1.657642883110     0.812293091610     1.868770490762
    H                2.073498883110     1.745571091610     2.252193490762
    H                2.281041883110    -0.008322908390     2.227624490762
    H                1.718251883110     0.831684091610     0.779835490762
    C                0.157002883110     0.602134091610     3.848596490762
    H               -0.868787116890     0.474714091610     4.197684490762
    H                0.752408883110    -0.223552908390     4.241150490762
    H                0.545296883110     1.530019091610     4.271502490762
    C               -0.337986116890    -0.675986908390     1.763435490762
    H               -1.372860116890    -0.826261908390     2.076107490762
    H                0.248470883110    -1.524959908390     2.119246490762
    H               -0.309206116890    -0.676438908390     0.673121490762
    C               -0.629979116890     1.798077091610     1.807526490762
    H               -0.608438116890     1.834107091610     0.717527490762
    H               -1.669768116890     1.694307091610     2.122008490762
    H               -0.254326116890     2.748771091610     2.189633490762

""")

GEOS['%s-%s-%s' % (dbse, '40-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.362067675150     0.695736618684    -2.241844550836
    H                0.792249675150     1.688082618684    -2.256977550836
    C                1.192295675150    -0.423361381316    -2.202481550836
    H                2.266624675150    -0.299536381316    -2.186193550836
    C                0.637917675150    -1.701973381316    -2.184399550836
    H                1.282025675150    -2.570128381316    -2.153321550836
    C               -0.746544324850    -1.861662381316    -2.205580550836
    H               -1.176660324850    -2.853730381316    -2.189906550836
    C               -1.576779324850    -0.742760381316    -2.244375550836
    H               -2.651008324850    -0.866636381316    -2.258728550836
    C               -1.022551324850     0.535919618684    -2.262569550836
    H               -1.666860324850     1.404114618684    -2.292623550836
    --
    0 1
    C                0.207974675150     0.630773618684     2.408436449164
    C                1.654222675150     0.809560618684     1.953475449164
    H                2.070079675150     1.742838618684     2.336898449164
    H                2.277622675150    -0.011055381316     2.312328449164
    H                1.714831675150     0.828951618684     0.864540449164
    C                0.153582675150     0.599402618684     3.933301449164
    H               -0.872207324850     0.471981618684     4.282389449164
    H                0.748988675150    -0.226284381316     4.325855449164
    H                0.541877675150     1.527286618684     4.356207449164
    C               -0.341405324850    -0.678718381316     1.848140449164
    H               -1.376279324850    -0.828993381316     2.160812449164
    H                0.245050675150    -1.527692381316     2.203951449164
    H               -0.312626324850    -0.679171381316     0.757826449164
    C               -0.633398324850     1.795344618684     1.892231449164
    H               -0.611857324850     1.831375618684     0.802232449164
    H               -1.673187324850     1.691575618684     2.206713449164
    H               -0.257746324850     2.746039618684     2.274337449164

""")

GEOS['%s-%s-%s' % (dbse, '40-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.371367013077     0.703166141283    -2.472187161830
    H                0.801549013077     1.695512141283    -2.487320161830
    C                1.201595013077    -0.415931858717    -2.432824161830
    H                2.275924013077    -0.292106858717    -2.416536161830
    C                0.647217013077    -1.694543858717    -2.414742161830
    H                1.291325013077    -2.562698858717    -2.383664161830
    C               -0.737244986923    -1.854232858717    -2.435923161830
    H               -1.167360986923    -2.846300858717    -2.420249161830
    C               -1.567479986923    -0.735330858717    -2.474718161830
    H               -2.641708986923    -0.859206858717    -2.489071161830
    C               -1.013251986923     0.543349141283    -2.492912161830
    H               -1.657560986923     1.411544141283    -2.522966161830
    --
    0 1
    C                0.197907013077     0.622730141283     2.657799838170
    C                1.644156013077     0.801518141283     2.202837838170
    H                2.060013013077     1.734796141283     2.586260838170
    H                2.267555013077    -0.019097858717     2.561691838170
    H                1.704765013077     0.820909141283     1.113903838170
    C                0.143516013077     0.591359141283     4.182664838170
    H               -0.882273986923     0.463939141283     4.531752838170
    H                0.738922013077    -0.234327858717     4.575218838170
    H                0.531810013077     1.519244141283     4.605569838170
    C               -0.351472986923    -0.686761858717     2.097502838170
    H               -1.386345986923    -0.837036858717     2.410174838170
    H                0.234984013077    -1.535734858717     2.453314838170
    H               -0.322692986923    -0.687213858717     1.007188838170
    C               -0.643465986923     1.787302141283     2.141593838170
    H               -0.621924986923     1.823333141283     1.051594838170
    H               -1.683254986923     1.683533141283     2.456076838170
    H               -0.267812986923     2.737997141283     2.523700838170

""")

GEOS['%s-%s-%s' % (dbse, '40-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.386474676123     0.715235956631    -2.846405976667
    H                0.816656676123     1.707581956631    -2.861538976667
    C                1.216702676123    -0.403862043369    -2.807042976667
    H                2.291031676123    -0.280037043369    -2.790754976667
    C                0.662324676123    -1.682474043369    -2.788960976667
    H                1.306432676123    -2.550629043369    -2.757882976667
    C               -0.722137323877    -1.842163043369    -2.810141976667
    H               -1.152253323877    -2.834231043369    -2.794467976667
    C               -1.552372323877    -0.723261043369    -2.848936976667
    H               -2.626601323877    -0.847137043369    -2.863289976667
    C               -0.998144323877     0.555418956631    -2.867130976667
    H               -1.642453323877     1.423613956631    -2.897184976667
    --
    0 1
    C                0.181551676123     0.609663956631     3.062919023333
    C                1.627800676123     0.788450956631     2.607957023333
    H                2.043657676123     1.721728956631     2.991380023333
    H                2.251199676123    -0.032164043369     2.966811023333
    H                1.688409676123     0.807842956631     1.519023023333
    C                0.127160676123     0.578292956631     4.587784023333
    H               -0.898629323877     0.450871956631     4.936872023333
    H                0.722566676123    -0.247394043369     4.980338023333
    H                0.515454676123     1.506176956631     5.010690023333
    C               -0.367828323877    -0.699828043369     2.502623023333
    H               -1.402701323877    -0.850103043369     2.815294023333
    H                0.218628676123    -1.548802043369     2.858434023333
    H               -0.339048323877    -0.700281043369     1.412309023333
    C               -0.659820323877     1.774235956631     2.546713023333
    H               -0.638279323877     1.810265956631     1.456715023333
    H               -1.699610323877     1.670465956631     2.861196023333
    H               -0.284168323877     2.724929956631     2.928820023333

""")

GEOS['%s-%s-%s' % (dbse, '40-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.415942262784     0.738778348920    -3.576321591317
    H                0.846124262784     1.731124348920    -3.591454591317
    C                1.246170262784    -0.380319651080    -3.536958591317
    H                2.320499262784    -0.256494651080    -3.520670591317
    C                0.691792262784    -1.658931651080    -3.518876591317
    H                1.335900262784    -2.527086651080    -3.487798591317
    C               -0.692669737216    -1.818620651080    -3.540057591317
    H               -1.122785737216    -2.810688651080    -3.524383591317
    C               -1.522904737216    -0.699718651080    -3.578852591317
    H               -2.597133737216    -0.823594651080    -3.593205591317
    C               -0.968676737216     0.578961348920    -3.597046591317
    H               -1.612985737216     1.447156348920    -3.627100591317
    --
    0 1
    C                0.149651262784     0.584177348920     3.853106408683
    C                1.595900262784     0.762965348920     3.398144408683
    H                2.011756262784     1.696243348920     3.781567408683
    H                2.219299262784    -0.057650651080     3.756998408683
    H                1.656509262784     0.782356348920     2.309210408683
    C                0.095259262784     0.552806348920     5.377971408683
    H               -0.930530737216     0.425386348920     5.727059408683
    H                0.690665262784    -0.272880651080     5.770525408683
    H                0.483554262784     1.480691348920     5.800876408683
    C               -0.399728737216    -0.725314651080     3.292810408683
    H               -1.434602737216    -0.875589651080     3.605481408683
    H                0.186727262784    -1.574287651080     3.648621408683
    H               -0.370948737216    -0.725766651080     2.202495408683
    C               -0.691721737216     1.748749348920     3.336900408683
    H               -0.670180737216     1.784779348920     2.246901408683
    H               -1.731510737216     1.644979348920     3.651383408683
    H               -0.316068737216     2.699443348920     3.719007408683

""")

GEOS['%s-%s-%s' % (dbse, '41-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.140803673734    -0.955672225840    -1.238870367882
    H               -0.156049673734    -1.964198225840    -1.168547367882
    C               -1.378240673734    -0.335670225840    -1.466972367882
    O               -2.393135673734    -1.001881225840    -1.575739367882
    C               -1.289680673734     1.112096774160    -1.545487367882
    H               -2.188470673734     1.676641774160    -1.723784367882
    C               -0.094901673734     1.718803774160    -1.392505367882
    H                0.021811326266     2.791356774160    -1.440281367882
    N                1.053553326266     1.009734774160    -1.162425367882
    H                1.935661326266     1.475838774160    -1.028847367882
    C                1.095122326266    -0.370259225840    -1.060984367882
    O                2.117298326266    -0.988484225840    -0.836371367882
    --
    0 1
    C                1.187535326266     2.363089774160     2.021972632118
    H                0.756686326266     2.674138774160     1.069190632118
    H                0.553301326266     2.768522774160     2.810220632118
    H                2.170415326266     2.823537774160     2.115094632118
    C                1.260551326266     0.844218774160     2.111490632118
    H                1.690935326266     0.554655774160     3.072781632118
    H                1.935141326266     0.453058774160     1.346372632118
    C               -0.104233673734     0.186052774160     1.956674632118
    H               -0.775477673734     0.554927774160     2.738293632118
    H               -0.551704673734     0.496317774160     1.006773632118
    C               -0.047944673734    -1.334634225840     2.016309632118
    H                0.375661326266    -1.637064225840     2.976945632118
    H                0.643577326266    -1.696513225840     1.251807632118
    C               -1.417227673734    -1.976507225840     1.832536632118
    H               -1.855606673734    -1.703049225840     0.872350632118
    H               -1.360288673734    -3.063275225840     1.876593632118
    H               -2.105190673734    -1.648161225840     2.611769632118

""")

GEOS['%s-%s-%s' % (dbse, '41-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.145618174632    -0.956297052487    -1.296146351241
    H               -0.160864174632    -1.964823052487    -1.225823351241
    C               -1.383055174632    -0.336295052487    -1.524248351241
    O               -2.397950174632    -1.002506052487    -1.633015351241
    C               -1.294495174632     1.111471947513    -1.602763351241
    H               -2.193285174632     1.676016947513    -1.781060351241
    C               -0.099716174632     1.718178947513    -1.449781351241
    H                0.016996825368     2.790731947513    -1.497557351241
    N                1.048738825368     1.009109947513    -1.219701351241
    H                1.930846825368     1.475213947513    -1.086123351241
    C                1.090307825368    -0.370884052487    -1.118260351241
    O                2.112483825368    -0.989109052487    -0.893647351241
    --
    0 1
    C                1.195015825368     2.364060947513     2.110974648759
    H                0.764167825368     2.675109947513     1.158191648759
    H                0.560782825368     2.769493947513     2.899221648759
    H                2.177896825368     2.824507947513     2.204095648759
    C                1.268032825368     0.845189947513     2.200492648759
    H                1.698416825368     0.555626947513     3.161783648759
    H                1.942622825368     0.454028947513     1.435374648759
    C               -0.096752174632     0.187022947513     2.045675648759
    H               -0.767997174632     0.555898947513     2.827295648759
    H               -0.544223174632     0.497287947513     1.095775648759
    C               -0.040463174632    -1.333663052487     2.105311648759
    H                0.383141825368    -1.636094052487     3.065947648759
    H                0.651057825368    -1.695542052487     1.340808648759
    C               -1.409746174632    -1.975536052487     1.921537648759
    H               -1.848125174632    -1.702079052487     0.961352648759
    H               -1.352807174632    -3.062305052487     1.965594648759
    H               -2.097709174632    -1.647190052487     2.700770648759

""")

GEOS['%s-%s-%s' % (dbse, '41-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.150414120116    -0.956919415865    -1.353203780728
    H               -0.165660120116    -1.965445415865    -1.282880780728
    C               -1.387851120116    -0.336917415865    -1.581305780728
    O               -2.402746120116    -1.003128415865    -1.690072780728
    C               -1.299291120116     1.110849584135    -1.659820780728
    H               -2.198081120116     1.675394584135    -1.838117780728
    C               -0.104512120116     1.717556584135    -1.506838780728
    H                0.012200879884     2.790109584135    -1.554614780728
    N                1.043942879884     1.008487584135    -1.276758780728
    H                1.926050879884     1.474591584135    -1.143180780728
    C                1.085511879884    -0.371506415865    -1.175317780728
    O                2.107687879884    -0.989731415865    -0.950704780728
    --
    0 1
    C                1.202468879884     2.365027584135     2.199636219272
    H                0.771619879884     2.676076584135     1.246854219272
    H                0.568234879884     2.770461584135     2.987884219272
    H                2.185348879884     2.825475584135     2.292758219272
    C                1.275485879884     0.846157584135     2.289154219272
    H                1.705868879884     0.556594584135     3.250445219272
    H                1.950074879884     0.454996584135     1.524036219272
    C               -0.089300120116     0.187990584135     2.134338219272
    H               -0.760544120116     0.556866584135     2.915957219272
    H               -0.536770120116     0.498255584135     1.184437219272
    C               -0.033011120116    -1.332696415865     2.193973219272
    H                0.390594879884    -1.635126415865     3.154609219272
    H                0.658510879884    -1.694575415865     1.429471219272
    C               -1.402294120116    -1.974569415865     2.010200219272
    H               -1.840673120116    -1.701111415865     1.050014219272
    H               -1.345355120116    -3.061337415865     2.054257219272
    H               -2.090256120116    -1.646223415865     2.789433219272

""")

GEOS['%s-%s-%s' % (dbse, '41-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.155221964548    -0.957543377802    -1.410400522842
    H               -0.170467964548    -1.966069377802    -1.340077522842
    C               -1.392658964548    -0.337541377802    -1.638502522842
    O               -2.407553964548    -1.003752377802    -1.747269522842
    C               -1.304098964548     1.110225622198    -1.717017522842
    H               -2.202888964548     1.674770622198    -1.895314522842
    C               -0.109319964548     1.716932622198    -1.564035522842
    H                0.007393035452     2.789485622198    -1.611811522842
    N                1.039135035452     1.007863622198    -1.333955522842
    H                1.921243035452     1.473967622198    -1.200377522842
    C                1.080704035452    -0.372130377802    -1.232514522842
    O                2.102880035452    -0.990355377802    -1.007901522842
    --
    0 1
    C                1.209940035452     2.365997622198     2.288514477158
    H                0.779091035452     2.677046622198     1.335732477158
    H                0.575706035452     2.771430622198     3.076762477158
    H                2.192820035452     2.826444622198     2.381636477158
    C                1.282956035452     0.847126622198     2.378033477158
    H                1.713340035452     0.557563622198     3.339324477158
    H                1.957546035452     0.455965622198     1.612914477158
    C               -0.081828964548     0.188959622198     2.223216477158
    H               -0.753072964548     0.557835622198     3.004835477158
    H               -0.529299964548     0.499224622198     1.273316477158
    C               -0.025539964548    -1.331726377802     2.282851477158
    H                0.398065035452    -1.634157377802     3.243488477158
    H                0.665982035452    -1.693605377802     1.518349477158
    C               -1.394822964548    -1.973599377802     2.099078477158
    H               -1.833201964548    -1.700142377802     1.138892477158
    H               -1.337883964548    -3.060368377802     2.143135477158
    H               -2.082785964548    -1.645253377802     2.878311477158

""")

GEOS['%s-%s-%s' % (dbse, '41-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.160992728763    -0.958292279203    -1.479054527077
    H               -0.176238728763    -1.966818279203    -1.408731527077
    C               -1.398429728763    -0.338290279203    -1.707156527077
    O               -2.413324728763    -1.004501279203    -1.815923527077
    C               -1.309869728763     1.109476720797    -1.785671527077
    H               -2.208659728763     1.674021720797    -1.963968527077
    C               -0.115090728763     1.716183720797    -1.632689527077
    H                0.001622271237     2.788736720797    -1.680465527077
    N                1.033364271237     1.007114720797    -1.402609527077
    H                1.915472271237     1.473218720797    -1.269031527077
    C                1.074933271237    -0.372879279203    -1.301168527077
    O                2.097109271237    -0.991104279203    -1.076555527077
    --
    0 1
    C                1.218907271237     2.367161720797     2.395196472923
    H                0.788058271237     2.678209720797     1.442414472923
    H                0.584673271237     2.772594720797     3.183444472923
    H                2.201787271237     2.827608720797     2.488318472923
    C                1.291923271237     0.848290720797     2.484715472923
    H                1.722307271237     0.558727720797     3.446006472923
    H                1.966513271237     0.457129720797     1.719596472923
    C               -0.072861728763     0.190123720797     2.329898472923
    H               -0.744105728763     0.558999720797     3.111517472923
    H               -0.520332728763     0.500388720797     1.379998472923
    C               -0.016572728763    -1.330563279203     2.389533472923
    H                0.407032271237    -1.632993279203     3.350170472923
    H                0.674949271237    -1.692442279203     1.625031472923
    C               -1.385855728763    -1.972436279203     2.205760472923
    H               -1.824234728763    -1.698978279203     1.245574472923
    H               -1.328916728763    -3.059204279203     2.249817472923
    H               -2.073818728763    -1.644090279203     2.984993472923

""")

GEOS['%s-%s-%s' % (dbse, '41-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.177819877051    -0.960475996987    -1.679242398327
    H               -0.193065877051    -1.969001996987    -1.608919398327
    C               -1.415256877051    -0.340473996987    -1.907344398327
    O               -2.430151877051    -1.006684996987    -2.016111398327
    C               -1.326696877051     1.107293003013    -1.985859398327
    H               -2.225486877051     1.671838003013    -2.164156398327
    C               -0.131917877051     1.714000003013    -1.832877398327
    H               -0.015204877051     2.786553003013    -1.880653398327
    N                1.016537122949     1.004931003013    -1.602797398327
    H                1.898645122949     1.471035003013    -1.469219398327
    C                1.058106122949    -0.375062996987    -1.501356398327
    O                2.080282122949    -0.993287996987    -1.276743398327
    --
    0 1
    C                1.245055122949     2.370555003013     2.706270601673
    H                0.814206122949     2.681603003013     1.753487601673
    H                0.610821122949     2.775988003013     3.494517601673
    H                2.227935122949     2.831002003013     2.799391601673
    C                1.318071122949     0.851684003013     2.795788601673
    H                1.748455122949     0.562121003013     3.757079601673
    H                1.992661122949     0.460523003013     2.030670601673
    C               -0.046713877051     0.193517003013     2.640971601673
    H               -0.717957877051     0.562393003013     3.422591601673
    H               -0.494184877051     0.503782003013     1.691071601673
    C                0.009575122949    -1.327169996987     2.700607601673
    H                0.433180122949    -1.629599996987     3.661243601673
    H                0.701096122949    -1.689047996987     1.936104601673
    C               -1.359707877051    -1.969042996987     2.516833601673
    H               -1.798086877051    -1.695584996987     1.556648601673
    H               -1.302768877051    -3.055810996987     2.560890601673
    H               -2.047670877051    -1.640696996987     3.296066601673

""")

GEOS['%s-%s-%s' % (dbse, '41-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.204479740165    -0.963935792930    -1.996409676551
    H               -0.219725740165    -1.972461792930    -1.926086676551
    C               -1.441916740165    -0.343933792930    -2.224511676551
    O               -2.456811740165    -1.010144792930    -2.333278676551
    C               -1.353356740165     1.103833207070    -2.303026676551
    H               -2.252146740165     1.668378207070    -2.481323676551
    C               -0.158577740165     1.710540207070    -2.150044676551
    H               -0.041864740165     2.783093207070    -2.197820676551
    N                0.989877259835     1.001471207070    -1.919964676551
    H                1.871985259835     1.467575207070    -1.786386676551
    C                1.031446259835    -0.378522792930    -1.818523676551
    O                2.053622259835    -0.996747792930    -1.593910676551
    --
    0 1
    C                1.286481259835     2.375931207070     3.199119323449
    H                0.855633259835     2.686979207070     2.246336323449
    H                0.652248259835     2.781364207070     3.987366323449
    H                2.269362259835     2.836378207070     3.292240323449
    C                1.359498259835     0.857060207070     3.288637323449
    H                1.789882259835     0.567497207070     4.249928323449
    H                2.034088259835     0.465899207070     2.523519323449
    C               -0.005286740165     0.198893207070     3.133820323449
    H               -0.676530740165     0.567769207070     3.915440323449
    H               -0.452757740165     0.509158207070     2.183920323449
    C                0.051002259835    -1.321793792930     3.193456323449
    H                0.474607259835    -1.624223792930     4.154092323449
    H                0.742523259835    -1.683671792930     2.428953323449
    C               -1.318280740165    -1.963666792930     3.009682323449
    H               -1.756659740165    -1.690208792930     2.049497323449
    H               -1.261341740165    -3.050434792930     3.053739323449
    H               -2.006243740165    -1.635320792930     3.788915323449

""")

GEOS['%s-%s-%s' % (dbse, '41-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N               -0.255003560235    -0.970492552952    -2.597480699168
    H               -0.270249560235    -1.979018552952    -2.527157699168
    C               -1.492440560235    -0.350490552952    -2.825582699168
    O               -2.507335560235    -1.016701552952    -2.934349699168
    C               -1.403880560235     1.097276447048    -2.904097699168
    H               -2.302670560235     1.661821447048    -3.082394699168
    C               -0.209101560235     1.703983447048    -2.751115699168
    H               -0.092388560235     2.776536447048    -2.798891699168
    N                0.939353439765     0.994914447048    -2.521035699168
    H                1.821461439765     1.461018447048    -2.387457699168
    C                0.980922439765    -0.385079552952    -2.419594699168
    O                2.003098439765    -1.003304552952    -2.194981699168
    --
    0 1
    C                1.364991439765     2.386119447048     4.133128300832
    H                0.934142439765     2.697168447048     3.180345300832
    H                0.730757439765     2.791552447048     4.921375300832
    H                2.347871439765     2.846566447048     4.226249300832
    C                1.438007439765     0.867248447048     4.222646300832
    H                1.868391439765     0.577685447048     5.183937300832
    H                2.112597439765     0.476087447048     3.457528300832
    C                0.073222439765     0.209081447048     4.067829300832
    H               -0.598021560235     0.577957447048     4.849449300832
    H               -0.374248560235     0.519346447048     3.117929300832
    C                0.129511439765    -1.311604552952     4.127465300832
    H                0.553116439765    -1.614035552952     5.088101300832
    H                0.821033439765    -1.673483552952     3.362962300832
    C               -1.239771560235    -1.953477552952     3.943691300832
    H               -1.678150560235    -1.680020552952     2.983506300832
    H               -1.182832560235    -3.040246552952     3.987748300832
    H               -1.927734560235    -1.625131552952     4.722924300832

""")

GEOS['%s-%s-%s' % (dbse, '42-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.173464126480    -0.872836441074    -1.133601875749
    H                0.428132126480    -1.824900441074    -0.907679875749
    C               -1.201314873520    -0.601831441074    -1.197964875749
    O               -2.015911873520    -1.484412441074    -0.990546875749
    C               -1.498980873520     0.783008558926    -1.519887875749
    H               -2.528621873520     1.087503558926    -1.593552875749
    C               -0.490378873520     1.654814558926    -1.722556875749
    H               -0.660700873520     2.693495558926    -1.963611875749
    N                0.823363126480     1.277843558926    -1.630616875749
    H                1.567427126480     1.940731558926    -1.771487875749
    C                1.231995126480    -0.008685441074    -1.321203875749
    O                2.400038126480    -0.329863441074    -1.225568875749
    --
    0 1
    C                1.025828126480    -0.886956441074     2.161301124251
    H                1.479681126480    -0.979012441074     1.173318124251
    H                1.570331126480    -1.550936441074     2.830498124251
    C                1.080142126480     0.578995558926     2.594237124251
    H                2.038432126480     1.051894558926     2.383512124251
    H                0.892027126480     0.660633558926     3.667460124251
    C               -0.087209873520     1.191810558926     1.823465124251
    H                0.187095126480     1.280591558926     0.769994124251
    H               -0.371623873520     2.182951558926     2.175099124251
    C               -1.204153873520     0.154238558926     1.994494124251
    H               -1.929693873520     0.177852558926     1.183122124251
    H               -1.748239873520     0.358935558926     2.916742124251
    C               -0.481117873520    -1.212309441074     2.098151124251
    H               -0.723781873520    -1.856465441074     1.254791124251
    H               -0.796223873520    -1.739711441074     2.997237124251

""")

GEOS['%s-%s-%s' % (dbse, '42-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.171620525804    -0.871831978161    -1.193423344861
    H                0.426288525804    -1.823895978161    -0.967501344861
    C               -1.203158474196    -0.600826978161    -1.257786344861
    O               -2.017755474196    -1.483407978161    -1.050368344861
    C               -1.500824474196     0.784013021839    -1.579709344861
    H               -2.530465474196     1.088508021839    -1.653374344861
    C               -0.492222474196     1.655819021839    -1.782378344861
    H               -0.662544474196     2.694500021839    -2.023433344861
    N                0.821519525804     1.278848021839    -1.690438344861
    H                1.565583525804     1.941736021839    -1.831309344861
    C                1.230151525804    -0.007680978161    -1.381025344861
    O                2.398194525804    -0.328858978161    -1.285390344861
    --
    0 1
    C                1.028775525804    -0.888561978161     2.256931655139
    H                1.482628525804    -0.980617978161     1.268948655139
    H                1.573278525804    -1.552541978161     2.926129655139
    C                1.083089525804     0.577390021839     2.689867655139
    H                2.041379525804     1.050289021839     2.479143655139
    H                0.894973525804     0.659028021839     3.763090655139
    C               -0.084262474196     1.190204021839     1.919096655139
    H                0.190041525804     1.278986021839     0.865624655139
    H               -0.368676474196     2.181346021839     2.270730655139
    C               -1.201207474196     0.152633021839     2.090124655139
    H               -1.926746474196     0.176246021839     1.278752655139
    H               -1.745292474196     0.357330021839     3.012373655139
    C               -0.478170474196    -1.213914978161     2.193781655139
    H               -0.720835474196    -1.858070978161     1.350421655139
    H               -0.793276474196    -1.741317978161     3.092867655139

""")

GEOS['%s-%s-%s' % (dbse, '42-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.169805786795    -0.870843062064    -1.252311933522
    H                0.424473786795    -1.822907062064    -1.026389933522
    C               -1.204973213205    -0.599838062064    -1.316674933522
    O               -2.019570213205    -1.482419062064    -1.109256933522
    C               -1.502639213205     0.785001937936    -1.638597933522
    H               -2.532280213205     1.089496937936    -1.712262933522
    C               -0.494037213205     1.656807937936    -1.841266933522
    H               -0.664359213205     2.695488937936    -2.082321933522
    N                0.819704786795     1.279836937936    -1.749326933522
    H                1.563768786795     1.942724937936    -1.890197933522
    C                1.228336786795    -0.006692062064    -1.439913933522
    O                2.396379786795    -0.327870062064    -1.344278933522
    --
    0 1
    C                1.031676786795    -0.890143062064     2.351071066478
    H                1.485529786795    -0.982199062064     1.363089066478
    H                1.576178786795    -1.554123062064     3.020269066478
    C                1.085989786795     0.575808937936     2.784007066478
    H                2.044279786795     1.048707937936     2.573283066478
    H                0.897874786795     0.657446937936     3.857230066478
    C               -0.081361213205     1.188623937936     2.013236066478
    H                0.192942786795     1.277404937936     0.959764066478
    H               -0.365776213205     2.179764937936     2.364870066478
    C               -1.198306213205     0.151051937936     2.184264066478
    H               -1.923845213205     0.174665937936     1.372892066478
    H               -1.742391213205     0.355748937936     3.106513066478
    C               -0.475269213205    -1.215496062064     2.287921066478
    H               -0.717934213205    -1.859652062064     1.444562066478
    H               -0.790375213205    -1.742898062064     3.187008066478

""")

GEOS['%s-%s-%s' % (dbse, '42-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.168010987109    -0.869865223312    -1.310550122852
    H                0.422678987109    -1.821929223312    -1.084628122852
    C               -1.206768012891    -0.598860223312    -1.374913122852
    O               -2.021365012891    -1.481441223312    -1.167495122852
    C               -1.504434012891     0.785979776688    -1.696836122852
    H               -2.534075012891     1.090474776688    -1.770501122852
    C               -0.495832012891     1.657785776688    -1.899505122852
    H               -0.666154012891     2.696466776688    -2.140560122852
    N                0.817909987109     1.280814776688    -1.807565122852
    H                1.561973987109     1.943702776688    -1.948436122852
    C                1.226541987109    -0.005714223312    -1.498152122852
    O                2.394584987109    -0.326892223312    -1.402517122852
    --
    0 1
    C                1.034545987109    -0.891706223312     2.444170877148
    H                1.488397987109    -0.983762223312     1.456187877148
    H                1.579047987109    -1.555686223312     3.113367877148
    C                1.088858987109     0.574245776688     2.877106877148
    H                2.047148987109     1.047144776688     2.666381877148
    H                0.900743987109     0.655883776688     3.950329877148
    C               -0.078492012891     1.187060776688     2.106335877148
    H                0.195811987109     1.275841776688     1.052863877148
    H               -0.362907012891     2.178201776688     2.457969877148
    C               -1.195437012891     0.149488776688     2.277363877148
    H               -1.920976012891     0.173102776688     1.465991877148
    H               -1.739522012891     0.354185776688     3.199611877148
    C               -0.472400012891    -1.217059223312     2.381020877148
    H               -0.715065012891    -1.861215223312     1.537660877148
    H               -0.787507012891    -1.744462223312     3.280106877148

""")

GEOS['%s-%s-%s' % (dbse, '42-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.166236797291    -0.868898478508    -1.368120678866
    H                0.420904797291    -1.820962478508    -1.142198678866
    C               -1.208542202709    -0.597893478508    -1.432483678866
    O               -2.023139202709    -1.480474478508    -1.225065678866
    C               -1.506208202709     0.786946521492    -1.754406678866
    H               -2.535849202709     1.091441521492    -1.828071678866
    C               -0.497606202709     1.658752521492    -1.957075678866
    H               -0.667928202709     2.697433521492    -2.198130678866
    N                0.816135797291     1.281781521492    -1.865135678866
    H                1.560199797291     1.944669521492    -2.006006678866
    C                1.224767797291    -0.004747478508    -1.555722678866
    O                2.392810797291    -0.325925478508    -1.460087678866
    --
    0 1
    C                1.037381797291    -0.893251478508     2.536203321134
    H                1.491234797291    -0.985307478508     1.548220321134
    H                1.581884797291    -1.557231478508     3.205400321134
    C                1.091695797291     0.572700521492     2.969139321134
    H                2.049985797291     1.045599521492     2.758414321134
    H                0.903579797291     0.654338521492     4.042362321134
    C               -0.075656202709     1.185514521492     2.198368321134
    H                0.198647797291     1.274296521492     1.144896321134
    H               -0.360070202709     2.176656521492     2.550002321134
    C               -1.192600202709     0.147943521492     2.369396321134
    H               -1.918140202709     0.171556521492     1.558024321134
    H               -1.736686202709     0.352640521492     3.291644321134
    C               -0.469564202709    -1.218604478508     2.473053321134
    H               -0.712229202709    -1.862760478508     1.629693321134
    H               -0.784670202709    -1.746007478508     3.372139321134

""")

GEOS['%s-%s-%s' % (dbse, '42-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.161000367648    -0.866045236164    -1.538038169495
    H                0.415668367648    -1.818109236164    -1.312116169495
    C               -1.213778632352    -0.595040236164    -1.602401169495
    O               -2.028375632352    -1.477621236164    -1.394983169495
    C               -1.511444632352     0.789799763836    -1.924324169495
    H               -2.541085632352     1.094294763836    -1.997989169495
    C               -0.502842632352     1.661605763836    -2.126993169495
    H               -0.673164632352     2.700286763836    -2.368048169495
    N                0.810899367648     1.284634763836    -2.035053169495
    H                1.554963367648     1.947522763836    -2.175924169495
    C                1.219531367648    -0.001894236164    -1.725640169495
    O                2.387574367648    -0.323072236164    -1.630005169495
    --
    0 1
    C                1.045753367648    -0.897812236164     2.807833830505
    H                1.499605367648    -0.989869236164     1.819850830505
    H                1.590255367648    -1.561793236164     3.477031830505
    C                1.100066367648     0.568139763836     3.240769830505
    H                2.058356367648     1.041037763836     3.030045830505
    H                0.911951367648     0.649776763836     4.313992830505
    C               -0.067284632352     1.180953763836     2.469998830505
    H                0.207019367648     1.269735763836     1.416526830505
    H               -0.351699632352     2.172094763836     2.821632830505
    C               -1.184229632352     0.143381763836     2.641026830505
    H               -1.909768632352     0.166995763836     1.829654830505
    H               -1.728315632352     0.348078763836     3.563275830505
    C               -0.461193632352    -1.223166236164     2.744683830505
    H               -0.703857632352    -1.867322236164     1.901323830505
    H               -0.776299632352    -1.750568236164     3.643769830505

""")

GEOS['%s-%s-%s' % (dbse, '42-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.152464692262    -0.861394275021    -1.815012784465
    H                0.407132692262    -1.813458275021    -1.589090784465
    C               -1.222314307738    -0.590389275021    -1.879375784465
    O               -2.036911307738    -1.472970275021    -1.671957784465
    C               -1.519980307738     0.794450724979    -2.201298784465
    H               -2.549621307738     1.098945724979    -2.274963784465
    C               -0.511378307738     1.666256724979    -2.403967784465
    H               -0.681700307738     2.704937724979    -2.645022784465
    N                0.802363692262     1.289285724979    -2.312027784465
    H                1.546427692262     1.952173724979    -2.452898784465
    C                1.210995692262     0.002756724979    -2.002614784465
    O                2.379038692262    -0.318421275021    -1.906979784465
    --
    0 1
    C                1.059398692262    -0.905247275021     3.250606215535
    H                1.513250692262    -0.997304275021     2.262623215535
    H                1.603900692262    -1.569228275021     3.919803215535
    C                1.113711692262     0.560704724979     3.683542215535
    H                2.072001692262     1.033602724979     3.472817215535
    H                0.925596692262     0.642341724979     4.756765215535
    C               -0.053640307738     1.173518724979     2.912771215535
    H                0.220664692262     1.262300724979     1.859299215535
    H               -0.338054307738     2.164659724979     3.264405215535
    C               -1.170584307738     0.135946724979     3.083799215535
    H               -1.896123307738     0.159560724979     2.272427215535
    H               -1.714670307738     0.340643724979     4.006048215535
    C               -0.447548307738    -1.230601275021     3.187456215535
    H               -0.690212307738    -1.874757275021     2.344096215535
    H               -0.762654307738    -1.758003275021     4.086542215535

""")

GEOS['%s-%s-%s' % (dbse, '42-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.135727323404    -0.852274373635    -2.358124310979
    H                0.390395323404    -1.804338373635    -2.132202310979
    C               -1.239051676596    -0.581269373635    -2.422487310979
    O               -2.053648676596    -1.463850373635    -2.215069310979
    C               -1.536717676596     0.803570626365    -2.744410310979
    H               -2.566358676596     1.108065626365    -2.818075310979
    C               -0.528115676596     1.675376626365    -2.947079310979
    H               -0.698437676596     2.714057626365    -3.188134310979
    N                0.785626323404     1.298405626365    -2.855139310979
    H                1.529690323404     1.961293626365    -2.996010310979
    C                1.194258323404     0.011876626365    -2.545726310979
    O                2.362301323404    -0.309301373635    -2.450091310979
    --
    0 1
    C                1.086154323404    -0.919826373635     4.118825689021
    H                1.540007323404    -1.011883373635     3.130843689021
    H                1.630657323404    -1.583807373635     4.788023689021
    C                1.140468323404     0.546125626365     4.551761689021
    H                2.098758323404     1.019023626365     4.341037689021
    H                0.952352323404     0.627762626365     5.624984689021
    C               -0.026883676596     1.158939626365     3.780990689021
    H                0.247420323404     1.247721626365     2.727518689021
    H               -0.311297676596     2.150080626365     4.132624689021
    C               -1.143827676596     0.121367626365     3.952018689021
    H               -1.869367676596     0.144981626365     3.140646689021
    H               -1.687913676596     0.326064626365     4.874267689021
    C               -0.420791676596    -1.245180373635     4.055675689021
    H               -0.663456676596    -1.889336373635     3.212315689021
    H               -0.735897676596    -1.772582373635     4.954762689021

""")

GEOS['%s-%s-%s' % (dbse, '43-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.327824207720    -0.875953308520    -1.345755359991
    H                0.110933207720    -1.836541308520    -1.117270359991
    C               -0.730712792280    -0.112376308520    -1.857007359991
    O               -1.829028792280    -0.613443308520    -2.028078359991
    C               -0.365131792280     1.266234691480    -2.132034359991
    H               -1.108000792280     1.926769691480    -2.544511359991
    C                0.885285207720     1.692894691480    -1.861138359991
    H                1.203593207720     2.708830691480    -2.041838359991
    N                1.835873207720     0.861561691480    -1.332589359991
    H                2.757079207720     1.198860691480    -1.108243359991
    C                1.604526207720    -0.468218308520    -1.023359359991
    O                2.445549207720    -1.188964308520    -0.523084359991
    --
    0 1
    C               -0.830928792280    -0.049365308520     2.325301640009
    C               -2.147674792280     0.691923691480     2.540174640009
    H               -2.854745792280     0.073717691480     3.095036640009
    H               -1.988174792280     1.612107691480     3.104359640009
    H               -2.605384792280     0.953420691480     1.585020640009
    C               -0.215491792280    -0.402803308520     3.676578640009
    H                0.727670207720    -0.935176308520     3.545797640009
    H               -0.018470792280     0.496938691480     4.261381640009
    H               -0.886785792280    -1.040540308520     4.253315640009
    C                0.134196207720     0.844112691480     1.549512640009
    H                1.082766207720     0.331517691480     1.379158640009
    H                0.339486207720     1.762248691480     2.102938640009
    H               -0.288199792280     1.122598691480     0.582047640009
    C               -1.092245792280    -1.329399308520     1.534936640009
    H               -1.555983792280    -1.107566308520     0.572311640009
    H               -0.159669792280    -1.867798308520     1.355299640009
    H               -1.762753792280    -1.991213308520     2.085121640009

""")

GEOS['%s-%s-%s' % (dbse, '43-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.348734113357    -0.874714444679    -1.404310037531
    H                0.131843113357    -1.835302444679    -1.175825037531
    C               -0.709802886643    -0.111137444679    -1.915562037531
    O               -1.808118886643    -0.612204444679    -2.086633037531
    C               -0.344221886643     1.267473555321    -2.190589037531
    H               -1.087090886643     1.928008555321    -2.603066037531
    C                0.906195113357     1.694133555321    -1.919693037531
    H                1.224503113357     2.710069555321    -2.100393037531
    N                1.856783113357     0.862800555321    -1.391144037531
    H                2.777989113357     1.200099555321    -1.166798037531
    C                1.625436113357    -0.466979444679    -1.081914037531
    O                2.466459113357    -1.187725444679    -0.581639037531
    --
    0 1
    C               -0.863420886643    -0.051290444679     2.416289962469
    C               -2.180166886643     0.689998555321     2.631162962469
    H               -2.887237886643     0.071792555321     3.186025962469
    H               -2.020666886643     1.610183555321     3.195347962469
    H               -2.637875886643     0.951495555321     1.676009962469
    C               -0.247983886643    -0.404728444679     3.767567962469
    H                0.695178113357    -0.937101444679     3.636785962469
    H               -0.050962886643     0.495013555321     4.352369962469
    H               -0.919277886643    -1.042465444679     4.344303962469
    C                0.101704113357     0.842187555321     1.640500962469
    H                1.050274113357     0.329592555321     1.470146962469
    H                0.306994113357     1.760324555321     2.193927962469
    H               -0.320691886643     1.120674555321     0.673036962469
    C               -1.124737886643    -1.331324444679     1.625924962469
    H               -1.588475886643    -1.109490444679     0.663300962469
    H               -0.192160886643    -1.869723444679     1.446288962469
    H               -1.795245886643    -1.993138444679     2.176109962469

""")

GEOS['%s-%s-%s' % (dbse, '43-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.369298002688    -0.873496289559    -1.461896193582
    H                0.152407002688    -1.834084289559    -1.233411193582
    C               -0.689238997312    -0.109919289559    -1.973148193582
    O               -1.787554997312    -0.610986289559    -2.144219193582
    C               -0.323657997312     1.268691710441    -2.248175193582
    H               -1.066526997312     1.929226710441    -2.660652193582
    C                0.926759002688     1.695351710441    -1.977279193582
    H                1.245067002688     2.711287710441    -2.157979193582
    N                1.877347002688     0.864018710441    -1.448730193582
    H                2.798553002688     1.201317710441    -1.224384193582
    C                1.646000002688    -0.465761289559    -1.139500193582
    O                2.487023002688    -1.186507289559    -0.639225193582
    --
    0 1
    C               -0.895374997312    -0.053183289559     2.505773806418
    C               -2.212120997312     0.688105710441     2.720646806418
    H               -2.919192997312     0.069899710441     3.275508806418
    H               -2.052621997312     1.608289710441     3.284831806418
    H               -2.669830997312     0.949602710441     1.765492806418
    C               -0.279937997312    -0.406621289559     3.857050806418
    H                0.663224002688    -0.938994289559     3.726269806418
    H               -0.082917997312     0.493120710441     4.441852806418
    H               -0.951232997312    -1.044358289559     4.433787806418
    C                0.069749002688     0.840294710441     1.729984806418
    H                1.018320002688     0.327699710441     1.559630806418
    H                0.275039002688     1.758430710441     2.283410806418
    H               -0.352645997312     1.118780710441     0.762519806418
    C               -1.156691997312    -1.333217289559     1.715408806418
    H               -1.620430997312    -1.111384289559     0.752783806418
    H               -0.224115997312    -1.871616289559     1.535771806418
    H               -1.827199997312    -1.995031289559     2.265593806418

""")

GEOS['%s-%s-%s' % (dbse, '43-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.389615493823    -0.872292573287    -1.518791926529
    H                0.172724493823    -1.832880573287    -1.290306926529
    C               -0.668921506177    -0.108715573287    -2.030043926529
    O               -1.767237506177    -0.609782573287    -2.201114926529
    C               -0.303340506177     1.269895426713    -2.305070926529
    H               -1.046209506177     1.930430426713    -2.717547926529
    C                0.947076493823     1.696555426713    -2.034174926529
    H                1.265384493823     2.712491426713    -2.214874926529
    N                1.897664493823     0.865222426713    -1.505625926529
    H                2.818870493823     1.202521426713    -1.281279926529
    C                1.666317493823    -0.464557573287    -1.196395926529
    O                2.507340493823    -1.185303573287    -0.696120926529
    --
    0 1
    C               -0.926946506177    -0.055053573287     2.594184073471
    C               -2.243692506177     0.686235426713     2.809057073471
    H               -2.950764506177     0.068028426713     3.363920073471
    H               -2.084193506177     1.606419426713     3.373242073471
    H               -2.701402506177     0.947732426713     1.853904073471
    C               -0.311509506177    -0.408491573287     3.945462073471
    H                0.631652493823    -0.940864573287     3.814680073471
    H               -0.114489506177     0.491250426713     4.530264073471
    H               -0.982804506177    -1.046228573287     4.522199073471
    C                0.038177493823     0.838424426713     1.818395073471
    H                0.986748493823     0.325829426713     1.648041073471
    H                0.243467493823     1.756560426713     2.371822073471
    H               -0.384217506177     1.116910426713     0.850931073471
    C               -1.188263506177    -1.335088573287     1.803820073471
    H               -1.652002506177    -1.113254573287     0.841195073471
    H               -0.255687506177    -1.873486573287     1.624183073471
    H               -1.858771506177    -1.996901573287     2.354005073471

""")

GEOS['%s-%s-%s' % (dbse, '43-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.409692851673    -0.871103105305    -1.575014991740
    H                0.192801851673    -1.831691105305    -1.346529991740
    C               -0.648844148327    -0.107526105305    -2.086266991740
    O               -1.747160148327    -0.608593105305    -2.257337991740
    C               -0.283263148327     1.271084894695    -2.361293991740
    H               -1.026132148327     1.931619894695    -2.773770991740
    C                0.967153851673     1.697744894695    -2.090397991740
    H                1.285461851673     2.713680894695    -2.271097991740
    N                1.917741851673     0.866411894695    -1.561848991740
    H                2.838947851673     1.203710894695    -1.337502991740
    C                1.686394851673    -0.463368105305    -1.252618991740
    O                2.527417851673    -1.184114105305    -0.752343991740
    --
    0 1
    C               -0.958145148327    -0.056902105305     2.681550008260
    C               -2.274891148327     0.684386894695     2.896423008260
    H               -2.981962148327     0.066180894695     3.451285008260
    H               -2.115391148327     1.604571894695     3.460608008260
    H               -2.732600148327     0.945883894695     1.941269008260
    C               -0.342708148327    -0.410340105305     4.032827008260
    H                0.600453851673    -0.942713105305     3.902045008260
    H               -0.145687148327     0.489401894695     4.617629008260
    H               -1.014002148327    -1.048077105305     4.609564008260
    C                0.006979851673     0.836575894695     1.905761008260
    H                0.955549851673     0.323980894695     1.735407008260
    H                0.212269851673     1.754711894695     2.459187008260
    H               -0.415416148327     1.115062894695     0.938296008260
    C               -1.219462148327    -1.336936105305     1.891185008260
    H               -1.683200148327    -1.115103105305     0.928560008260
    H               -0.286885148327    -1.875335105305     1.711548008260
    H               -1.889970148327    -1.998750105305     2.441370008260

""")

GEOS['%s-%s-%s' % (dbse, '43-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.468884178686    -0.867596409614    -1.740770635582
    H                0.251993178686    -1.828184409614    -1.512285635582
    C               -0.589652821314    -0.104019409614    -2.252022635582
    O               -1.687968821314    -0.605086409614    -2.423093635582
    C               -0.224071821314     1.274591590386    -2.527049635582
    H               -0.966940821314     1.935126590386    -2.939526635582
    C                1.026345178686     1.701251590386    -2.256153635582
    H                1.344653178686     2.717187590386    -2.436853635582
    N                1.976933178686     0.869918590386    -1.727604635582
    H                2.898139178686     1.207217590386    -1.503258635582
    C                1.745586178686    -0.459861409614    -1.418374635582
    O                2.586609178686    -1.180607409614    -0.918099635582
    --
    0 1
    C               -1.050122821314    -0.062351409614     2.939119364418
    C               -2.366868821314     0.678937590386     3.153991364418
    H               -3.073939821314     0.060731590386     3.708854364418
    H               -2.207368821314     1.599122590386     3.718176364418
    H               -2.824578821314     0.940434590386     2.198838364418
    C               -0.434685821314    -0.415788409614     4.290396364418
    H                0.508476178686    -0.948162409614     4.159614364418
    H               -0.237664821314     0.483952590386     4.875198364418
    H               -1.105979821314    -1.053525409614     4.867133364418
    C               -0.084998821314     0.831126590386     2.163329364418
    H                0.863572178686     0.318532590386     1.992975364418
    H                0.120292178686     1.749263590386     2.716756364418
    H               -0.507393821314     1.109613590386     1.195865364418
    C               -1.311439821314    -1.342385409614     2.148754364418
    H               -1.775177821314    -1.120551409614     1.186129364418
    H               -0.378863821314    -1.880784409614     1.969117364418
    H               -1.981947821314    -2.004199409614     2.698939364418

""")

GEOS['%s-%s-%s' % (dbse, '43-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.565246624216    -0.861887723319    -2.010617718985
    H                0.348355624216    -1.822475723319    -1.782132718985
    C               -0.493290375784    -0.098310723319    -2.521869718985
    O               -1.591606375784    -0.599377723319    -2.692940718985
    C               -0.127709375784     1.280300276681    -2.796896718985
    H               -0.870578375784     1.940835276681    -3.209373718985
    C                1.122707624216     1.706960276681    -2.526000718985
    H                1.441015624216     2.722896276681    -2.706700718985
    N                2.073295624216     0.875627276681    -1.997451718985
    H                2.994501624216     1.212926276681    -1.773105718985
    C                1.841948624216    -0.454152723319    -1.688221718985
    O                2.682971624216    -1.174898723319    -1.187946718985
    --
    0 1
    C               -1.199861375784    -0.071221723319     3.358436281015
    C               -2.516607375784     0.670067276681     3.573309281015
    H               -3.223678375784     0.051860276681     4.128171281015
    H               -2.357107375784     1.590251276681     4.137494281015
    H               -2.974316375784     0.931564276681     2.618155281015
    C               -0.584424375784    -0.424659723319     4.709714281015
    H                0.358737624216    -0.957032723319     4.578932281015
    H               -0.387403375784     0.475082276681     5.294516281015
    H               -1.255718375784    -1.062396723319     5.286450281015
    C               -0.234736375784     0.822256276681     2.582647281015
    H                0.713833624216     0.309661276681     2.412293281015
    H               -0.029446375784     1.740392276681     3.136073281015
    H               -0.657132375784     1.100742276681     1.615183281015
    C               -1.461178375784    -1.351256723319     2.568071281015
    H               -1.924916375784    -1.129422723319     1.605447281015
    H               -0.528601375784    -1.889655723319     2.388434281015
    H               -2.131686375784    -2.013070723319     3.118256281015

""")

GEOS['%s-%s-%s' % (dbse, '43-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                0.753566044470    -0.850731176825    -2.537975459237
    H                0.536675044470    -1.811319176825    -2.309490459237
    C               -0.304970955530    -0.087154176825    -3.049227459237
    O               -1.403286955530    -0.588221176825    -3.220298459237
    C                0.060610044470     1.291456823175    -3.324254459237
    H               -0.682258955530     1.951991823175    -3.736731459237
    C                1.311027044470     1.718116823175    -3.053358459237
    H                1.629335044470     2.734052823175    -3.234058459237
    N                2.261615044470     0.886783823175    -2.524809459237
    H                3.182821044470     1.224082823175    -2.300463459237
    C                2.030268044470    -0.442996176825    -2.215579459237
    O                2.871291044470    -1.163742176825    -1.715304459237
    --
    0 1
    C               -1.492491955530    -0.088558176825     4.177901540763
    C               -2.809237955530     0.652730823175     4.392774540763
    H               -3.516309955530     0.034524823175     4.947637540763
    H               -2.649738955530     1.572914823175     4.956959540763
    H               -3.266947955530     0.914227823175     3.437621540763
    C               -0.877055955530    -0.441996176825     5.529179540763
    H                0.066107044470    -0.974369176825     5.398397540763
    H               -0.680034955530     0.457745823175     6.113981540763
    H               -1.548349955530    -1.079733176825     6.105916540763
    C               -0.527367955530     0.804919823175     3.402112540763
    H                0.421203044470     0.292324823175     3.231758540763
    H               -0.322077955530     1.723055823175     3.955539540763
    H               -0.949762955530     1.083405823175     2.434648540763
    C               -1.753808955530    -1.368592176825     3.387536540763
    H               -2.217547955530    -1.146759176825     2.424912540763
    H               -0.821232955530    -1.906991176825     3.207900540763
    H               -2.424316955530    -2.030406176825     3.937721540763

""")

GEOS['%s-%s-%s' % (dbse, '44-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666376944546     0.164415358648    -2.521316019984
    H                1.228858944546    -0.349278641352    -1.754793019984
    H                1.228012944546     0.677812358648    -3.288663019984
    C               -0.665997055454     0.163577358648    -2.521441019984
    H               -1.227945055454    -0.350894641352    -1.754950019984
    H               -1.228207055454     0.676250358648    -3.288801019984
    --
    0 1
    C               -2.532765055454    -0.397469641352     0.964546980016
    H               -2.562258055454    -1.010490641352     0.063356980016
    H               -2.568899055454    -1.071689641352     1.820163980016
    H               -3.433936055454     0.213547358648     0.981792980016
    C               -1.271329055454     0.455206358648     1.000364980016
    H               -1.271734055454     1.075299358648     1.899758980016
    H               -1.262940055454     1.142114358648     0.151304980016
    C               -0.000054055454    -0.382351641352     0.973421980016
    H               -0.000208055454    -1.069024641352     1.825253980016
    H                0.000086944546    -1.009929641352     0.076779980016
    C                1.271165944546     0.455234358648     1.000826980016
    H                1.271438944546     1.075045358648     1.900311980016
    H                1.262970944546     1.142309358648     0.151918980016
    C                2.532617944546    -0.397489641352     0.965001980016
    H                2.562240944546    -1.010346641352     0.063692980016
    H                3.433795944546     0.213446358648     0.982579980016
    H                2.568535944546    -1.071945641352     1.820507980016

""")

GEOS['%s-%s-%s' % (dbse, '44-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666385325165     0.171667716462    -2.632877062287
    H                1.228867325165    -0.342026283538    -1.866354062287
    H                1.228021325165     0.685064716462    -3.400224062287
    C               -0.665988674835     0.170829716462    -2.633002062287
    H               -1.227936674835    -0.343642283538    -1.866511062287
    H               -1.228198674835     0.683502716462    -3.400362062287
    --
    0 1
    C               -2.532768674835    -0.400289283538     1.007923937713
    H               -2.562261674835    -1.013310283538     0.106733937713
    H               -2.568902674835    -1.074510283538     1.863540937713
    H               -3.433939674835     0.210726716462     1.025169937713
    C               -1.271331674835     0.452385716462     1.043741937713
    H               -1.271737674835     1.072479716462     1.943135937713
    H               -1.262943674835     1.139294716462     0.194681937713
    C               -0.000057674835    -0.385171283538     1.016798937713
    H               -0.000211674835    -1.071844283538     1.868630937713
    H                0.000083325165    -1.012749283538     0.120155937713
    C                1.271163325165     0.452414716462     1.044202937713
    H                1.271436325165     1.072225716462     1.943688937713
    H                1.262968325165     1.139489716462     0.195295937713
    C                2.532614325165    -0.400309283538     1.008378937713
    H                2.562237325165    -1.013166283538     0.107069937713
    H                3.433792325165     0.210626716462     1.025956937713
    H                2.568532325165    -1.074765283538     1.863884937713

""")

GEOS['%s-%s-%s' % (dbse, '44-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666393385559     0.178749285404    -2.741808647441
    H                1.228875385559    -0.334944714596    -1.975285647441
    H                1.228029385559     0.692146285404    -3.509155647441
    C               -0.665980614441     0.177911285404    -2.741933647441
    H               -1.227928614441    -0.336560714596    -1.975442647441
    H               -1.228190614441     0.690584285404    -3.509293647441
    --
    0 1
    C               -2.532771614441    -0.403042714596     1.050278352559
    H               -2.562264614441    -1.016063714596     0.149088352559
    H               -2.568905614441    -1.077263714596     1.905895352559
    H               -3.433942614441     0.207973285404     1.067524352559
    C               -1.271334614441     0.449632285404     1.086096352559
    H               -1.271740614441     1.069726285404     1.985490352559
    H               -1.262946614441     1.136541285404     0.237035352559
    C               -0.000060614441    -0.387924714596     1.059153352559
    H               -0.000214614441    -1.074597714596     1.910985352559
    H                0.000080385559    -1.015502714596     0.162510352559
    C                1.271159385559     0.449661285404     1.086557352559
    H                1.271432385559     1.069472285404     1.986043352559
    H                1.262964385559     1.136736285404     0.237650352559
    C                2.532611385559    -0.403062714596     1.050733352559
    H                2.562234385559    -1.015919714596     0.149424352559
    H                3.433789385559     0.207873285404     1.068311352559
    H                2.568529385559    -1.077518714596     1.906239352559

""")

GEOS['%s-%s-%s' % (dbse, '44-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666401545671     0.185706427838    -2.848826382951
    H                1.228883545671    -0.327987572162    -2.082303382951
    H                1.228037545671     0.699103427838    -3.616173382951
    C               -0.665972454329     0.184868427838    -2.848951382951
    H               -1.227920454329    -0.329603572162    -2.082460382951
    H               -1.228182454329     0.697541427838    -3.616311382951
    --
    0 1
    C               -2.532774454329    -0.405747572162     1.091888617049
    H               -2.562267454329    -1.018768572162     0.190698617049
    H               -2.568908454329    -1.079968572162     1.947505617049
    H               -3.433945454329     0.205268427838     1.109134617049
    C               -1.271338454329     0.446927427838     1.127706617049
    H               -1.271743454329     1.067021427838     2.027100617049
    H               -1.262949454329     1.133835427838     0.278645617049
    C               -0.000063454329    -0.390629572162     1.100763617049
    H               -0.000217454329    -1.077302572162     1.952595617049
    H                0.000077545671    -1.018207572162     0.204120617049
    C                1.271156545671     0.446956427838     1.128167617049
    H                1.271429545671     1.066767427838     2.027653617049
    H                1.262961545671     1.134031427838     0.279260617049
    C                2.532607545671    -0.405768572162     1.092343617049
    H                2.562231545671    -1.018624572162     0.191034617049
    H                3.433786545671     0.205168427838     1.109921617049
    H                2.568526545671    -1.080224572162     1.947849617049

""")

GEOS['%s-%s-%s' % (dbse, '44-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666409346234     0.192549254485    -2.954086506892
    H                1.228891346234    -0.321144745515    -2.187563506892
    H                1.228045346234     0.705946254485    -3.721433506892
    C               -0.665964653766     0.191711254485    -2.954211506892
    H               -1.227912653766    -0.322760745515    -2.187720506892
    H               -1.228174653766     0.704384254485    -3.721571506892
    --
    0 1
    C               -2.532777653766    -0.408408745515     1.132815493108
    H               -2.562270653766    -1.021429745515     0.231625493108
    H               -2.568911653766    -1.082628745515     1.988432493108
    H               -3.433948653766     0.202608254485     1.150061493108
    C               -1.271341653766     0.444267254485     1.168633493108
    H               -1.271746653766     1.064360254485     2.068027493108
    H               -1.262952653766     1.131175254485     0.319572493108
    C               -0.000066653766    -0.393290745515     1.141690493108
    H               -0.000220653766    -1.079963745515     1.993522493108
    H                0.000074346234    -1.020868745515     0.245047493108
    C                1.271153346234     0.444296254485     1.169094493108
    H                1.271426346234     1.064106254485     2.068580493108
    H                1.262958346234     1.131370254485     0.320187493108
    C                2.532605346234    -0.408428745515     1.133270493108
    H                2.562228346234    -1.021284745515     0.231961493108
    H                3.433783346234     0.202507254485     1.150848493108
    H                2.568523346234    -1.082884745515     1.988776493108

""")

GEOS['%s-%s-%s' % (dbse, '44-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666432507284     0.212585018327    -3.262286323774
    H                1.228914507284    -0.301108981673    -2.495763323774
    H                1.228068507284     0.725982018327    -4.029633323774
    C               -0.665941492716     0.211747018327    -3.262411323774
    H               -1.227889492716    -0.302724981673    -2.495920323774
    H               -1.228151492716     0.724420018327    -4.029771323774
    --
    0 1
    C               -2.532786492716    -0.416198981673     1.252648676226
    H               -2.562279492716    -1.029219981673     0.351458676226
    H               -2.568920492716    -1.090418981673     2.108265676226
    H               -3.433957492716     0.194818018327     1.269894676226
    C               -1.271350492716     0.436477018327     1.288466676226
    H               -1.271755492716     1.056570018327     2.187860676226
    H               -1.262961492716     1.123385018327     0.439405676226
    C               -0.000075492716    -0.401080981673     1.261523676226
    H               -0.000229492716    -1.087753981673     2.113355676226
    H                0.000065507284    -1.028658981673     0.364880676226
    C                1.271144507284     0.436506018327     1.288927676226
    H                1.271417507284     1.056316018327     2.188413676226
    H                1.262949507284     1.123580018327     0.440020676226
    C                2.532595507284    -0.416218981673     1.253103676226
    H                2.562219507284    -1.029074981673     0.351794676226
    H                3.433774507284     0.194717018327     1.270681676226
    H                2.568514507284    -1.090674981673     2.108609676226

""")

GEOS['%s-%s-%s' % (dbse, '44-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666469699469     0.244917152810    -3.759636245136
    H                1.228951699469    -0.268776847190    -2.993113245136
    H                1.228105699469     0.758314152810    -4.526983245136
    C               -0.665904300531     0.244079152810    -3.759761245136
    H               -1.227852300531    -0.270392847190    -2.993270245136
    H               -1.228114300531     0.756752152810    -4.527121245136
    --
    0 1
    C               -2.532801300531    -0.428769847190     1.446026754864
    H               -2.562294300531    -1.041790847190     0.544836754864
    H               -2.568935300531    -1.102990847190     2.301643754864
    H               -3.433972300531     0.182246152810     1.463272754864
    C               -1.271364300531     0.423905152810     1.481844754864
    H               -1.271770300531     1.043999152810     2.381238754864
    H               -1.262976300531     1.110813152810     0.632783754864
    C               -0.000090300531    -0.413651847190     1.454900754864
    H               -0.000244300531    -1.100324847190     2.306732754864
    H                0.000050699469    -1.041229847190     0.558258754864
    C                1.271129699469     0.423934152810     1.482305754864
    H                1.271402699469     1.043745152810     2.381790754864
    H                1.262935699469     1.111009152810     0.633397754864
    C                2.532581699469    -0.428789847190     1.446481754864
    H                2.562204699469    -1.041646847190     0.545172754864
    H                3.433759699469     0.182146152810     1.464059754864
    H                2.568499699469    -1.103246847190     2.301987754864

""")

GEOS['%s-%s-%s' % (dbse, '44-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.666541833136     0.307592077976    -4.723729343502
    H                1.229023833136    -0.206101922024    -3.957206343502
    H                1.228177833136     0.820989077976    -5.491076343502
    C               -0.665832166864     0.306754077976    -4.723854343502
    H               -1.227780166864    -0.207717922024    -3.957363343502
    H               -1.228042166864     0.819427077976    -5.491214343502
    --
    0 1
    C               -2.532829166864    -0.453138922024     1.820881656498
    H               -2.562322166864    -1.066159922024     0.919692656498
    H               -2.568963166864    -1.127359922024     2.676498656498
    H               -3.434000166864     0.157877077976     1.838127656498
    C               -1.271393166864     0.399536077976     1.856699656498
    H               -1.271798166864     1.019630077976     2.756093656498
    H               -1.263004166864     1.086445077976     1.007639656498
    C               -0.000118166864    -0.438020922024     1.829756656498
    H               -0.000272166864    -1.124693922024     2.681588656498
    H                0.000022833136    -1.065598922024     0.933114656498
    C                1.271101833136     0.399565077976     1.857161656498
    H                1.271374833136     1.019376077976     2.756646656498
    H                1.262906833136     1.086640077976     1.008253656498
    C                2.532553833136    -0.453158922024     1.821336656498
    H                2.562176833136    -1.066015922024     0.920027656498
    H                3.433731833136     0.157777077976     1.838915656498
    H                2.568471833136    -1.127614922024     2.676842656498

""")

GEOS['%s-%s-%s' % (dbse, '45-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606157322785     0.045908762034    -2.435774628304
    H               -1.668005322785     0.045810762034    -2.435767628304
    C                0.605880677215     0.045575762034    -2.435513628304
    H                1.667709677215     0.044897762034    -2.435051628304
    --
    0 1
    C               -2.530391322785    -0.351248237966     0.864813371696
    H               -2.538757322785    -1.013202237966    -0.001597628304
    H               -2.582309322785    -0.977518237966     1.755403371696
    H               -3.432805322785     0.257655762034     0.832051371696
    C               -1.269859322785     0.503351762034     0.875882371696
    H               -1.286510322785     1.176351762034     1.736291371696
    H               -1.244592322785     1.137567762034    -0.012913628304
    C                0.000059677215    -0.334979237966     0.916337371696
    H                0.000062677215    -0.952768237966     1.819399371696
    H                0.000126677215    -1.023278237966     0.067096371696
    C                1.269958677215     0.503396762034     0.875969371696
    H                1.286585677215     1.176363762034     1.736390371696
    H                1.244813677215     1.137569762034    -0.012831628304
    C                2.530480677215    -0.351239237966     0.865022371696
    H                2.538860677215    -1.013222237966    -0.001356628304
    H                3.432859677215     0.257691762034     0.832296371696
    H                2.582297677215    -0.977454237966     1.755656371696

""")

GEOS['%s-%s-%s' % (dbse, '45-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606164466600     0.048202892913    -2.558018700525
    H               -1.668012466600     0.048104892913    -2.558011700525
    C                0.605873533400     0.047869892913    -2.557757700525
    H                1.667702533400     0.047191892913    -2.557295700525
    --
    0 1
    C               -2.530388466600    -0.352076107087     0.908925299475
    H               -2.538755466600    -1.014030107087     0.042515299475
    H               -2.582307466600    -0.978346107087     1.799516299475
    H               -3.432803466600     0.256827892913     0.876164299475
    C               -1.269856466600     0.502523892913     0.919995299475
    H               -1.286508466600     1.175523892913     1.780404299475
    H               -1.244589466600     1.136739892913     0.031199299475
    C                0.000062533400    -0.335807107087     0.960450299475
    H                0.000064533400    -0.953596107087     1.863511299475
    H                0.000129533400    -1.024106107087     0.111209299475
    C                1.269960533400     0.502568892913     0.920082299475
    H                1.286588533400     1.175536892913     1.780503299475
    H                1.244815533400     1.136741892913     0.031280299475
    C                2.530483533400    -0.352067107087     0.909135299475
    H                2.538862533400    -1.014050107087     0.042756299475
    H                3.432861533400     0.256863892913     0.876409299475
    H                2.582300533400    -0.978282107087     1.799769299475

""")

GEOS['%s-%s-%s' % (dbse, '45-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606171162255     0.050468549345    -2.678753706124
    H               -1.668019162255     0.050370549345    -2.678746706124
    C                0.605866837745     0.050135549345    -2.678492706124
    H                1.667695837745     0.049457549345    -2.678030706124
    --
    0 1
    C               -2.530386162255    -0.352894450655     0.952494293876
    H               -2.538752162255    -1.014847450655     0.086083293876
    H               -2.582304162255    -0.979163450655     1.843084293876
    H               -3.432800162255     0.256009549345     0.919732293876
    C               -1.269854162255     0.501706549345     0.963563293876
    H               -1.286505162255     1.174705549345     1.823972293876
    H               -1.244587162255     1.135922549345     0.074767293876
    C                0.000064837745    -0.336624450655     1.004018293876
    H                0.000067837745    -0.954414450655     1.907080293876
    H                0.000131837745    -1.024923450655     0.154777293876
    C                1.269962837745     0.501751549345     0.963650293876
    H                1.286590837745     1.174718549345     1.824071293876
    H                1.244818837745     1.135923549345     0.074848293876
    C                2.530485837745    -0.352884450655     0.952703293876
    H                2.538865837745    -1.014867450655     0.086324293876
    H                3.432864837745     0.256046549345     0.919977293876
    H                2.582302837745    -0.979099450655     1.843337293876

""")

GEOS['%s-%s-%s' % (dbse, '45-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606178173484     0.052714752832    -2.798444527704
    H               -1.668026173484     0.052616752832    -2.798437527704
    C                0.605859826516     0.052381752832    -2.798183527704
    H                1.667688826516     0.051703752832    -2.797721527704
    --
    0 1
    C               -2.530383173484    -0.353704247168     0.995685472296
    H               -2.538750173484    -1.015658247168     0.129274472296
    H               -2.582302173484    -0.979974247168     1.886276472296
    H               -3.432798173484     0.255199752832     0.962923472296
    C               -1.269851173484     0.500895752832     1.006754472296
    H               -1.286503173484     1.173895752832     1.867163472296
    H               -1.244584173484     1.135111752832     0.117958472296
    C                0.000066826516    -0.337435247168     1.047209472296
    H                0.000069826516    -0.955224247168     1.950271472296
    H                0.000133826516    -1.025734247168     0.197968472296
    C                1.269965826516     0.500940752832     1.006841472296
    H                1.286593826516     1.173908752832     1.867262472296
    H                1.244820826516     1.135113752832     0.118040472296
    C                2.530487826516    -0.353695247168     0.995895472296
    H                2.538867826516    -1.015678247168     0.129515472296
    H                3.432866826516     0.255235752832     0.963168472296
    H                2.582305826516    -0.979910247168     1.886528472296

""")

GEOS['%s-%s-%s' % (dbse, '45-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606184919544     0.054940697594    -2.917061343471
    H               -1.668032919544     0.054842697594    -2.917054343471
    C                0.605853080456     0.054607697594    -2.916800343471
    H                1.667682080456     0.053929697594    -2.916338343471
    --
    0 1
    C               -2.530380919544    -0.354508302406     1.038488656529
    H               -2.538747919544    -1.016461302406     0.172078656529
    H               -2.582299919544    -0.980777302406     1.929079656529
    H               -3.432795919544     0.254396697594     1.005727656529
    C               -1.269848919544     0.500092697594     1.049558656529
    H               -1.286500919544     1.173091697594     1.909967656529
    H               -1.244581919544     1.134308697594     0.160762656529
    C                0.000069080456    -0.338238302406     1.090013656529
    H                0.000072080456    -0.956027302406     1.993074656529
    H                0.000137080456    -1.026537302406     0.240772656529
    C                1.269968080456     0.500137697594     1.049645656529
    H                1.286596080456     1.173104697594     1.910066656529
    H                1.244823080456     1.134310697594     0.160843656529
    C                2.530491080456    -0.354498302406     1.038698656529
    H                2.538870080456    -1.016481302406     0.172319656529
    H                3.432869080456     0.254432697594     1.005972656529
    H                2.582308080456    -0.980713302406     1.929332656529

""")

GEOS['%s-%s-%s' % (dbse, '45-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606204933645     0.061531546426    -3.268273764067
    H               -1.668052933645     0.061433546426    -3.268266764067
    C                0.605833066355     0.061198546426    -3.268012764067
    H                1.667662066355     0.060520546426    -3.267550764067
    --
    0 1
    C               -2.530373933645    -0.356886453574     1.165227235933
    H               -2.538740933645    -1.018839453574     0.298816235933
    H               -2.582291933645    -0.983155453574     2.055817235933
    H               -3.432788933645     0.252017546426     1.132465235933
    C               -1.269841933645     0.497714546426     1.176296235933
    H               -1.286492933645     1.170713546426     2.036705235933
    H               -1.244574933645     1.131930546426     0.287500235933
    C                0.000077066355    -0.340616453574     1.216751235933
    H                0.000080066355    -0.958406453574     2.119813235933
    H                0.000144066355    -1.028915453574     0.367510235933
    C                1.269975066355     0.497759546426     1.176383235933
    H                1.286603066355     1.170726546426     2.036804235933
    H                1.244830066355     1.131931546426     0.287581235933
    C                2.530498066355    -0.356877453574     1.165436235933
    H                2.538878066355    -1.018859453574     0.299057235933
    H                3.432877066355     0.252054546426     1.132710235933
    H                2.582315066355    -0.983091453574     2.056070235933

""")

GEOS['%s-%s-%s' % (dbse, '45-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606237949660     0.072321850403    -3.843273722370
    H               -1.668085949660     0.072223850403    -3.843266722370
    C                0.605800050340     0.071988850403    -3.843012722370
    H                1.667629050340     0.071310850403    -3.842550722370
    --
    0 1
    C               -2.530361949660    -0.360780149597     1.372720277630
    H               -2.538728949660    -1.022734149597     0.506309277630
    H               -2.582280949660    -0.987049149597     2.263311277630
    H               -3.432776949660     0.248123850403     1.339958277630
    C               -1.269829949660     0.493820850403     1.383789277630
    H               -1.286481949660     1.166819850403     2.244198277630
    H               -1.244562949660     1.128036850403     0.494993277630
    C                0.000089050340    -0.344510149597     1.424244277630
    H                0.000091050340    -0.962300149597     2.327306277630
    H                0.000156050340    -1.032810149597     0.575003277630
    C                1.269987050340     0.493865850403     1.383876277630
    H                1.286615050340     1.166832850403     2.244297277630
    H                1.244842050340     1.128037850403     0.495075277630
    C                2.530510050340    -0.360771149597     1.372930277630
    H                2.538889050340    -1.022754149597     0.506551277630
    H                3.432888050340     0.248159850403     1.340203277630
    H                2.582327050340    -0.986986149597     2.263563277630

""")

GEOS['%s-%s-%s' % (dbse, '45-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C               -0.606302492441     0.093523131678    -4.973024922246
    H               -1.668150492441     0.093425131678    -4.973017922246
    C                0.605735507559     0.093190131678    -4.972763922246
    H                1.667564507559     0.092512131678    -4.972301922246
    --
    0 1
    C               -2.530338492441    -0.368430868322     1.780400077754
    H               -2.538705492441    -1.030383868322     0.913989077754
    H               -2.582257492441    -0.994699868322     2.670990077754
    H               -3.432753492441     0.240473131678     1.747638077754
    C               -1.269806492441     0.486170131678     1.791469077754
    H               -1.286458492441     1.159169131678     2.651878077754
    H               -1.244539492441     1.120386131678     0.902673077754
    C                0.000111507559    -0.352160868322     1.831924077754
    H                0.000114507559    -0.969950868322     2.734986077754
    H                0.000179507559    -1.040459868322     0.982683077754
    C                1.270010507559     0.486215131678     1.791556077754
    H                1.286638507559     1.159182131678     2.651977077754
    H                1.244865507559     1.120387131678     0.902754077754
    C                2.530533507559    -0.368421868322     1.780609077754
    H                2.538912507559    -1.030403868322     0.914230077754
    H                3.432911507559     0.240510131678     1.747883077754
    H                2.582350507559    -0.994635868322     2.671243077754

""")

GEOS['%s-%s-%s' % (dbse, '46-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.377866410581     1.024914797076    -1.625786657364
    H                0.957851410581     2.026487797076    -1.563494657364
    H                1.953097410581     0.938953797076    -2.545893657364
    H                2.057377410581     0.864262797076    -0.793659657364
    C                0.332412410581    -0.065209202924    -1.608412657364
    O                0.624496410581    -1.240043202924    -1.425330657364
    N               -0.944353589419     0.357324797076    -1.822696657364
    H               -1.098996589419     1.334462797076    -1.990499657364
    C               -2.054178589419    -0.564931202924    -1.916458657364
    H               -1.663680589419    -1.554358202924    -1.699427657364
    H               -2.828916589419    -0.318942202924    -1.192950657364
    H               -2.485295589419    -0.566487202924    -2.916550657364
    --
    0 1
    C                2.649030410581     0.463066797076     2.053957342636
    H                2.765869410581     1.217482797076     1.275213342636
    H                2.562920410581     0.987953797076     3.005613342636
    H                3.561122410581    -0.131175202924     2.080765342636
    C                1.420758410581    -0.400320202924     1.796406342636
    H                1.326189410581    -1.145774202924     2.589457342636
    H                1.537178410581    -0.953785202924     0.862562342636
    C                0.138220410581     0.418293797076     1.731104342636
    H                0.026650410581     0.996028797076     2.654188342636
    H                0.217454410581     1.151108797076     0.921452342636
    C               -1.106135589419    -0.432042202924     1.514225342636
    H               -1.196937589419    -1.146528202924     2.335642342636
    H               -0.978328589419    -1.020979202924     0.604010342636
    C               -2.372974589419     0.408249797076     1.424110342636
    H               -2.306060589419     1.115300797076     0.596341342636
    H               -3.258316589419    -0.207768202924     1.270277342636
    H               -2.523330589419     0.984460797076     2.337329342636

""")

GEOS['%s-%s-%s' % (dbse, '46-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.372501711536     1.021159968465    -1.688629433959
    H                0.952486711536     2.022732968465    -1.626337433959
    H                1.947732711536     0.935198968465    -2.608736433959
    H                2.052012711536     0.860507968465    -0.856502433959
    C                0.327047711536    -0.068964031535    -1.671255433959
    O                0.619131711536    -1.243798031535    -1.488173433959
    N               -0.949718288464     0.353569968465    -1.885539433959
    H               -1.104361288464     1.330707968465    -2.053342433959
    C               -2.059543288464    -0.568686031535    -1.979301433959
    H               -1.669045288464    -1.558113031535    -1.762270433959
    H               -2.834281288464    -0.322697031535    -1.255793433959
    H               -2.490660288464    -0.570242031535    -2.979393433959
    --
    0 1
    C                2.654466711536     0.466871968465     2.117636566041
    H                2.771305711536     1.221287968465     1.338891566041
    H                2.568355711536     0.991757968465     3.069292566041
    H                3.566557711536    -0.127370031535     2.144444566041
    C                1.426194711536    -0.396516031535     1.860084566041
    H                1.331624711536    -1.141970031535     2.653135566041
    H                1.542614711536    -0.949980031535     0.926240566041
    C                0.143656711536     0.422097968465     1.794783566041
    H                0.032085711536     0.999832968465     2.717866566041
    H                0.222890711536     1.154912968465     0.985130566041
    C               -1.100700288464    -0.428237031535     1.577903566041
    H               -1.191501288464    -1.142723031535     2.399320566041
    H               -0.972893288464    -1.017175031535     0.667688566041
    C               -2.367538288464     0.412054968465     1.487788566041
    H               -2.300625288464     1.119105968465     0.660019566041
    H               -3.252880288464    -0.203963031535     1.333955566041
    H               -2.517894288464     0.988265968465     2.401008566041

""")

GEOS['%s-%s-%s' % (dbse, '46-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.367159611876     1.017420654849    -1.751208478380
    H                0.947144611876     2.018993654849    -1.688916478380
    H                1.942390611876     0.931459654849    -2.671315478380
    H                2.046670611876     0.856768654849    -0.919081478380
    C                0.321705611876    -0.072703345151    -1.733834478380
    O                0.613789611876    -1.247537345151    -1.550752478380
    N               -0.955060388124     0.349830654849    -1.948118478380
    H               -1.109703388124     1.326968654849    -2.115921478380
    C               -2.064885388124    -0.572425345151    -2.041880478380
    H               -1.674387388124    -1.561852345151    -1.824849478380
    H               -2.839623388124    -0.326436345151    -1.318372478380
    H               -2.496002388124    -0.573981345151    -3.041972478380
    --
    0 1
    C                2.659879611876     0.470660654849     2.181047521620
    H                2.776718611876     1.225076654849     1.402302521620
    H                2.573769611876     0.995546654849     3.132703521620
    H                3.571971611876    -0.123581345151     2.207855521620
    C                1.431607611876    -0.392726345151     1.923496521620
    H                1.337038611876    -1.138180345151     2.716546521620
    H                1.548027611876    -0.946191345151     0.989652521620
    C                0.149069611876     0.425887654849     1.858194521620
    H                0.037499611876     1.003621654849     2.781278521620
    H                0.228303611876     1.158701654849     1.048542521620
    C               -1.095286388124    -0.424448345151     1.641314521620
    H               -1.186088388124    -1.138934345151     2.462731521620
    H               -0.967479388124    -1.013385345151     0.731100521620
    C               -2.362125388124     0.415843654849     1.551200521620
    H               -2.295211388124     1.122894654849     0.723430521620
    H               -3.247467388124    -0.200174345151     1.397367521620
    H               -2.512481388124     0.992054654849     2.464419521620

""")

GEOS['%s-%s-%s' % (dbse, '46-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.361824590309     1.013686466610    -1.813704381292
    H                0.941809590309     2.015259466610    -1.751412381292
    H                1.937055590309     0.927725466610    -2.733811381292
    H                2.041335590309     0.853034466610    -0.981577381292
    C                0.316370590309    -0.076437533390    -1.796330381292
    O                0.608454590309    -1.251271533390    -1.613248381292
    N               -0.960395409691     0.346096466610    -2.010614381292
    H               -1.115038409691     1.323234466610    -2.178417381292
    C               -2.070220409691    -0.576159533390    -2.104376381292
    H               -1.679722409691    -1.565586533390    -1.887345381292
    H               -2.844958409691    -0.330170533390    -1.380868381292
    H               -2.501337409691    -0.577715533390    -3.104468381292
    --
    0 1
    C                2.665285590309     0.474444466610     2.244374618708
    H                2.782124590309     1.228860466610     1.465629618708
    H                2.579175590309     0.999331466610     3.196030618708
    H                3.577377590309    -0.119797533390     2.271182618708
    C                1.437013590309    -0.388942533390     1.986823618708
    H                1.342444590309    -1.134396533390     2.779873618708
    H                1.553433590309    -0.942407533390     1.052979618708
    C                0.154475590309     0.429671466610     1.921521618708
    H                0.042905590309     1.007406466610     2.844605618708
    H                0.233709590309     1.162486466610     1.111869618708
    C               -1.089880409691    -0.420664533390     1.704641618708
    H               -1.180682409691    -1.135150533390     2.526059618708
    H               -0.962073409691    -1.009601533390     0.794427618708
    C               -2.356719409691     0.419627466610     1.614527618708
    H               -2.289805409691     1.126678466610     0.786757618708
    H               -3.242061409691    -0.196390533390     1.460694618708
    H               -2.507075409691     0.995838466610     2.527746618708

""")

GEOS['%s-%s-%s' % (dbse, '46-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.356506973965     1.009964523500    -1.875995141435
    H                0.936491973965     2.011537523500    -1.813703141435
    H                1.931737973965     0.924003523500    -2.796102141435
    H                2.036017973965     0.849312523500    -1.043868141435
    C                0.311052973965    -0.080159476500    -1.858621141435
    O                0.603136973965    -1.254993476500    -1.675539141435
    N               -0.965713026035     0.342374523500    -2.072905141435
    H               -1.120356026035     1.319512523500    -2.240708141435
    C               -2.075538026035    -0.579881476500    -2.166667141435
    H               -1.685040026035    -1.569308476500    -1.949636141435
    H               -2.850276026035    -0.333892476500    -1.443159141435
    H               -2.506655026035    -0.581437476500    -3.166759141435
    --
    0 1
    C                2.670673973965     0.478216523500     2.307493858565
    H                2.787512973965     1.232632523500     1.528748858565
    H                2.584563973965     1.003102523500     3.259149858565
    H                3.582764973965    -0.116025476500     2.334301858565
    C                1.442401973965    -0.385171476500     2.049942858565
    H                1.347831973965    -1.130625476500     2.842992858565
    H                1.558821973965    -0.938635476500     1.116098858565
    C                0.159863973965     0.433442523500     1.984640858565
    H                0.048293973965     1.011177523500     2.907724858565
    H                0.239097973965     1.166257523500     1.174988858565
    C               -1.084492026035    -0.416892476500     1.767760858565
    H               -1.175294026035    -1.131379476500     2.589178858565
    H               -0.956685026035    -1.005830476500     0.857546858565
    C               -2.351331026035     0.423398523500     1.677646858565
    H               -2.284418026035     1.130449523500     0.849876858565
    H               -3.236673026035    -0.192618476500     1.523813858565
    H               -2.501687026035     0.999609523500     2.590865858565

""")

GEOS['%s-%s-%s' % (dbse, '46-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.340617639646     0.998842886309    -2.062127150334
    H                0.920602639646     2.000415886309    -1.999835150334
    H                1.915848639646     0.912881886309    -2.982234150334
    H                2.020128639646     0.838190886309    -1.230000150334
    C                0.295163639646    -0.091281113691    -2.044753150334
    O                0.587247639646    -1.266115113691    -1.861671150334
    N               -0.981602360354     0.331252886309    -2.259037150334
    H               -1.136245360354     1.308390886309    -2.426840150334
    C               -2.091427360354    -0.591003113691    -2.352799150334
    H               -1.700929360354    -1.580430113691    -2.135768150334
    H               -2.866165360354    -0.345014113691    -1.629291150334
    H               -2.522544360354    -0.592559113691    -3.352891150334
    --
    0 1
    C                2.686774639646     0.489485886309     2.496101849666
    H                2.803613639646     1.243901886309     1.717356849666
    H                2.600664639646     1.014371886309     3.447757849666
    H                3.598865639646    -0.104756113691     2.522909849666
    C                1.458502639646    -0.373902113691     2.238549849666
    H                1.363932639646    -1.119356113691     3.031600849666
    H                1.574922639646    -0.927366113691     1.304705849666
    C                0.175964639646     0.444711886309     2.173248849666
    H                0.064394639646     1.022446886309     3.096332849666
    H                0.255198639646     1.177526886309     1.363595849666
    C               -1.068391360354    -0.405623113691     1.956368849666
    H               -1.159193360354    -1.120109113691     2.777785849666
    H               -0.940584360354    -0.994561113691     1.046154849666
    C               -2.335230360354     0.434668886309     1.866253849666
    H               -2.268317360354     1.141718886309     1.038484849666
    H               -3.220572360354    -0.181349113691     1.712420849666
    H               -2.485586360354     1.010879886309     2.779473849666

""")

GEOS['%s-%s-%s' % (dbse, '46-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.310617621831     0.977844624081    -2.413555001394
    H                0.890602621831     1.979417624081    -2.351263001394
    H                1.885848621831     0.891883624081    -3.333662001394
    H                1.990128621831     0.817192624081    -1.581428001394
    C                0.265163621831    -0.112279375919    -2.396181001394
    O                0.557247621831    -1.287113375919    -2.213099001394
    N               -1.011602378169     0.310254624081    -2.610465001394
    H               -1.166245378169     1.287392624081    -2.778268001394
    C               -2.121427378169    -0.612001375919    -2.704227001394
    H               -1.730929378169    -1.601428375919    -2.487196001394
    H               -2.896165378169    -0.366012375919    -1.980719001394
    H               -2.552544378169    -0.613557375919    -3.704319001394
    --
    0 1
    C                2.717173621831     0.510763624081     2.852202998606
    H                2.834012621831     1.265179624081     2.073458998606
    H                2.631063621831     1.035649624081     3.803858998606
    H                3.629265621831    -0.083478375919     2.879010998606
    C                1.488901621831    -0.352624375919     2.594651998606
    H                1.394332621831    -1.098078375919     3.387702998606
    H                1.605321621831    -0.906088375919     1.660807998606
    C                0.206363621831     0.465989624081     2.529350998606
    H                0.094793621831     1.043724624081     3.452433998606
    H                0.285597621831     1.198804624081     1.719697998606
    C               -1.037992378169    -0.384345375919     2.312470998606
    H               -1.128794378169    -1.098832375919     3.133887998606
    H               -0.910185378169    -0.973283375919     1.402255998606
    C               -2.304831378169     0.455945624081     2.222355998606
    H               -2.237917378169     1.162996624081     1.394586998606
    H               -3.190173378169    -0.160071375919     2.068522998606
    H               -2.455187378169     1.032156624081     3.135574998606

""")

GEOS['%s-%s-%s' % (dbse, '46-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                1.251624246328     0.936552719425    -3.104617966946
    H                0.831609246328     1.938125719425    -3.042325966946
    H                1.826855246328     0.850591719425    -4.024724966946
    H                1.931135246328     0.775900719425    -2.272490966946
    C                0.206170246328    -0.153571280575    -3.087243966946
    O                0.498254246328    -1.328405280575    -2.904161966946
    N               -1.070595753672     0.268962719425    -3.301527966946
    H               -1.225238753672     1.246100719425    -3.469330966946
    C               -2.180420753672    -0.653293280575    -3.395289966946
    H               -1.789922753672    -1.642720280575    -3.178258966946
    H               -2.955158753672    -0.407304280575    -2.671781966946
    H               -2.611537753672    -0.654849280575    -4.395381966946
    --
    0 1
    C                2.776952246328     0.552604719425     3.552458033054
    H                2.893791246328     1.307020719425     2.773713033054
    H                2.690841246328     1.077490719425     4.504114033054
    H                3.689043246328    -0.041637280575     3.579266033054
    C                1.548679246328    -0.310783280575     3.294906033054
    H                1.454110246328    -1.056237280575     4.087957033054
    H                1.665099246328    -0.864247280575     2.361062033054
    C                0.266141246328     0.507830719425     3.229605033054
    H                0.154571246328     1.085565719425     4.152688033054
    H                0.345376246328     1.240645719425     2.419952033054
    C               -0.978214753672    -0.342504280575     3.012725033054
    H               -1.069016753672    -1.056991280575     3.834142033054
    H               -0.850407753672    -0.931442280575     2.102510033054
    C               -2.245052753672     0.497786719425     2.922610033054
    H               -2.178139753672     1.204837719425     2.094841033054
    H               -3.130395753672    -0.118230280575     2.768777033054
    H               -2.395408753672     1.073997719425     3.835830033054

""")

GEOS['%s-%s-%s' % (dbse, '47-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.756918829440     1.238975168191    -2.483102035900
    H                1.330945829440     2.140096168191    -2.650670035900
    C                1.402816829440     0.006508168191    -2.397053035900
    H                2.478244829440    -0.048749831809    -2.496523035900
    C                0.662769829440    -1.154678831809    -2.180445035900
    H                1.164063829440    -2.110142831809    -2.109613035900
    C               -0.723256170560    -1.083779831809    -2.051940035900
    H               -1.296796170560    -1.985538831809    -1.885631035900
    C               -1.369304170560     0.148545168191    -2.136740035900
    H               -2.444695170560     0.204362168191    -2.035832035900
    C               -0.629167170560     1.310170168191    -2.351491035900
    H               -1.130098170560     2.266461168191    -2.414734035900
    --
    0 1
    C                0.143276829440     0.216405168191     0.911741964100
    H                0.265403829440     0.441888168191    -0.138684035900
    C               -0.947804170560    -0.538493831809     1.339077964100
    H               -1.668309170560    -0.895007831809     0.615383964100
    C               -1.108504170560    -0.832618831809     2.692027964100
    H               -1.955612170560    -1.418587831809     3.022275964100
    C               -0.176622170560    -0.371565831809     3.620521964100
    H               -0.300515170560    -0.599531831809     4.670529964100
    C                0.914962829440     0.383536168191     3.195281964100
    H                1.638081829440     0.741369168191     3.915664964100
    C                1.073766829440     0.677021168191     1.841790964100
    H                1.921024829440     1.263096168191     1.511742964100

""")

GEOS['%s-%s-%s' % (dbse, '47-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.747405996107     1.225036668191    -2.564532522986
    H                1.321432996107     2.126157668191    -2.732100522986
    C                1.393303996107    -0.007430331809    -2.478483522986
    H                2.468731996107    -0.062688331809    -2.577953522986
    C                0.653256996107    -1.168617331809    -2.261875522986
    H                1.154550996107    -2.124081331809    -2.191043522986
    C               -0.732769003893    -1.097718331809    -2.133370522986
    H               -1.306309003893    -1.999477331809    -1.967061522986
    C               -1.378817003893     0.134606668191    -2.218170522986
    H               -2.454208003893     0.190423668191    -2.117262522986
    C               -0.638680003893     1.296231668191    -2.432921522986
    H               -1.139611003893     2.252522668191    -2.496164522986
    --
    0 1
    C                0.152789996107     0.230343668191     0.993172477014
    H                0.274915996107     0.455826668191    -0.057254522986
    C               -0.938292003893    -0.524555331809     1.420508477014
    H               -1.658796003893    -0.881069331809     0.696814477014
    C               -1.098992003893    -0.818680331809     2.773458477014
    H               -1.946099003893    -1.404649331809     3.103706477014
    C               -0.167109003893    -0.357627331809     3.701952477014
    H               -0.291003003893    -0.585593331809     4.751960477014
    C                0.924475996107     0.397474668191     3.276712477014
    H                1.647594996107     0.755307668191     3.997094477014
    C                1.083279996107     0.690959668191     1.923221477014
    H                1.930537996107     1.277034668191     1.593173477014

""")

GEOS['%s-%s-%s' % (dbse, '47-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.738090489650     1.211386821945    -2.644276029443
    H                1.312117489650     2.112507821945    -2.811844029443
    C                1.383988489650    -0.021080178055    -2.558227029443
    H                2.459416489650    -0.076338178055    -2.657697029443
    C                0.643941489650    -1.182267178055    -2.341619029443
    H                1.145235489650    -2.137731178055    -2.270787029443
    C               -0.742084510350    -1.111368178055    -2.213114029443
    H               -1.315624510350    -2.013127178055    -2.046805029443
    C               -1.388132510350     0.120956821945    -2.297914029443
    H               -2.463523510350     0.176773821945    -2.197006029443
    C               -0.647995510350     1.282581821945    -2.512665029443
    H               -1.148926510350     2.238872821945    -2.575908029443
    --
    0 1
    C                0.162105489650     0.243992821945     1.072915970557
    H                0.284231489650     0.469476821945     0.022488970557
    C               -0.928976510350    -0.510905178055     1.500251970557
    H               -1.649480510350    -0.867419178055     0.776557970557
    C               -1.089676510350    -0.805031178055     2.853201970557
    H               -1.936783510350    -1.390999178055     3.183449970557
    C               -0.157793510350    -0.343977178055     3.781695970557
    H               -0.281686510350    -0.571943178055     4.831703970557
    C                0.933791489650     0.411124821945     3.356455970557
    H                1.656910489650     0.768957821945     4.076838970557
    C                1.092595489650     0.704609821945     2.002964970557
    H                1.939853489650     1.290684821945     1.672916970557

""")

GEOS['%s-%s-%s' % (dbse, '47-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.728914996107     1.197942661735    -2.722818196110
    H                1.302941996107     2.099063661735    -2.890386196110
    C                1.374812996107    -0.034524338265    -2.636769196110
    H                2.450240996107    -0.089782338265    -2.736239196110
    C                0.634765996107    -1.195711338265    -2.420161196110
    H                1.136059996107    -2.151175338265    -2.349329196110
    C               -0.751260003893    -1.124812338265    -2.291656196110
    H               -1.324800003893    -2.026571338265    -2.125347196110
    C               -1.397308003893     0.107512661735    -2.376456196110
    H               -2.472699003893     0.163329661735    -2.275548196110
    C               -0.657171003893     1.269137661735    -2.591207196110
    H               -1.158102003893     2.225428661735    -2.654450196110
    --
    0 1
    C                0.171280996107     0.257437661735     1.151457803890
    H                0.293406996107     0.482920661735     0.101031803890
    C               -0.919801003893    -0.497461338265     1.578793803890
    H               -1.640305003893    -0.853975338265     0.855099803890
    C               -1.080501003893    -0.791586338265     2.931744803890
    H               -1.927608003893    -1.377555338265     3.261991803890
    C               -0.148618003893    -0.330533338265     3.860238803890
    H               -0.272512003893    -0.558499338265     4.910245803890
    C                0.942966996107     0.424568661735     3.434997803890
    H                1.666085996107     0.782402661735     4.155380803890
    C                1.101770996107     0.718053661735     2.081506803890
    H                1.949028996107     1.304128661735     1.751459803890

""")

GEOS['%s-%s-%s' % (dbse, '47-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.719870746107     1.184690321945    -2.800239035900
    H                1.293897746107     2.085811321945    -2.967807035900
    C                1.365768746107    -0.047776678055    -2.714190035900
    H                2.441196746107    -0.103034678055    -2.813660035900
    C                0.625721746107    -1.208963678055    -2.497582035900
    H                1.127015746107    -2.164427678055    -2.426750035900
    C               -0.760304253893    -1.138064678055    -2.369077035900
    H               -1.333844253893    -2.039823678055    -2.202768035900
    C               -1.406352253893     0.094260321945    -2.453877035900
    H               -2.481743253893     0.150077321945    -2.352969035900
    C               -0.666215253893     1.255885321945    -2.668628035900
    H               -1.167146253893     2.212176321945    -2.731871035900
    --
    0 1
    C                0.180324746107     0.270689321945     1.228878964100
    H                0.302451746107     0.496173321945     0.178452964100
    C               -0.910756253893    -0.484208678055     1.656214964100
    H               -1.631261253893    -0.840722678055     0.932520964100
    C               -1.071456253893    -0.778334678055     3.009164964100
    H               -1.918564253893    -1.364302678055     3.339412964100
    C               -0.139574253893    -0.317280678055     3.937658964100
    H               -0.263467253893    -0.545246678055     4.987666964100
    C                0.952011746107     0.437821321945     3.512418964100
    H                1.675130746107     0.795654321945     4.232801964100
    C                1.110814746107     0.731306321945     2.158927964100
    H                1.958072746107     1.317381321945     1.828879964100

""")

GEOS['%s-%s-%s' % (dbse, '47-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.693305746107     1.145765841314    -3.027638542356
    H                1.267332746107     2.046886841314    -3.195206542356
    C                1.339203746107    -0.086701158686    -2.941589542356
    H                2.414631746107    -0.141959158686    -3.041059542356
    C                0.599156746107    -1.247888158686    -2.724981542356
    H                1.100450746107    -2.203352158686    -2.654149542356
    C               -0.786869253893    -1.176989158686    -2.596476542356
    H               -1.360409253893    -2.078748158686    -2.430167542356
    C               -1.432917253893     0.055335841314    -2.681276542356
    H               -2.508308253893     0.111152841314    -2.580368542356
    C               -0.692780253893     1.216960841314    -2.896027542356
    H               -1.193711253893     2.173251841314    -2.959270542356
    --
    0 1
    C                0.206889746107     0.309613841314     1.456278457644
    H                0.329016746107     0.535097841314     0.405852457644
    C               -0.884191253893    -0.445284158686     1.883614457644
    H               -1.604696253893    -0.801799158686     1.159920457644
    C               -1.044891253893    -0.739410158686     3.236564457644
    H               -1.891999253893    -1.325378158686     3.566812457644
    C               -0.113009253893    -0.278356158686     4.165058457644
    H               -0.236902253893    -0.506323158686     5.215066457644
    C                0.978576746107     0.476745841314     3.739818457644
    H                1.701695746107     0.834578841314     4.460201457644
    C                1.137379746107     0.770230841314     2.386327457644
    H                1.984637746107     1.356304841314     2.056280457644

""")

GEOS['%s-%s-%s' % (dbse, '47-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.650265996107     1.082702001524    -3.396063529443
    H                1.224292996107     1.983823001524    -3.563631529443
    C                1.296163996107    -0.149764998476    -3.310014529443
    H                2.371591996107    -0.205022998476    -3.409484529443
    C                0.556116996107    -1.310951998476    -3.093406529443
    H                1.057410996107    -2.266415998476    -3.022574529443
    C               -0.829909003893    -1.240052998476    -2.964901529443
    H               -1.403449003893    -2.141811998476    -2.798592529443
    C               -1.475957003893    -0.007727998476    -3.049701529443
    H               -2.551348003893     0.048089001524    -2.948793529443
    C               -0.735820003893     1.153897001524    -3.264452529443
    H               -1.236751003893     2.110188001524    -3.327695529443
    --
    0 1
    C                0.249929996107     0.372678001524     1.824703470557
    H                0.372055996107     0.598161001524     0.774276470557
    C               -0.841152003893    -0.382220998476     2.252039470557
    H               -1.561656003893    -0.738734998476     1.528345470557
    C               -1.001852003893    -0.676345998476     3.604989470557
    H               -1.848959003893    -1.262313998476     3.935237470557
    C               -0.069969003893    -0.215292998476     4.533483470557
    H               -0.193863003893    -0.443258998476     5.583491470557
    C                1.021615996107     0.539810001524     4.108243470557
    H                1.744734996107     0.897643001524     4.828626470557
    C                1.180419996107     0.833295001524     2.754752470557
    H                2.027677996107     1.419369001524     2.424704470557

""")

GEOS['%s-%s-%s' % (dbse, '47-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.566530496107     0.960008328401    -4.112850272986
    H                1.140557496107     1.861129328401    -4.280418272986
    C                1.212428496107    -0.272458671599    -4.026801272986
    H                2.287856496107    -0.327716671599    -4.126271272986
    C                0.472381496107    -1.433645671599    -3.810193272986
    H                0.973675496107    -2.389109671599    -3.739361272986
    C               -0.913644503893    -1.362746671599    -3.681688272986
    H               -1.487184503893    -2.264505671599    -3.515379272986
    C               -1.559692503893    -0.130421671599    -3.766488272986
    H               -2.635083503893    -0.074604671599    -3.665580272986
    C               -0.819555503893     1.031203328401    -3.981239272986
    H               -1.320486503893     1.987494328401    -4.044482272986
    --
    0 1
    C                0.333665496107     0.495371328401     2.541489727014
    H                0.455791496107     0.720855328401     1.491063727014
    C               -0.757416503893    -0.259526671599     2.968826727014
    H               -1.477920503893    -0.616041671599     2.245131727014
    C               -0.918116503893    -0.553652671599     4.321776727014
    H               -1.765223503893    -1.139620671599     4.652023727014
    C                0.013766496107    -0.092598671599     5.250270727014
    H               -0.110127503893    -0.320564671599     6.300277727014
    C                1.105351496107     0.662503328401     4.825029727014
    H                1.828470496107     1.020336328401     5.545412727014
    C                1.264155496107     0.955988328401     3.471538727014
    H                2.111413496107     1.542063328401     3.141491727014

""")

GEOS['%s-%s-%s' % (dbse, '48-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.268171970694    -0.006564712293    -1.851573828337
    C                0.596694970694    -1.145180712293    -2.073956828337
    H                1.145826970694    -2.061716712293    -1.897084828337
    C               -0.727272029306    -1.190900712293    -2.504100828337
    H               -1.211785029306    -2.143509712293    -2.664293828337
    C               -1.401788029306     0.006945287707    -2.718613828337
    H               -2.429948029306     0.012216287707    -3.053050828337
    C               -0.719149029306     1.197902287707    -2.491755828337
    H               -1.197217029306     2.155369287707    -2.642036828337
    C                0.604304970694     1.138790287707    -2.061897828337
    H                1.159511970694     2.049721287707    -1.875319828337
    --
    0 1
    N                0.484189970694     0.006870287707     3.610107171663
    C                1.354857970694     0.001257287707     2.590694171663
    H                2.402714970694     0.002506287707     2.864996171663
    C                0.981471970694    -0.005809712293     1.248866171663
    H                1.728932970694    -0.009973712293     0.467766171663
    C               -0.374576029306    -0.007223712293     0.936450171663
    H               -0.704305029306    -0.012679712293    -0.093830828337
    C               -1.292680029306    -0.001508712293     1.981620171663
    H               -2.357130029306    -0.002358712293     1.792884171663
    C               -0.817076029306     0.005408287707     3.290579171663
    H               -1.509259029306     0.009991287707     4.123770171663

""")

GEOS['%s-%s-%s' % (dbse, '48-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.289041236592    -0.006150965441    -1.922207328337
    C                0.617564236592    -1.144766965441    -2.144590328337
    H                1.166696236592    -2.061302965441    -1.967718328337
    C               -0.706402763408    -1.190486965441    -2.574734328337
    H               -1.190915763408    -2.143095965441    -2.734927328337
    C               -1.380918763408     0.007359034559    -2.789247328337
    H               -2.409078763408     0.012630034559    -3.123684328337
    C               -0.698279763408     1.198316034559    -2.562389328337
    H               -1.176347763408     2.155783034559    -2.712670328337
    C                0.625174236592     1.139204034559    -2.132531328337
    H                1.180381236592     2.050135034559    -1.945953328337
    --
    0 1
    N                0.463320236592     0.006457034559     3.680740671663
    C                1.333988236592     0.000844034559     2.661327671663
    H                2.381845236592     0.002092034559     2.935629671663
    C                0.960603236592    -0.006223965441     1.319499671663
    H                1.708063236592    -0.010387965441     0.538399671663
    C               -0.395445763408    -0.007637965441     1.007083671663
    H               -0.725174763408    -0.013093965441    -0.023197328337
    C               -1.313548763408    -0.001922965441     2.052253671663
    H               -2.377998763408    -0.002771965441     1.863517671663
    C               -0.837944763408     0.004995034559     3.361212671663
    H               -1.530128763408     0.009578034559     4.194403671663

""")

GEOS['%s-%s-%s' % (dbse, '48-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.309632888410    -0.005742478191    -1.991900423371
    C                0.638155888410    -1.144358478191    -2.214283423371
    H                1.187287888410    -2.060894478191    -2.037411423371
    C               -0.685811111590    -1.190078478191    -2.644427423371
    H               -1.170324111590    -2.142687478191    -2.804620423371
    C               -1.360327111590     0.007767521809    -2.858940423371
    H               -2.388487111590     0.013038521809    -3.193377423371
    C               -0.677688111590     1.198724521809    -2.632082423371
    H               -1.155756111590     2.156191521809    -2.782363423371
    C                0.645765888410     1.139612521809    -2.202224423371
    H                1.200972888410     2.050543521809    -2.015646423371
    --
    0 1
    N                0.442728888410     0.006048521809     3.750433576629
    C                1.313396888410     0.000435521809     2.731020576629
    H                2.361253888410     0.001684521809     3.005323576629
    C                0.940010888410    -0.006632478191     1.389192576629
    H                1.687471888410    -0.010796478191     0.608093576629
    C               -0.416037111590    -0.008046478191     1.076777576629
    H               -0.745766111590    -0.013501478191     0.046495576629
    C               -1.334140111590    -0.002331478191     2.121946576629
    H               -2.398590111590    -0.003180478191     1.933211576629
    C               -0.858537111590     0.004586521809     3.430905576629
    H               -1.550720111590     0.009169521809     4.264096576629

""")

GEOS['%s-%s-%s' % (dbse, '48-1.05', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.330034559353    -0.005337978191    -2.060951334712
    C                0.658557559353    -1.143953978191    -2.283334334712
    H                1.207689559353    -2.060489978191    -2.106462334712
    C               -0.665409440647    -1.189673978191    -2.713478334712
    H               -1.149922440647    -2.142282978191    -2.873671334712
    C               -1.339925440647     0.008172021809    -2.927991334712
    H               -2.368085440647     0.013443021809    -3.262428334712
    C               -0.657286440647     1.199129021809    -2.701133334712
    H               -1.135354440647     2.156596021809    -2.851414334712
    C                0.666167559353     1.140017021809    -2.271275334712
    H                1.221374559353     2.050948021809    -2.084697334712
    --
    0 1
    N                0.422327559353     0.005644021809     3.819484665288
    C                1.292995559353     0.000031021809     2.800071665288
    H                2.340851559353     0.001280021809     3.074374665288
    C                0.919609559353    -0.007036978191     1.458243665288
    H                1.667070559353    -0.011200978191     0.677143665288
    C               -0.436439440647    -0.008450978191     1.145827665288
    H               -0.766167440647    -0.013905978191     0.115546665288
    C               -1.354542440647    -0.002735978191     2.190997665288
    H               -2.418992440647    -0.003584978191     2.002261665288
    C               -0.878938440647     0.004182021809     3.499956665288
    H               -1.571122440647     0.008765021809     4.333147665288

""")

GEOS['%s-%s-%s' % (dbse, '48-1.1', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.350239230217    -0.004937471816    -2.129335328337
    C                0.678762230217    -1.143553471816    -2.351718328337
    H                1.227894230217    -2.060089471816    -2.174846328337
    C               -0.645204769783    -1.189273471816    -2.781862328337
    H               -1.129717769783    -2.141882471816    -2.942055328337
    C               -1.319720769783     0.008572528184    -2.996375328337
    H               -2.347880769783     0.013843528184    -3.330812328337
    C               -0.637081769783     1.199529528184    -2.769517328337
    H               -1.115149769783     2.156996528184    -2.919798328337
    C                0.686372230217     1.140417528184    -2.339659328337
    H                1.241579230217     2.051348528184    -2.153081328337
    --
    0 1
    N                0.402122230217     0.005243528184     3.887868671663
    C                1.272790230217    -0.000369471816     2.868455671663
    H                2.320647230217     0.000878528184     3.142757671663
    C                0.899405230217    -0.007437471816     1.526627671663
    H                1.646865230217    -0.011601471816     0.745527671663
    C               -0.456643769783    -0.008851471816     1.214211671663
    H               -0.786371769783    -0.014306471816     0.183930671663
    C               -1.374746769783    -0.003136471816     2.259381671663
    H               -2.439196769783    -0.003985471816     2.070645671663
    C               -0.899142769783     0.003781528184     3.568340671663
    H               -1.591326769783     0.008364528184     4.401531671663

""")

GEOS['%s-%s-%s' % (dbse, '48-1.25', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.410007394785    -0.003752370486    -2.331624005656
    C                0.738530394785    -1.142368370486    -2.554007005656
    H                1.287662394785    -2.058904370486    -2.377135005656
    C               -0.585436605215    -1.188088370486    -2.984151005656
    H               -1.069949605215    -2.140697370486    -3.144344005656
    C               -1.259952605215     0.009757629514    -3.198664005656
    H               -2.288112605215     0.015028629514    -3.533101005656
    C               -0.577313605215     1.200714629514    -2.971806005656
    H               -1.055381605215     2.158181629514    -3.122087005656
    C                0.746140394785     1.141602629514    -2.541948005656
    H                1.301347394785     2.052533629514    -2.355370005656
    --
    0 1
    N                0.342354394785     0.004057629514     4.090156994344
    C                1.213022394785    -0.001554370486     3.070743994344
    H                2.260879394785    -0.000306370486     3.345046994344
    C                0.839636394785    -0.008622370486     1.728915994344
    H                1.587097394785    -0.012786370486     0.947816994344
    C               -0.516411605215    -0.010036370486     1.416500994344
    H               -0.846140605215    -0.015492370486     0.386218994344
    C               -1.434514605215    -0.004321370486     2.461670994344
    H               -2.498965605215    -0.005171370486     2.272934994344
    C               -0.958911605215     0.002596629514     3.770628994344
    H               -1.651094605215     0.007179629514     4.603820994344

""")

GEOS['%s-%s-%s' % (dbse, '48-1.5', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.507738888410    -0.001814471816    -2.662401910621
    C                0.836261888410    -1.140430471816    -2.884784910621
    H                1.385393888410    -2.056966471816    -2.707912910621
    C               -0.487705111590    -1.186150471816    -3.314928910621
    H               -0.972218111590    -2.138759471816    -3.475121910621
    C               -1.162221111590     0.011695528184    -3.529441910621
    H               -2.190381111590     0.016966528184    -3.863878910621
    C               -0.479582111590     1.202652528184    -3.302583910621
    H               -0.957650111590     2.160119528184    -3.452864910621
    C                0.843871888410     1.143540528184    -2.872725910621
    H                1.399078888410     2.054471528184    -2.686147910621
    --
    0 1
    N                0.244622888410     0.002120528184     4.420935089379
    C                1.115290888410    -0.003492471816     3.401522089379
    H                2.163147888410    -0.002244471816     3.675825089379
    C                0.741904888410    -0.010560471816     2.059694089379
    H                1.489365888410    -0.014724471816     1.278594089379
    C               -0.614143111590    -0.011974471816     1.747279089379
    H               -0.943872111590    -0.017429471816     0.716997089379
    C               -1.532246111590    -0.006259471816     2.792448089379
    H               -2.596696111590    -0.007108471816     2.603712089379
    C               -1.056643111590     0.000658528184     4.101407089379
    H               -1.748826111590     0.005241528184     4.934598089379

""")

GEOS['%s-%s-%s' % (dbse, '48-2.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    N                1.699533983445     0.001988528184    -3.311543834712
    C                1.028056983445    -1.136627471816    -3.533926834712
    H                1.577188983445    -2.053163471816    -3.357054834712
    C               -0.295910016555    -1.182347471816    -3.964070834712
    H               -0.780423016555    -2.134956471816    -4.124263834712
    C               -0.970426016555     0.015498528184    -4.178583834712
    H               -1.998586016555     0.020769528184    -4.513020834712
    C               -0.287787016555     1.206455528184    -3.951725834712
    H               -0.765855016555     2.163922528184    -4.102006834712
    C                1.035666983445     1.147343528184    -3.521867834712
    H                1.590873983445     2.058274528184    -3.335289834712
    --
    0 1
    N                0.052827983445    -0.001682471816     5.070077165288
    C                0.923495983445    -0.007295471816     4.050664165288
    H                1.971351983445    -0.006047471816     4.324967165288
    C                0.550109983445    -0.014363471816     2.708836165288
    H                1.297570983445    -0.018527471816     1.927736165288
    C               -0.805938016555    -0.015777471816     2.396420165288
    H               -1.135667016555    -0.021232471816     1.366139165288
    C               -1.724042016555    -0.010062471816     3.441590165288
    H               -2.788492016555    -0.010911471816     3.252854165288
    C               -1.248438016555    -0.003144471816     4.750549165288
    H               -1.940622016555     0.001438528184     5.583740165288

""")

GEOS['%s-%s-%s' % (dbse, '49-0.9', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.845253410999     1.020410493593    -2.107118882438
    H                1.506582410999     1.865713493593    -1.973191882438
    C                1.375685410999    -0.254963506407    -2.295412882438
    H                2.447360410999    -0.398981506407    -2.303721882438
    C                0.524244410999    -1.344552506407    -2.473381882438
    H                0.935903410999    -2.333534506407    -2.621650882438
    C               -0.857539589001    -1.158971506407    -2.460189882438
    H               -1.518204589001    -2.003966506407    -2.598210882438
    C               -1.387869589001     0.116125493593    -2.268960882438
    H               -2.459541589001     0.259907493593    -2.256541882438
    C               -0.536436589001     1.205913493593    -2.093834882438
    H               -0.948747589001     2.195297493593    -1.949207882438
    --
    0 1
    N                0.027767410999     0.386640493593     3.613114117562
    C                0.182451410999     1.324601493593     2.668237117562
    H                0.324221410999     2.337127493593     3.026139117562
    C                0.168241410999     1.057312493593     1.301738117562
    H                0.298481410999     1.858337493593     0.587853117562
    C               -0.016096589001    -0.254317506407     0.876448117562
    H               -0.032662589001    -0.497436506407    -0.176899882438
    C               -0.177945589001    -1.239351506407     1.845307117562
    H               -0.324488589001    -2.274562506407     1.570535117562
    C               -0.148742589001    -0.872018506407     3.188396117562
    H               -0.272339589001    -1.619532506407     3.962497117562

""")

GEOS['%s-%s-%s' % (dbse, '49-0.95', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.846283395453     1.037016444330    -2.188821088914
    H                1.507612395453     1.882319444330    -2.054894088914
    C                1.376715395453    -0.238357555670    -2.377115088914
    H                2.448390395453    -0.382375555670    -2.385424088914
    C                0.525274395453    -1.327946555670    -2.555084088914
    H                0.936933395453    -2.316928555670    -2.703353088914
    C               -0.856509604547    -1.142365555670    -2.541892088914
    H               -1.517174604547    -1.987360555670    -2.679913088914
    C               -1.386839604547     0.132731444330    -2.350663088914
    H               -2.458511604547     0.276513444330    -2.338244088914
    C               -0.535406604547     1.222519444330    -2.175537088914
    H               -0.947717604547     2.211903444330    -2.030910088914
    --
    0 1
    N                0.026750395453     0.370243444330     3.693787911086
    C                0.181434395453     1.308205444330     2.748910911086
    H                0.323204395453     2.320731444330     3.106812911086
    C                0.167224395453     1.040915444330     1.382411911086
    H                0.297464395453     1.841941444330     0.668525911086
    C               -0.017113604547    -0.270714555670     0.957120911086
    H               -0.033679604547    -0.513832555670    -0.096227088914
    C               -0.178962604547    -1.255748555670     1.925979911086
    H               -0.325505604547    -2.290959555670     1.651208911086
    C               -0.149759604547    -0.888415555670     3.269069911086
    H               -0.273356604547    -1.635929555670     4.043170911086

""")

GEOS['%s-%s-%s' % (dbse, '49-1.0', 'dimer')] = qcdb.Molecule("""
    units Angstrom
    0 1
    C                0.847292151223     1.053280336611    -2.268840447569
    H                1.508621151223     1.898583336611    -2.134913447569
    C                1.377724151223    -0.222093663389    -2.457134447569
    H                2.449399151223    -0.366111663389    -2.465443447569
    C                0.526283151223    -1.311682663389    -2.635103447569
    H                0.937942151223    -2.300664663389    -2.783372447569
    C               -0.855500848777    -1.126101663389    -2.621911447569
    H               -1.516165848777    -1.971096663389    -2.759932447569
    C               -1.385830848777     0.148995336611    -2.430682447569
    H               -2.457502848777     0.292777336611    -2.418263447569
    C               -0.534397848777     1.238783336611    -2.255556447569
    H               -0.946708848777     2.228167336611    -2.110929447569
    --
    0 1
    N                0.025754151223     0.354184336611     3.772799552431
    C                0.180438151223     1.292146336611     2.827922552431
    H                0.322209151223     2.304672336611     3.185824552431
    C                0.166229151223     1.024856336611     1.461423552431
    H                0.296469151223     1.825882336611     0.747537552431
    C               -0.018109848777    -0.286773663389     1.036132552431
    H               -0.034675848777    -0.529891663389    -0.017214447569
    C               -0.179958848777    -1.271807663389     2.004992552431
    H               -0.326501848777    -2.307018663389     1.730220552431
    C               -0.150755848777    -0.904474663389     3.348081552431
    H               -0.274351848777    -1.651988663389     4.122182552431

""")

GEOS['%s-%s-%s' % (dbse, '49-1.05', 'dimer')] = qcdb