---
title: StrictYAML
---

{{< github-stars user="crdoconnor" project="strictyaml" >}}


StrictYAML is a [type-safe](https://en.wikipedia.org/wiki/Type_safety) YAML parser
that parses and validates a [restricted subset](features-removed) of the [YAML](what-is-yaml)
specification.

Priorities:

- Beautiful API
- Refusing to parse [the ugly, hard to read and insecure features of YAML](features-removed) like [the Norway problem](why/implicit-typing-removed).
- Strict validation of markup and straightforward type casting.
- Clear, readable exceptions with **code snippets** and **line numbers**.
- Acting as a near-drop in replacement for pyyaml, ruamel.yaml or poyo.
- Ability to read in YAML, make changes and write it out again with comments preserved.
- [Not speed](why/speed-not-a-priority), currently.


Simple example:

```yaml
# All about the character
name: Ford Prefect
age: 42
possessions:
- Towel

```


```python
from strictyaml import load, Map, Str, Int, Seq, YAMLError

```





Default parse result:


```python
>>> load(yaml_snippet)
YAML({'name': 'Ford Prefect', 'age': '42', 'possessions': ['Towel']})
```



All data is string, list or OrderedDict:


```python
>>> load(yaml_snippet).data
{'name': 'Ford Prefect', 'age': '42', 'possessions': ['Towel']}
```



Quickstart with schema:


```python
from strictyaml import load, Map, Str, Int, Seq, YAMLError

schema = Map({"name": Str(), "age": Int(), "possessions": Seq(Str())})

```





42 is now parsed as an integer:


```python
>>> person = load(yaml_snippet, schema)
>>> person.data
{'name': 'Ford Prefect', 'age': 42, 'possessions': ['Towel']}
```



A YAMLError will be raised if there are syntactic problems, violations of your schema or use of disallowed YAML features:

```yaml
# All about the character
name: Ford Prefect
age: 42

```






For example, a schema violation:


```python
try:
    person = load(yaml_snippet, schema)
except YAMLError as error:
    print(error)

```

```yaml
while parsing a mapping
  in "<unicode string>", line 1, column 1:
    # All about the character
     ^ (line: 1)
required key(s) 'possessions' not found
  in "<unicode string>", line 3, column 1:
    age: '42'
    ^ (line: 3)
```





If parsed correctly:


```python
from strictyaml import load, Map, Str, Int, Seq, YAMLError, as_document

schema = Map({"name": Str(), "age": Int(), "possessions": Seq(Str())})

```





You can modify values and write out the YAML with comments preserved:


```python
person = load(yaml_snippet, schema)
person['age'] = 43
print(person.as_yaml())

```

```yaml
# All about the character
name: Ford Prefect
age: 43
possessions:
- Towel
```





As well as look up line numbers:


```python
>>> person = load(yaml_snippet, schema)
>>> person['possessions'][0].start_line
5
```



And construct YAML documents from dicts or lists:


```python
print(as_document({"x": 1}).as_yaml())

```

```yaml
x: 1
```







## Install

```sh
$ pip install strictyaml
```


## Why StrictYAML?

There are a number of formats and approaches that can achieve more or
less the same purpose as StrictYAML. I've tried to make it the best one.
Below is a series of documented justifications:

- [Why avoid using environment variables as configuration?](why-not/environment-variables-as-config)
- [Why not use HJSON?](why-not/hjson)
- [Why not HOCON?](why-not/hocon)
- [Why not use INI files?](why-not/ini)
- [Why not use JSON Schema for validation?](why-not/json-schema)
- [Why not JSON for simple configuration files?](why-not/json)
- [Why not JSON5?](why-not/json5)
- [Why not use the YAML 1.2 standard? - we don't need a new standard!](why-not/ordinary-yaml)
- [Why not use kwalify with standard YAML to validate my YAML?](why-not/pykwalify)
- [Why not use Python's schema library (or similar) for validation?](why-not/python-schema)
- [Why not use SDLang?](why-not/sdlang)
- [What is wrong with TOML?](why-not/toml)
- [Why shouldn't I just use Python code for configuration?](why-not/turing-complete-code)
- [Why not use XML for configuration or DSLs?](why-not/xml)



## Using StrictYAML

How to:

- [Build a YAML document from scratch in code](using/alpha/howto/build-yaml-document)
- [Either/or schema validation of different, equally valid different kinds of YAML](using/alpha/howto/either-or-validation)
- [Labeling exceptions](using/alpha/howto/label-exceptions)
- [Merge YAML documents](using/alpha/howto/merge-yaml-documents)
- [Revalidate an already validated document](using/alpha/howto/revalidation)
- [Reading in YAML, editing it and writing it back out](using/alpha/howto/roundtripping)
- [Get line numbers of YAML elements](using/alpha/howto/what-line)
- [Parsing YAML without a schema](using/alpha/howto/without-a-schema)


Compound validators:

- [Fixed length sequences (FixedSeq)](using/alpha/compound/fixed-length-sequences)
- [Mappings combining defined and undefined keys (MapCombined)](using/alpha/compound/map-combined)
- [Mappings with arbitrary key names (MapPattern)](using/alpha/compound/map-pattern)
- [Mapping with defined keys and a custom key validator (Map)](using/alpha/compound/mapping-with-slug-keys)
- [Using a YAML object of a parsed mapping](using/alpha/compound/mapping-yaml-object)
- [Mappings with defined keys (Map)](using/alpha/compound/mapping)
- [Optional keys with defaults (Map/Optional)](using/alpha/compound/optional-keys-with-defaults)
- [Validating optional keys in mappings (Map)](using/alpha/compound/optional-keys)
- [Sequences of unique items (UniqueSeq)](using/alpha/compound/sequences-of-unique-items)
- [Sequence/list validator (Seq)](using/alpha/compound/sequences)
- [Updating document with a schema](using/alpha/compound/update)


Scalar validators:

- [Boolean (Bool)](using/alpha/scalar/boolean)
- [Parsing comma separated items (CommaSeparated)](using/alpha/scalar/comma-separated)
- [Datetimes (Datetime)](using/alpha/scalar/datetime)
- [Decimal numbers (Decimal)](using/alpha/scalar/decimal)
- [Email and URL validators](using/alpha/scalar/email-and-url)
- [Empty key validation](using/alpha/scalar/empty)
- [Enumerated scalars (Enum)](using/alpha/scalar/enum)
- [Floating point numbers (Float)](using/alpha/scalar/float)
- [Hexadecimal Integers (HexInt)](using/alpha/scalar/hexadecimal-integer)
- [Integers (Int)](using/alpha/scalar/integer)
- [Validating strings with regexes (Regex)](using/alpha/scalar/regular-expressions)
- [Parsing strings (Str)](using/alpha/scalar/string)


Restrictions:

- [Disallowed YAML](using/alpha/restrictions/disallowed-yaml)
- [Duplicate keys](using/alpha/restrictions/duplicate-keys)
- [Dirty load](using/alpha/restrictions/loading-dirty-yaml)



## Design justifications

There are some design decisions in StrictYAML which are controversial
and/or not obvious. Those are documented here:

- [What is wrong with duplicate keys?](why/duplicate-keys-disallowed)
- [What is wrong with explicit tags?](why/explicit-tags-removed)
- [What is wrong with flow-style YAML?](why/flow-style-removed)
- [The Norway Problem - why StrictYAML refuses to do implicit typing and so should you](why/implicit-typing-removed)
- [What is wrong with node anchors and references?](why/node-anchors-and-references-removed)
- [Why does StrictYAML not parse direct representations of Python objects?](why/not-parse-direct-representations-of-python-objects)
- [Why does StrictYAML only parse from strings and not files?](why/only-parse-strings-not-files)
- [Why is parsing speed not a high priority for StrictYAML?](why/speed-not-a-priority)
- [What is syntax typing?](why/syntax-typing-bad)
- [Why does StrictYAML make you define a schema in Python - a Turing-complete language?](why/turing-complete-schema)



## Star Contributors

- @wwoods
- @chrisburr
- @jnichols0

## Other Contributors

- @eulores
- @WaltWoods
- @ChristopherGS
- @gvx
- @AlexandreDecan
- @lots0logs
- @tobbez
- @jaredsampson
- @BoboTIG

StrictYAML also includes code from [ruamel.yaml](https://yaml.readthedocs.io/en/latest/), Copyright Anthon van der Neut.

## Contributing

- Before writing any code, please read the tutorial on [contributing to hitchdev libraries](https://hitchdev.com/approach/contributing-to-hitch-libraries/).
- Before writing any code, if you're proposing a new feature, please raise it on github. If it's an existing feature / bug, please comment and briefly describe how you're going to implement it.
- All code needs to come accompanied with a story that exercises it or a modification to an existing story. This is used both to test the code and build the documentation.