"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TsicTable = void 0;
const bin_util_1 = require("@ot-builder/bin-util");
const errors_1 = require("@ot-builder/errors");
const ot_vtt_private_1 = require("@ot-builder/ot-vtt-private");
const primitive_1 = require("@ot-builder/primitive");
const iconv = require("iconv-lite");
exports.TsicTable = {
    ...(0, bin_util_1.Read)((view, ds) => {
        const majorVersion = view.uint16();
        const minorVersion = view.uint16();
        errors_1.Assert.SubVersionSupported("TSIC", majorVersion, minorVersion, [1, 1]);
        const flags = view.uint16();
        const axisCount = view.uint16();
        errors_1.Assert.SizeMatch("TSIC::axisCount", axisCount, ds.length);
        const recordCount = view.uint16();
        const reserved = view.uint16();
        for (const [iDim, dim] of ds.entries()) {
            const axisArrayItem = view.next(primitive_1.Tag);
            errors_1.Assert.TagMatch(`TSIC::AxisArray[${iDim}]`, axisArrayItem, dim.tag);
        }
        const locations = [];
        for (let iRecord = 0; iRecord < recordCount; iRecord++) {
            locations[iRecord] = new Map();
            for (const dim of ds)
                locations[iRecord].set(dim, view.next(primitive_1.F2D14));
        }
        const table = new ot_vtt_private_1.TSIC.Table();
        for (let iRecord = 0; iRecord < recordCount; iRecord++) {
            table.records[iRecord] = view.next(TsicRecord, locations[iRecord]);
        }
        return table;
    }),
    ...(0, bin_util_1.Write)((frag, table, ds) => {
        frag.uint16(1).uint16(1); // Version
        frag.uint16(0); // flags
        frag.uint16(ds.length); // axisCount
        frag.uint16(table.records.length); // recordCount
        frag.uint16(0); // reserved
        for (const dim of ds)
            frag.push(primitive_1.Tag, dim.tag); // xisArray
        for (let iRecord = 0; iRecord < table.records.length; iRecord++) {
            for (const dim of ds) {
                frag.push(primitive_1.F2D14, table.records[iRecord].location.get(dim) || 0); // RecordLocations
            }
        }
        for (let iRecord = 0; iRecord < table.records.length; iRecord++) {
            frag.push(TsicRecord, table.records[iRecord]);
        }
    })
};
const TsicRecord = {
    ...(0, bin_util_1.Read)((view, location) => {
        const flags = view.uint16();
        const numCVTEntries = view.uint16();
        const nameLength = view.uint16();
        const name = iconv.decode(view.bytes(2 * nameLength), `utf16-be`);
        const cvtArray = view.array(numCVTEntries, primitive_1.UInt16);
        const cvtValueArray = view.array(numCVTEntries, primitive_1.Int16);
        const cvtMappings = new Map();
        for (let iCvt = 0; iCvt < numCVTEntries; iCvt++) {
            cvtMappings.set(cvtArray[iCvt], cvtValueArray[iCvt]);
        }
        const record = { name, location, cvtValues: cvtMappings };
        return record;
    }),
    ...(0, bin_util_1.Write)((frag, record) => {
        // Do not sort -- order may carry semantics
        const cvtEntriesArray = Array.from(record.cvtValues);
        const name = record.name || "";
        frag.uint16(0); // flags
        frag.uint16(cvtEntriesArray.length); // numCVTEntries
        frag.uint16(name.length); // nameLength
        frag.bytes(iconv.encode(name, "utf16-be")); // nameArray
        for (let iCvt = 0; iCvt < cvtEntriesArray.length; iCvt++)
            frag.uint16(cvtEntriesArray[iCvt][0]); // CVTArray
        for (let iCvt = 0; iCvt < cvtEntriesArray.length; iCvt++)
            frag.int16(cvtEntriesArray[iCvt][1]); // CVTValueArray
    })
};
//# sourceMappingURL=index.js.map