/*
 * @(#)xrubik.c
 *
 * Copyright 1993 - 2025  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Rubik */

#ifndef WINVER
#include "version.h"
static const char aboutHelp[] = {
"Rubik Version " VERSION "\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd AT verizon.net>\n"
"The latest version is at: "
"https://www.sillycycle.com/puzzles.html"
};

static const char synopsisHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}]\n"
"[-frame {color}] [-delay msecs] [-[no]sound]\n"
"[-moveSound {filename}] [-{font|fn} {fontname}]\n"
"[-view {int}] [-size{X|Y|Z} {int}] [-[no]orient]\n"
"[-[no]practice] [-userName {string}]\n"
"[-scoreFile {filename}] [-scores] [-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA) || defined(WINVER)
static const char descriptionHelp[] = {
"The original puzzle has 9 squares per face (size = 3).  "
"The puzzle was designed by Erno Rubik and called the\n"
"Rubik's Cube.  This has 8!*12!*3^8*2^12/12 or 4.3 * 10^19 "
"different combinations.\n"
"The Pocket Cube has 4 squares per face (size = 2) also "
"designed by Erno Rubik.  This has 7!*3^6 or 3,674,160\n"
"different combinations.\n"
"Rubik's Revenge has 16 squares per face (size = 4) also "
"designed by Erno Rubik.  This has 7!*3^6*24!*24!/(4!)^6\n"
"or 7.4 * 10^46 different combinations.\n"
"5x5x5 Cube.  This has 8!*12!*3^7*2^10*(24!)^3/(4!)^12 "
"or 2.83 * 10^74 different combinations.\n"
"There is also the Magic Domino 3x3x2 cube which has "
"(8!)^2/4 or 406,425,600 combinations.\n"
"A physical 6x6x6 cube is possible but to my knowledge no "
"one has been too successful in building one.  7x7x7 is\n"
"also possible, but here one must make the center most "
"cubes smaller than the outside cubes, so the corners do\n"
"not fall off when turned."
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a piece.  Release "
"\"mouse-left\" button on a piece on the same face and\n"
"in the same row.  The pieces will then turn towards "
"where the mouse button was released.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode\n"
"the record should say \"practice\").  This is good for "
"learning moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to read (get) a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"Press \"S\" or \"s\" keys to start auto-solver.  Only works "
"on 2x2x2, 3x3x3, and 4x4x4 cubes.\n"
"\n"
"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
"has to orient the faces in orient mode, besides\n"
"getting all the faces to be the same color.  To do this "
"one has to get the lines to be oriented in\n"
"the same direction, this only matters with center "
"piece, if at all (i.e. those that are not on a\n"
"corner or edge).  This does add complexity so there are "
"2 sets of records.\n"
"\n"
"Press \"I\" or \"i\" keys to increase the number of pieces.\n"
"\n"
"Press \"D\" or \"d\" keys to decrease the number of pieces.\n"
"\n"
"Press \"x\" key to increase the number of pieces along "
"the x axis.\n"
"\n"
"Press \"X\" key to decrease the number of pieces along "
"the x axis.\n"
"\n"
"Press \"y\" key to increase the number of pieces along "
"the y axis.\n"
"\n"
"Press \"Y\" key to decrease the number of pieces along "
"the y axis.\n"
"\n"
"Press \"j\" key to increase the number of pieces along "
"the z axis.\n"
"\n"
"Press \"J\" key to decrease the number of pieces along "
"the z axis.\n"
"\n"
"Press \"V\" or \"v\" keys to change the view of the puzzle.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of pieces.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of pieces.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Rubik2d as:\n"
"  /    Counterclockwise\n"
"  8    Up\n"
"  ^\n"
"4<5>6  Left, Clockwise, Right\n"
"  v\n"
"  2    Down\n"
"\n"
"Key pad for Rubik3d, use must use your intuition (is this "
"a cop out or what?).  The key pad is defined\n"
"differently depending on which side of the cube your mouse "
"is pointing at.  One thing that stays the\n"
"same is \"5\" is Clockwise and \"/\" is Counterclockwise.\n"
"\n"
"Use the control key and the left mouse button, keypad, or "
"arrow keys to move the whole cube.  This is\n"
"not recorded as a turn."
};

static const char referencesHelp[] = {
"Inside Rubik's Cube and Beyond by Christoph Bandelow, "
"Birkhauser, 1982 pp 44, 45, 88, 89.\n"
"Magic Cubes 1996 Catalog of Dr. Christoph Bandelow.\n"
"The Simple Solution To Rubik's Cube, James G. Nourse, "
"June 1981.\n"
"Rubik's Cube Newsletter by Ideal Aug 1982 Vol.1 No. 2.\n"
"Rubik's Cube The Solution, Ideal Toy Corporation, 1981.\n"
"Rubik's Revenge Puzzle The Solution, Ideal Toy "
"Corporation, 1982."
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented for size nxnxn where "
"n < 5."
};

#include "file.h"
#ifdef WINVER
#include "RubikP.h"
#define TITLE "wrubik"

static RubikRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "C:/ProgramData/wpuzzles"
#else
#define SCOREPATH "C:\\ProgramData\\wpuzzles"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->rubik.started = b
#else
#include "xwin.h"
#include "xgui.h"
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Rubik.h"
#include "Rubik2d.h"
#ifdef HAVE_OPENGL
#include "RubikGL.h"
#else
#include "Rubik3d.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#ifdef CONSTPIXMAPS
#include "icons/16x16/rubik.xpm"
#include "icons/22x22/rubik.xpm"
#include "icons/24x24/rubik.xpm"
#include "icons/32x32/rubik.xpm"
#include "icons/48x48/rubik.xpm"
#include "icons/64x64/rubik.xpm"
#else
#include "pixmaps/16x16/rubik.xpm"
#include "pixmaps/22x22/rubik.xpm"
#include "pixmaps/24x24/rubik.xpm"
#include "pixmaps/32x32/rubik.xpm"
#include "pixmaps/48x48/rubik.xpm"
#include "pixmaps/64x64/rubik.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?rubik_16x16:rubik_22x22):\
(((s)<=24)?rubik_24x24:rubik_32x32)):\
(((s)<=48)?rubik_48x48:rubik_64x64)))
#endif
#include "pixmaps/64x64/rubik.xbm"
#define DEFINE_XBM (char *) rubik_64x64_bits, rubik_64x64_width, rubik_64x64_height
#ifdef HAVE_EDITRES
#ifdef __VMS
#include <Xmu/Editres.h>
#else
#include <X11/Xmu/Editres.h>
#endif
#endif
#ifndef SCOREPATH
#ifdef __VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef SCOREFILE
#define SCOREFILE "rubik.scores"
#endif

#define MAX_FACETS 6
#define NEVER (-1)
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[2][MAX_FACETS - MIN_FACETS + 1]
[MAX_FACETS - MIN_FACETS + 1][MAX_FACETS - MIN_FACETS + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static Widget movesText, recordText, message;
static Widget orientToggle, practiceToggle;
static Widget soundMenuItem;
static Widget practiceDialog, randomizeDialog, solveDialog;
static Widget descriptionDialog, featuresDialog;
static Widget synopsisDialog, referencesDialog, aboutDialog;
static char buff[MESSAGE_LENGTH];
static const char *sizeLabels[] =
{
	"Size X:",
	"Size Y:",
	"Size Z:"
};
static const char *speedLabel = "Animation Speed:";
#define PRINT_MESSAGE(b) printState(message, b)
#define MIN_SPEED 1
#define MAX_SPEED 50
#elif !defined(WINVER)
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#ifdef HAVE_MOTIF
static Widget facetXChanger, facetYChanger, facetZChanger, speedChanger;
#elif defined(HAVE_ATHENA)
static Widget facetXSlider, facetYSlider, facetZSlider, speedSlider;
static Widget facetXSliderLabel, facetYSliderLabel, facetZSliderLabel;
static Widget speedSliderLabel;
static const char *fileTypes[] =
{
	"Get",
	"Write",
	"Exit"
};
#define numFileTypes (sizeof(fileTypes)/sizeof(fileTypes[0]))
static const char *playTypes[] =
{
	"Undo",
	"Redo",
	"Clear",
#ifdef EXTRA
	"Practice",
#endif
	"Randomize",
	"Auto-solve",
#ifdef EXTRA
	"Increment Size",
	"Decrement Size",
	"Oriented",
	"Speed >",
	"Slow <",
#endif
	"Sound @"
#ifdef HAVE_OPENGL
	, "View"
#endif
};
#define numPlayTypes (sizeof(playTypes)/sizeof(playTypes[0]))
static const char *helpTypes[] =
{
	"Description...",
	"Features...",
	"Synopsis...",
	"References...",
	"About..."
};
#define numHelpTypes (sizeof(helpTypes)/sizeof(helpTypes[0]))
#else
#ifndef WINVER
static Widget shell;
#endif
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle2d, puzzle3d;
static char userNameDsp[USER_NAME_LENGTH] = "";
#define WINDOW_HEIGHT 300

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w,
		XmNlabelString, xmstr, NULL);
	XmStringFree(xmstr);
}
#elif defined(HAVE_ATHENA)
static void
printState(Widget w, char *msg)
{
	XtVaSetValues(w,
		XtNlabel, msg, NULL);
	if (w == movesText)
		XtVaSetValues(w,
			XtNwidth, 64, NULL);
}
#endif

static void
printRecords(void)
{
	int i, j, k, orient;

	(void) printf("            RUBIK  HALL OF FAME\n\n");
	(void) printf("ORIENT   X   Y   Z USER            MOVES\n");
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++)
			for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++)
				for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++) {
					if (puzzleRecord[orient][i][j][k].score > 0)
						(void) printf("%6d%4d%4d%4d %-12s%9d\n",
							orient, i + 1, j + 1, k + 1,
							puzzleRecord[orient][i][j][k].name,
							puzzleRecord[orient][i][j][k].score);
				}
}
#endif

static void
initRecords(void)
{
	int i, j, k, orient;

	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++)
			for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++)
				for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++) {
					puzzleRecord[orient][k][j][i].score = puzzleRecord[orient][k][i][j].score =
						puzzleRecord[orient][j][k][i].score = puzzleRecord[orient][j][i][k].score =
						puzzleRecord[orient][i][k][j].score = puzzleRecord[orient][i][j][k].score = NEVER;
					(void) strncpy(puzzleRecord[orient][k][j][i].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][k][i][j].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][j][k][i].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][j][i][k].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][i][k][j].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][i][j][k].name,
						NOACCESS, USER_NAME_LENGTH);
				}
}

static void
readRecords(void)
{
	FILE *fp;
	int i, j, k, n, orient;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++)
			for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++)
				for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++) {
					if (fscanf(fp, "%d %s\n", &n, userName) != 2) {
						(void) fprintf(stderr,
							"corrupt record, expecting integer and name for %d %d %d %d\n",
							orient, i, j, k);
						(void) fclose(fp);
						return;
					}
					if (n <= puzzleRecord[orient][i][j][k].score ||
							puzzleRecord[orient][i][j][k].score <= NEVER) {
						puzzleRecord[orient][k][j][i].score = puzzleRecord[orient][k][i][j].score =
							puzzleRecord[orient][j][k][i].score = puzzleRecord[orient][j][i][k].score =
							puzzleRecord[orient][i][k][j].score = puzzleRecord[orient][i][j][k].score = n;
						(void) strncpy(puzzleRecord[orient][k][j][i].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][k][i][j].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][j][k][i].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][j][i][k].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][i][k][j].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][i][j][k].name,
							userName, USER_NAME_LENGTH);
					}
				}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int i, j, k, orient;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Cannot write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		lockFile[FILE_NAME_LENGTH - 6] = '\0';
		(void) strncat(lockFile, ".lock", 6);
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (orient = 0; orient < 2; orient++) {
			for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++) {
				for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++) {
					for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++)
						(void) fprintf(fp, "%d %s\n",
							puzzleRecord[orient][i][j][k].score,
							puzzleRecord[orient][i][j][k].name);
					if (j < MAX_FACETS - MIN_FACETS ||
							i < MAX_FACETS - MIN_FACETS ||
							orient < 1)
						(void) fprintf(fp, "\n");
				}
				if (i < MAX_FACETS - MIN_FACETS ||
						orient < 1)
					(void) fprintf(fp, "\n");
			}
			if (orient < 1)
				(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int sizeX, int sizeY, int sizeZ, Boolean orient, Boolean practice)
{
	int i = sizeX - MIN_FACETS, j = sizeY - MIN_FACETS, k = sizeZ - MIN_FACETS;
	int l = (orient) ? 1 : 0;

	if (practice) {
		(void) strncpy(recordDsp, "practice", 9);
	} else if (sizeX > MAX_FACETS ||
			sizeY > MAX_FACETS || sizeZ > MAX_FACETS) {
		(void) strncpy(recordDsp, "NOT RECORDED", 13);
	} else if (puzzleRecord[l][i][j][k].score <= NEVER) {
		(void) strncpy(recordDsp, "NEVER ", 7);
		(void) strncat(recordDsp, NOACCESS, strlen(NOACCESS) + 1);
	} else {
#ifdef HAVE_SNPRINTF
		(void) snprintf(recordDsp, MESSAGE_LENGTH, "%d %s",
			puzzleRecord[l][i][j][k].score,
			puzzleRecord[l][i][j][k].name);
#else
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[l][i][j][k].score,
			puzzleRecord[l][i][j][k].name);
#endif
	}
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(recordText, recordDsp);
#endif
}

#ifndef WINVER
/* There is probably a better way to assure that they are the same
 * but I do not know it off hand. */
static void
makeEquivalent(String *userName, String *scoreFile,
		int *sizeX, int *sizeY, int *sizeZ,
		Boolean *orient, Boolean *practice, int *delay)
{
	Boolean mono, reverse;
	Boolean scoreOnly, versionOnly;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAX_FACES];

	XtVaGetValues(puzzle2d,
		XtNuserName, userName,
		XtNscoreFile, scoreFile,
		XtNsizeX, sizeX,
		XtNsizeY, sizeY,
		XtNsizeZ, sizeZ,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverseVideo, &reverse,
		XtNdelay, delay,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]),
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf("%s\n", aboutHelp);
		exit(0);
	}
	if (strcmp(*scoreFile, "")) {
		(void) strncpy(scoreFileName, *scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
	XtVaSetValues(puzzle2d,
		XtNdirection, IGNORE_DIR,
		XtNstart, False, NULL);
	XtVaSetValues(puzzle3d,
		XtNuserName, *userName,
		XtNsizeX, *sizeX,
		XtNsizeY, *sizeY,
		XtNsizeZ, *sizeZ,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverseVideo, reverse,
		XtNdelay, *delay,
		XtNdirection, IGNORE_DIR,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
}
#endif

static void
printStatus(char *msg, int nMoves
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		, int dim, int sizeX, int sizeY, int sizeZ
#endif
		)
{
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(message, msg);
	intCpy(buff, nMoves);
	printState(movesText, buff);
#else
	char titleDsp[TITLE_LENGTH];
	char dimDsp[3];
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) strncpy(dimDsp, "GL", 3);
	else
#endif
	if (dim == 3)
		(void) strncpy(dimDsp, "3d", 3);
	else
		(void) strncpy(dimDsp, "2d", 3);
#ifdef HAVE_SNPRINTF
	(void) snprintf(titleDsp, TITLE_LENGTH,
		"%s%s: %dx%dx%d@ (%d/%s) - %s",
		progDsp, dimDsp, sizeX, sizeY, sizeZ,
		nMoves, recordDsp, msg);
#else
	(void) sprintf(titleDsp,
		"%s%s: %dx%dx%d@ (%d/%s) - %s",
		progDsp, dimDsp, sizeX, sizeY, sizeZ,
		nMoves, recordDsp, msg);
#endif
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent((dim == 2) ? puzzle2d : puzzle3d),
		XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
handleSolved(int counter, int sizeX, int sizeY, int sizeZ, Boolean orient)
{
	int i = sizeX - MIN_FACETS, j = sizeY - MIN_FACETS, k = sizeZ - MIN_FACETS;
	int l = (orient) ? 1 : 0;

	if (sizeX <= MAX_FACETS && sizeY <= MAX_FACETS && sizeZ <= MAX_FACETS &&
			(counter < puzzleRecord[l][i][j][k].score ||
			puzzleRecord[l][i][j][k].score <= NEVER)) {
		readRecords();	/* Maybe its been updated by another */
		puzzleRecord[l][i][j][k].score = counter;
		(void) strncpy(puzzleRecord[l][i][j][k].name, userNameDsp,
			USER_NAME_LENGTH);
		if ((sizeX <= 2 && sizeY <= 2) || (sizeY <= 2 && sizeZ <= 2) ||
				(sizeZ <= 2 && sizeX <= 2) || (orient &&
				(counter < puzzleRecord[!l][i][j][k].score ||
				puzzleRecord[!l][i][j][k].score <= NEVER))) {
			puzzleRecord[!l][i][j][k].score = counter;
			(void) strncpy(puzzleRecord[!l][i][j][k].name,
				userNameDsp, USER_NAME_LENGTH);
		}
		writeRecords();
		printRecord(sizeX, sizeY, sizeZ, orient, False);
		return True;
	}
	return False;
}

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static int
getSpeed(int delay)
{
	return MAX_SPEED + MIN_SPEED - delay - 1;
}

static int
getDelay(int speed)
{
	return MAX_SPEED + MIN_SPEED - speed - 1;
}

static void
orientCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int sizeX, sizeY, sizeZ;
	Boolean orient, practice, old;

#ifdef HAVE_MOTIF
	orient = cbs->set;
#elif defined(HAVE_ATHENA)
	orient = state;
#endif
	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNorient, &old,
		XtNpractice, &practice, NULL);
	if (old == orient)
		return;
	XtVaSetValues(puzzle3d,
		XtNorient, orient, NULL);
	XtVaSetValues(puzzle2d,
		XtNorient, orient, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}

static void
practiceCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int sizeX, sizeY, sizeZ;
	Boolean orient, practice, old;

#ifdef HAVE_MOTIF
	practice = cbs->set;
#elif defined(HAVE_ATHENA)
	practice = state;
#endif
	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNorient, &orient,
		XtNpractice, &old, NULL);
	if (old == practice)
		return;
	XtVaSetValues(puzzle3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(puzzle2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ, orient, practice);
	if (practice)
		messageDsp[0] = '\0';
	else
		(void) strncpy(messageDsp, "Randomize (z) to start",
			MESSAGE_LENGTH);
	printState(message, messageDsp);
}

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
pieceXUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeX;

	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX, NULL);
	sizeX++;
	setScale(facetXSlider, facetXSliderLabel, sizeX,
		MIN_FACETS, MAX_FACETS, True);
	XtVaSetValues(puzzle2d,
		XtNsizeX, sizeX, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeX, sizeX, NULL);
}

static void
pieceXDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeX;

	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX, NULL);
	sizeX--;
	if (sizeX < MIN_FACETS) {
		return;
	}
	setScale(facetXSlider, facetXSliderLabel, sizeX,
		MIN_FACETS, MAX_FACETS, True);
	XtVaSetValues(puzzle2d,
		XtNsizeX, sizeX, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeX, sizeX, NULL);
}

#else

static void
pieceXChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int sizeX, sizeY, sizeZ, old;
	Boolean orient, practice;
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	int limit;
	sizeX = cbs->position;
#else
	sizeX = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &sizeX,
			MIN_FACETS, MAX_FACETS, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(puzzle2d,
		XtNsizeX, &old,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old == sizeX)
		return;
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	XtVaGetValues(facetXChanger,
		XmNmaximumValue, &limit, NULL);
	if (sizeX >= limit)
		XtVaSetValues(facetXChanger,
			XmNmaximumValue, sizeX + 1,
			XmNposition, sizeX, NULL);
#elif defined(HAVE_ATHENA)
	setScale(w, label, sizeX, MIN_FACETS, MAX_FACETS, False);
#endif
	XtVaSetValues(puzzle2d,
		XtNsizeX, sizeX, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeX, sizeX, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}
#endif

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
pieceYUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeY;

	XtVaGetValues(puzzle2d,
		XtNsizeY, &sizeY, NULL);
	sizeY++;
	setScale(facetYSlider, facetYSliderLabel, sizeY,
		MIN_FACETS, MAX_FACETS, True);
	XtVaSetValues(puzzle2d,
		XtNsizeY, sizeY, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeY, sizeY, NULL);
}

static void
pieceYDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeY;

	XtVaGetValues(puzzle2d,
		XtNsizeY, &sizeY, NULL);
	sizeY--;
	if (sizeY < MIN_FACETS) {
		return;
	}
	setScale(facetYSlider, facetYSliderLabel, sizeY,
		MIN_FACETS, MAX_FACETS, True);
	XtVaSetValues(puzzle2d,
		XtNsizeY, sizeY, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeY, sizeY, NULL);
}

#else

static void
pieceYChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int sizeX, sizeY, sizeZ, old;
	Boolean orient, practice;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	int limit;
	sizeY = cbs->position;
#else
	sizeY = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &sizeY,
			MIN_FACETS, MAX_FACETS, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX,
		XtNsizeY, &old,
		XtNsizeZ, &sizeZ,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old == sizeY)
		return;
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	XtVaGetValues(facetYChanger,
		XmNmaximumValue, &limit, NULL);
	if (sizeY >= limit)
		XtVaSetValues(facetYChanger,
			XmNmaximumValue, sizeY + 1,
			XmNposition, sizeY, NULL);
#elif defined(HAVE_ATHENA)
	setScale(w, label, sizeX, MIN_FACETS, MAX_FACETS, False);
#endif
	XtVaSetValues(puzzle2d,
		XtNsizeY, sizeY, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeY, sizeY, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}
#endif

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
pieceZUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeZ;

	XtVaGetValues(puzzle2d,
		XtNsizeZ, &sizeZ, NULL);
	sizeZ++;
	setScale(facetZSlider, facetZSliderLabel, sizeZ,
		MIN_FACETS, MAX_FACETS, True);
	XtVaSetValues(puzzle2d,
		XtNsizeZ, sizeZ, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeZ, sizeZ, NULL);
}

static void
pieceZDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int sizeZ;

	XtVaGetValues(puzzle2d,
		XtNsizeZ, &sizeZ, NULL);
	sizeZ--;
	if (sizeZ < MIN_FACETS) {
		return;
	}
	setScale(facetZSlider, facetZSliderLabel, sizeZ,
		MIN_FACETS, MAX_FACETS, True);
	XtVaSetValues(puzzle2d,
		XtNsizeZ, sizeZ, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeZ, sizeZ, NULL);
}

#else

static void
pieceZChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int sizeX, sizeY, sizeZ, old;
	Boolean orient, practice;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	int limit;
	sizeZ = cbs->position;
#else
	sizeZ = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &sizeZ,
			MIN_FACETS, MAX_FACETS, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &old,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old == sizeZ)
		return;
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	XtVaGetValues(facetZChanger,
		XmNmaximumValue, &limit, NULL);
	if (sizeZ >= limit)
		XtVaSetValues(facetZChanger,
			XmNmaximumValue, sizeZ + 1,
			XmNposition, sizeZ, NULL);
#elif defined(HAVE_ATHENA)
	setScale(w, label, sizeX, MIN_FACETS, MAX_FACETS, False);
#endif
	XtVaSetValues(puzzle2d,
		XtNsizeZ, sizeZ, NULL);
	XtVaSetValues(puzzle3d,
		XtNsizeZ, sizeZ, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}
#endif

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
speedUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle2d,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed++;
	if (speed > MAX_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}

static void
speedDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle2d,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed--;
	if (speed < MIN_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}

#else

static void
speedChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int delay, speed, old;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	speed = cbs->position;
#else
	speed = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &speed,
			MIN_SPEED, MAX_SPEED, SCROLL_SIZE)) {
		return;
	}
#endif
	delay = getDelay(speed);
	XtVaGetValues(puzzle2d,
		XtNdelay, &old, NULL);
	if (old == delay)
		return;
#ifdef HAVE_ATHENA
	setScale(w, label, speed, MIN_SPEED, MAX_SPEED, True);
#endif
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}
#endif

static void
puzzlePracticeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(practiceDialog);
#endif
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_PRACTICE, NULL);
}

static void
puzzleRandomizeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(randomizeDialog);
#endif
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_RANDOMIZE, NULL);
}

static void
fileMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_GET;

	if (value == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle2d,
		XtNmenu, value, NULL);
}

static void
setSoundCheck(Widget w)
{
	Boolean sound;
	XtVaGetValues(puzzle2d,
		XtNsound, &sound, NULL);
#ifdef HAVE_MOTIF
	setToggle(w, sound, False);
#elif defined(HAVE_ATHENA)
	menuCheck(topLevel, w, sound);
#endif
}

static void
playMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_UNDO;

#ifndef EXTRA
	if (value >= ACTION_PRACTICE)
		value += ACTION_RANDOMIZE - ACTION_PRACTICE;
	if (value >= ACTION_INCREMENT)
		value += ACTION_SOUND - ACTION_INCREMENT;
#endif
	XtVaSetValues(puzzle2d,
		XtNmenu, value, NULL);
	if (value == ACTION_SOUND)
		setSoundCheck(w);
}

static void
helpMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	Widget dialog;

#ifdef HAVE_ATHENA
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
#endif
	switch (value) {
	case 0:
		dialog = descriptionDialog;
		break;
	case 1:
		dialog = featuresDialog;
		break;
	case 2:
		dialog = synopsisDialog;
		break;
	case 3:
		dialog = referencesDialog;
		break;
	case 4:
		dialog = aboutDialog;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuCallback: %d", value);
			XtWarning(buf);
			free(buf);
			return;
		}
	}
#ifdef HAVE_MOTIF
	XtManageChild(dialog);
#elif defined(HAVE_ATHENA)
	XtPopup(dialog, XtGrabNone);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(dialog), &wmDeleteWindow, 1);
#endif
}
#endif

static void
initialize(
#ifdef WINVER
RubikWidget w, HBRUSH brush
#else
void
#endif
)
{
	int sizeX, sizeY, sizeZ;
	Boolean orient, practice;
	char *userName, *scoreFile;
#ifdef WINVER
	int dim;

	initializePuzzle(w, brush);

	sizeX = w->rubik.sizeX;
	sizeY = w->rubik.sizeY;
	sizeZ = w->rubik.sizeZ;
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	userName = w->rubik.userName;
	scoreFile = w->rubik.scoreFile;
	dim = w->rubik.dim;
	SET_STARTED(w, False);
	if (strcmp(scoreFile, "")) {
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
#else
	int delay;
	makeEquivalent(&userName, &scoreFile, &sizeX, &sizeY, &sizeZ,
		&orient, &practice, &delay);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	{
		int limit;
		XtVaGetValues(facetXChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeX >= limit)
			XtVaSetValues(facetXChanger,
				XmNposition, sizeX,
				XmNmaximumValue, sizeX + 1, NULL);
		else
			XtVaSetValues(facetXChanger,
				XmNposition, sizeX, NULL);
		XtVaGetValues(facetYChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeY >= limit)
			XtVaSetValues(facetYChanger,
				XmNposition, sizeY,
				XmNmaximumValue, sizeY + 1, NULL);
		else
			XtVaSetValues(facetYChanger,
				XmNposition, sizeY, NULL);
		XtVaGetValues(facetZChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeZ >= limit)
			XtVaSetValues(facetZChanger,
				XmNposition, sizeZ,
				XmNmaximumValue, sizeZ + 1, NULL);
		else
			XtVaSetValues(facetZChanger,
				XmNposition, sizeZ, NULL);
	}
#else
	if (sizeX > MAX_FACETS)
		XtVaSetValues(facetXChanger,
			XmNmaximum, sizeX, NULL);
	XmScaleSetValue(facetXChanger, sizeX);
	if (sizeY > MAX_FACETS)
		XtVaSetValues(facetYChanger,
			XmNmaximum, sizeY, NULL);
	XmScaleSetValue(facetYChanger, sizeY);
	if (sizeZ > MAX_FACETS)
		XtVaSetValues(facetZChanger,
			XmNmaximum, sizeZ, NULL);
	XmScaleSetValue(facetZChanger, sizeZ);
#endif
	setToggle(orientToggle, orient, False);
	setToggle(practiceToggle, practice, False);
	{
		int speed;
		speed = getSpeed(delay);
#ifdef USE_SPIN
		XtVaSetValues(speedChanger,
			XmNposition, speed, NULL);
#else
		XmScaleSetValue(speedChanger, speed);
#endif
	}
	setSoundCheck(soundMenuItem);
#elif defined(HAVE_ATHENA)
	XtVaSetValues(orientToggle,
		XtNstate, orient, NULL);
	XtVaSetValues(practiceToggle,
		XtNstate, practice, NULL);
	{
		int speed;
		speed = getSpeed(delay);
		setScale(speedSlider, speedSliderLabel, speed,
			MIN_SPEED, MAX_SPEED, True);
	}
	setSoundCheck(soundMenuItem);
#endif
#endif
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
	userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
		userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#else
		char *login;

#ifdef HAVE_CUSERID
		login = cuserid(NULL);
#else
		login = getlogin();
#endif
		if (login == NULL) {
			userNameDsp[0] = '\0';
		} else {
			(void) strncpy(userNameDsp, login, USER_NAME_LENGTH - 1);
			userNameDsp[USER_NAME_LENGTH - 1] = '\0';
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) strncpy(userNameDsp, "guest", 6);
#endif
	}
	printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, sizeX, sizeY, sizeZ);
#else
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, 2, sizeX, sizeY, sizeZ);
#ifdef HAVE_OPENGL
	printStatus(messageDsp, movesDsp, 4, sizeX, sizeY, sizeZ);
#else
	printStatus(messageDsp, movesDsp, 3, sizeX, sizeY, sizeZ);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzle(RubikWidget w, int reason)
#else
static void
puzzleCallback(Widget w, caddr_t clientData,
		rubikCallbackStruct *callData)
#endif
{
	int sizeX, sizeY, sizeZ;
	Boolean orient, practice, cheat;
#ifdef WINVER
	int dim = 0;
#else
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	int limit;
#endif
	Boolean start;
	int reason = callData->reason;
	Widget otherw = (Widget) NULL;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	int dim = 0, otherdim = 0;
#endif

	if (w == puzzle2d) {
		otherw = puzzle3d;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		dim = 2;
#ifdef HAVE_OPENGL
		otherdim = 4;
#else
		otherdim = 3;
#endif
#endif
	} else if (w == puzzle3d) {
		otherw = puzzle2d;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		otherdim = 2;
#ifdef HAVE_OPENGL
		dim = 4;
#else
		dim = 3;
#endif
#endif
	}
#endif
	messageDsp[0] = '\0';
#ifdef WINVER
	sizeX = w->rubik.sizeX;
	sizeY = w->rubik.sizeY;
	sizeZ = w->rubik.sizeZ;
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	cheat = w->rubik.cheat;
	/*start = w->rubik.started;*/
	dim = w->rubik.dim;
#else
	XtVaGetValues(w,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		(void) XIconifyWindow(XtDisplay(shell),
			XtWindow(shell),
			XScreenNumberOfScreen(XtScreen(shell)));
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(practiceDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(practiceDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(practiceDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_PRACTICE, NULL);
#endif
		break;
	case ACTION_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(randomizeDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(randomizeDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(randomizeDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_RANDOMIZE, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(solveDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(solveDialog), &wmDeleteWindow, 1);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGE_LENGTH);
#endif
#endif
		break;
	case ACTION_RESTORE:
		if (practice) {
			(void) strncpy(recordDsp, "practice",
				MESSAGE_LENGTH);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
			printState(recordText, recordDsp);
#endif
		}
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
		XtVaSetValues(otherw,
			XtNdirection, RESTORE_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, RESTORE_DIR, NULL);
#endif
		break;
	case ACTION_CLEAR:
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
		XtVaSetValues(otherw,
			XtNdirection, CLEAR_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, CLEAR_DIR, NULL);
#endif
		break;
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_AMBIGUOUS:
		(void) strncpy(messageDsp, "Ambiguous move",
			MESSAGE_LENGTH);
		break;
	case ACTION_ILLEGAL:
		if (practice || randomized)
			(void) strncpy(messageDsp, "Illegal move",
				MESSAGE_LENGTH);
		else
			(void) strncpy(messageDsp,
				"Randomize (z) to start", MESSAGE_LENGTH);
		break;
#ifndef WINVER
	case ACTION_MOVED:
		if (!callData->control)
			movesDsp++;
#ifdef DEBUG
		(void) printf("MOVED: face %d, position %d, direction %d, control %d\n",
			callData->face, callData->position, callData->direction,
			callData->control);
#endif
		XtVaSetValues(otherw,
			XtNface, callData->face,
			XtNpos, callData->position,
			XtNdirection, callData->direction,
			XtNcontrol, callData->control,
			XtNfast, callData->fast,
			XtNstart, True, NULL);
		SET_STARTED(w, True);
		break;
#endif
	case ACTION_SOLVED:
		if (practice) {
			movesDsp = 0;
		} else if (cheat) {
			(void) strncpy(messageDsp, "No cheating ", 13);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else if (handleSolved(movesDsp, sizeX, sizeY, sizeZ,
				orient)) {
			(void) strncpy(messageDsp, "Congratulations ", 17);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else {
			(void) strncpy(messageDsp, "Solved!", 8);
		}
		SET_STARTED(w, False);
#ifndef WINVER
		SET_STARTED(otherw, False);
#endif
		randomized = False;
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
#ifndef WINVER
		SET_STARTED(otherw, False);
#endif
		break;
	case ACTION_PRACTICE:
		movesDsp = 0;
		randomized = False;
		practice = !practice;
		if (!practice)
			(void) strncpy(messageDsp, "Randomize (z) to start",
				MESSAGE_LENGTH);
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.practice = practice;
		w->rubik.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, practice,
			XtNstart, False, NULL);
		XtVaSetValues(otherw,
			XtNpractice, practice,
			XtNstart, False, NULL);
#ifdef HAVE_MOTIF
		setToggle(practiceToggle, practice, False);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(practiceToggle,
			XtNstate, practice, NULL);
#endif
#endif
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
#ifdef WINVER
		w->rubik.practice = False;
		w->rubik.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, False,
			XtNstart, False, NULL);
		XtVaSetValues(otherw,
			XtNpractice, False,
			XtNstart, False, NULL);
#endif
		break;
#ifdef HAVE_OPENGL
	case ACTION_VIEW:
		{
			int view;

#ifdef WINVER
			view = w->rubik.view;
#else
			XtVaGetValues(puzzle3d,
				XtNview, &view, NULL);
#endif
			view = (view + 1) % (2 * MAX_VIEWS);
#ifdef WINVER
			w->rubik.view = view;
#else
			XtVaSetValues(puzzle3d,
				XtNview, view, NULL);
#endif
		}
		break;
#endif
	case ACTION_ORIENTIZE:
		movesDsp = 0;
		orient = !orient;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.orient = orient;
#else
		XtVaSetValues(w,
			XtNorient, orient, NULL);
		XtVaSetValues(otherw,
			XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
		setToggle(orientToggle, orient, False);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(orientToggle,
			XtNstate, orient, NULL);
#endif
#endif
		break;
	case ACTION_INCX:
		movesDsp = 0;
		sizeX++;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.sizeX = sizeX;
#else
		XtVaSetValues(w,
			XtNsizeX, sizeX, NULL);
		XtVaSetValues(otherw,
			XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaGetValues(facetXChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeX >= limit)
			XtVaSetValues(facetXChanger,
				XmNmaximumValue, sizeX + 1,
				XmNposition, sizeX, NULL);
		else
			XtVaSetValues(facetXChanger,
				XmNposition, sizeX, NULL);
#else
		if (sizeX > MAX_FACETS)
			XtVaSetValues(facetXChanger,
				XmNmaximum, sizeX, NULL);
		XmScaleSetValue(facetXChanger, sizeX);
#endif
#elif defined(HAVE_ATHENA)
		setScale(facetXSlider, facetXSliderLabel, sizeX,
			MIN_FACETS, MAX_FACETS, False);
#endif
#endif
		break;
	case ACTION_DECX:
		movesDsp = 0;
		sizeX--;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.sizeX = sizeX;
#else
		XtVaSetValues(w,
			XtNsizeX, sizeX, NULL);
		XtVaSetValues(otherw,
			XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		if (sizeX > MAX_FACETS)
			XtVaSetValues(facetXChanger,
				XmNmaximumValue, sizeX,
				XmNposition, sizeX, NULL);
		else
			XtVaSetValues(facetXChanger,
				XmNposition, sizeX, NULL);
#else
		if (sizeX > MAX_FACETS)
			XtVaSetValues(facetXChanger,
				XmNmaximum, sizeX, NULL);
		XmScaleSetValue(facetXChanger, sizeX);
#endif
#elif defined(HAVE_ATHENA)
		setScale(facetXSlider, facetXSliderLabel, sizeX,
			MIN_FACETS, MAX_FACETS, False);
#endif
#endif
		break;
	case ACTION_INCY:
		movesDsp = 0;
		sizeY++;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.sizeY = sizeY;
#else
		XtVaSetValues(w,
			XtNsizeY, sizeY, NULL);
		XtVaSetValues(otherw,
			XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaGetValues(facetYChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeY >= limit)
			XtVaSetValues(facetYChanger,
				XmNmaximumValue, sizeY + 1,
				XmNposition, sizeY, NULL);
		else
			XtVaSetValues(facetYChanger,
				XmNposition, sizeY, NULL);
#else
		if (sizeY > MAX_FACETS)
			XtVaSetValues(facetYChanger,
				XmNmaximum, sizeY, NULL);
		XmScaleSetValue(facetYChanger, sizeY);
#endif
#elif defined(HAVE_ATHENA)
		setScale(facetYSlider, facetYSliderLabel, sizeY,
			MIN_FACETS, MAX_FACETS, False);
#endif
#endif
		break;
	case ACTION_DECY:
		movesDsp = 0;
		sizeY--;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.sizeY = sizeY;
#else
		XtVaSetValues(w,
			XtNsizeY, sizeY, NULL);
		XtVaSetValues(otherw,
			XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		if (sizeY > MAX_FACETS)
			XtVaSetValues(facetYChanger,
				XmNmaximumValue, sizeY,
				XmNposition, sizeY, NULL);
		else
			XtVaSetValues(facetYChanger,
				XmNposition, sizeY, NULL);
#else
		if (sizeY > MAX_FACETS)
			XtVaSetValues(facetYChanger,
				XmNmaximum, sizeY, NULL);
		XmScaleSetValue(facetYChanger, sizeY);
#endif
#elif defined(HAVE_ATHENA)
		setScale(facetYSlider, facetYSliderLabel, sizeY,
			MIN_FACETS, MAX_FACETS, False);
#endif
#endif
		break;
	case ACTION_INCZ:
		movesDsp = 0;
		sizeZ++;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.sizeZ = sizeZ;
#else
		XtVaSetValues(w,
			XtNsizeZ, sizeZ, NULL);
		XtVaSetValues(otherw,
			XtNsizeZ, sizeZ, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaGetValues(facetZChanger,
			XmNmaximumValue, &limit, NULL);
		if (sizeZ >= limit)
			XtVaSetValues(facetZChanger,
				XmNmaximumValue, sizeZ + 1,
				XmNposition, sizeZ, NULL);
		else
			XtVaSetValues(facetZChanger,
				XmNposition, sizeZ, NULL);
#else
		if (sizeZ > MAX_FACETS)
			XtVaSetValues(facetZChanger,
				XmNmaximum, sizeZ, NULL);
		XmScaleSetValue(facetZChanger, sizeZ);
#endif
#elif defined(HAVE_ATHENA)
		setScale(facetZSlider, facetZSliderLabel, sizeZ,
			MIN_FACETS, MAX_FACETS, False);
#endif
#endif
		break;
	case ACTION_DECZ:
		movesDsp = 0;
		sizeZ--;
		printRecord(sizeX, sizeY, sizeZ, orient, practice);
#ifdef WINVER
		w->rubik.sizeZ = sizeZ;
#else
		XtVaSetValues(w,
			XtNsizeZ, sizeZ, NULL);
		XtVaSetValues(otherw,
			XtNsizeZ, sizeZ, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		if (sizeZ > MAX_FACETS)
			XtVaSetValues(facetZChanger,
				XmNmaximumValue, sizeZ,
				XmNposition, sizeZ, NULL);
		else
			XtVaSetValues(facetZChanger,
				XmNposition, sizeZ, NULL);
#else
		if (sizeZ > MAX_FACETS)
			XtVaSetValues(facetZChanger,
				XmNmaximum, sizeZ, NULL);
		XmScaleSetValue(facetZChanger, sizeZ);
#endif
#elif defined(HAVE_ATHENA)
		setScale(facetZSlider, facetZSliderLabel, sizeZ,
			MIN_FACETS, MAX_FACETS, False);
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_UNDO:
		if (callData->control) {
			XtVaSetValues(otherw,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
		} else {
			movesDsp--;
			XtVaSetValues(otherw,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
			SET_STARTED(w, True);
		}
		break;
	case ACTION_REDO:
		if (callData->control) {
			XtVaSetValues(otherw,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
		} else {
			movesDsp++;
			XtVaSetValues(otherw,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
			SET_STARTED(w, True);
		}
		break;
#endif
#ifdef WINVER
	case ACTION_DIM:
		dim++;
#ifdef HAVE_OPENGL
		if (dim == 3)
			dim = 4;
		else if (dim > 4)
#else
		if (dim > 3)
#endif
			dim = 2;
		w->rubik.dim = dim;
		break;
#endif
#ifndef WINVER
	case ACTION_CHEAT:
		XtVaSetValues(w,
			XtNcheat, cheat, NULL);
		XtVaSetValues(otherw,
			XtNcheat, cheat, NULL);
		break;
#endif
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	case ACTION_SOUND:
		setSoundCheck(soundMenuItem);
		break;
#endif
	}
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, sizeX, sizeY, sizeZ);
#else
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, dim, sizeX, sizeY, sizeZ);
	printStatus(messageDsp, movesDsp, otherdim, sizeX, sizeY, sizeZ);
#endif
#endif
}

#ifdef WINVER
void
setPuzzleMove(RubikWidget w, int reason, int face, int position, int direction,
		Boolean control, int fast)
{
	int sizeX, sizeY, sizeZ;
	/* Boolean orient, practice, cheat; */
	int dim;

	messageDsp[0] = '\0';
	sizeX = w->rubik.sizeX;
	sizeY = w->rubik.sizeY;
	sizeZ = w->rubik.sizeZ;
#if 0
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	cheat = w->rubik.cheat;
#endif
	dim = w->rubik.dim;
	switch (reason) {
	case ACTION_MOVED:
		if (!control) {
			movesDsp++;
			SET_STARTED(w, True);
		}
		break;
	case ACTION_UNDO:
		if (!control) {
			movesDsp--;
			SET_STARTED(w, True);
		}
		break;
	case ACTION_REDO:
		if (!control) {
			movesDsp++;
			SET_STARTED(w, True);
		}
		break;
	}
	printStatus(messageDsp, movesDsp, dim, sizeX, sizeY, sizeZ);
}

static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.rubik.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.rubik.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		randomizePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					TOP,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					BOTTOM,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			clearPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_PRACTICE:
			practicePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_PRINT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			printPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_ORIENTIZE:
			orientizePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DIM:
			(void) dimPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_VIEW:
			(void) viewPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_TOP:
		case ACTION_TR:
		case ACTION_RIGHT:
		case ACTION_BR:
		case ACTION_BOTTOM:
		case ACTION_BL:
		case ACTION_LEFT:
		case ACTION_TL:
		case ACTION_CW:
		case ACTION_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) - ACTION_TOP,
					FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_CONTROL_TOP:
		case ACTION_CONTROL_TR:
		case ACTION_CONTROL_RIGHT:
		case ACTION_CONTROL_BOTTOM:
		case ACTION_CONTROL_BL:
		case ACTION_CONTROL_LEFT:
		case ACTION_CONTROL_CW:
		case ACTION_CONTROL_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) - ACTION_CONTROL_TOP,
					TRUE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_INCREMENT:
			incrementPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECREMENT:
			decrementPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCX:
			incrementAxisPuzzle(&widget, 'x');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECX:
			incrementAxisPuzzle(&widget, 'X');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCY:
			incrementAxisPuzzle(&widget, 'y');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECY:
			incrementAxisPuzzle(&widget, 'Y');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCZ:
			incrementAxisPuzzle(&widget, 'z');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECZ:
			incrementAxisPuzzle(&widget, 'Z');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK | MB_ICONQUESTION);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK | MB_ICONEXCLAMATION);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK | MB_ICONINFORMATION);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(XtPointer) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (int) msg.wParam;
}

#else

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*rubik.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*rubik.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "rubik.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "rubik.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*rubik.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*rubik.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*rubik.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*rubik.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*rubik.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*rubik.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*rubik.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*rubik.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-frame", (char *) "*rubik.frameColor", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*rubik.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*rubik.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*rubik.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*rubik.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*rubik.font", XrmoptionSepArg, NULL},
	{(char *) "-view", (char *) "*rubik.view", XrmoptionSepArg, NULL},
	{(char *) "-sizeX", (char *) "*rubik.sizeX", XrmoptionSepArg, NULL},
	{(char *) "-sizeX", (char *) "*rubik.sizeX", XrmoptionSepArg, NULL},
	{(char *) "-sizeY", (char *) "*rubik.sizeY", XrmoptionSepArg, NULL},
	{(char *) "-sizeZ", (char *) "*rubik.sizeZ", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*rubik.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*rubik.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*rubik.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*rubik.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*rubik.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*rubik.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*rubik.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*rubik.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
soundCallback(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	Boolean value = cbs->set;

	XtVaSetValues(puzzle2d,
		XtNsound, value, NULL);
}

#elif defined(HAVE_ATHENA)
static void
puzzlePracticeCancelCallback(Widget w, XtPointer clientData,
		XtPointer callData)
{
	XtPopdown(practiceDialog);
}

static void
createPracticeQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	practiceDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, practiceDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzlePracticeCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzlePracticeCancelCallback, dialog);
	XtRealizeWidget(practiceDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(practiceDialog), &wmDeleteWindow, 1);
}

static void
puzzleRandomizeCancelCallback(Widget w, XtPointer clientData,
		XtPointer callData)
{
	XtPopdown(randomizeDialog);
}

static void
createRandomizeQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	randomizeDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, randomizeDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzleRandomizeCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzleRandomizeCancelCallback, dialog);
	XtRealizeWidget(randomizeDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(randomizeDialog), &wmDeleteWindow, 1);
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
	XtAppContext appCon;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	Widget mainPanel, controlPanel, menuBar;
	Widget movesRowCol;
	Widget changerRowCol, changer2RowCol;
	Widget speedChangerRowCol, toggleRowCol;
	Widget playMenu;
#endif
#ifdef HAVE_MOTIF
	Widget pullDownMenu, widget;
	Widget menuBarPanel;
	Widget facetXChangerRowCol, facetYChangerRowCol, facetZChangerRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString, clearString;
#ifdef EXTRA
	XmString practiceString;
#endif
	XmString randomizeString, solveString;
#ifdef EXTRA
	XmString incrementString, decrementString, orientString;
	XmString speedString, slowString;
#endif
	XmString soundString;
#ifdef HAVE_OPENGL
	XmString viewString;
#endif
#elif defined(HAVE_ATHENA)
	XtActionsRec actions[] = {
		{(char *) "DeleteWindowProc", deleteWindowProc},
		{(char *) "ClosePopupPanel", (XtActionProc) closePopupPanel}
	};
	String fallbackResources[] = {
		(char *) "?.translations: #override <Message>WM_PROTOCOLS: DeleteWindowProc()",
		(char *) "?.TransientShell.translations: #override <Message>WM_PROTOCOLS: ClosePopupPanel()",
		NULL
	};
	Widget facetXBox, facetYBox, facetZBox, speedBox;
	Widget orientBox, practiceBox;
	Widget movesLabel, recordLabel;
	Widget w;
	Widget fileLabel, playLabel, helpLabel;
	int sizeX, sizeY, sizeZ, delay;
	Boolean orient, practice;
	unsigned int i;
#else
	char titleDsp[TITLE_FILE_LENGTH];
#endif

#ifdef __VMS
	int n;
	progDsp = strrchr(argv[0], ']');
	for (n = 0; progDsp[n] != '\0' && progDsp[n] != '.'; n++);
	progDsp[n] = '\0';
#else
	progDsp = strrchr(argv[0], '/');
#endif
	if (progDsp != NULL)
		progDsp++;
	if (progDsp == NULL)
		progDsp = argv[0];
	topLevel = XtVaAppInitialize(NULL, "XRubik",
		options, XtNumber(options), &argc, argv,
#ifdef HAVE_ATHENA
		fallbackResources,
#else
		NULL,
#endif
		NULL);
	appCon = XtWidgetToApplicationContext(topLevel);
	if (argc != 1)
		usage(argv[0], synopsisHelp);
#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P', NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "fileMenu",
		0, fileMenuCallback,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL, NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
#ifdef EXTRA
	practiceString = XmStringCreateSimple((char *) "Practice");
#endif
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Auto-solve");
#ifdef EXTRA
	incrementString = XmStringCreateSimple((char *) "Increment Size");
	decrementString = XmStringCreateSimple((char *) "Decrement Size");
	orientString = XmStringCreateSimple((char *) "Oriented");
	speedString = XmStringCreateSimple((char *) "Speed >");
	slowString = XmStringCreateSimple((char *) "Slow <");
#endif
	soundString = XmStringCreateSimple((char *) "Sound @");
#ifdef HAVE_OPENGL
	viewString = XmStringCreateSimple((char *) "View");
#endif
	playMenu = XmVaCreateSimplePulldownMenu(menuBar, (char *) "playMenu",
		1, playMenuCallback,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
#endif
		XmVaPUSHBUTTON, randomizeString, 'z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 's', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
#endif
		XmVaTOGGLEBUTTON, soundString, '@', NULL, NULL,
#ifdef HAVE_OPENGL
		XmVaPUSHBUTTON, viewString, 'V', NULL, NULL,
#endif
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
#ifdef EXTRA
	XmStringFree(practiceString);
#endif
	XmStringFree(randomizeString);
	XmStringFree(solveString);
#ifdef EXTRA
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(orientString);
	XmStringFree(speedString);
	XmStringFree(slowString);
#endif
	XmStringFree(soundString);
#ifdef HAVE_OPENGL
	XmStringFree(viewString);
#endif
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL); /* mnemonic warning on Cygwin */
	XtVaSetValues(menuBar,
		XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 0);
	widget = XtVaCreateManagedWidget("Features...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 1);
	widget = XtVaCreateManagedWidget("Synopsis...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'S', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 2);
	widget = XtVaCreateManagedWidget("References...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 3);
	widget = XtVaCreateManagedWidget("About...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 4);
	XtManageChild(menuBar);
	solveDialog = createHelp(menuBar, "Solve", (char *) solveHelp);
	practiceDialog = createQuery(topLevel, "Practice Query",
		(char *) "Are you sure you want to toggle the practice mode?",
		(XtCallbackProc) puzzlePracticeCallback);
	randomizeDialog = createQuery(topLevel, "Randomize Query",
		(char *) "Are you sure you want to randomize?",
		(XtCallbackProc) puzzleRandomizeCallback);
	mainPanel = XtVaCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel, NULL);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate and out of date. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("nullText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, " ", 2, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	facetXChangerRowCol = XtVaCreateManagedWidget("facetXChangerRowCol",
		xmRowColumnWidgetClass, changerRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(facetXChangerRowCol, &facetXChanger,
		(char *) sizeLabels[0], DEFAULT_FACETS,
		MIN_FACETS, MAX_FACETS, 2,
		(XtCallbackProc) pieceXChangeCallback);
#else
	createSlider(facetXChangerRowCol, &facetXChanger,
		(char *) sizeLabels[0], DEFAULT_FACETS,
		MIN_FACETS, MAX_FACETS, 2, SCROLL_SIZE,
		(XtCallbackProc) pieceXChangeCallback);
#endif
	facetYChangerRowCol = XtVaCreateManagedWidget("facetYChangerRowCol",
		xmRowColumnWidgetClass, changerRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(facetYChangerRowCol, &facetYChanger,
		(char *) sizeLabels[1], DEFAULT_FACETS,
		MIN_FACETS, MAX_FACETS, 2,
		(XtCallbackProc) pieceYChangeCallback);
#else
	createSlider(facetYChangerRowCol, &facetYChanger,
		(char *) sizeLabels[1], DEFAULT_FACETS,
		MIN_FACETS, MAX_FACETS, 2, SCROLL_SIZE,
		(XtCallbackProc) pieceYChangeCallback);
#endif
	changer2RowCol = XtVaCreateManagedWidget("changer2RowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	facetZChangerRowCol = XtVaCreateManagedWidget("facetZChangerRowCol",
		xmRowColumnWidgetClass, changer2RowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(facetZChangerRowCol, &facetZChanger,
		(char *) sizeLabels[2], DEFAULT_FACETS,
		MIN_FACETS, MAX_FACETS, 2,
		(XtCallbackProc) pieceZChangeCallback);
#else
	createSlider(facetZChangerRowCol, &facetZChanger,
		(char *) sizeLabels[2], DEFAULT_FACETS,
		MIN_FACETS, MAX_FACETS, 2, SCROLL_SIZE,
		(XtCallbackProc) pieceZChangeCallback);
#endif
	speedChangerRowCol = XtVaCreateManagedWidget("speedChangerRowCol",
		xmRowColumnWidgetClass, changer2RowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2,
		(XtCallbackProc) speedChangeCallback);
#else
	createSlider(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2, SCROLL_SIZE,
		(XtCallbackProc) speedChangeCallback);
#endif
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	orientToggle = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, toggleRowCol,
		XmNset, DEFAULT_ORIENT, NULL);
	XtAddCallback(orientToggle, XmNvalueChangedCallback,
		(XtCallbackProc) orientCallback, (XtPointer) NULL);
	practiceToggle = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, toggleRowCol,
		XmNset, DEFAULT_PRACTICE, NULL);
	XtAddCallback(practiceToggle, XmNvalueChangedCallback,
		(XtCallbackProc) practiceCallback, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Rubik's Cube! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle2d = XtVaCreateManagedWidget("rubik",
		rubik2dWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("rubik",
		rubikGLWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
#else
	puzzle3d = XtVaCreateManagedWidget("rubik",
		rubik3dWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
#endif
#elif defined(HAVE_ATHENA)
	XtAppAddActions(appCon, actions, XtNumber(actions));
	createPracticeQuery((char *) "Practice Query",
		(char *) "Are you sure you want to toggle the practice mode?");
	createRandomizeQuery((char *) "Randomize Query",
		(char *) "Are you sure you want to randomize?");
	createHelp(topLevel, &solveDialog, (char *) "Auto-solve",
		(char *) solveHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &descriptionDialog, (char *) "Description",
		(char *) descriptionHelp, (XtCallbackProc) closePopupPanel2);
	createScrollHelp(topLevel, &featuresDialog, (char *) "Features",
		(char *) featuresHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &synopsisDialog, (char *) "Synopsis",
		(char *) synopsisHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &referencesDialog, (char *) "References",
		(char *) referencesHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &aboutDialog, (char *) "About",
		(char *) aboutHelp, (XtCallbackProc) closePopupPanel2);
	mainPanel = XtVaCreateManagedWidget("form",
		panedWidgetClass, topLevel, NULL);
	menuBar = XtVaCreateManagedWidget("MenuBar",
		formWidgetClass, mainPanel,
		XtNborderWidth, 1, NULL);
	createMenu(menuBar, &fileLabel, NULL,
		fileTypes, "File", numFileTypes,
		0, False, fileMenuCallback);
	playLabel = XtVaCreateManagedWidget("Play",
		menuButtonWidgetClass, menuBar,
		XtNborderWidth, 0,
		XtNfromHoriz, fileLabel, NULL);
	playMenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, playLabel, NULL);
	for (i = 0; i < numPlayTypes; i++) {
#ifdef HAVE_OPENGL
		if (i == numPlayTypes - 2)
#else
		if (i == numPlayTypes - 1)
#endif
		{
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu,
				XtNleftMargin, 20, NULL); /* for check */
			soundMenuItem = w;
		} else
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu, NULL);
		XtAddCallback(w,
			XtNcallback, (XtCallbackProc) playMenuCallback,
			(XtPointer) (size_t) i);
	}
	createMenu(menuBar, &helpLabel, playLabel,
		helpTypes, "Help", numHelpTypes,
		0, False, helpMenuCallback);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		formWidgetClass, mainPanel,
		XtNborderWidth, 0, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0, NULL);
	movesLabel = XtVaCreateManagedWidget("Moves",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0, NULL);
	movesText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 32,
		XtNfromHoriz, movesLabel, NULL);
	recordLabel = XtVaCreateManagedWidget("Record",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0,
		XtNfromHoriz, movesText, NULL);
	recordText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 108,
		XtNfromHoriz, recordLabel, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, movesRowCol, NULL);
	facetXBox = XtVaCreateManagedWidget("facetXBox",
		boxWidgetClass, changerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	facetYBox = XtVaCreateManagedWidget("facetYBox",
		boxWidgetClass, changerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromHoriz, facetXBox, NULL);
	changer2RowCol = XtVaCreateManagedWidget("changerRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, changerRowCol, NULL);
	facetZBox = XtVaCreateManagedWidget("facetZBox",
		boxWidgetClass, changer2RowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	speedChangerRowCol = XtVaCreateManagedWidget("speedChangerRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, changer2RowCol, NULL);
	speedBox = XtVaCreateManagedWidget("speedBox",
		boxWidgetClass, speedChangerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, speedChangerRowCol, NULL);
	orientBox = XtVaCreateManagedWidget("Oriented",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNstate, DEFAULT_ORIENT, NULL);
	practiceBox = XtVaCreateManagedWidget("Practice",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNstate, DEFAULT_PRACTICE,
		XtNfromHoriz, orientBox, NULL);
	message = XtVaCreateManagedWidget("Welcome",
		labelWidgetClass, controlPanel,
		XtNjustify, XtJustifyLeft,
		XtNborderWidth, 0,
		XtNwidth, 180,
		XtNfromVert, toggleRowCol, NULL);
	puzzle2d = XtVaCreateManagedWidget("rubik",
		rubik2dWidgetClass, mainPanel,
		XtNfromVert, controlPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtVaGetValues(puzzle2d,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNdelay, &delay, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("rubik",
		rubikGLWidgetClass, mainPanel,
		XtNfromVert, puzzle2d,
		XtNheight, WINDOW_HEIGHT, NULL);
#else
	puzzle3d = XtVaCreateManagedWidget("rubik",
		rubik3dWidgetClass, mainPanel,
		XtNfromVert, puzzle2d,
		XtNheight, WINDOW_HEIGHT, NULL);
#endif
#ifdef USE_SPIN
	createSpinner(facetXBox, &facetXSliderLabel,
		sizeLabels[0], 0, sizeX, MIN_FACETS, MAX_FACETS, False,
		pieceXUpCallback, pieceXDownCallback);
	createSpinner(facetYBox, &facetYSliderLabel,
		sizeLabels[1], 0, sizeY, MIN_FACETS, MAX_FACETS, False,
		pieceYUpCallback, pieceYDownCallback);
	createSpinner(facetZBox, &facetZSliderLabel,
		sizeLabels[2], 0, sizeZ, MIN_FACETS, MAX_FACETS, False,
		pieceZUpCallback, pieceZDownCallback);
	createSpinner(speedBox, &speedSliderLabel,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		speedUpCallback, speedDownCallback);
#else
	createSlider(facetXBox, &facetXSliderLabel, &facetXSlider,
		sizeLabels[0], 0, sizeX, MIN_FACETS, MAX_FACETS, False,
		SCROLL_SIZE, pieceXChangeCallback, pieceXChangeCallback);
	createSlider(facetYBox, &facetYSliderLabel, &facetYSlider,
		sizeLabels[1], 0, sizeY, MIN_FACETS, MAX_FACETS, False,
		SCROLL_SIZE, pieceYChangeCallback, pieceYChangeCallback);
	createSlider(facetZBox, &facetZSliderLabel, &facetZSlider,
		sizeLabels[2], 0, sizeZ, MIN_FACETS, MAX_FACETS, False,
		SCROLL_SIZE, pieceZChangeCallback, pieceZChangeCallback);
	createSlider(speedBox, &speedSliderLabel, &speedSlider,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		SCROLL_SIZE, speedChangeCallback, speedChangeCallback);
#endif
	createToggle(orientBox, &orientToggle, "Oriented:", 0,
		orient, (XtCallbackProc) orientCallback);
	createToggle(practiceBox, &practiceToggle, "Practice:", 0,
		practice, (XtCallbackProc) practiceCallback);
	XtRealizeWidget(topLevel);
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
	XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel),
		&wmDeleteWindow, 1);
#else
	puzzle2d = XtVaCreateManagedWidget("rubik",
		rubik2dWidgetClass, topLevel, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
	shell = XtAppCreateShell(progDsp, titleDsp,
		topLevelShellWidgetClass, XtDisplay(topLevel), NULL, 0);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("rubik",
		rubikGLWidgetClass, shell, NULL);
#else
	puzzle3d = XtVaCreateManagedWidget("rubik",
		rubik3dWidgetClass, shell, NULL);
#endif
#endif
	XtVaGetValues(puzzle2d,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel,
			XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
#else
	XtVaSetValues(topLevel,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
	XtVaSetValues(shell,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
#endif
	XtAddCallback(puzzle3d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_MOTIF
	updateToggle(playMenu, &soundMenuItem,
		False,
#ifdef EXTRA
		11,
#else
		5,
#endif
		(XtCallbackProc) soundCallback);
	descriptionDialog = createHelp(menuBar, (char *) "Description",
		(char *) descriptionHelp);
	featuresDialog = createScrollHelp(menuBar, (char *) "Features",
		(char *) featuresHelp, pixmap);
	synopsisDialog = createHelp(menuBar, (char *) "Synopsis",
		(char *) synopsisHelp);
	referencesDialog = createHelp(menuBar, (char *) "References",
		(char *) referencesHelp);
	aboutDialog = createHelp(menuBar, (char *) "About",
		(char *) aboutHelp);
#endif
	initialize();
	XtRealizeWidget(topLevel);
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	XtRealizeWidget(shell);
#endif
	XGrabButton(XtDisplay(puzzle2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle2d),
		XCreateFontCursor(XtDisplay(puzzle2d), XC_hand2));
	XGrabButton(XtDisplay(puzzle3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle3d),
		XCreateFontCursor(XtDisplay(puzzle3d), XC_hand2));
#ifdef HAVE_EDITRES
	XtAddEventHandler(topLevel, (EventMask) 0, TRUE,
		(XtEventHandler) _XEditResCheckMessages, NULL);
#endif
	XtAppMainLoop(appCon);

#ifdef __VMS
	return 1;
#else
	return 0;
#endif
}
#endif
